/*
** Copyright (c) 2009  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "custombutton.h"
#include <QPainter>
#include <QDebug>
#include <QGraphicsSceneMouseEvent>

//-----------------------------------------------------------------------------
/**
** Constructor.
**
** \param strBackground The name of the background image.
** \param strIcon The name of the icon which is placed on the button.
** \param iconOffset The icon offset from the center.
*/
CCustomButton::CCustomButton(const QString& strBackground, const QString& strIcon, QPointF iconOffset) : QObject(NULL), QGraphicsItem()
{
    m_pxBackground = QPixmap::fromImage(QImage(strBackground));
    m_pxIcon = QPixmap::fromImage(QImage(strIcon));
    m_State = State_Normal;
    m_IconOffset = iconOffset;

#ifndef Q_WS_HILDON
    setAcceptHoverEvents(true);
#endif
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CCustomButton::~CCustomButton()
{
}

//-----------------------------------------------------------------------------
/**
** Sets the icon used in the button.
**
** \param strIcon The name of the icon.
*/
void CCustomButton::setIcon(const QString& strIcon)
{
    m_pxIcon = QPixmap::fromImage(QImage(strIcon));
    update();
}

//-----------------------------------------------------------------------------
/**
** Returns the bounding rect of the item. This is relative to the current position.
**
** \return The bounding rectangle.
*/
QRectF CCustomButton::boundingRect() const
{
    return QRectF(QPointF(0, 0), m_pxBackground.size());
}

//-----------------------------------------------------------------------------
/**
** Paints the button with the given painter.
**
** \param pPainter The painter to be used.
** \param pOption The style options.
** \param pWidget The widget where the painting will be done.
*/
void CCustomButton::paint(QPainter* pPainter, const QStyleOptionGraphicsItem* pOption, QWidget* pWidget)
{
    pPainter->drawPixmap(0, 0, m_pxBackground.width(), m_pxBackground.height(), m_pxBackground);

    QPointF pos((m_pxBackground.width() - m_pxIcon.width() / 3) / 2, (m_pxBackground.height() - m_pxIcon.height()) / 2);
    pPainter->drawPixmap(pos + m_IconOffset, m_pxIcon, QRectF(m_State * m_pxIcon.width() / 3, 0, m_pxIcon.width() / 3, m_pxIcon.height()));
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse press events. Changes the state of the button.
**
** \param pEvent The mouse event.
*/
void CCustomButton::mousePressEvent(QGraphicsSceneMouseEvent* pEvent)
{
    m_State = State_Pushed;
    update();
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse release events. Changes the state of the button.
**
** \param pEvent The mouse event.
*/
void CCustomButton::mouseReleaseEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (boundingRect().contains(pEvent->pos()))
    {
#ifndef Q_WS_HILDON
        m_State = State_Hover;
#else
        m_State = State_Normal;
#endif
        emit clicked();
    }
    else
    {
        m_State = State_Normal;
    }

    update();
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse hover enter events. Changes the state of the button.
**
** \param pEvent The mouse event.
*/
void CCustomButton::hoverEnterEvent(QGraphicsSceneHoverEvent* pEvent)
{
    m_State = State_Hover;
    update();
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse hover leave events. Changes the state of the button.
**
** \param pEvent The mouse event.
*/
void CCustomButton::hoverLeaveEvent(QGraphicsSceneHoverEvent* pEvent)
{
    m_State = State_Normal;
    update();
}

// EOF
