/*
** Copyright (c) 2009  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "numberitem.h"
#include "rotator.h"
#include <QPainter>
#include <QDebug>
#include <QGraphicsSceneMouseEvent>

#define NUMBER_HEIGHT 128
#define NUMBER_WIDTH 100

//-----------------------------------------------------------------------------
/**
** The constructor.
**
** \param pRotator Pointer to the rotator which contains this item.
** \param x The x position of the item.
** \param y The y position of the item.
** \param bSeparator If true this is a separator (:).
** \param pixmapNumbers The image which contains the numbers.
*/
CNumberItem::CNumberItem(CRotator* pRotator, int x, int y, bool bSeparator, QPixmap pixmapNumbers) : QGraphicsItem()
{
    m_OrigY = y;
    m_W = bSeparator ? SEPARATOR_WIDTH : NUMBER_ITEM_WIDTH;
    m_H = NUMBER_ITEM_HEIGHT;
    m_bSeparator = bSeparator;
    m_pixmapNumbers = pixmapNumbers;
    m_Value = 0;
    m_bShowBackground = false;
    m_pRotator = pRotator;

    setPos(x, y);
}

//-----------------------------------------------------------------------------
/**
** The destructor.
*/
CNumberItem::~CNumberItem()
{
}

//-----------------------------------------------------------------------------
/**
** Sets the offset for the item. The offset is added to the original y-position.
**
** \param value The new offset value.
*/
void CNumberItem::setOffset(int offset)
{
    setPos(pos().x(), m_OrigY + offset);
}

//-----------------------------------------------------------------------------
/**
** Sets a new value for the item. The value is shown as a number in the view.
**
** \param value The new value.
*/
void CNumberItem::setValue(int value)
{
    if (m_Value != value)
    {
        m_Value = value;
        update();
    }
}

//-----------------------------------------------------------------------------
/**
** Enables/disables the background for the item.
**
** \param bShow Set to true to display the background and false to hide it.
*/
void CNumberItem::showBackground(bool bShow)
{
    if (m_bShowBackground != bShow)
    {
        m_bShowBackground = bShow;
        update();
    }
}

//-----------------------------------------------------------------------------
/**
** Returns the bounding rect of the item. This is relative to the current position.
**
** \return The bounding rectangle.
*/
QRectF CNumberItem::boundingRect() const
{
    return QRectF(0, 0, m_W, m_H);
}

//-----------------------------------------------------------------------------
/**
** Paints the number item with the given painter.
**
** \param pPainter The painter to be used.
** \param pOption The style options.
** \param pWidget The widget where the painting will be done.
*/
void CNumberItem::paint(QPainter* pPainter, const QStyleOptionGraphicsItem* pOption, QWidget* pWidget)
{
    QPoint p(0, (m_H - NUMBER_HEIGHT) / 2);
    if (m_bSeparator)
    {
        pPainter->drawPixmap(p + QPoint(10, 0), m_pixmapNumbers, QRectF(0, 10 * NUMBER_HEIGHT, 20, NUMBER_HEIGHT));
    }
    else
    {
        if (m_bShowBackground > 0)
        {
            // Background alpha depends on how far the item is from the center
            int alpha = 50 - abs(pos().y() + m_H / 2) / 7;
            alpha = qMax(0, alpha);
            pPainter->fillRect(0, 0, m_W, m_H, QColor(0, 0, 0, alpha));
        }
        pPainter->drawPixmap(p, m_pixmapNumbers, QRectF(0, (m_Value / 10) * NUMBER_HEIGHT, NUMBER_WIDTH, NUMBER_HEIGHT));
        pPainter->drawPixmap(p + QPoint(NUMBER_WIDTH, 0), m_pixmapNumbers, QRectF(0, (m_Value % 10) * NUMBER_HEIGHT, NUMBER_WIDTH, NUMBER_HEIGHT));
    }
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse move events. Forwards the event to the rotator.
**
** \param pEvent The mouse event.
*/
void CNumberItem::mouseMoveEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (m_pRotator)
    {
        pEvent->setPos(pEvent->pos() + pos());      // Map to scene coordinates
        m_pRotator->mouseMoveEvent(pEvent);
    }
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse press events. Forwards the event to the rotator.
**
** \param pEvent The mouse event.
*/
void CNumberItem::mousePressEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (m_pRotator)
    {
        pEvent->setPos(pEvent->pos() + pos());      // Map to scene coordinates
        m_pRotator->mousePressEvent(pEvent);
    }
}

//-----------------------------------------------------------------------------
/**
** Handler for the mouse release events. Forwards the event to the rotator.
**
** \param pEvent The mouse event.
*/
void CNumberItem::mouseReleaseEvent(QGraphicsSceneMouseEvent* pEvent)
{
    if (m_pRotator)
    {
        pEvent->setPos(pEvent->pos() + pos());      // Map to scene coordinates
        m_pRotator->mouseReleaseEvent(pEvent);
    }
}

// EOF
