// Distributed under the Thrift Software License
//
// See accompanying file LICENSE or visit the Thrift site at:
// http://developers.apache.com/thrift/

#include <cstring>
#include <sys/socket.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <iostream>
#include <sstream>

#include "TSSLTransport.h"
#include <boost/shared_ptr.hpp>

namespace apache { namespace thrift { namespace transport {

/**
 * Common functions used by both client and server SSL 
 * implementations.
 */
TSSLTransport::TSSLTransport() :
  paranoid_(true) // check to make sure that the names are right.	
{}

TSSLTransport::~TSSLTransport(){}

/*
 * Checks to see if the connection is OK.
 * Uses SSL_get_verify_result. method.
 * Behaivor is tunable using the SSL_CTX_set_verify().
 */
bool TSSLTransport::basicVerify(SSL *ssl, const std::string &host){ 

  X509 *peerCertificate;

  /**
   * Right now, this fails with the reasons here
   * http://openssl.org/docs/apps/verify.html
   * 
   * Should there be an option where an out-of-date cert is still considered 
   * valid?
   */
  if(SSL_get_verify_result(ssl) == X509_V_OK){ 
    peerCertificate = SSL_get_peer_certificate(ssl);
  } else {
    std::stringstream s;
    s << SSL_get_verify_result(ssl);
    GlobalOutput(("Basic verification of X509 cert failed with: ERROR CODE " + s.str()).c_str());
    X509_free(peerCertificate);
    throw TTransportException(TTransportException::UNKNOWN, 
			      "Cert. verification failed.");
  }
  
  /* Do extra checks if so desired. */
  if (paranoid_ && !extendedVerify(peerCertificate, host)){
    X509_free(peerCertificate);
    throw TTransportException(TTransportException::UNKNOWN, 
			      "Paranoid verification failed.");
  }
  
  // Get rid of the cert.
  if (peerCertificate != NULL)
    X509_free(peerCertificate);
  
  return true;
}

/**
 * If desired, perform extra checks on the server's cert.
 * Right now, make sure that the cert's subject name matches the passed in
 * hostname.
 */
bool TSSLTransport::extendedVerify(X509 *peerCertificate, const std::string &host){
  
  X509_NAME * name;
  int loc;
  int lastpos=-1;
  X509_NAME_ENTRY *e; 

  if (peerCertificate == NULL){
    GlobalOutput("Peer did not present a cert. Failing extended verify.");
    return false;
  }

  /* Get the subject name off the cert. */
  name = X509_get_subject_name(peerCertificate);
  
  // Go through the stack of certs, and check that they are all good.
  loc = -1;
  for (;;){
    // try to get the subjectAltName.dNSName.
    lastpos = X509_NAME_get_index_by_NID(name, NID_subject_alt_name, lastpos);
    if (lastpos == -1){
      // And if this doesn't exist, fall back on commonName.
      lastpos = X509_NAME_get_index_by_NID(name, NID_commonName, lastpos);
      if (lastpos == -1){
	break;
      }
    }

    // Now that we have the entry, see if it matches.
    e = X509_NAME_get_entry(name, lastpos);
    ASN1_STRING *name_str = X509_NAME_ENTRY_get_data(e);
    char data_buff[ASN1_STRING_length(name_str)+1];
    memcpy(data_buff, ASN1_STRING_data(name_str), ASN1_STRING_length(name_str));
    data_buff[ASN1_STRING_length(name_str)] = '\0';
    if (host.compare(std::string(data_buff)) != 0){
      GlobalOutput(("Hostname \'" + host + 
		    "\' does not match cert. name \'" + 
		    std::string(data_buff) + "\'").c_str());
      return false;
    }
  }

  // If we get here, everything checks out.
  return true;
}

}}} // apache::thrift::transport
