/* The MIT License:

Copyright (c) 2010 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */


#include <cstdlib>
#include <cmath>

#include <gtkmm/dialog.h>
#include <gtkmm/main.h>
#include <gtkmm/drawingarea.h>

#include <ting/Exc.hpp>
#include <ting/math.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>
#include <ting/Timer.hpp>
#include <ting/Thread.hpp>
#include <ting/utils.hpp>
#include <pugixml/pugixml.hpp>

#include "stdafx.hpp"
#include "Preferences.hpp"
#include "dialogs/OptionsDialog.hpp"
#include "ThereminWindow.hpp"
#include "File.hpp"
#include "FSFile.hpp"
#include "MainWindow.hpp"
#include "MainThread.hpp"
#include "utils.hpp"



using namespace ting;



ThereminWindow::ThereminWindow() :
		tappedX(0)
{
	//options menu
#ifndef M_NON_MAEMO_LINUX
	this->optionsMenuItem.signal_clicked().connect(
			sigc::mem_fun(*this, &ThereminWindow::ShowOptionsDialog)
		);
#else
	this->optionsMenuItem.signal_activate().connect(
			sigc::mem_fun(*this, &ThereminWindow::ShowOptionsDialog)
		);
#endif
	this->optionsMenuItem.set_sensitive(true);


	this->add_events(
			Gdk::BUTTON_PRESS_MASK |
			Gdk::BUTTON_RELEASE_MASK |
			Gdk::POINTER_MOTION_MASK |
			Gdk::BUTTON_MOTION_MASK
		);

	this->signal_button_press_event().connect(
			sigc::mem_fun(*this, &ThereminWindow::OnButtonPressEvent)
		);
	this->signal_button_release_event().connect(
			sigc::mem_fun(*this, &ThereminWindow::OnButtonReleaseEvent)
		);
	this->signal_motion_notify_event().connect(
			sigc::mem_fun(*this, &ThereminWindow::OnMotionEvent)
		);

	//set CAN_FOCUS flag to be able to receive keyboard events
	this->set_flags(Gtk::CAN_FOCUS);

	//NOTE: if use simple connect() then Arrow Keys press events does not come,
	//this is the bug in Gtkmm. Thus, process arrow keys in OnKeyPress_Notify()
	//while process all other keys in OnKeyPress().
	this->signal_key_press_event().connect_notify(
			sigc::mem_fun(*this, &ThereminWindow::OnKeyPress_Notify)
		);

	//set default timbre
	this->instr.ApplyTimbre(Preferences::Inst().GetTimbre());

	{
		ThereminView *tw = new ThereminView();
		this->add(*Gtk::manage(tw));
		tw->show();
	}
}



ThereminWindow::ThereminView::ThereminView(){
	this->signal_expose_event().connect(
			sigc::mem_fun(*this, &ThereminView::OnExposeEvent)
		);
}



void ThereminWindow::ShowOptionsDialog(){
//	TRACE(<< "ThereminWindow::ShowOptionsDialog(): enter" << std::endl)
	OptionsDialog d(MainWindow::Inst());
	d.SetNumOctaves(Preferences::Inst().GetNumOctaves());
	d.SetTimbre(Preferences::Inst().GetTimbre());
	d.SetLockToTappedOctave(Preferences::Inst().GetLockCurrentOctave());

	switch(d.run()){
		case Gtk::RESPONSE_OK: //OK
			Preferences::Inst().SetNumOctaves(d.GetNumOctaves());
			Preferences::Inst().SetTimbre(d.GetTimbre());
			Preferences::Inst().SetLockCurrentOctave(d.GetLockToTappedOctave());
			this->instr.ApplyTimbre(Preferences::Inst().GetTimbre());

			Preferences::Inst().SaveToFile();
			break;
		default: //Cancel and all other cases
			break;
	}

	d.hide();
	this->queue_draw();//request redrawing entire window
//	TRACE(<< "ThereminWindow::ShowOptionsDialog(): exit" << std::endl)
}



bool ThereminWindow::ThereminView::OnExposeEvent(GdkEventExpose* event){
//	TRACE(<< "ThereminWindow::OnExposeEvent(): enter" << std::endl)

	Glib::RefPtr<Gdk::Window> window = this->get_window();
	if(!window)
		return true;

	Gtk::Allocation allocation = get_allocation();
	const int width = allocation.get_width();
	const int height = allocation.get_height();

	Cairo::RefPtr<Cairo::Context> c = window->create_cairo_context();

	if(event){
		// clip to the area indicated by the expose event so that we only
		// redraw the portion of the window that needs to be redrawn
		c->rectangle(event->area.x, event->area.y, event->area.width, event->area.height);
		c->clip();
	}

	c->set_line_width(1.0);
	c->set_source_rgb(0, 0, 0);//black
	c->select_font_face("Sans", Cairo::FONT_SLANT_NORMAL, Cairo::FONT_WEIGHT_NORMAL);
	c->set_font_size(50.0);

	for(unsigned i = 0; i < Preferences::Inst().GetNumOctaves(); ++i){
		c->save();
		c->translate(0, (i * height) / Preferences::Inst().GetNumOctaves());
		ThereminWindow::ThereminView::DrawOctave(
				c,
				width,
				height / Preferences::Inst().GetNumOctaves(),
				Preferences::Inst().GetBasePitch() + (Preferences::Inst().GetNumOctaves() - 1 - i) * 12
			);
		c->restore();
	}

	//draw tapped indicator
//	c->set_line_width(2.0);
//	c->set_source_rgb(0.7f, 0, 0);//darker red
//	c->move_to(this->tappedX, 0);
//	c->line_to(this->tappedX, width);
//	c->stroke();


//	TRACE(<< "ThereminWindow::OnExposeEvent(): exit" << std::endl)

	return true;
}



static const char* notes[] = {
	"A", 0, "B", "C", 0, "D", 0, "E", "F", 0, "G", 0
};



//static
void ThereminWindow::ThereminView::DrawOctave(
		Cairo::RefPtr<Cairo::Context> c,
		unsigned w,
		unsigned h,
		unsigned basePitch
	)
{
	float step = float(w) / 12.0f;
	float curX = 0;
	float height = float(h);

	//draw white background
	c->set_source_rgb(1, 1, 1);//white
	c->rectangle(0, 0, w, h);
	c->fill();

	for(unsigned i = 0, n = basePitch % 12; i < 12; ++i, ++n){
		if(n >= 12)
			n -= 12;

		ASSERT(n < 12)
		//draw notes and black bars for sharp notes
		if(notes[n]){
			//draw note
			Cairo::TextExtents extents;
			c->get_text_extents(notes[n], extents);
			c->set_source_rgb(0, 0, 0);//black
			c->move_to(curX + step / 2 - (extents.width / 2) - extents.x_bearing, height / 2);
			c->show_text(notes[n]);
		}else{
			//draw black bar (sharp note)
			c->set_source_rgb(0.7, 0.7, 0.7);//grey
			c->rectangle(curX, 0, step, height);
			c->fill();
		}

		c->set_source_rgb(0, 0, 0);//black
		c->move_to(curX, 0);
		c->line_to(curX, height);

		c->save();
		c->set_identity_matrix();
		c->set_line_width(2);
		c->stroke();
		c->restore();

		curX += step;
	}

	//draw horizontal line delimiting octaves
	c->move_to(0, h);
	c->line_to(w, h);
	c->save();
	c->set_identity_matrix();
	c->set_line_width(3);
	c->stroke();
	c->restore();

	//draw octave number
	c->set_source_rgb(0.5f, 0, 0);//dark red
	c->move_to(5, h - 5);
	char buf[33];
	snprintf(buf, sizeof(buf)-1, "%u", basePitch / 12);
	buf[32] = 0;
	c->show_text(buf);
}



void ThereminWindow::OnKeyPress_Notify(GdkEventKey* event){
	TRACE(<<"OnKeyPress_Notify(): keyval = " << event->keyval << std::endl)
	switch(event->keyval){
		case GDK_Up:
			//DHighestOctave() + 1 is the number of octaves
			if(Preferences::Inst().GetBasePitch() + this->PitchRange() >= DHighestOctave() * 12){
				Preferences::Inst().SetBasePitch(
						(DHighestOctave() + 1 - Preferences::Inst().GetNumOctaves()) * 12
					);
			}else{
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() + 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Down:
			if(Preferences::Inst().GetBasePitch() <= 12){
				Preferences::Inst().SetBasePitch(0);
			}else{
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() - 12
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Left:
			if(Preferences::Inst().GetBasePitch() != 0){
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() - 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		case GDK_Right:
			//octaves index starts from 0, here need the number of octaves,
			//i.e. DHighestOctave() + 1
			if(Preferences::Inst().GetBasePitch() + this->PitchRange() < 12 * (DHighestOctave() + 1)){
				Preferences::Inst().SetBasePitch(
						Preferences::Inst().GetBasePitch() + 1
					);
			}

			this->queue_draw();//request redrawing etire window
			break;

		default:
			//NOTE: no need to call OnKeyPress_Notify() handler
			//of the base class here. because it is connected to the signal
			//and will be called later, or already has been called.
			//this->BasicWindow::OnKeyPress_Notify(event);
			break;
	}
}



//static
ting::u8 ThereminWindow::CalcVol(real y, real height){
	real factor = real(1) - y / height;
	//quadric dependency on factor allows more precise volume control by finger
	return ting::u8(s32(real(0xff) * factor * factor));
}



ThereminWindow::FreqVol ThereminWindow::CalcFreqAndVol(
		real width,
		real height,
		real x,
		real y,
		bool isPressEvent
	)
{
	FreqVol ret;
	x -= width / real(24); //subtract half of halfstep onscreen width

	unsigned octHeight = unsigned(height / real(Preferences::Inst().GetNumOctaves()));

	ting::ClampBottom(y, real(0));
	ASSERT(y >= real(0))

	//find the screen octave number of the touched octave
	unsigned touchedOctave = unsigned(y) / octHeight;
	ting::ClampTop(touchedOctave, Preferences::Inst().GetNumOctaves() - 1);

	if(Preferences::Inst().GetLockCurrentOctave()){
		//if press event then update current playing octave, this is to prevent volume leaps
		if(isPressEvent)
			this->curOctave = touchedOctave;
	}else{
		this->curOctave = touchedOctave;
	}

	unsigned octavesShift = (Preferences::Inst().GetNumOctaves() - 1 - this->curOctave) * 12;

	ret.freq = CalcFreq(
			real(27.5f),
			x / width + real(Preferences::Inst().GetBasePitch() + octavesShift) / real(12)
		);

	if(touchedOctave == this->curOctave)
		ret.vol = ThereminWindow::CalcVol(y - real(octHeight * this->curOctave), octHeight);
	else if(touchedOctave < this->curOctave)
		ret.vol = 0xff;//max volume
	else //touchedOctave > this->curOctave
		ret.vol = 0;

	return ret;
}



bool ThereminWindow::OnButtonPressEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< this->get_width() << " height = "<< this->get_height() << std::endl)

	this->tappedX = float(event->x);
//	this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

	FreqVol fv = this->CalcFreqAndVol(
			float(this->get_width()),
			float(this->get_height()),
			float(event->x),
			float(event->y),
			true //press event
		);

	this->instr.Stop();

#ifndef M_NO_AUDIO
	this->instr.RenewChannel();
	this->instr.SetFrequency(fv.freq);
	this->instr.SetVolume(fv.vol);
	this->instr.Play();
#endif

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_ON music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_ON;
		m->timeStamp = ting::GetTicks();
		m->volume = fv.vol;
		m->freq = fv.freq;
		m->timbre = Preferences::Inst().GetTimbre();
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool ThereminWindow::OnButtonReleaseEvent(GdkEventButton* event){
//	TRACE(<<"onButtonPressEvent(): invoked, x = "<< event->x << " y = "<< event->y << std::endl)
//	TRACE(<<"onButtonPressEvent(): width = "<< window->get_width() << " height = "<< window->get_height() << std::endl)

//	this->queue_draw_area(this->tappedX - 2, 0, this->tappedX + 2, this->get_height());
	this->tappedX = -1;

	this->instr.Stop();

	if(Preferences::Inst().GetBroadcast()){
		//send NOTE_OFF music action event to main thread
		Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
		m->type = MusicActionMessage::NOTE_OFF;
		m->timeStamp = ting::GetTicks();
		m->timbre = Preferences::Inst().GetTimbre();
		MainThread::Inst().PushMessage(m);
	}

	return true;
}



bool ThereminWindow::OnMotionEvent(GdkEventMotion* event){
	if(event->state & GDK_BUTTON1_MASK){
		this->tappedX = float(event->x);
//		this->queue_draw_area (this->tappedX - 2, 0, this->tappedX + 2, this->get_height());

		if(this->instr.IsChannelValid()){
			FreqVol fv = this->CalcFreqAndVol(
					float(this->get_width()),
					float(this->get_height()),
					float(event->x),
					float(event->y),
					false //not a press event
				);

			this->instr.SetFrequency(fv.freq);
			this->instr.SetVolume(fv.vol);

			if(Preferences::Inst().GetBroadcast()){
				//send NOTE_ON music action event to main thread
				Ptr<MusicActionMessage> m(new MusicActionMessage(&MainThread::Inst()));
				m->type = MusicActionMessage::NOTE_ON;
				m->timeStamp = ting::GetTicks();
				m->volume = fv.vol;
				m->freq = fv.freq;
				m->timbre = Preferences::Inst().GetTimbre();
				MainThread::Inst().PushMessage(m);
			}
		}
	}else{
		this->tappedX = -1;
	}
	return true;
}
