/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/

/*
 * STK library (http://ccrma.stanford.edu/software/stk/index.html) was used as
 * a reference while creating this class.
 */

#pragma once

#include <ting/debug.hpp>
#include <ting/math.hpp>
#include <ting/utils.hpp>

#include "config.hpp"

#include "FilterBiquad.hpp"
#include "FilterOnePole.hpp"
#include "DelayAllPass1.hpp"
//#include "NoiseGen.hpp"


namespace aumiks{


class Bowed{
	DelayAllPass1 neckDelay;
	DelayAllPass1 bridgeDelay;
	FilterOnePole stringFilter;
	FilterBiquad bodyFilter;

	real betaRatio;
	real bowVelocity;

	class BowTable{
		real offset;
		real slope;
		
	public:
		BowTable() :
				offset(0),
				slope(0.1)
		{}

		//The table offset is a bias which controls the
		//symmetry of the friction.  If you want the
		//friction to vary with direction, use a non-zero
		//value for the offset.  The default value is zero.
		inline void SetOffset(real theOffset){
			this->offset = theOffset;
		}

		//The table slope controls the width of the friction
		//pulse, which is related to bow force.
		inline void SetSlope(real theSlope){
			this->slope = theSlope;
		}

		real Tick(real input){
			// The input represents differential string vs. bow velocity.
			real sample;
			sample = input + this->offset;// add bias to input
			sample *= this->slope;// then scale it
			real ret = ting::Abs(sample) + real(0.75);
			ret = ting::Pow(ret, real(-4.0));

			// Set minimum friction to 0.0
			// if (lastOutput < 0.0 ) lastOutput = 0.0;
			// Set maximum friction to 1.0.
			ting::ClampTop(ret, real(1));

			return ret;
		}
	} bowTable;

public:

	Bowed(real lowestFrequency = 20){
		ting::uint length = ting::uint(real(44100) / lowestFrequency + 1);
		this->neckDelay.Init(length, 100.0);

		length /= 2;
		this->bridgeDelay.Init(length, 29.0);

		this->bowTable.SetSlope(3.0);

//		vibrato_.setFrequency( 6.12723 );
//		vibratoGain_ = 0.0;

		this->stringFilter.SetPole(0.6 - (0.1 * 22050.0 / (44100.0)));
//		this->stringFilter.setGain( 0.95 );

		this->bodyFilter.SetResonance(500.0, 0.85);
//		this->bodyFilter.SetGain(0.2);

//		adsr_.setAllTimes( 0.02, 0.005, 0.9, 0.01 );

		this->betaRatio = 0.127236;

		// Necessary to initialize internal variables.
		this->SetFrequency(220.0);
	}


	/**
	 * @brief Start bowing.
	 * @param amplitude - bowing intensity, a value from [0:1]
	 */
	void StartBowing(real amplitude){
		this->bowVelocity = 0.03 + ( 0.2 * amplitude );
	}


	void SetFrequency(real frequency){
		if(frequency <= 0.0){
			TRACE(<< "[ERROR] Bowed::SetFrequency(): parameter is less than or equal to zero!" << std::endl);
			frequency = 220.0;
		}

		// Delay = length - approximate filter delay.
		real baseDelay = (44100.0) / frequency - 4.0;
		if(baseDelay <= 0.0)
			baseDelay = 0.3;
		this->bridgeDelay.SetDelay(baseDelay * this->betaRatio);// bow to bridge length
		this->neckDelay.SetDelay(baseDelay * (1.0 - this->betaRatio));// bow to nut (finger) length
	}


	inline aumiks::real Tick(){
		real bridgeRefl = -this->stringFilter.Tick(real(0.95) * this->bridgeDelay.LastOutValue());
		real nutRefl = -this->neckDelay.LastOutValue();
		real stringVel = bridgeRefl + nutRefl;// Sum is String Velocity
		real velDiff = this->bowVelocity - stringVel;// Differential Velocity
		real newVel = velDiff * this->bowTable.Tick(velDiff);// Non-Linear Bow Function

		// Do string propagations
		this->neckDelay.Tick(bridgeRefl + newVel);
		this->bridgeDelay.Tick(nutRefl + newVel);


		return this->bodyFilter.Tick(this->bridgeDelay.LastOutValue());
	}
};


}//~namespace
