import time
import datetime
import logging

import telepathy

import util.coroutines as coroutines
import gtk_toolbox


_moduleLogger = logging.getLogger("channel.text")


class TextChannel(telepathy.server.ChannelTypeText):
	"""
	Look into implementing ChannelInterfaceMessages for rich text formatting
	"""

	def __init__(self, connection, manager, props, contactHandle):
		self.__manager = manager
		self.__props = props

		try:
			# HACK Older python-telepathy way
			telepathy.server.ChannelTypeText.__init__(self, connection, contactHandle)
			self._requested = props[telepathy.interfaces.CHANNEL_INTERFACE + '.Requested']
			self._implement_property_get(
				telepathy.interfaces.CHANNEL_INTERFACE,
				{"Requested": lambda: self._requested}
			)
		except TypeError:
			# HACK Newer python-telepathy way
			telepathy.server.ChannelTypeText.__init__(self, connection, manager, props)
		self.__nextRecievedId = 0
		self.__lastMessageTimestamp = datetime.datetime(1, 1, 1)

		self.__otherHandle = contactHandle

		# HACK Older python-telepathy doesn't provide this
		self._immutable_properties = {
			'ChannelType': telepathy.server.interfaces.CHANNEL_INTERFACE,
			'TargetHandle': telepathy.server.interfaces.CHANNEL_INTERFACE,
			'Interfaces': telepathy.server.interfaces.CHANNEL_INTERFACE,
			'TargetHandleType': telepathy.server.interfaces.CHANNEL_INTERFACE,
			'TargetID': telepathy.server.interfaces.CHANNEL_INTERFACE,
			'Requested': telepathy.server.interfaces.CHANNEL_INTERFACE
		}

		self.__callback = coroutines.func_sink(
			coroutines.expand_positional(
				self._on_conversations_updated
			)
		)
		self._conn.session.voicemails.updateSignalHandler.register_sink(
			self.__callback
		)
		self._conn.session.texts.updateSignalHandler.register_sink(
			self.__callback
		)

		# The only reason there should be anything in the conversation is if
		# its new, so report it all
		try:
			mergedConversations = self._conn.session.voicemails.get_conversation(self._contactKey)
		except KeyError:
			_moduleLogger.debug("No voicemails in the conversation yet for %r" % (self._contactKey, ))
		else:
			self._report_conversation(mergedConversations)
		try:
			mergedConversations = self._conn.session.texts.get_conversation(self._contactKey)
		except KeyError:
			_moduleLogger.debug("No texts conversation yet for %r" % (self._contactKey, ))
		else:
			self._report_conversation(mergedConversations)

	def get_props(self):
		# HACK Older python-telepathy doesn't provide this
		props = dict()
		for prop, iface in self._immutable_properties.items():
			props[iface + '.' + prop] = \
				self._prop_getters[iface][prop]()
		return props

	@gtk_toolbox.log_exception(_moduleLogger)
	def Send(self, messageType, text):
		if messageType != telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL:
			raise telepathy.errors.NotImplemented("Unhandled message type: %r" % messageType)

		_moduleLogger.info("Sending message to %r" % (self.__otherHandle, ))
		self._conn.session.backend.send_sms(self.__otherHandle.phoneNumber, text)
		self._conn.session.textsStateMachine.reset_timers()

		self.Sent(int(time.time()), messageType, text)

	@gtk_toolbox.log_exception(_moduleLogger)
	def Close(self):
		self.close()

	def close(self):
		self._conn.session.voicemails.updateSignalHandler.unregister_sink(
			self.__callback
		)
		self._conn.session.texts.updateSignalHandler.unregister_sink(
			self.__callback
		)
		self.__callback = None

		telepathy.server.ChannelTypeText.Close(self)
		if self.__manager.channel_exists(self.__props):
			# HACK Older python-telepathy requires doing this manually
			self.__manager.remove_channel(self)
		self.remove_from_connection()

	@property
	def _contactKey(self):
		contactKey = self.__otherHandle.contactID, self.__otherHandle.phoneNumber
		return contactKey

	@gtk_toolbox.log_exception(_moduleLogger)
	def _on_conversations_updated(self, conv, conversationIds):
		if self._contactKey not in conversationIds:
			return
		_moduleLogger.debug("Incoming messages from %r for existing conversation" % (self._contactKey, ))
		mergedConversations = conv.get_conversation(self._contactKey)
		self._report_conversation(mergedConversations)

	def _report_conversation(self, mergedConversations):
		# Can't filter out messages in a texting conversation that came in
		# before the last one sent because that creates a race condition of two
		# people sending at about the same time, which happens quite a bit
		newConversations = mergedConversations.conversations
		if not newConversations:
			_moduleLogger.debug(
				"No messages ended up existing for %r" % (self._contactKey, )
			)
			return
		newConversations = self._filter_out_reported(newConversations)
		newConversations = self._filter_out_read(newConversations)
		newConversations = list(newConversations)
		if not newConversations:
			_moduleLogger.debug(
				"New messages for %r have already been read externally" % (self._contactKey, )
			)
			return
		self.__lastMessageTimestamp = newConversations[-1].time

		messages = [
			newMessage
			for newConversation in newConversations
			for newMessage in newConversation.messages
			if newMessage.whoFrom != "Me:"
		]
		if not newConversations:
			_moduleLogger.debug(
				"All incoming messages were really outbound messages for %r" % (self._contactKey, )
			)
			return

		for newMessage in messages:
			formattedMessage = self._format_message(newMessage)
			self._report_new_message(formattedMessage)

	def _filter_out_reported(self, conversations):
		return (
			conversation
			for conversation in conversations
			if self.__lastMessageTimestamp < conversation.time
		)

	def _filter_out_read(self, conversations):
		return (
			conversation
			for conversation in conversations
			if not conversation.isRead and not conversation.isArchived
		)

	def _format_message(self, message):
		return " ".join(part.text.strip() for part in message.body)

	def _report_new_message(self, message):
		currentReceivedId = self.__nextRecievedId
		timestamp = int(time.time())
		type = telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL

		_moduleLogger.info("Received message from User %r" % self.__otherHandle)
		self.Received(currentReceivedId, timestamp, self.__otherHandle, type, 0, message)

		self.__nextRecievedId += 1
