#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "xlib.h"
#include "object.h"
#include "cmd.h"

#ifdef CASE_SENSITIVE
#define xcase(X) X
#else
#define xcase(X) tolower(X)
#endif

struct xlib_set_t xlib_set[] = {
  /* DUMMY XLIB */

  { 0, 0, "Invalid", NULL }, 

  /* GENERAL */

  { 0, 0, "NOP", CMD_NOP }, 
  { 0, 0, "DEPTH", CMD_DEPTH }, 
  { 0, 0, ".s", CMD_DSTACK },
  { 0, 0, ".", CMD_DISPLAY },
  { 0, 0, "bye", CMD_OFF }, 
  { 0, 0, "MEM", CMD_MEM },
  { 0, 0, "ROOM", CMD_ROOM },
  { 1, 0, "EVAL", CMD_EVAL }, 
  { 1, 0, "COMPEVAL", CMD_COMPEVAL }, 
  { 0, 0, "SLOW", CMD_SLOW }, 
  { 0, 0, "VERYSLOW", CMD_VERYSLOW }, 
  { 0, 0, "CATCH", CMD_CATCH }, 
  { 0, 0, "ERRJMP", CMD_THROW }, 
  { 0, 0, "THROW", CMD_THROW_MSG },
  { 0, 0, ".dbg", CMD_DBG }, 
  { 0, 0, ".trc", CMD_TRACE }, 
  { 0, 0, ".run", CMD_CONT },
  { 1, 0, "LOAD", CMD_LOAD }, 
  { 1, 0, "EXEC", CMD_EXEC },
  { 1, 0, "TYPE", CMD_TYPE }, 
  { 1, 0, "DUPTYPE", CMD_DUPTYPE },
  { 0, 0, "CLOCK", CMD_CLOCK }, 
  { 0, 0, "TIME", CMD_TIME },
  { 1, 0, "CKREF", CMD_CKREF }, 
  { 1, 0, "NEWOB", CMD_NEWOB },
  { 0, 0, "SELF", CMD_SELF },
  { 0, 0, "RECURSE", CMD_RECURSE },
  { 2, 0, "SAME", CMD_SAME },
  { 2, 0, "EQ", CMD_EQ },

  /* FLAGS */

  { 0, 0, ".dal", CMD_MOD_DAL },
  { 0, 0, ".lisp", CMD_MOD_LISP },
  { 0, 0, ".rpn", CMD_MOD_RPN },
  { 0, 0, ".rad", CMD_MOD_RAD },
  { 0, 0, ".deg", CMD_MOD_DEG },
  { 0, 0, ".radix16", CMD_MOD_HEX },
  { 0, 0, ".radix10", CMD_MOD_DEC },
  { 0, 0, ".radix8", CMD_MOD_OCT },
  { 0, 0, ".radix2", CMD_MOD_BIN },
  { 0, 0, ".r@", CMD_MOD_POLAR },
  { 0, 0, ".xy", CMD_MOD_RECT },

  /* STACK ARG CHECKING */

  { 1, 0, "CK1", CMD_CK1 },
  { 2, 0, "CK2", CMD_CK2 },
  { 3, 0, "CK3", CMD_CK3 },
  { 1, 0, "CKN", CMD_CKN },

  /* RUNSTREAM */

  { 0, 0, "'", CMD_QUOTE }, 
  { 0, 0, "SKIP", CMD_SKIP }, 
  { 0, 0, "'R", CMD_QUOTE_R },
  { 0, 0, "ticR", CMD_TIC_R }, 
  { 1, 0, ">R", CMD_TO_R }, 
  { 0, 0, "R>", CMD_QUOTE_R },
  { 0, 0, "'T", CMD_TAIL_R },
  { 0, 0, "R@", CMD_RCL_R }, 
  { 0, 0, "RDROP", CMD_RDROP }, 
  { 0, 0, "COLA", CMD_COLA },
  { 1, 0, "IT", CMD_IT }, 
  { 1, 0, "?SKIP", CMD_QSKIP }, 
  { 1, 0, "ITE", CMD_ITE }, 
  { 1, 0, "CASE", CMD_CASE }, 
  { 1, 0, "?SEMI", CMD_QSEMI },

  /* INDEFINITE LOOP */

  { 0, 0, "IDUP", CMD_IDUP }, 
  { 0, 0, "BEGIN", CMD_BEGIN }, 
  { 0, 0, "AGAIN", CMD_AGAIN }, 
  { 0, 0, "REPEAT", CMD_REPEAT },
  { 1, 0, "WHILE", CMD_WHILE }, 
  { 1, 0, "UNTIL", CMD_UNTIL },

  /* COUNTED LOOP */

  { 2, 0, "DO", CMD_DO }, 
  { 0, 0, "LOOP", CMD_LOOP },
  { 1, 0, "ZERO_DO", CMD_ZERO_DO }, 
  { 1, 0, "ONE_DO", CMD_ONE_DO },
  { 0, 0, "INDEX@", CMD_INDEX_RCL }, 
  { 1, 0, "INDEX!", CMD_INDEX_STO }, 
  { 0, 0, "ISTOP@", CMD_ISTOP_RCL }, 
  { 1, 0, "ISTOP!", CMD_ISTOP_STO }, 

  /* LOGIC */

  { 0, 0, "Nil", CMD_NIL },
  { 0, 0, "True", CMD_TRUE }, 
  { 0, 0, "False", CMD_FALSE }, 
  { 1, 0, "NOT", CMD_NOT },
  { 2, 0, "AND", CMD_AND }, 
  { 2, 0, "OR", CMD_OR }, 
  { 2, 0, "XOR", CMD_XOR },

  /* LAMBDA */

  { 1, 0, "@?", CMD_EXIST }, 
  { 1, 0, "@", CMD_RCL },
  { 2, 0, "!", CMD_STO }, 
  { 1, 0, "PURGE", CMD_PURGE },
  { -1, 0, "BIND", CMD_BIND }, 
  { 0, 0, "ABND", CMD_ABND },
  { -1, 0, "CACHE", CMD_CACHE }, 
  { 0, 0, "DUMP", CMD_DUMP },
  { 0, 0, "VARS", CMD_VARS },

  /* STACK */

  { 1, 0, "DUP", CMD_DUP }, 
  { 1, 0, "DROP", CMD_DROP }, 
  { 2, 0, "SWAP", CMD_SWAP },
  { 2, 0, "OVER", CMD_OVER }, 
  { 3, 0, "ROT", CMD_ROT }, 
  { 3, 0, "UNROT", CMD_UNROT },
  { -1, 0, "PICK", CMD_PICK }, 
  { -1, 0, "ROLL", CMD_ROLL }, 
  { -1, 0, "UNROLL", CMD_UNROLL },

  /* INTEGER */

  { 0, 0, "_BINT", CMD__BINT }, 
  { 0, 0, "_REAL", CMD__REAL },
  { 0, 0, "_CPLX", CMD__CPLX }, 
  { 0, 0, "_CSTR", CMD__CSTR },
  { 0, 0, "_ZINT", CMD__ZINT }, 
  { 0, 0, "_CHAR", CMD__CHAR },
  { 0, 0, "_IDNT", CMD__IDNT }, 
  { 0, 0, "_LAM" , CMD__LAM  },
  { 0, 0, "_CODE", CMD__CODE }, 
  { 0, 0, "_ARRY", CMD__ARRY }, 
  { 0, 0, "_SYMB", CMD__SYMB }, 
  { 0, 0, "_LIST", CMD__LIST }, 
  { 0, 0, "_PROG", CMD__PROG },

  { 0, 0, "ZERO", CMD_BINT_ZERO }, 
  { 0, 0, "ONE", CMD_BINT_ONE }, 
  { 0, 0, "TWO", CMD_BINT_TWO }, 
  { 0, 0, "THREE", CMD_BINT_THREE }, 
  { 0, 0, "FOUR", CMD_BINT_FOUR }, 
  { 0, 0, "FIVE", CMD_BINT_FIVE }, 

  { 2, 0, "#+", CMD_BINT_PLUS }, 
  { 2, 0, "#-", CMD_BINT_MINUS }, 
  { 2, 0, "#*", CMD_BINT_TIMES }, 
  { 2, 0, "#/", CMD_BINT_DIVIDE },

  { 2, 0, "#|", CMD_BINT_OR },
  { 2, 0, "#&", CMD_BINT_AND },
  { 2, 0, "#^", CMD_BINT_XOR },
  { 2, 0, "#~", CMD_BINT_NOT },

  { 2, 0, "#<", CMD_BINT_LESS }, 
  { 2, 0, "#<=", CMD_BINT_LESS_EQ },
  { 2, 0, "#>", CMD_BINT_MORE }, 
  { 2, 0, "#>=", CMD_BINT_MORE_EQ },
  { 2, 0, "#=", CMD_BINT_EQUAL }, 
  { 2, 0, "#0=", CMD_BINT_Q_ZERO },
  { 2, 0, "#<>", CMD_BINT_NOT_EQUAL }, 
  { 1, 0, "#0<>", CMD_BINT_Q_NOT_ZERO },
  { 1, 0, "#0>", CMD_BINT_MORE_ZERO }, 
  { 1, 0, "#0<", CMD_BINT_LESS_ZERO },
  { 1, 0, "#0>=", CMD_BINT_MORE_EQ_ZERO }, 
  { 1, 0, "#0<=", CMD_BINT_LESS_EQ_ZERO },

  /* ZINT */

  { 2, 0, "Z+", CMD_ZINT_PLUS }, 
  { 2, 0, "Z-", CMD_ZINT_MINUS }, 
  { 2, 0, "Z*", CMD_ZINT_TIMES }, 
  { 2, 0, "Z/", CMD_ZINT_DIVIDE },
  { 2, 0, "Z/qr", CMD_ZINT_DIV_QR },
  { 2, 0, "Z/r", CMD_ZINT_MOD },
  { 2, 0, "Zgcd", CMD_ZINT_GCD },
  { 2, 0, "Zegcd", CMD_ZINT_EGCD },

  { 1, 0, "Zneg", CMD_ZINT_NEG },
  { 1, 0, "Zabs", CMD_ZINT_ABS },
  { 1, 0, "Zsgn", CMD_ZINT_SGN },
  { 1, 0, "Z!", CMD_ZINT_FACTORIAL },

  { 2, 0, "Z<", CMD_ZINT_LESS }, 
  { 2, 0, "Z<=", CMD_ZINT_LESS_EQ },
  { 2, 0, "Z>", CMD_ZINT_MORE }, 
  { 2, 0, "Z>=", CMD_ZINT_MORE_EQ },
  { 2, 0, "Z=", CMD_ZINT_EQUAL }, 
  { 2, 0, "Z0=", CMD_ZINT_Q_ZERO },
  { 2, 0, "Z<>", CMD_ZINT_NOT_EQUAL }, 
  { 1, 0, "Z0<>", CMD_ZINT_Q_NOT_ZERO },
  { 1, 0, "Z0>", CMD_ZINT_MORE_ZERO }, 
  { 1, 0, "Z0<", CMD_ZINT_LESS_ZERO },
  { 1, 0, "Z0>=", CMD_ZINT_MORE_EQ_ZERO }, 
  { 1, 0, "Z0<=", CMD_ZINT_LESS_EQ_ZERO },

  /* ZINT <-> BINT */

  { 1, 0, "Z>#", CMD_ZINT2BINT },
  { 1, 0, "#>Z", CMD_BINT2ZINT },

  /* ZINT <-> REAL */

  { 1, 0, "Z>%", CMD_ZINT2REAL },
  { 1, 0, "%>Z", CMD_REAL2ZINT },

  /* BINT <-> REAL */

  { 1, 0, "#>%", CMD_BINT2REAL },
  { 1, 0, "%>#", CMD_REAL2BINT },

  /* REAL */

  { 0, 0, "%0", CMD_REAL_ZERO }, 
  { 0, 0, "%1", CMD_REAL_ONE },
  { 0, 0, "%-1", CMD_REAL_NEG_ONE }, 
  { 0, 0, "%e", CMD_REAL_E }, 
  { 0, 0, "%pi", CMD_REAL_PI }, 
  { 0, 0, "%g", CMD_REAL_G },
  { 2, 0, "%+", CMD_REAL_PLUS }, 
  { 2, 0, "%-", CMD_REAL_MINUS },
  { 2, 0, "%*", CMD_REAL_TIMES }, 
  { 2, 0, "%/", CMD_REAL_DIVIDE },
  { 2, 0, "%^", CMD_REAL_POWER }, 
  { 1, 0, "%abs", CMD_REAL_ABS },
  { 1, 0, "%neg", CMD_REAL_NEG }, 
  { 1, 0, "%sgn", CMD_REAL_SGN },
  { 1, 0, "%1/", CMD_REAL_INV }, 
  { 1, 0, "%sqrt", CMD_REAL_SQRT },
  { 2, 0, "%nroot", CMD_REAL_NROOT },

  { 1, 0, "%exp", CMD_REAL_EXP }, 
  { 1, 0, "%expm1", CMD_REAL_EXPM1 },
  { 1, 0, "%ln", CMD_REAL_LN }, 
  { 1, 0, "%lnp1", CMD_REAL_LNP1 },
  { 1, 0, "%log", CMD_REAL_LOG }, 
  { 1, 0, "%alog", CMD_REAL_ALOG },

  { 1, 0, "%sin", CMD_REAL_SIN }, 
  { 1, 0, "%sinh", CMD_REAL_SINH }, 
  { 1, 0, "%asin", CMD_REAL_ASIN }, 
  { 1, 0, "%asinh", CMD_REAL_ASINH }, 

  { 1, 0, "%cos", CMD_REAL_COS }, 
  { 1, 0, "%cosh", CMD_REAL_COSH }, 
  { 1, 0, "%acos", CMD_REAL_ACOS },
  { 1, 0, "%acosh", CMD_REAL_ACOSH },

  { 1, 0, "%tan", CMD_REAL_TAN }, 
  { 1, 0, "%atan", CMD_REAL_ATAN }, 
  { 1, 0, "%tanh", CMD_REAL_TANH }, 
  { 1, 0, "%atanh", CMD_REAL_ATANH }, 

  { 1, 0, "%sec", CMD_REAL_SEC }, 
  { 1, 0, "%sech", CMD_REAL_SECH }, 
  { 1, 0, "%asec", CMD_REAL_ASEC }, 
  { 1, 0, "%asech", CMD_REAL_ASECH }, 

  { 1, 0, "%csc", CMD_REAL_CSC }, 
  { 1, 0, "%csch", CMD_REAL_CSCH }, 
  { 1, 0, "%acsc", CMD_REAL_ACSC },
  { 1, 0, "%acsch", CMD_REAL_ACSCH },

  { 1, 0, "%cot", CMD_REAL_COT }, 
  { 1, 0, "%acot", CMD_REAL_ACOT }, 
  { 1, 0, "%coth", CMD_REAL_COTH }, 
  { 1, 0, "%acoth", CMD_REAL_ACOTH }, 

  { 1, 0, "%ip", CMD_REAL_IP }, 
  { 1, 0, "%fp", CMD_REAL_FP }, 
  { 1, 0, "%floor", CMD_REAL_FLOOR }, 
  { 1, 0, "%ceil", CMD_REAL_CEIL }, 

  { 2, 0, "%mod", CMD_REAL_MOD }, 
  { 2, 0, "%angle", CMD_REAL_ANGLE }, 

  { 2, 0, "%comb", CMD_REAL_COMB }, 
  { 2, 0, "%perm", CMD_REAL_PERM }, 
  { 1, 0, "%!", CMD_REAL_FACTORIAL }, 
  { 1, 0, "%gamma", CMD_REAL_GAMMA }, 
  { 1, 0, "%erf", CMD_REAL_ERF },
  { 1, 0, "%erfc", CMD_REAL_ERFC },
  { 1, 0, "%ierfc", CMD_REAL_IERFC },

  { 2, 0, "%<", CMD_REAL_LESS }, 
  { 2, 0, "%<=", CMD_REAL_LESS_EQ },
  { 2, 0, "%>", CMD_REAL_MORE }, 
  { 2, 0, "%>=", CMD_REAL_MORE_EQ },
  { 2, 0, "%=", CMD_REAL_EQUAL }, 
  { 1, 0, "%0=", CMD_REAL_Q_ZERO },
  { 2, 0, "%<>", CMD_REAL_NOT_EQUAL }, 
  { 1, 0, "%0<>", CMD_REAL_Q_NOT_ZERO },
  { 1, 0, "%0>", CMD_REAL_MORE_ZERO }, 
  { 1, 0, "%0<", CMD_REAL_LESS_ZERO },
  { 1, 0, "%0>=", CMD_REAL_MORE_EQ_ZERO }, 
  { 1, 0, "%0<=", CMD_REAL_LESS_EQ_ZERO },

  /* CPLX */

  { 0, 0, "C%0", CMD_CPLX_ZERO }, 
  { 0, 0, "C%1", CMD_CPLX_ONE },
  { 0, 0, "C%-1", CMD_CPLX_NEG_ONE }, 
  { 0, 0, "C%j", CMD_CPLX_J }, 

  { 2, 0, "C%+", CMD_CPLX_PLUS }, 
  { 2, 0, "C%-", CMD_CPLX_MINUS },
  { 2, 0, "C%*", CMD_CPLX_TIMES }, 
  { 2, 0, "C%/", CMD_CPLX_DIVIDE },
  { 2, 0, "C%mod", CMD_CPLX_MOD },
  { 2, 0, "C%^", CMD_CPLX_POWER }, 

  { 1, 0, "C%angle", CMD_CPLX_ANGLE }, 
  { 1, 0, "C%abs", CMD_CPLX_ABS },
  { 1, 0, "C%neg", CMD_CPLX_NEG }, 
  { 1, 0, "C%sgn", CMD_CPLX_SGN },
  { 1, 0, "C%real", CMD_CPLX_REAL },
  { 1, 0, "C%imag", CMD_CPLX_IMAG },
  { 1, 0, "C%conj", CMD_CPLX_CONJ },

  { 1, 0, "C%1/", CMD_CPLX_INV }, 
  { 1, 0, "C%sqrt", CMD_CPLX_SQRT }, 
  { 2, 0, "C%nroot", CMD_CPLX_NROOT },

  { 1, 0, "C%exp", CMD_CPLX_EXP }, 
  { 1, 0, "C%ln", CMD_CPLX_LN }, 
  { 1, 0, "C%log", CMD_CPLX_LOG }, 
  { 1, 0, "C%alog", CMD_CPLX_ALOG },

  { 1, 0, "C%sin", CMD_CPLX_SIN }, 
  { 1, 0, "C%sinh", CMD_CPLX_SINH }, 
  { 1, 0, "C%asin", CMD_CPLX_ASIN }, 
  { 1, 0, "C%asinh", CMD_CPLX_ASINH }, 

  { 1, 0, "C%cos", CMD_CPLX_COS }, 
  { 1, 0, "C%cosh", CMD_CPLX_COSH }, 
  { 1, 0, "C%acos", CMD_CPLX_ACOS },
  { 1, 0, "C%acosh", CMD_CPLX_ACOSH },

  { 1, 0, "C%tan", CMD_CPLX_TAN }, 
  { 1, 0, "C%atan", CMD_CPLX_ATAN }, 
  { 1, 0, "C%tanh", CMD_CPLX_TANH }, 
  { 1, 0, "C%atanh", CMD_CPLX_ATANH }, 

  { 1, 0, "C%sec", CMD_CPLX_SEC }, 
  { 1, 0, "C%sech", CMD_CPLX_SECH }, 
  { 1, 0, "C%asec", CMD_CPLX_ASEC }, 
  { 1, 0, "C%asech", CMD_CPLX_ASECH }, 

  { 1, 0, "C%csc", CMD_CPLX_CSC }, 
  { 1, 0, "C%csch", CMD_CPLX_CSCH }, 
  { 1, 0, "C%acsc", CMD_CPLX_ACSC },
  { 1, 0, "C%acsch", CMD_CPLX_ACSCH },

  { 1, 0, "C%cot", CMD_CPLX_COT }, 
  { 1, 0, "C%acot", CMD_CPLX_ACOT }, 
  { 1, 0, "C%coth", CMD_CPLX_COTH }, 
  { 1, 0, "C%acoth", CMD_CPLX_ACOTH }, 

  { 1, 0, "C%!", CMD_CPLX_FACTORIAL }, 
  { 1, 0, "C%gamma", CMD_CPLX_GAMMA }, 

  { 2, 0, "C%=", CMD_CPLX_EQUAL }, 
  { 1, 0, "C%0=", CMD_CPLX_Q_ZERO },
  { 2, 0, "C%<>", CMD_CPLX_NOT_EQUAL }, 
  { 1, 0, "C%0<>", CMD_CPLX_Q_NOT_ZERO },

  /* ARRAY */

  { -1, 0, "[]N", CMD_MAKEARRY }, 
  {  1, 0, "ONE[]N", CMD_ARRY1 },
  {  2, 0, "GETATELN", CMD_GETATELN }, 

  { -1, 0, "AMAKE", CMD_MAKEARRY },
  {  2, 0, "AREF", CMD_AREF },
  {  1, 0, "ADIMS", CMD_ARSIZE },
  {  1, 0, "RANK", CMD_RANK },
  {  1, 0, "ALEN", CMD_ALEN },

  /* COMPOSITE */

  {  1, 0, "INNER", CMD_INNER }, 
  { -1, 0, "::N", CMD_META_PROG }, 
  { -1, 0, "{}N", CMD_META_LIST },
  {  1, 0, "CAR", CMD_CAR },
  {  1, 0, "CDR", CMD_CDR },
  {  1, 0, "NEXT", CMD_NEXT },

  /* STRING */

  { 1, 0, "DOSTR>", CMD_EVAL_STR }, 
  { 1, 0, "PARSE$", CMD_PARSE_STR },
  { 1, 0, "NEXT-TOKEN", CMD_NEXT_TOKEN }, 
  { 1, 0, "DO>STR", CMD_DO_STR },

  /* SYMBOLIC */

  { -1, 0, "SYMBN", CMD_SYMBN }, 
  { 1, 0, "SYMB1", CMD_SYMB1 },
  { 1, 0, "Simplify", CMD_SIMPLIFY },

  /* SYMBOLIC FUNCTIONS */

  { 1, 0, "1>x", CMD_COERCE1 }, 
  { 2, 0, "2>xy", CMD_COERCE2 },

  { 0, 0, "x(", CMD_xOPAREN }, 
  { 0, 0, "x)", CMD_xCPAREN },

  { 1, 1, "xNEG", CMD_xNEGATIVE }, 

  { 2, 0, "x**", CMD_xPOW },
  { 2, 0, "x*", CMD_xTIMES }, 
  { 2, 0, "x/", CMD_xDIVIDE }, 
  { 2, 0, "x%", CMD_xMOD }, 
  { 2, 0, "x+", CMD_xPLUS }, 
  { 2, 0, "x-", CMD_xMINUS },
  
  { 2, 0, "x=", CMD_xASSIGN }, 
  { 1, 0, "x,", CMD_xCOMMA },

  { 1, 1, "xABS", CMD_xABS }, 
  { 1, 1, "xSGN", CMD_xSGN },
  { 1, 1, "xINV", CMD_xINV }, 
  { 1, 1, "xSQRT", CMD_xSQRT },
  { 2, 1, "xNROOT", CMD_xNROOT },

  { 1, 1, "xEXP", CMD_xEXP }, 
  { 1, 1, "xLN", CMD_xLN },
  { 1, 1, "xLOG", CMD_xLOG }, 
  { 1, 1, "xALOG", CMD_xALOG },
  { 1, 1, "xSIN", CMD_xSIN }, 
  { 1, 1, "xCOS", CMD_xCOS }, 
  { 1, 1, "xSINH", CMD_xSINH }, 
  { 1, 1, "xCOSH", CMD_xCOSH }, 
  { 1, 1, "xASIN", CMD_xASIN }, 
  { 1, 1, "xACOS", CMD_xACOS },
  { 1, 1, "xASINH", CMD_xASINH }, 
  { 1, 1, "xACOSH", CMD_xACOSH },
  { 1, 1, "xTAN", CMD_xTAN }, 
  { 1, 1, "xATAN", CMD_xATAN }, 
  { 1, 1, "xTANH", CMD_xTANH }, 
  { 1, 1, "xATANH", CMD_xATANH }, 

  { 2, 1, "xMOD", CMD_xMOD }, 
  { 2, 1, "xANGLE", CMD_xANGLE }, 

  { 2, 1, "xCOMB", CMD_xCOMB }, 
  { 2, 1, "xPERM", CMD_xPERM }, 
  { 1, 1, "xFACT", CMD_xFACTORIAL }, 
  { 1, 1, "xGAMMA", CMD_xGAMMA }, 

  /* END MARKER */

  { 0, 0, NULL, NULL }
};

size_t xlib_set_len = sizeof(xlib_set) / sizeof(xlib_set[0]);

int xstrcmp(const char *s1, const char *s2) {
  while(*s1 && *s2 && (xcase(*s1) == xcase(*s2))) {
    s1 ++, s2 ++;
  }

  return (int)(xcase(*s1) - xcase(*s2));
}

int xstrncmp(const char *s1, const char *s2, size_t n) {
  if(n > 0) {
    while((-- n) && *s1 && *s2 && (xcase(*s1) == xcase(*s2))) {
      s1 ++, s2 ++;
    }

    return (int)(xcase(*s1) - xcase(*s2));
  }

  return 0;
}

T_CODE code_find(const char *id) {
  struct xlib_set_t *ptr;
  size_t ndx;

  /* skip first (invalid. ndx = 0) id */
  for(ptr = &xlib_set[1], ndx = 1; ptr->id; ptr ++, ndx ++)
    if(0 == xstrcmp(id, ptr->id))
      return ptr->call;

  return (T_CODE)0;
}

const char *code_getid(const T_CODE code) {
  struct xlib_set_t *ptr;

  for(ptr = &xlib_set[0]; ptr->id; ptr ++)
    if(ptr->call == code)
      return ptr->id;

  return NULL;
}

signed int code_arity(const T_CODE code) {
  struct xlib_set_t *ptr;

  for(ptr = &xlib_set[0]; ptr->id; ptr ++)
    if(ptr->call == code)
      return ptr->arity;

  return -1;
}

int code_function(const T_CODE code) {
  struct xlib_set_t *ptr;

  for(ptr = &xlib_set[0]; ptr->id; ptr ++)
    if(ptr->call == code)
      return ptr->function;

  return 0;
}
