#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "console.h"

#include "object.h"
#include "stack.h"
#include "cmd.h"
#include "mem.h"
#include "dump.h"
#include "xlib.h"
#include "symb.h"
#include "main.h"

static inline T_CODE find_prefix(char token) {
  switch(token) {
  case '-': return CMD_xNEGATIVE;
  }

  return NULL;
}

static inline T_CODE find_operator(const char *token, size_t len) {
  switch(len) {
  case 1:
    switch(token[0]) {
    case '(': return CMD_xOPAREN;
    case ')': return CMD_xCPAREN;
    case '*': return CMD_xTIMES;
    case '/': return CMD_xDIVIDE;
    case '%': return CMD_xMOD;
    case '+': return CMD_xPLUS;
    case '-': return CMD_xMINUS;
    case '=': return CMD_xASSIGN;
    case ',': return CMD_xCOMMA;
    }

  case 2:
    switch(token[0]) {
    case '*':
      switch(token[1]) {
      case '*': return CMD_xPOW;
      }
    }
  }

  return NULL;
}

static inline int __priority(T_CODE op) {
  if(op == CMD_xNEGATIVE)
    return 40;
  else if(op == CMD_xPOW)
    return 33;
  else if(op == CMD_xTIMES || op == CMD_xDIVIDE || op == CMD_xMOD)
    return 32;
  else if(op == CMD_xPLUS || op == CMD_xMINUS)
    return 30;
  else if(op == CMD_xASSIGN)
    return 2;
  else if(op == CMD_xCOMMA)
    return 1;

  return 0;
}

/* return next token (number, name, function) */
static const char* alg_token(const char *symb, T_ATOM **ob) {
  T_ATOM *token = NULL;

  if(symb) {
    T_CODE op;

    if(*symb == '(') {
      token = atom_make_code(CMD_xOPAREN);
      symb ++;
    } 
    else if(*symb == ')') {
      token = atom_make_code(CMD_xCPAREN);
      symb ++;
    }
    else if((op = find_prefix(*symb)) != NULL) {
      token = atom_make_code(op);
      symb ++;
    }
    else { /* (number, name, function) */
      double x;
      char *endptr;

      x = strtod(symb, &endptr);
      if(endptr > symb) { /* number */
	token = atom_make_real(x);
	symb = endptr;
      } else { /* name or function */
	const char *id;
	size_t len;

	for(id = symb, len = 0; *symb && !find_operator(symb, 1); len ++)
	  symb ++;

	if(find_operator(symb, 1) == CMD_xOPAREN)
	  token = atom_make_func((T_NAME)id, len);
	else
	  token = atom_make_name((T_NAME)id, len);
      }
    }
  }

  if(ob)
    *ob = token;
  
  return symb;
}

/* return next operator */
static const char* alg_oper(const char *symb, T_ATOM **ob) {
  T_ATOM *token = NULL;

  if(symb) {
    T_CODE op1 = NULL, op2;
    size_t len;
    
    for(len = 0; (op2 = find_operator(symb, len + 1)) != NULL; len ++)
      op1 = op2;
    
    token = atom_make_code(op1);
    symb += len;
  }
  
  if(ob)
    *ob = token;
  
  return symb;
}

/* convert symb to (meta) rpn program, return number of objects pushed */
size_t symb_meta(const T_SYMB *root, T_CONS **STK) {
  size_t z = 0;

  if(root) {
    if(root->car)
      z += symb_meta(root->car, STK);

    if(root->cdr)
      z += symb_meta(root->cdr, STK);

    if(root->ob) {
      STK_add(STK, atom_link(root->ob));
      z ++;
    }
  }

  return z;
}

/* convert symb to rpn expression */
char *symb_postorder(const T_SYMB *root, char **stored) {
  if(root) {
    symb_postorder(root->car, stored);
    symb_postorder(root->cdr, stored);

    if(root->ob) {
      char *word = atom_string(root->ob, &word);

      if(**stored) {
	*stored = realloc(*stored, strlen(*stored) + 2);
	strcat(*stored, " ");
      }
      
      *stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
      strcat(*stored, word);
    }
  }

  return *stored;
}


/* convert symb to (infix) algebraic expr */
char *symb_inorder(const T_SYMB *root, char **stored) {
  if(root) {
    if(root->ob == NULL) { /* function argument, number, etc. */
      if(root->car) {
	char *word = atom_string(root->ob, &word);

	*stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
	strcat(*stored, word);
      }
    }
    else if(atom_q_function(root->ob)) { /* f[a], f[a,b] */
      const T_SYMB *cdr;
      char *word = atom_string(root->ob, &word);

      *stored = realloc(*stored, strlen(*stored) + strlen(word) + 3);
      strcat(*stored, word);
      strcat(*stored, "[");

      if(root->car)
	symb_inorder(root->car, stored);

      for(cdr = root->cdr; cdr; cdr = cdr->cdr) {
	*stored = realloc(*stored, strlen(*stored) + 2);
	strcat(*stored, ",");
	symb_inorder(cdr->car, stored);
      }

      *stored = realloc(*stored, strlen(*stored) + 2);
      strcat(*stored, "]");
    } 
    else { /* a <op> b */
      T_ATOM *op = root->ob;
      int priority = 0;

      if(op->prolog == DOCODE)
	priority = __priority(atom_get_code(op));

      if(priority == 0)
	priority = 1000;

      if(root->car) {
	T_ATOM *op1 = (root->car)->ob;
	int child_priority = 0;
	int paren = 0;

	if(op1 && op1->prolog == DOCODE)
	  child_priority = __priority(atom_get_code(op1));

	if(child_priority == 0)
	  child_priority = 1000;

	paren = child_priority < priority;

	if(paren) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, "(");
	}

	symb_inorder(root->car, stored);

	if(paren) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, ")");
	}
      }

      if(root->ob) {
	char *word = atom_string(root->ob, &word);

	*stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);

	strcat(*stored, word);
      }

      if(root->cdr) {
	T_ATOM *op2 = (root->cdr)->ob;
	int child_priority = 0;
	int paren = 0;

	if(op2 && op2->prolog == DOCODE)
	  child_priority = __priority(atom_get_code(op2));

	if(child_priority == 0)
	  child_priority = 1000;

	paren = child_priority < priority;

	if(paren) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, "(");
	}

	symb_inorder(root->cdr, stored);

	if(paren) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, ")");
	}
      }
    }
  }

  return *stored;
}

/* convert symb to lisp-like expression */
char *symb_preorder(const T_SYMB *root, char **stored) {
  if(root) {
    if(root->ob == NULL) {
      if(root->car) {
	char *word = atom_string((root->car)->ob, &word);

	*stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
	strcat(*stored, word);
      }
    }
    else if(! atom_q_exec(root->ob)) {
      if(root->ob) {
	char *word = atom_string(root->ob, &word);

	*stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
	strcat(*stored, word);
      }
    }
    else if(atom_q_function(root->ob)) { /* function */
      char *word = atom_string(root->ob, &word);

      *stored = realloc(*stored, strlen(*stored) + strlen(word) + 2);
      strcat(*stored, "(");
      strcat(*stored, word);

      if(root->car) {
	const T_SYMB *cdr;

	*stored = realloc(*stored, strlen(*stored) + 2);
	strcat(*stored, " ");
	symb_preorder(root->car, stored);

	for(cdr = root->cdr; cdr; cdr = cdr->cdr) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, " ");
	  symb_preorder(cdr, stored);
	}
      }

      *stored = realloc(*stored, strlen(*stored) + 2);
      strcat(*stored, ")");
    }
    else { /* code */
      char *word = atom_string(root->ob, &word);

      *stored = realloc(*stored, strlen(*stored) + strlen(word) + 2);
      strcat(*stored, "(");
      strcat(*stored, word);

      if(root->car) {
	*stored = realloc(*stored, strlen(*stored) + 2);
	strcat(*stored, " ");
	symb_preorder(root->car, stored);

	if(root->cdr) {
	  *stored = realloc(*stored, strlen(*stored) + 2);
	  strcat(*stored, " ");
	  symb_preorder(root->cdr, stored);
	}
      }

      *stored = realloc(*stored, strlen(*stored) + 2);
      strcat(*stored, ")");
    }
  }

  return *stored;
}

/* public, convert algebraic expr (CSTR) into (SYMB) */
T_SYMB *symb_parse(const char *word) {
  T_ATOM *stream;
  T_ATOM *rpn;
  T_SYMB *symb;

  stream = parse_expr(word);

  rpn = symb_eval(stream);
  atom_unlink(stream);

  symb = symb_gen(atom_get_cons(rpn));
  atom_unlink(rpn);

  return symb;
}

/* public, convert SYMB into CSTR */
char *symb_expr(const T_SYMB *symb) {
  char *alg = malloc(256);

  alg[0] = '\0';
  
  if(symb) {
    switch(FLAGS.mod_symb) {
    case MOD_DAL:
      alg = symb_inorder(symb, &alg);
      break;

    case MOD_LISP:
      alg = symb_preorder(symb, &alg);
      break;

    case MOD_RPN:
      alg = symb_postorder(symb, &alg);
      break;
    }
  }

  //alg = realloc(alg, strlen(alg) + 1);

  return alg;
}

void cons_append(T_CONS ***tail, T_ATOM *ob) {
  T_CONS *s = cons_alloc();
  
  s->car = ob;
  s->cdr = NULL;

  if(**tail == NULL) {
    **tail = s;
  } else {
    (**tail)->cdr = s;
  }

  *tail = &((**tail)->cdr);
}

/* public, convert CSTR into tokenized stream (PROG) */
T_ATOM *parse_expr(const char *word) {
  T_ATOM *infix;
  T_CONS **tail;

  enum { TOKEN, OPERATOR } expecting;

  infix = atom_make_cons(DOPROG);
  tail = &(atom_get_cons(infix));
  *tail = NULL;
  
  expecting = TOKEN;

  while(word && *word) {
    T_ATOM *ob;

    word = (expecting == TOKEN) ? 
      alg_token(word, &ob) : 
      alg_oper(word, &ob);

    cons_append(&tail, ob);

    if(ob->prolog == DOREAL || ob->prolog == DOIDNT) {
      expecting = OPERATOR;
    }
    else if(ob->prolog == DOFUNC) {
      expecting = TOKEN;
    }
    else if(ob->prolog == DOCODE) {
      expecting = (atom_get_code(ob) == CMD_xCPAREN) ? OPERATOR : TOKEN;
    }
    else {
      xprintf(win_cmd, "(parse-expr) got '");
      atom_dump(win_cmd, ob);
      xprintf(win_cmd, "' when expecting an %s\n",
	      expecting == TOKEN ? "operand" : "operator\n");
    }
  }

  if(expecting == TOKEN) {
    xprintf(win_cmd, "(parse-expr) got nothing when expecting an operand\n");
  }

  return infix;
}

/* public, convert rpn expression (PROG) into (SYMB) */
T_SYMB *symb_gen(const T_CONS *rpn) {
  static T_SYMB *data[256];
  size_t  depth = 0;

  for( ; rpn; rpn = rpn->cdr) {
    T_ATOM *el = atom_link(rpn->car);

    if(atom_q_number(el) || atom_q_name(el) || atom_q_nop(el)) {
      T_SYMB *n = symb_alloc();

      n->ob = el;
      n->car = NULL;
      n->cdr = NULL;

      data[depth ++] = n;
    }
    else if(atom_q_function(el)) {
      T_SYMB *n, *arg;
      signed int arity, arity_given, ndx;

      if(el->prolog == DOCODE)
	arity = code_arity(atom_get_code(el));
      else
	arity = 1; /* default for user-defined function */

      for(arity_given = 0; depth > arity_given; arity_given ++) {
	T_ATOM *arg = (data[depth - arity_given - 1])->ob;

	if(atom_q_nop(arg)) {
	  /* unlink NOP because it won't be used.. */
	  symb_unlink(data[depth - arity_given - 1]);
	  break;
	}
      }

      if(arity_given > arity)
	arity = arity_given;

      if(arity > depth) {
	Exception(NULL, "Not enough arguments", 0);
      }

      n = symb_alloc();
      n->ob = el;

      for(arg = n, ndx = arity; ndx > 0; ) {
	arg->car = data[depth - ndx];
	ndx --;

	if(ndx > 0) {
	  arg = arg->cdr = symb_alloc();
	  arg->ob = NULL;
	}
      }

      depth -= (arity + 1); /* dump NULL marker as well */
      data[depth ++] = n;
    }
    else if(el->prolog == DOCODE) {
      T_SYMB *n;

      n = symb_alloc();
      n->ob = el;

      n->cdr = data[-- depth];
      n->car = data[-- depth];

      data[depth ++] = n;
    }
  }

  if(depth > 0)
    return data[-- depth];

  return NULL;
}

/* public, convert tokenized stream into postfix */
T_ATOM *symb_eval(T_ATOM *stream) {
  T_CONS *opSTK = NULL; /* operator stack */
  T_ATOM *rpn; /* rpn object */

  T_CONS **tail;
  T_CONS *seq;

  rpn = atom_make_cons(DOPROG);
  tail = &(atom_get_cons(rpn));
  *tail = NULL;

  for(seq = atom_get_cons(stream); seq; seq = seq->cdr) {
    T_ATOM *popped = atom_link(seq->car);

    if(atom_q_number(popped) || atom_q_name(popped)) { /* operand */
      cons_append(&tail, popped);
    }
    else if(atom_q_function(popped)) { /* function */
      STK_add(&opSTK, popped);

      /* marker (NOP) for first function arg */
      cons_append(&tail, atom_make_code(CMD_NOP));
    }
    else if(popped->prolog == DOCODE 
	    && atom_get_code(popped) == CMD_xOPAREN) {
      STK_add(&opSTK, popped);
    }
    else if(popped->prolog == DOCODE
	    && atom_get_code(popped) == CMD_xCPAREN) {
      atom_unlink(popped);

      while(opSTK) {
	T_ATOM *op = STK_pop(&opSTK);

	if(op->prolog == DOCODE && atom_get_code(op) == CMD_xOPAREN) {
	  atom_unlink(op);

	  if(opSTK) {
	    T_ATOM *function = STK_pop(&opSTK);

	    if((function->prolog == DOFUNC) ||
	       ((function->prolog == DOCODE) && 
		(atom_get_code(function) != CMD_xOPAREN))) {
	      cons_append(&tail, function);
	    } else {
	      atom_unlink(function);
	    }
	  }
	  break;
	} else {
	  cons_append(&tail, op);
	}
      }
    } else {
      int me = __priority(atom_get_code(popped));

      while(opSTK && __priority(atom_get_code(STK_peek(opSTK, 0))) >= me) {
	cons_append(&tail, STK_pop(&opSTK));
      }

      if(! atom_q_comma(popped))
	STK_add(&opSTK, popped);
      else
	atom_unlink(popped);
    }
  }

  while(opSTK)
    cons_append(&tail, STK_pop(&opSTK));

  return rpn;
}

/* public, convert SYMB to (executable) PROG */

T_ATOM *symb_rpn(const T_SYMB *symb) {
  T_CONS *STK = NULL;
  size_t N = symb_meta(symb, &STK);
  T_ATOM *RPN = stk_meta2ob(&STK, N, DOPROG);

  while(STK) 
    STK_drop(&STK);

  return RPN;
}

/* Copy */
T_SYMB *symb_copy(const T_SYMB *symb) {
  if(symb) {
    T_SYMB *newsymb = symb_alloc();

    newsymb->ob = atom_copy(symb->ob);

    newsymb->car = symb_copy(symb->car);
    newsymb->cdr = symb_copy(symb->cdr);

    return newsymb;
  }

  return NULL;
}

int symb_q_zero(const T_SYMB *symb) {
  int zero = 0;

  if(symb)
    zero = atom_q_null(symb->ob);

  return zero;
}

int symb_q_one(const T_SYMB *symb) {
  int one = 0;

  if(symb) {
    const T_ATOM *ob = symb->ob;

    switch(atom_q_type(ob)) {
    case BINT:
      one = atom_get_bint(ob) == 1;
      break;

    case ZINT:
      if(1) {
	T_ZINT z;

	mpz_init(z);
	mpz_set(z, atom_get_zint(ob));
	one = (mpz_cmp_ui(z, 1)) == 0;
	mpz_clear(z);
      }
      break;

    case REAL:
      one = atom_get_real(ob) == 1;
      break;

    case CPLX:
      if(1) {
	T_CPLX z = atom_get_cplx(ob);

	one = (z.x == 1.0) && (z.y == 0.0);
      }
      break;

    default:
      break;
    }
  }

  return one;
}

/* Symplify SYMB */
T_SYMB *symb_reduce(const T_SYMB *symb) {
  if(symb) {
    T_SYMB *newsymb = NULL;
    T_SYMB *nleft = NULL, *nright = NULL;
    T_CODE op;
    int symb_OK;

    op = (atom_q_type(symb->ob) == CODE) ? atom_get_code(symb->ob) : 0;

    if(symb->car)
      nleft = symb_reduce(symb->car);

    if(symb->cdr)
      nright = symb_reduce(symb->cdr);

    symb_OK = 0;

    if(! symb_OK && op == CMD_xNEGATIVE) {
      if(symb_q_zero(nleft)) { /* -0 = 0 */
	newsymb = nleft;

	if(nright)
	  symb_unlink(nright);

	symb_OK = 1;
      } else if(atom_q_type(nleft->ob) == CODE &&
		atom_get_code(nleft->ob) == CMD_xNEGATIVE) { /* -(-x) = x */
	newsymb = symb_reduce(nleft->car);

	symb_unlink(nleft);

	if(nright)
	  symb_unlink(nright);

	symb_OK = 1;
      } else if(atom_q_type(nleft->ob) == CODE &&
		atom_get_code(nleft->ob) == CMD_xMINUS) { /* -(a-b) = b-a */
	newsymb = symb_alloc();
	newsymb->ob = atom_link(nleft->ob);

	newsymb->car = symb_reduce(nleft->cdr);
	newsymb->cdr = symb_reduce(nleft->car);

	symb_unlink(nleft);

	if(nright)
	  symb_unlink(nright);

	symb_OK = 1;
      }
    }

    if(! symb_OK && op == CMD_xPLUS) {
      if(symb_q_zero(nright)) { /* a+0 = a */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_zero(nleft)) { /* 0+a = a */
	newsymb = nright;
	symb_unlink(nleft);

	symb_OK = 1;
      }
    }

    if(! symb_OK && op == CMD_xMINUS) {
      if(symb_q_zero(nright)) { /* a-0 = a */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_zero(nleft)) { /* 0-a = NEG(a) */
	newsymb = symb_alloc();

	newsymb->ob = atom_make_code(CMD_xNEGATIVE);
	newsymb->car = nright;
	newsymb->cdr = NULL;

	symb_unlink(nleft);

	symb_OK = 1;
      }
    }

    if(! symb_OK && op == CMD_xTIMES) {
      if(symb_q_zero(nright)) { /* a*0 = 0 */
	newsymb = nright;
	symb_unlink(nleft);

	symb_OK = 1;
      } else if(symb_q_zero(nleft)) { /* 0*a = 0 */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_one(nright)) { /* a*1 = a */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_one(nleft)) { /* 1*a = a */
	newsymb = nright;
	symb_unlink(nleft);

	symb_OK = 1;
      } else if(atom_q_type(nleft->ob) == CODE && 
		atom_get_code(nleft->ob) == CMD_xNEGATIVE &&
		atom_q_type(nright->ob) == CODE && 
		atom_get_code(nright->ob) == CMD_xNEGATIVE) {
	/* (-a)*(-b) = a*b */

	newsymb = symb_alloc();
	newsymb->ob = atom_link(symb->ob); /* CMD_xTIMES */

	newsymb->car = symb_reduce(nleft->car);
	newsymb->cdr = symb_reduce(nright->car);

	symb_unlink(nleft);
	symb_unlink(nright);

	symb_OK = 1;
      }
    }

    if(! symb_OK && op == CMD_xDIVIDE) {
      if(symb_q_one(nright)) { /* a/1 = a */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else
      if(symb_q_zero(nleft)) { /* 0/a = 0 */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } 
      else if(atom_q_type(nleft->ob) == CODE && 
		atom_get_code(nleft->ob) == CMD_xNEGATIVE &&
		atom_q_type(nright->ob) == CODE && 
		atom_get_code(nright->ob) == CMD_xNEGATIVE) {
	/* (-a)/(-b) = a/b */

	newsymb = symb_alloc();
	newsymb->ob = atom_link(symb->ob); /* CMD_xDIVIDE */

	newsymb->car = symb_reduce(nleft->car);
	newsymb->cdr = symb_reduce(nright->car);

	symb_unlink(nleft);
	symb_unlink(nright);

	symb_OK = 1;
      }
    }

    if(! symb_OK && op == CMD_xPOW) {
      if(symb_q_zero(nright)) { /* a^0 = 1 */
	newsymb = symb_alloc();

	newsymb->ob = atom_make_real(1);

	symb_unlink(nleft);
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_zero(nleft)) { /* 0^a = 0 */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_one(nright)) { /* a^1 = a */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      } else if(symb_q_one(nleft)) { /* 1^a = 1 */
	newsymb = nleft;
	symb_unlink(nright);

	symb_OK = 1;
      }
    }

    if(! symb_OK) {
      newsymb = symb_alloc();

      newsymb->ob = atom_link(symb->ob);

      newsymb->car = nleft;
      newsymb->cdr = nright;

      symb_OK = 1;
    }

    return newsymb;
  }

  return NULL;
}

/* Simplify OBJECT */
T_ATOM* symb_simplify(const T_ATOM *ob) {
  T_ATOM *newob = atom_make_symb();

  atom_get_symb(newob) = symb_reduce(atom_get_symb(ob));

  return newob;
}
