#include <stdio.h>
#include <assert.h>

#include "console.h"

#include "object.h"
#include "mem.h"
#include "dump.h"
#include "xlib.h"
#include "main.h"
#include "stack.h"

/* Data Stack := data stack object */ 
T_CONS *DSTK;

/* Return Stack := return stack object */

T_CONS *ISTK[RSTK_MAX], *RSTK[RSTK_MAX], *IPTR;
int RSTK_LEVEL = -1, ISTK_LEVEL = -1;

/* Loop Stack */

T_LOOP LOOP[LOOP_MAX];
int LOOP_LEVEL;

void LOOP_add(signed long int Index, signed long int Stop) {
  if(LOOP_LEVEL + 1 < LOOP_MAX) {
    LOOP_LEVEL ++;

    LOOP[LOOP_LEVEL].Index = Index;
    LOOP[LOOP_LEVEL].Stop = Stop;
  } else
    Exception(NULL, "LOOP Stack Overflow", 0);
}

void LOOP_drop(void) {
  if(LOOP_LEVEL >= 0) {
    LOOP_LEVEL --;
  } else
    Exception(NULL, "LOOP Stack Underflow", 0);
}

signed long int LOOP_Index(size_t level) {
  if(LOOP_LEVEL - level >= 0)
    return LOOP[LOOP_LEVEL - level].Index;
  else
    Exception(NULL, "LOOP Stack Underflow", 0);
}

signed long int LOOP_Stop(size_t level) {
  if(LOOP_LEVEL - level >= 0)
    return LOOP[LOOP_LEVEL - level].Stop;
  else
    Exception(NULL, "LOOP Stack Underflow", 0);
}

/* Data Stack */

void DSTK_add(T_ATOM *o) { /* LIFO */
  register T_CONS *tos = cons_alloc();

  tos->cdr = DSTK;
  tos->car = o;

  DSTK = tos;
}

void DSTK_dup(void) {
  register T_CONS *tos = cons_alloc();

  tos->cdr = DSTK;
  tos->car = atom_link(DSTK->car);

  DSTK = tos;
}

void DSTK_roll(size_t n) {
  if(n > 0) {
    register T_CONS *level_0, *level_n1, *level_n;

    level_0 = DSTK;

    for(level_n1 = level_0; n > 1; n --)
      level_n1 = level_n1->cdr;

    level_n = level_n1->cdr;

    level_n1->cdr = level_n->cdr;
    level_n->cdr = level_0;

    DSTK = level_n;
  }
}

void DSTK_swap(void) {
  register T_CONS *level0, *level1;

  level0 = DSTK;
  level1 = level0->cdr;

  level0->cdr = level1->cdr;
  level1->cdr = level0;

  DSTK = level1;
}

void DSTK_unroll(size_t n) {
  if(n > 0) {
    T_CONS *level_0, *level_1, *level_n, *level_n1;

    level_0 = DSTK;
    level_n = level_0;
    level_1 = level_0->cdr;

    for( ; n > 0; n --)
      level_n = level_n->cdr;

    level_n1 = level_n->cdr;

    level_0->cdr = level_n1;
    level_n->cdr = level_0;

    DSTK = level_1;
  }
}

void DSTK_drop(void) {
  register T_CONS *tos = DSTK;

  DSTK = DSTK->cdr;

  tos->cdr = NULL;
  cons_unlink(tos);
}

T_ATOM *DSTK_pop(void) {
  register T_CONS *tos = DSTK;
  T_ATOM *ob = atom_link(tos->car);

  DSTK = tos->cdr;

  tos->cdr = NULL;
  cons_unlink(tos);

  return ob;
}

T_ATOM *DSTK_peek(size_t level) {
  register T_CONS *stk = DSTK;

  for(stk = DSTK; stk && (level > 0); level --)
    stk = stk->cdr;

  return stk ? stk->car : NULL;
}

void DSTK_dump(WINDOW *stream) {
  size_t maxlevel = 16;
  size_t depth = STK_depth(DSTK);

  if(depth > maxlevel)
    depth = maxlevel;

#ifdef __GNUC__
  T_ATOM *revSTK[depth];
#else
  T_ATOM **revSTK = calloc(depth, sizeof(T_ATOM *));
#endif
 
  size_t ndx;
  T_CONS *ptr;

  for(ptr = DSTK, ndx = 0;
      ptr && (ndx < depth); ndx ++, ptr = ptr->cdr) {

    revSTK[ndx] = ptr->car;
  }

  for( ; ndx > 0; ndx --) {
    xprintf(stream, " %zu: ", ndx);
    atom_dump(stream, revSTK[ndx - 1]);
    xprintf(stream, "\n");
  }

#ifndef __GNUC__
  free(revSTK);
#endif
}

/* runstream & return stack */

void ISTK_dump(WINDOW *stream, const T_CONS *rs) {
  for( ; rs; rs = rs->cdr) {
    atom_dump(stream, rs->car);

    waddch(FLAGS.video ? stream : stdscr, ' ');
    /*
    if(FLAGS.video)
      waddch(stream, ' ');
    else
      putchar(' ');
    */
  }

 
  waddch(FLAGS.video ? stream : stdscr, ';');
  /*
  if(FLAGS.video)
    waddch(stream, ';');
  else
    putchar(' ');
  */
}

void ISTK_add(T_CONS *rs) {
  if(ISTK_LEVEL + 1 < RSTK_MAX && RSTK_LEVEL + 1 < RSTK_MAX) {
    ISTK[++ ISTK_LEVEL] = rs;
    RSTK[++ RSTK_LEVEL] = IPTR;
    IPTR = rs;
  } else
    Exception(NULL, "RSTK Overflow", 0);
}

void RSTK_dup(void) {
  if(ISTK_LEVEL + 1 < RSTK_MAX && RSTK_LEVEL + 1 < RSTK_MAX) {
    RSTK[++ RSTK_LEVEL] = IPTR;
  } else
    Exception(NULL, "RSTK Overflow", 0);
}

void RSTK_pop(void) {
  if(RSTK_LEVEL >= 0)
    RSTK_LEVEL --;
  else Exception(NULL, "RSTK Underflow", 0);
}

T_CONS *RSTK_peek(void) {
  if(RSTK_LEVEL >= 0)
    return RSTK[RSTK_LEVEL];
  else Exception(NULL, "RSTK Underflow", 0);
  
  return NULL;
}

void ISTK_kill(void) {
  if(RSTK_LEVEL >= 0) {
    T_CONS *rs = ISTK[ISTK_LEVEL --];

    if(rs)
      cons_unlink(rs);

    IPTR = RSTK[RSTK_LEVEL --];
  } else
    Exception(NULL, "Empty RSTK", 0);
}

void RSTK_dump(void) {
  int level;

  for(level = RSTK_LEVEL; level >= 0; level --) {
    xprintf(win_cmd, "%3d: ", level);
    ISTK_dump(win_cmd, RSTK[level]);
    xprintf(win_cmd, "\n");
  }
}

/* generic for LIFO */

size_t STK_depth(const T_CONS *stk) {
  register size_t ndx;

  for(ndx = 0; stk; stk = stk->cdr)
    ndx ++;

  return ndx;
}

void STK_add(T_CONS **STK, T_ATOM *o) { /* LIFO */
  register T_CONS *tos = cons_alloc();

  tos->cdr = *STK;
  tos->car = o;

  *STK = tos;
}

void STK_drop(T_CONS **STK) {
  register T_CONS *tos = *STK;

  *STK = tos->cdr;

  tos->cdr = NULL;
  cons_unlink(tos);
}

T_ATOM *STK_pop(T_CONS **STK) {
  register T_CONS *tos = *STK;
  T_ATOM *ob = atom_link(tos->car);

  *STK = tos->cdr;

  tos->cdr = NULL;
  cons_unlink(tos);

  return ob;
}

T_ATOM *STK_peek(const T_CONS *STK, size_t level) {
  for( ; STK && (level > 0); level --)
    STK = STK->cdr;

  return STK->car;
}

void STK_dump(T_CONS *STK, WINDOW *stream) {
  size_t maxlevel = 8;
  size_t depth = STK_depth(STK);

  if(depth > maxlevel)
    depth = maxlevel;

#ifdef __GNUC__
  T_ATOM *revSTK[depth];
#else
  T_ATOM **revSTK = calloc(depth, sizeof(T_ATOM *));
#endif
 
  size_t ndx;
  T_CONS *ptr;

  for(ptr = STK, ndx = 0;
      ptr && (ndx < depth); ndx ++, ptr = ptr->cdr) {

    revSTK[ndx] = ptr->car;
  }

  for( ; ndx > 0; ndx --) {
    xprintf(stream, " %zu: ", ndx);
    atom_dump(stream, revSTK[ndx - 1]);
    xprintf(stream, "\n");
  }

#ifndef __GNUC__
  free(revSTK);
#endif
}

/* Helper (FUNCTION) */

int DSTK_CK1(T_ATOM *self, size_t T1) {
#ifndef __SAFE__
  return 1;
#else
  static char msg[64];

  if(__peek_type(0) & T1)
    return 1;

  sprintf(msg, "Expected [ 1:%04x ]", T1);
  Exception(self, msg, 0);

  return 0;
#endif

}

int DSTK_CK2(T_ATOM *self, size_t T2, size_t T1) {
#ifndef __SAFE__
  return 1;
#else
  static char msg[64];

  if((__peek_type(1) & T2) &&
     (__peek_type(0) & T1))
    return 1;

  sprintf(msg, "Expected [ 2:%04x 1:%04x ]", T2, T1);
  Exception(self, msg, 0);

  return 0;
#endif
}

int DSTK_CK3(T_ATOM *self, size_t T3, size_t T2, size_t T1) {
#ifndef __SAFE__
  return 1;
#else
  static char msg[64];

  if((__peek_type(2) & T3) &&
     (__peek_type(1) & T2) &&
     (__peek_type(0) & T1))
    return 1;

  sprintf(msg, "Expected [ 3:%04x 2:%04x 1:%04x ]", T3, T2, T1);
  Exception(self, msg, 0);

  return 0;
#endif
}

int DSTK_CKN(T_ATOM *self) {
#ifndef __SAFE__
  return 1;
#else
  static char msg[64];

  if(DSTK_CK1(self, BINT)) {
    const T_ATOM *o;
    T_BINT n;
    
    o = DSTK_peek(0);
    n = atom_get_bint(o);

    if(n < 0) {
      sprintf(msg, "%ld: Expected non-negative argument", n);
      Exception(self, msg, 0);
    }
    else if(STK_depth(DSTK) > n) {
      return 1;
    } else {
      sprintf(msg, "Expected (DEPTH > %lu)", n);
      Exception(self, msg, 0);
    }
  }

  return 0;
#endif
}

int DSTK_CKANY(T_ATOM *self, size_t N) {
#ifndef __SAFE__
  return N;
#else
  static char msg[64];

  if(STK_depth(DSTK) >= N) {
    return N;
  }

  sprintf(msg, "Expected (DEPTH >= %zu)", N);
  Exception(self, msg, 0);

  return 0;
#endif
}

T_BINT __pop_bint(T_BINT *b) {
  T_BINT x = 0;

  if(DSTK_CK1(NULL, BINT)) {
    register T_ATOM *ob = DSTK_pop();
    
    x = atom_get_bint(ob);
    atom_unlink(ob);
  }

  if(b)
    *b = x;

  return x;
}

void __pop_zint(T_ZINT *pz) {
  T_ZINT p;

  mpz_init_set_ui(p, 0);

  if(DSTK_CK1(NULL, ZINT)) {
    register T_ATOM *ob = DSTK_pop();

    mpz_set(p, atom_get_zint(ob));
    atom_unlink(ob);
  }

  if(pz) {
    /* mpz_init(*pz); */
    mpz_set(*pz, p);
  }

  mpz_clear(p);
}

T_BINT __pop_flag(T_BINT *q) {
  T_BINT q_true = 0;

  if(DSTK_CKANY(NULL, 1)) {
    register T_ATOM *ob  = DSTK_pop();
    
    q_true = ! atom_q_false(ob);
    atom_unlink(ob);
  }

  if(q)
    *q = q_true;

  return q_true;
}

T_CPLX __peek_cplx(size_t level) {
  T_CPLX z;

  z.x = z.y = 0;

  if(DSTK_CK1(NULL, CPLX))
    z = atom_get_cplx(DSTK->car);

  return z;
}

T_CPLX __pop_cplx(T_CPLX *pz) {
  T_CPLX z;

  z.x = z.y = 0;

  if(DSTK_CK1(NULL, CPLX)) {
    T_ATOM *ob = DSTK_pop();

    z = atom_get_cplx(ob);
    atom_unlink(ob);
  }

  if(pz)
    *pz = z;

  return z;
}

T_REAL __pop_real(T_REAL *px) {
  T_REAL x = 0.0;

  if(DSTK_CK1(NULL, REAL)) {
    T_ATOM *ob = DSTK_pop();

    x = atom_get_real(ob);
    atom_unlink(ob);
  }

  return x;
}
