#ifndef __OBJECT_H__
#define __OBJECT_H__

#include <sys/types.h>
#include <gmp.h>

struct atom_t;

typedef enum { 
  BINT =     1,
  REAL =     2,
  CPLX =     4,
  CSTR =    16,
  ZINT =    32,
  CHAR =    64,
  IDNT =   128,
  LAM  =   256,
  ARRY =   512,
  CODE =  1024,
  LIST =  4096,
  PROG =  8192,
  SYMB = 16384,
  FUNC = 32768,
} T_OBTYPE;

typedef void (T_PROLOG)(struct atom_t *);

/* object core */

typedef void *T_CODE;

typedef char T_CHAR;
typedef T_CHAR* T_NAME;
typedef T_NAME T_FUNC;

typedef mpz_t T_ZINT;

typedef signed long T_BINT;
typedef double T_REAL;

typedef struct {
  T_REAL x, y;
} T_CPLX;

typedef struct {
  unsigned char dims;
  unsigned long *dim;
  struct atom_t **data; /* will be dinamically allocated */
} T_ARRY;

typedef struct cons_t {
  struct atom_t *car;
  struct cons_t *cdr;
} T_CONS;

typedef struct symb_t {
  struct atom_t *ob;

  struct symb_t *car;
  struct symb_t *cdr;
} T_SYMB;

/* data (DSTK, ISTK) object */

union core_t {
  T_CODE  code;

  T_BINT  bint;
  T_REAL  real;
  T_CPLX  cplx;
  T_ZINT  zint;

  T_CHAR  chr ;
  T_NAME  name; /* pointer */

  T_CONS* cons;
  T_SYMB* symb;

  T_ARRY  arry;
};

typedef struct atom_t {
  T_PROLOG *prolog;
  union core_t u;
} T_ATOM;

/* stored word (WORD) */
typedef struct word_t {
  struct word_t *next;

  T_NAME id;
  T_ATOM *ob;
} T_WORD;

/* lambda environment (FLDR) */
typedef struct fldr_t {
  struct fldr_t *parent;

  T_NAME  id;
  T_WORD *fst;
} T_FLDR;

/* Prolog */

void DOBINT(T_ATOM *);
void DOREAL(T_ATOM *);
void DOCPLX(T_ATOM *);
void DOARRY(T_ATOM *);
void DOCSTR(T_ATOM *);
void DOZINT(T_ATOM *);
void DOCHAR(T_ATOM *);
void DOLIST(T_ATOM *);
void DOSYMB(T_ATOM *); void DOSYMBEVAL(T_ATOM *); // prolog for COMPEVAL
void DOIDNT(T_ATOM *);
void DOLAM (T_ATOM *);
void DOCODE(T_ATOM *);
void DOPROG(T_ATOM *);
void DOFUNC(T_ATOM *);

/* Object building */

T_ATOM* atom_make_bint(T_BINT n);
T_ATOM* atom_make_real(T_REAL x);
T_ATOM* atom_make_cplx(T_REAL x, T_REAL y);
T_ATOM* atom_make_zint(T_ZINT s);

T_ATOM* atom_make_code(T_CODE code);

T_ATOM* atom_make_char(T_CHAR c);

T_ATOM* atom_make_idnt(const T_NAME s);
T_ATOM* atom_make_lam (const T_NAME s);
T_ATOM* atom_make_cstr(const T_NAME s);

T_ATOM *atom_make_name(const T_NAME s, size_t len);
T_ATOM *atom_make_func(const T_NAME s, size_t len);

T_ATOM *atom_make_flag(int flag);
T_ATOM* atom_make_cons(T_PROLOG *prolog);
T_ATOM* atom_make_symb(void);

T_ATOM *atom_make_arry(unsigned long dims, 
		       unsigned long *dim,
		       unsigned long *total,
		       T_ATOM ***data);

/* Object fetching */

#define atom_get_bint(a) ((a)->u.bint)
#define atom_get_real(a) ((a)->u.real)
#define atom_get_cplx(a) ((a)->u.cplx)
#define atom_get_zint(a) ((a)->u.zint)
#define atom_get_name(a) ((a)->u.name)
#define atom_get_char(a) ((a)->u.chr)
#define atom_get_code(a) ((a)->u.code)
#define atom_get_cons(a) ((a)->u.cons)
#define atom_get_symb(a) ((a)->u.symb)
#define atom_get_arry(a) (&((a)->u.arry))

/*
T_BINT  atom_get_bint(const T_ATOM *);
T_REAL  atom_get_real(const T_ATOM *);
T_CPLX  atom_get_cplx(const T_ATOM *);
T_ZINT  atom_get_zint(const T_ATOM *);

T_NAME  atom_get_name(const T_ATOM *);
T_CHAR  atom_get_char(const T_ATOM *);

T_CODE  atom_get_code(const T_ATOM *);

T_CONS* atom_get_cons(const T_ATOM *);
T_SYMB* atom_get_symb(const T_ATOM *);

T_ARRY* atom_get_arry(T_ATOM *);
*/

/* Object testing */

int atom_q_function(const T_ATOM *);  /* function? */
int atom_q_nop(const T_ATOM *);       /* NOP? */
int atom_q_comma(const T_ATOM *);     /* comma? */
int atom_q_atomic(const T_ATOM *);    /* atomic? */
int atom_q_exec(const T_ATOM *);      /* executable? */
int atom_q_cons(const T_ATOM *);      /* composite? */
int atom_q_matrix(const T_ATOM *);    /* matrix? */
int atom_q_name(const T_ATOM *);      /* text-like? */
int atom_q_number(const T_ATOM *);    /* numeric? */
int atom_q_null(const T_ATOM *);      /* null object? */
int atom_q_false(const T_ATOM *);     /* false object? */
int atom_q_zint(const T_ATOM *);      /* zint? */

T_OBTYPE  atom_q_type(const T_ATOM *);
T_OBTYPE  q_prolog_type(const T_PROLOG *);

// const T_PROLOG *q_type_prolog(T_OBTYPE);

/* Convert to string */

char *bint_string(T_BINT);
char *zint_string(T_ZINT);
char *real_string(T_REAL);
char *cplx_string(T_CPLX);
char *atom_string(const T_ATOM *, char **);

/* Add object to composite */

T_CONS *cons_add(T_CONS *tail, T_ATOM *o);

/* Object (deep) copy & compare */

T_ATOM *atom_copy(const T_ATOM *);
int atom_equal(const T_ATOM *, const T_ATOM *);

/* Object evaluation */

void atom_eval(T_ATOM *, T_PROLOG *);

#endif
