#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "console.h"
#include "gstack.h"
#include "debug.h"

#include "object.h"
#include "mem.h"
#include "dump.h"
#include "xlib.h"
#include "cmd.h"
#include "main.h"
#include "symb.h"

T_ATOM *atom_make_bint(T_BINT n) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOBINT;
  ob->u.bint = n;

  return ob;
}

T_ATOM *atom_make_real(T_REAL x) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOREAL;
  ob->u.real = x;

  return ob;
}

T_ATOM *atom_make_cplx(T_REAL x, T_REAL y) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOCPLX;
  ob->u.cplx.x = x;
  ob->u.cplx.y = y;

  return ob;
}

T_ATOM *atom_make_name(const T_NAME s, size_t len) { 
  T_NAME id;
  T_ATOM *ob;

  id = xm_alloc(len + 1);
  strncpy(id, s, len);
  id[len] = '\0';

  ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOIDNT;
  ob->u.name = id;

  return ob;
}

T_ATOM *atom_make_func(const T_NAME s, size_t len) { 
  T_NAME id;
  T_CODE code;
  T_ATOM *ob;

  id = xm_alloc(len + 2);
  id[0] = 'x';
  id[1] = '\0';
  strncpy(id + 1, s, len);
  id[len + 1] = '\0';

  ob = atom_alloc();

  code = code_find(id);
  if(code) {
    ob->prolog = (T_PROLOG *)DOCODE;
    ob->u.code = code;
  } else {
    ob->prolog = (T_PROLOG *)DOFUNC;
    ob->u.name = xm_strdup(id + 1);
  }

  xm_free(id);
  return ob;
}

T_ATOM *atom_make_idnt(const T_NAME s) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOIDNT;
  ob->u.name = xm_strdup(s);

  return ob;
}

T_ATOM *atom_make_lam (const T_NAME s) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOLAM;
  ob->u.name = xm_strdup(s);

  return ob;
}

T_ATOM *atom_make_cstr(const T_NAME s) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOCSTR;
  ob->u.name = xm_strdup(s);

  return ob;
}

T_ATOM *atom_make_zint(T_ZINT n) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOZINT;
  mpz_init_set(ob->u.zint, n);

  return ob;
}

T_ATOM *atom_make_char(T_CHAR c) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOCHAR;
  ob->u.chr = c;

  return ob;
}

T_ATOM *atom_make_code(T_CODE code) { 
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOCODE;
  ob->u.code = code;

  return ob;
}

T_ATOM *atom_make_flag(int flag) {
  return atom_make_code(flag ? CMD_TRUE : CMD_FALSE);
}

T_ATOM* atom_make_cons(T_PROLOG *prolog) {
  T_ATOM *ob = atom_alloc();

  ob->prolog = prolog;
  return ob;
}

T_ATOM* atom_make_symb(void) {
  T_ATOM *ob = atom_alloc();

  ob->prolog = (T_PROLOG *)DOSYMB;
  return ob;
}

T_ATOM* atom_make_arry(unsigned long dims,
		       unsigned long *dim,
		       unsigned long *total,
		       T_ATOM ***data) {
  T_ATOM *ob = atom_alloc();
  T_ARRY *arry;
  unsigned long count;

  ob->prolog = (T_PROLOG *)DOARRY;
  arry = atom_get_arry(ob);

  arry->dims = dims;
  count = 0;

  if(dims > 0) {
    unsigned long j;

    if(total)
      count = *total;

    arry->dim = xm_calloc(dims, sizeof(*dim));
    for(j = 0; j < dims; j ++)
      arry->dim[j] = dim[j];

    if(count == 0)
      for(count = 1, j = 0; j < dims; j ++)
	count *= dim[j];

    arry->data = xm_calloc(count, sizeof(T_ATOM*));
  }

  if(total)
    *total = count;

  if(data)
    *data = arry ? arry->data : NULL;

  return ob;
}

/* object type */

T_OBTYPE atom_q_type(const T_ATOM *ob) {
  return ob ? q_prolog_type((const T_PROLOG *)ob->prolog) : 0;
}

T_OBTYPE q_prolog_type(const T_PROLOG *p) {
  if(p == NULL)
    return 0;
  else if(p == DOBINT)
    return BINT;
  else if(p == DOREAL)
    return REAL;
  else if(p == DOCPLX)
    return CPLX;
  else if(p == DOCSTR)
    return CSTR;
  else if(p == DOZINT)
    return ZINT;
  else if(p == DOCHAR)
    return CHAR;
  else if(p == DOIDNT)
    return IDNT;
  else if(p == DOFUNC)
    return FUNC;
  else if(p == DOLAM)
    return LAM;
  else if(p == DOCODE)
    return CODE;
  else if(p == DOARRY)
    return ARRY;
  else if(p == DOSYMB)
    return SYMB;
  else if(p == DOLIST)
    return LIST;
  else if(p == DOPROG)
    return PROG;

  return 0;
}

/*
const T_PROLOG *q_type_prolog(T_OBTYPE t) {
  switch(t) {
  case BINT: return (const T_PROLOG *)DOBINT;
  case REAL: return (const T_PROLOG *)DOREAL;
  case CPLX: return (const T_PROLOG *)DOCPLX;
  case CSTR: return (const T_PROLOG *)DOCSTR;
  case ZINT: return (const T_PROLOG *)DOZINT;
  case CHAR: return (const T_PROLOG *)DOCHAR;
  case IDNT: return (const T_PROLOG *)DOIDNT;
  case FUNC: return (const T_PROLOG *)DOFUNC;
  case LAM : return (const T_PROLOG *)DOLAM ;
  case CODE: return (const T_PROLOG *)DOCODE;
  case ARRY: return (const T_PROLOG *)DOARRY;
  case SYMB: return (const T_PROLOG *)DOSYMB;
  case LIST: return (const T_PROLOG *)DOLIST;
  case PROG: return (const T_PROLOG *)DOPROG;
  default:
    return NULL;
  }

  return NULL;
}
*/

/* function? */
int atom_q_function(const T_ATOM *ob) {
  int function = 0;

  if(ob) {
    switch(atom_q_type(ob)) {
    case FUNC:
      function = 1;
      break;

    case CODE:
      function = code_function(atom_get_code(ob));
      break;
      
    default:
      function = 0;
      break;
    }
  }

  return function;
}

int atom_q_comma(const T_ATOM *ob) {
  return ob && ob->prolog == DOCODE && atom_get_code(ob) == CMD_xCOMMA 
    ? 1 : 0;
}

int atom_q_nop(const T_ATOM *ob) {
  return ob && ob->prolog == DOCODE && atom_get_code(ob) == CMD_NOP 
    ? 1 : 0;
}

/* atomic? */
int atom_q_atomic(const T_ATOM *ob) {
  switch(atom_q_type(ob)) {
  case ZINT:
  case CSTR:
  case IDNT:
  case FUNC:
  case LAM:
  case ARRY:
  case SYMB:
  case LIST:
  case PROG:
    return 0;

  default:
    return 1;
  }

  return 1;
}

/* exectable? */
int atom_q_exec(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOCODE || t == DOFUNC) ? 1 : 0 ;
  }

  return 0;
}

/* composite? */
int atom_q_cons(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOLIST || t == DOPROG) ? 1 : 0 ;
  }

  return 0;
}

/* text? */
int atom_q_name(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOCSTR || t == DOIDNT || t == DOLAM) ? 1 : 0 ;
  }

  return 0;
}

/* number? */
int atom_q_number(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOBINT || t == DOREAL
       || t == DOCPLX
       || t == DOZINT) ? 1 : 0 ;
  }

  return 0;
}

/* matrix? */
int atom_q_matrix(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOARRY) ? 1 : 0 ;
  }

  return 0;
}

/* zint? */
int atom_q_zint(const T_ATOM *ob) {
  if(ob) {
    const T_PROLOG *t = (const T_PROLOG *)ob->prolog;

    return 
      (t == DOZINT) ? 1 : 0 ;
  }

  return 0;
}

int symb_equal(const T_SYMB *s1, const T_SYMB *s2) {
  int flag = 0;

  if(s1 == s2)
    return 1;

  if((s1 && s2 == NULL) || (s2 && s1 == NULL))
    return 0;

  flag = atom_equal(s1->ob, s2->ob) &&
    symb_equal(s1->car, s2->car) &&
    symb_equal(s1->cdr, s2->cdr);

  return flag;
}

/* equal? */
int atom_equal(const T_ATOM *a1, const T_ATOM *a2) {
  int flag = 0;

  if(a1 == a2)
    return 1;

  if(a1 && a2 && (a1->prolog == a2->prolog)) {
    switch(atom_q_type(a1)) {
    case BINT:
      flag = atom_get_bint(a1) == atom_get_bint(a2);
      break;

    case ZINT:
      flag = mpz_cmp(atom_get_zint(a1), atom_get_zint(a2)) == 0;
      break;

    case REAL:
      flag = atom_get_real(a1) == atom_get_real(a2);
      break;

    case CPLX:
      if(1) {
	T_CPLX c1, c2;

	c1 = atom_get_cplx(a1);
	c2 = atom_get_cplx(a2);

	flag = (c1.x == c2.x) && (c1.y == c2.y);
      }
      break;

    case CHAR:
      flag = atom_get_char(a1) == atom_get_char(a2);
      break;

    case CODE:
      flag = atom_get_code(a1) == atom_get_code(a2);
      break;

    case CSTR:
    case IDNT:
    case LAM:
    case FUNC:
      if(1) {
	const char *s1, *s2;
 
	s1 = atom_get_name(a1);
	s2 = atom_get_name(a2);

	flag = (s1 == s2) || !xstrcmp(s1, s2);
      }
      break;

    case LIST:
    case PROG:
      if(1) {
	T_CONS *seq1, *seq2;

	seq1 = atom_get_cons(a1);
	seq2 = atom_get_cons(a2);

	for(flag = 1; flag && seq1 && seq2; ) {
	  flag = atom_equal(seq1->car, seq2->car);
	  seq1 = seq1->cdr, seq2 = seq2->cdr;
	}

	if(flag)
	  flag = (seq1 == NULL) && (seq2 == NULL);
      }
 
      break;

    case ARRY:
      if(1) {
	const T_ARRY *ar1, *ar2;

	ar1 = atom_get_arry((T_ATOM *)a1);
	ar2 = atom_get_arry((T_ATOM *)a2);

	if(ar1->dims == ar2->dims) {
	  unsigned long total = 1;
	  unsigned long ndx, dims = ar1->dims;

	  for(flag = 1, ndx = 0; flag && (ndx < dims); ndx ++) {
	    flag = ar1->dim[ndx] == ar2->dim[ndx];

	    if(flag)
	      total *= ar1->dim[ndx];
	  }

	  if(flag) { /* same shape, compare elements */
	    for(ndx = 0; flag && (ndx < total); ndx ++)
	      flag = atom_equal(ar1->data[ndx], ar2->data[ndx]);
	  }
	}
      }
      break;

    case SYMB:
      flag = symb_equal(atom_get_symb(a1), atom_get_symb(a2));
      break;
    }
  }

  return flag;
}

/* null? */
int atom_q_null(const T_ATOM *ob) {
  if(ob == NULL) {
    return 1;
  }
  else if(ob->prolog == DOBINT) {
    return atom_get_bint(ob) == 0;
  }
  else if(ob->prolog == DOREAL) {
    return atom_get_real(ob) == 0;
  }
  else if(ob->prolog == DOCPLX) {
    T_CPLX z = atom_get_cplx(ob);

    return (z.x == 0.0) && (z.y == 0);
  }
  else if(ob->prolog == DOIDNT ||
	  ob->prolog == DOFUNC ||
	  ob->prolog == DOLAM ||
	  ob->prolog == DOCSTR) {
    T_NAME s = atom_get_name(ob);

    return !(s && *s);
  }
  else if(ob->prolog == DOCODE) {
    return 0;
  }
  else if(ob->prolog == DOARRY) {
    T_ARRY *arry = atom_get_arry((T_ATOM *)ob);

    return (arry == NULL || arry->dims == 0);
  }
  else if(ob->prolog == DOSYMB) {
    T_SYMB *symb = atom_get_symb(ob);

    return symb_q_zero(symb);
  }
  else if(atom_q_cons(ob)) {
    T_CONS *prog = atom_get_cons(ob);

    return (prog == NULL);
  }

  return 0;
}

/* false? */
int atom_q_false(const T_ATOM *ob) {
  if(ob == NULL) {
    return 1;
  }
  else if(ob->prolog == DOCODE) {
    return atom_get_code(ob) == CMD_FALSE;
  }

  return atom_q_null(ob);
}

/* object quick evaluation */
void atom_quick_eval(T_ATOM *ob, T_PROLOG *prolog) {
  register T_PROLOG *e_prolog = prolog ? prolog : ob->prolog;
  (e_prolog)(ob);
}

/* object evaluation */
void atom_eval(T_ATOM *ob, T_PROLOG *prolog) {
  if(ob) {
    T_PROLOG *e_prolog = prolog ? prolog : ob->prolog;

    if(e_prolog) {
      int mode = KEY_ENTER;

      if(FLAGS.mod_run == M_TRC || FLAGS.mod_run == M_DBG) {
	T_CONS *rs;
	char *obs;

	obs = atom_string(ob, NULL);
	gstack();
	set_footer(obs);
	free(obs);

	for(rs = IPTR; rs; rs = rs->cdr) {
	  obs = atom_string(rs->car, NULL);
	  add_footer(" ");
	  add_footer(obs);
	  free(obs);
	}
      }

      if(FLAGS.mod_run == M_TRC) {
	micro_delay(10000); /* 10 ms */
      }

      if(FLAGS.mod_run == M_DBG)
	mode = atom_debug(IPTR);

      if(mode == KEY_ENTER) { /* exec prolog */
	(e_prolog)(ob);
      }

    } else
      Exception(ob, "EVAL: Null Prolog", 0);
  } else
    Exception(NULL, "EVAL: Null Object", 0);
}

/* object (deep) copy */

T_ATOM *atom_copy(const T_ATOM *ob) {
  T_ATOM *newob = NULL;

  if(ob) {
    newob = atom_alloc();
    newob->prolog = ob->prolog;

    if(atom_q_atomic(ob)) { /* atom, memcpy() OK */
      newob->u = ob->u;
    }
    else if(ob->prolog == DOZINT) {
      mpz_init_set(newob->u.zint, atom_get_zint(ob));
    }
    else if(ob->prolog == DOSYMB) {
      newob->u.symb = symb_copy(atom_get_symb(ob));
    }
    else if(atom_q_cons(ob)) {
      T_ATOM *popped;
      T_CONS **dptr, *sptr;

      dptr = &(atom_get_cons(newob)); *dptr = NULL;

      for(sptr = atom_get_cons(ob); sptr &&
	    (popped = atom_copy(sptr->car)) != NULL; sptr = sptr->cdr) {
	*dptr = cons_alloc();

	(*dptr)->car = popped;
	(*dptr)->cdr = NULL;

	dptr = &((*dptr)->cdr);
      }
    }
    else if(atom_q_matrix(ob)) {
    }
    else if(atom_q_name(ob)) {
      newob->u.name = xm_strdup(atom_get_name(ob));
    }
    else
      assert(1);
  }

  return newob;
}

char *zint_string(T_ZINT x) {
  char *buf = NULL;

  switch(FLAGS.mod_bint) {
  case MOD_HEX:
    gmp_asprintf(&buf, "%#Zx", x);
    break;

  case MOD_OCT: 
    gmp_asprintf(&buf, "%#Zo", x);
    break;

  case MOD_BIN:
    buf = mpz_get_str(NULL, 2, x);
    break;

  case MOD_DEC:
  default:
    gmp_asprintf(&buf, "%Zd", x);
    break;
  }

  return buf;
}

char *bint_string(T_BINT x) {
  char *buf = NULL;

  switch(FLAGS.mod_bint) {
  case MOD_HEX:
    buf = malloc(32);
    sprintf(buf, "# %#lx", x);
    break;

  case MOD_DEC:
    buf = malloc(32);
    sprintf(buf, "# %ld", x);
    break;

  case MOD_OCT:
    buf = malloc(32);
    sprintf(buf, "# %#lo", x);
    break;

  case MOD_BIN:
    buf = malloc(130);

    if(1) {
      char *res, *ptr = &buf[129];
      int sign = 1;
      
      *ptr = '\0';

      if(x) {
	if(x < 0) {
	  sign = -1;
	  x = -x;
	}

	while(x > 0) {
	  *(-- ptr) = x%2 ? '1' : '0';
	  x >>= 1;
	}

	if(sign < 0)
	  *(-- ptr) = '-';
      } else {
	*(-- ptr) = '0';
      }

      *(-- ptr) = ' ';
      *(-- ptr) = '#';

      res = strdup(ptr);
      free(buf);

      buf = res;
    }
    break;
  }

  return buf;
}

char *real_string(T_REAL x) {
  char *buf = malloc(64);

  sprintf(buf, "%% %.16g", x);

  return buf;
}

char *cplx_string(T_CPLX z) {
  char *buf = malloc(64);
  double a, b;

  switch(FLAGS.mod_cplx) {
  case MOD_RECT:
    a = z.x;
    b = z.y;

    if(a && b)
      sprintf(buf, "C%% %.16g+%.16g*j", a, b);
    else if(a)
      sprintf(buf, "C%% %.16g", a);
    else if(b)
      sprintf(buf, "C%% %.16g*j", b);
    else
      sprintf(buf, "C%% 0");

    break;

  case MOD_POLAR:
    a = hypot(z.x, z.y);
    b = atan2(z.y, z.x);

    if(FLAGS.mod_angle == MOD_DEG)
      b *= 45.0/atan(1.0);

    if(a && b)
      sprintf(buf, "C%% %.16g@%.16g%s", a, b,
	      FLAGS.mod_angle == MOD_DEG ? " deg" : "");
    else if(a)
      sprintf(buf, "C%% %.16g", a);
    else
      sprintf(buf, "C%% 0");
    break;
  }

  return buf;
}

char *cons_string(T_OBTYPE t, const T_CONS *ob, char **stored) {
  if(! **stored) {
    *stored = realloc(*stored, strlen(*stored) + 
		      t == LIST ? 2 : 3);
    strcat(*stored, t == LIST ? "{" : "::");
  }

  if(ob) {
    char *word = atom_string(ob->car, &word);

    if(**stored) {
      *stored = realloc(*stored, strlen(*stored) + 2);
      strcat(*stored, " ");
    }

    *stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
    strcat(*stored, word);

    free(word);

    return cons_string(t, ob->cdr, stored);
  } else {
    *stored = realloc(*stored, strlen(*stored) + 3);
    strcat(*stored, t == LIST ? " }" : " ;");
  }

  return *stored;
}

char *arry_string(char **stored,
               unsigned long base,
               unsigned long dims,
               unsigned long *dim,
               T_ATOM **data,
               unsigned long *atom_ndx) {
  unsigned long ndx;

  *stored = realloc(*stored, strlen(*stored) + 2);
  strcat(*stored, "[");
  
  for(ndx = 0; ndx < dim[base]; ndx ++) {
    *stored = realloc(*stored, strlen(*stored) + 2);
    strcat(*stored, " ");

    if(base + 1 < dims) {
      *stored = arry_string(stored, base + 1, dims, dim, data, atom_ndx);
    } else {
      char *word = atom_string(data[(*atom_ndx) ++], &word);

      *stored = realloc(*stored, strlen(*stored) + strlen(word) + 1);
      strcat(*stored, word);
    }
  }

  *stored = realloc(*stored, strlen(*stored) + 3);
  strcat(*stored, " ]");

  return *stored;
}

char *atom_string(const T_ATOM *ob, char **stored) {
  char *buf = NULL;

  if(ob) {
    T_OBTYPE t = q_prolog_type((const T_PROLOG *)ob->prolog);

    switch(t) {
    case BINT:
      buf = bint_string(atom_get_bint(ob));
      break;

    case ZINT:
      buf = zint_string(atom_get_zint(ob));
      break;

    case REAL:
      buf = real_string(atom_get_real(ob));
      break;

    case CPLX:
      buf = cplx_string(atom_get_cplx(ob));
      break;

    case CHAR:
      buf = malloc(4);
      sprintf(buf, "%c", atom_get_char(ob));
      break;

    case CODE:
      if(1) {
	const char *id = code_getid(atom_get_code(ob));

	buf = strdup(id + ((*id == 'x') ? 1 : 0));
      }
      break;

    case ARRY:
      if(1) {
	T_ARRY *arry;
	T_ATOM **data;
	unsigned long dims;
	unsigned long *dim;

	arry = atom_get_arry((T_ATOM *)ob);

	dims = arry->dims;
	dim = arry->dim;
	data = arry->data;

	if(dims > 0) {
	  unsigned long atom_ndx = 0;

	  buf = malloc(256);
	  buf[0] = '\0';

	  arry_string(&buf, 0, dims, dim, data, &atom_ndx);
	} else {
	  buf = malloc(4);
	  strcpy(buf, "[ ]");
	}
      }
      break;

    case LIST:
    case PROG:
      buf = malloc(1);
      buf[0] = '\0';

      buf = cons_string(t, atom_get_cons(ob), &buf);
      break;

    case CSTR:
    case IDNT:
    case LAM:
      if(1) {
	const char *id = atom_get_name(ob);

	buf = malloc(strlen(id) + 1);
	sprintf(buf, "%s", id);
      }

      break;

    case FUNC:
      if(1) {
	const char *id = atom_get_name(ob);

	buf = malloc(strlen(id) + 1);
	sprintf(buf, "%s", id);
      }

      break;

    case SYMB:
      buf = symb_expr(atom_get_symb(ob));
      break;

    default:
      buf = strdup("<External>");
    }
  }

  if(stored)
    *stored = buf;

  return buf;
}

T_CONS *cons_add(T_CONS *tail, T_ATOM *o) {
  T_CONS *ntail = cons_alloc();

  ntail->car = o;
  ntail->cdr = NULL;

  if(tail)
    tail->cdr = ntail;

  return ntail;
}
