#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <stdlib.h>
#include <assert.h>

#include "console.h"

#include "object.h"
#include "stack.h"
#include "lambda.h"
#include "mem.h"
#include "xlib.h"
#include "main.h"
#include "dump.h"
#include "symb.h"

#ifndef __MAX_MEM__
#define __MAX_MEM__ 16384 /* ~576Kb */
#endif

/* for xm_alloc, etc. */

size_t xm_leak;

/* chained memory pool of objects */

#ifdef __STATZ__
struct _statz_t {
  size_t count, total, maxim;
  size_t moved;
} ;
#endif

#define MEM_NULL 0
#define MEM_ATOM 1
#define MEM_CONS 2
#define MEM_SYMB 3
#define MEM_WORD 4
#define MEM_FLDR 5

typedef struct _block_t {
  unsigned long T:4;
  unsigned long R:28;

  union {
    int any;

    T_ATOM atom;
    T_CONS cons;
    T_SYMB symb;
    T_WORD word;
    T_FLDR fldr;
  } data;
} T_BLOCK;

ptrdiff_t REF_DELTA;

static inline T_BLOCK *__self(void *data) {
  return (T_BLOCK *)(data - REF_DELTA);
}

static inline void *__unself(T_BLOCK *block) {
  return (void *)block + REF_DELTA;
}

typedef struct _mem_t {
  struct _mem_t *next; /* if needed */

  size_t mem;
  T_BLOCK *base;
  T_BLOCK *cache;
  T_BLOCK *mark;

#ifdef __STATZ__
  struct _statz_t statz;
#endif
} T_MEM;

T_MEM *MEMORY;

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))

void xm_init(void) {
  xprintf(win_cmd, "Initalizing memory.. ");

  MEMORY = malloc(sizeof(*MEMORY));

  MEMORY->next = NULL;
  MEMORY->mem = __MAX_MEM__;
  MEMORY->base = calloc(MEMORY->mem, sizeof(MEMORY->base[0]));
  MEMORY->mark = MEMORY->base + MEMORY->mem;
  MEMORY->cache = MEMORY->base;

#ifdef __STATZ__
  MEMORY->statz.count = 0;
  MEMORY->statz.total = 0;
  MEMORY->statz.maxim = 0;
  MEMORY->statz.moved = 0;
#endif

  xprintf(win_cmd, "OK, %zu %zu-byte objects\n",
	 MEMORY->mem,
	 sizeof(MEMORY->base[0]));

  xm_leak = 0;

  /* calculate REF_DELTA */

  if(REF_DELTA == 0) {
    T_BLOCK R;

    REF_DELTA = (void *)&(R.data.any) - (void *)&R;
  }

  folder_initialize();
}

void xm_statz(int leak) {
  size_t num[32], total;
  T_BLOCK *block;

  memset(num, 0, sizeof(num));
  total = 0;

  for(block = MEMORY->base; block < MEMORY->mark; block ++)
    if(block->R > 0) {
      num[block->T] ++;
      total ++;
    }

  if(leak) {
    if(total)
      xprintf(win_cmd, "Leaked (%zu):", total);
    else
      xprintf(win_cmd, "No leak");
  } else
    xprintf(win_cmd, "Memory:");

  if(leak && num[MEM_NULL])
    xprintf(win_cmd, " NULL = %zu", num[MEM_NULL]);

  if(!leak || num[MEM_ATOM])
    xprintf(win_cmd, " ATOM = %zu", num[MEM_ATOM]);

  if(!leak || num[MEM_CONS])
    xprintf(win_cmd, " CONS = %zu", num[MEM_CONS]);

  if(!leak || num[MEM_SYMB])
    xprintf(win_cmd, " SYMB = %zu", num[MEM_SYMB]);

  if(!leak || num[MEM_WORD])
    xprintf(win_cmd, " WORD = %zu", num[MEM_WORD]);

  if(!leak || num[MEM_FLDR])
    xprintf(win_cmd, " FLDR = %zu", num[MEM_FLDR]);

  xprintf(win_cmd, "\n");

#ifdef __STATZ__
  if(! leak) {
    struct _statz_t *t = &(MEMORY->statz);

    xprintf(win_cmd, "Object:");
    xprintf(win_cmd, " used = %zu", t->count);
    xprintf(win_cmd, " total = %zu", t->total);
    xprintf(win_cmd, " max = %zu", t->maxim);
    xprintf(win_cmd, " seek = %zu (avg = %.2g)\n", t->moved,
	   (double)(t->moved)/(t->total));
  }
#endif

  if(!leak || xm_leak) {
    xprintf(win_cmd, "Dynamic: %zu blocks\n", xm_leak);
  }

  video_update(0);
}

void xm_end(void) {
  xm_statz(0);

  /* clear HOME, LAM, RUNSTREAM, LOOP & DSTK */

  fldr_unlink(FOLDER_HOME);

  while(FOLDER_LAMBDA)
    lam_kill();

  while(RSTK_LEVEL >= 0) 
    ISTK_kill();

  while(LOOP_LEVEL >= 0)
    LOOP_drop();

  while(DSTK)
    DSTK_drop();

  /* shutdown memory system */

  xm_statz(1);

  free(MEMORY->base);
  free(MEMORY);
}

void *xm_calloc(size_t n, size_t z) {
  xm_leak ++;

  return calloc(n, z);
}

void *xm_alloc(size_t z) {
  return xm_calloc(1, z);
}

char *xm_strdup(const char *s) {
  if(s) {
    char *p = xm_alloc(strlen(s) + 1);

    return strcpy(p, s);
  }

  return NULL;
}

void xm_free(void *p) {
  if(p) {
    free(p);
    xm_leak --;
  }
}

void mem_clear(T_BLOCK *b) {
#ifdef __MALLOC__
  xm_free(b);
#else
  if(b) {
    memset(b, 0, sizeof(*b)); /* not really needed.. */

#ifdef __STATZ__
    MEMORY->statz.count --;
#endif
  }
#endif
}

size_t mem_avail(void) {
  size_t ref;
  T_BLOCK *block;

  for(ref = 0, block = MEMORY->base; block < MEMORY->mark; block ++)
    if(block->R > 0)
      ref ++;

  return (MEMORY->mem - ref);
}

void *mem_alloc(T_MEM *MEM, unsigned long T) {
  T_BLOCK *block = NULL;
  int found = 0;

  assert(MEM);
  assert(MEM->base);
  assert(MEM->mark);

#ifndef __MALLOC__
  assert(MEM->cache >= MEM->base && MEM->cache < MEM->mark);
#endif

#ifdef __MALLOC__

  block = xm_alloc(sizeof(*block));
  found = block == NULL ? 0 : 1;

#else

  for(block = MEM->cache; block < MEM->mark; block ++) {
    if(block->R == 0) {
      found = 1;
      break;
    }

#ifdef __STATZ__
    MEM->statz.moved ++;
#endif
  }

  if(! found) {
    for(block = MEM->base; block < MEM->cache; block ++) {
      if(block->R == 0) {
	found = 1;
	break;
      }

#ifdef __STATZ__
      MEM->statz.moved ++;
#endif
    }
  }

#endif

  if(found) {
#ifndef __MALLOC__
    assert(block >= MEM->base && block < MEM->mark);
#endif

#ifdef __STATZ__
    /* update statz */

    MEM->statz.count ++;
    MEM->statz.total ++;
    MEM->statz.maxim = max(MEM->statz.maxim, MEM->statz.count);
#endif

    memset(&(block->data), 0, sizeof(block->data));

    block->R = 1;
    block->T = T;

#ifndef __MALLOC__
    MEM->cache = block;
#endif

    dbg("(mem-alloc T %d) = %p\n", T, (void *) &(block->data.any));

    return (void *) &(block->data.any); // __unself(block);
  } else /* no mem */
    Exception(NULL, "Memory Exhausted", 1);

  return NULL;
}

void *mem_link(void *ob) {
  if(ob) {
    T_BLOCK *block = __self(ob);

    block->R ++;
  }

  return ob;
}

size_t mem_ref(void *ob) {
  if(ob) {
    T_BLOCK *block = __self(ob);

    return block->R;
  }

  return 0;
}

void mem_unlink(void *ob) {
  dbg("(mem-unlink %p)\n", ob);

  if(ob) {
    T_BLOCK *block = __self(ob);

    if(block->R > 0) {
      if(-- block->R == 0) {
	switch(block->T) {
	case MEM_ATOM : atom_kill((T_ATOM *)ob); break;
	case MEM_CONS : cons_kill((T_CONS *)ob); break;
	case MEM_SYMB : symb_kill((T_SYMB *)ob); break;
	case MEM_WORD : word_kill((T_WORD *)ob); break;
	case MEM_FLDR : fldr_kill((T_FLDR *)ob); break;

	default:
	  Exception(NULL, "(mem-unlink) got invalid T_BLOCK type", 1);
	  break;
	}

	mem_clear(block);
      }
    } else {
      Exception(NULL, "(mem-unlink) T_BLOCK with R < 1", 1);
    }
  } 
}

/* MEM_ATOM, alloc, link, unlink, kill */

T_ATOM *atom_alloc(void) {
  return mem_alloc(MEMORY, MEM_ATOM);
}

T_ATOM *atom_link(T_ATOM *o) {
  return (T_ATOM *)mem_link(o);
}

size_t atom_ref(T_ATOM *o) {
  return mem_ref(o);
}

void atom_unlink(T_ATOM *o) {
  if(o)
    mem_unlink(o);
}

void atom_kill(T_ATOM *o) {
  if(atom_q_zint(o)) {
    mpz_clear(atom_get_zint(o));
  }
  else if(atom_q_cons(o)) {
    cons_unlink(atom_get_cons(o));
  }
  else if(o->prolog == DOSYMB) {
    symb_unlink(atom_get_symb(o));
  }
  else if(atom_q_name(o) || o->prolog == DOFUNC) {
    xm_free(atom_get_name(o));
  }
  else if(atom_q_matrix(o)) {
    T_ARRY *arry = atom_get_arry(o);
    unsigned long j, total;

    if(arry->dims > 0) {
      for(total = 1, j = 0; j < arry->dims; j ++)
        total *= arry->dim[j];
    } else
      total = 0;

    if(arry->dim)
      xm_free(arry->dim);

    for(j = 0; j < total; j ++)
      atom_unlink(arry->data[j]);

    if(arry->data)
      xm_free(arry->data);
  }
}

/* MEM_CONS, alloc, link, unlink, kill */

T_CONS *cons_alloc(void) {
  return mem_alloc(MEMORY, MEM_CONS);
}

T_CONS *cons_link(T_CONS *s) {
  return (T_CONS *)mem_link(s);
}

size_t cons_ref(T_CONS *s) {
  return mem_ref(s);
}

void cons_unlink(T_CONS *s) {
  if(s)
    mem_unlink(s);
}

void cons_kill(T_CONS *s) {
  if(s) {
    if(s->cdr)
      cons_unlink(s->cdr);

    atom_unlink(s->car);
  }
}

/* MEM_SYMB, alloc, link, unlink, kill */

T_SYMB *symb_alloc(void) {
  return mem_alloc(MEMORY, MEM_SYMB);
}

T_SYMB *symb_link(T_SYMB *s) {
  return (T_SYMB *)mem_link(s);
}

size_t symb_ref(T_SYMB *s) {
  return mem_ref(s);
}

void symb_unlink(T_SYMB *s) {
  if(s) {
    mem_unlink(s);
  }
}

void symb_kill(T_SYMB *s) {
  if(s) {
    if(s->car)
      symb_unlink(s->car);

    if(s->cdr)
      symb_unlink(s->cdr);

    if(s->ob)
      atom_unlink(s->ob);
  }
}

/* MEM_WORD, alloc, link, unlink, kill */

T_WORD *word_alloc(void) {
  return mem_alloc(MEMORY, MEM_WORD);
}

T_WORD *word_link(T_WORD *W) {
  return (T_WORD *)mem_link(W);
}

size_t word_ref(T_WORD *W) {
  return mem_ref(W);
}

void word_unlink(T_WORD *W) {
  if(W)
    mem_unlink(W);
}

void word_kill(T_WORD *W) {
  if(W) {
    if(W->ob)
      atom_unlink(W->ob);

    if(W->id)
      xm_free(W->id);
  }
}

/* MEM_FLDR, alloc, link, unlink, kill */

T_FLDR *fldr_alloc(void) {
  return mem_alloc(MEMORY, MEM_FLDR);
}

T_FLDR *fldr_link(T_FLDR *F) {
  return (T_FLDR *)mem_link(F);
}

size_t fldr_ref(T_FLDR *F) {
  return mem_ref(F);
}

void fldr_unlink(T_FLDR *F) {
  if(F)
    mem_unlink(F);
}

void fldr_kill(T_FLDR *F) {
  if(F) {
    T_WORD *lam, *nxt;

    for(lam = F->fst; lam; lam = nxt) {
      nxt = lam->next;
      word_unlink(lam);
    }

    if(F->id)
      xm_free(F->id);
  }
}
