#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <assert.h>

#include "console.h"
#include "gstack.h"

#include "main.h"
#include "object.h"
#include "mem.h"
#include "stack.h"
#include "xlib.h"
#include "dump.h"
#include "cmd.h"
#include "symb.h"
#include "lambda.h"
#include "debug.h"

struct _flags_t FLAGS;
jmp_buf exception;

void Exception(T_ATOM *self, const char *msg, int fatal) {
  if(fatal)
    xprintf(win_cmd, "Fatal ");

  xprintf(win_cmd, "Exception");

  if(self) {
    xprintf(win_cmd, " at ");
    atom_dump(win_cmd, self);
  }

  if(msg)
    xprintf(win_cmd, " (%s)", msg);

  xprintf(win_cmd, "\n");

  RSTK_dump();
  
  if(fatal) {
    xprintf(win_cmd, "Exiting..\n");
    video_abandon();
    exit(1);
  } else {
    longjmp(exception, 1);

    /* should never happen*/
    xprintf(win_cmd, "I fucked up, aborting..");
    video_abandon();
    abort();
  }
}

void sigCtrlC(int q) {
  Exception(IPTR ? IPTR->car : NULL, "Ctrl-C signal", 0);
}

const char *next_command(const char *text, int state) {
  static int list_index = 0, len = 0;
  static T_WORD *lambda = NULL;
  static T_FLDR *pwd = NULL;

  const char *name;

  if(! state) {
    list_index = 1;
    len = strlen(text);

    pwd = FOLDER_CWD;
    lambda = pwd->fst;
  }

  while((name = xlib_set[list_index].id) && name[0]) {
    list_index ++;

    if(xstrncmp(name, text, len) == 0)
      return name;
  }

  while(pwd) {
    while(lambda) {
      name = lambda->id;
      lambda = lambda->next;

      if(xstrncmp(name, text, len) == 0)
	return name;
    }

    if((pwd = pwd->parent) != NULL)
      lambda = pwd->fst;
  }

  return NULL;
}

char *stream_read(FILE *stream) {
  char *buf;

  size_t allocated;
  size_t len = 0;

  int ch;
  int stop;

  if (feof(stream))
    return NULL;

  allocated = 0;
  buf = NULL;

  for(stop = 0; !stop; ) {
    if (len + 1 >= allocated) {
      allocated += 256; /* 256-byte jumps */
      buf = realloc(buf, allocated);
    }

    ch = fgetc(stream);

    switch(ch) {
    case EOF:
      stop = 1;

    default:
      buf[len ++] = (char) ch;
      break;
    }
  }

  buf[len] = '\0';

  /* trim excess memory */
  if(allocated > len + 1)
    buf = realloc(buf, len + 1);

  return buf;
}

char *file_read(char *file) {
  FILE *fp;

  fp = fopen(file, "rb");
  if(fp) {
    char *s;
    long len;

    fseek(fp, 0L, SEEK_END);
    len = ftell(fp);
    rewind(fp);

    s = malloc(len + 1);
    if(fread(s, len, 1, fp) == 1) {
      s[len] = '\0';
    } else {
      free(s);
      s = NULL;
    }

    fclose(fp);
    return s;
  }

  return NULL;
}

const char *next_word(const char *stream, char **word, int math) {
  if(stream) {
    const char *token;
    size_t len;

    while(*stream && isspace(*stream))
      stream ++;

    if(*stream == '"') { /* string literal is a special case */
      token = stream ++;

      while(*stream && *stream != '"')
	stream ++;

      if(*stream == '"')
	stream ++;
    }
    else if(*stream == '(' && !math) { /* comment is also a special case */
      size_t nest;

      for(nest = 1, token = stream ++; *stream && (nest > 0); stream ++) {
	switch(*stream) {
	case ')': nest --; break;
	case '(': nest ++; break;
	}
      }

      if(nest > 0)
	xprintf(win_cmd, "(COMMENT: reached end of stream)\n");
    }
    else if(*stream == '/' && *(stream + 1) == '*') { /* C-type comment */
      size_t nest;

      token = stream;

      for(nest = 1, stream += 2; *stream && (nest > 0); stream ++) {
	if(*stream == '/' && *(stream + 1) == '*') nest ++, stream ++;
	if(*stream == '*' && *(stream + 1) == '/') nest --, stream ++;
      }

      if(nest > 0)
	xprintf(win_cmd, "(COMMENT: reached end of stream)\n");
      else if(*stream)
	stream ++; /* skip past the '/' */
    }
    else {
      token = stream; /* token = beginning of word */
      
      while(*stream && (! isspace(*stream)))
	stream ++;
    }

    /* stream pointing to character after word */

    len = (size_t)(stream - token);
    
    if((len > 0) && word) { /* don't waste spacetime if word is null */
      *word = realloc(*word, len + 1);
      strncpy(*word, token, len);
      (*word)[len] = '\0';
    }
  }

  return stream;
}

int compile_code(T_ATOM **ob, const char *id, int force) {
  T_CODE code = code_find(id);

  if(code) {
    if(ob)
      *ob = atom_make_code(code);
    return 1;
  }

  if(force)
    Exception(NULL, "Not a built-in word", 0);

  return 0;
}

int compile_id(T_ATOM **ob, const char *id) {
  if(ob)
    *ob = atom_make_idnt((T_NAME)id);
  return 1;
}

int compile_lam(T_ATOM **ob, const char *lam) {
  if(ob)
    *ob = atom_make_lam((T_NAME)lam);
  return 1;
}

int compile_cstr(T_ATOM **ob, const char *cstr) {
  if(cstr[0] == '"') {
    char *ptr, *aux;

    ptr = aux = strdup(cstr + 1);

    while(*ptr) {
      if(*ptr == '"')
	*ptr = '\0';
      else
	ptr ++;
    }

    if(ob)
      *ob = atom_make_cstr((T_NAME)aux);

    free(aux);
  } else {
    if(ob)
      *ob = atom_make_cstr((T_NAME)cstr);
  }

  return 1;
}

int compile_chr(T_ATOM **ob, const char *word) {
  if(ob)
    *ob = atom_make_char(word[0]);

  return 1;
}

int compile_bint(T_ATOM **ob, const char *word, int force) {
  T_BINT x;
  char *endptr;

  x = strtol(word, &endptr, 0);

  if(*endptr == '\0') {
    if(ob)
      *ob = atom_make_bint(x);

    return 1;
  }

  if(force)
    Exception(NULL, "Invalid syntax", 0);

  return 0;
}

int compile_zint(T_ATOM **ob, const char *word, int force) {
  T_ZINT x;
  int q;

  mpz_init(x);

  q = (mpz_set_str(x, word, 0) == 0);
  
  if(q && ob)
    *ob = atom_make_zint(x);

  mpz_clear(x);

  if(!q && force)
    Exception(NULL, "Invalid syntax", 0);

  return q;
}

int compile_real(T_ATOM **ob, const char *word, int real, int force) {
  T_REAL g;
  char *endptr;

  g = strtod(word, &endptr);

  if(*endptr == '\0') {
    if(real) {
      if(ob)
	*ob = atom_make_real(g);
    } else {
      T_ZINT x;
      int q;

      mpz_init(x);

      /* check if it's a ZINT */

      q = (mpz_set_str(x, word, 0) == 0);

      if(ob)
	*ob = q ? atom_make_zint(x) : atom_make_real(g);

      mpz_clear(x);
    }

    return 1;
  }

  if(force)
    Exception(NULL, "Invalid syntax", 0);

  return 0;
}

int compile_cplx(T_ATOM **ob, const char *real, const char *imag) {
  T_REAL x, y;
  char *endptr1;

  x = strtod(real, &endptr1);

  if(*endptr1 == '\0') {
    char *endptr2;

    y = strtod(imag, &endptr2);

    if(*endptr2 == '\0') {
      if(ob)
	*ob = atom_make_cplx(x, y);

      return 1;
    } else {
      Exception(NULL, "Invalid imaginary part", 0);
    }
  } else {
    Exception(NULL, "Invalid real part", 0);
  }

  return 0;
}

int compile_symb(T_ATOM **ob, const char *word) {
  T_SYMB *symb = symb_parse(word);

  if(ob) {
    *ob = atom_make_symb();
    atom_get_symb(*ob) = symb;
  }

  return 1;
}

const char *compile_cons(T_ATOM **ob, const char *stream, T_PROLOG type) {
  T_ATOM *prog;
  T_CONS *cons;

  /* create (empty) composite object */

  prog = atom_alloc();
  prog->prolog = type;

  /* compile stream of objects comprised in composite object */
  cons = NULL;
  stream = string_compile(&cons, stream);
  atom_get_cons(prog) = cons;

  /* return composite object */
  if(ob)
    *ob = prog;

  return stream;
}

const char *next_token(T_ATOM **ob, const char *stream) {
  T_ATOM *newob = NULL;

 __next_token:
  if(stream && *stream) {
    char *word = NULL;

    stream = next_word(stream, &word, 0);

    if(word == NULL)
      newob = NULL;

    /* comments are ignored */
    else if((word[0] == '(') ||
       ((word[0] == '/') && (word[1] == '*'))) {
      free(word);
      goto __next_token;
    }

    /* special case: end of composite object */

    else if(! xstrcmp(word, ";") || ! xstrcmp(word, "}"))
      newob = NULL;

    /* check for explicit object definition */

    else if(! xstrcmp(word, "CODE")) {
      stream = next_word(stream, &word, 0);
      compile_code(&newob, word, /*force*/ 1);
    }
    else if(! xstrcmp(word, "ID")) {
      stream = next_word(stream, &word, 0);
      compile_id(&newob, word);
    }
    else if(! xstrcmp(word, "LAM")) {
      stream = next_word(stream, &word, 0);
      compile_lam(&newob, word);
    }
    else if(! xstrcmp(word, "CHR")) {
      stream = next_word(stream, &word, 0);
      compile_chr(&newob, word);
    }
    else if(! xstrcmp(word, "$")) {
      stream = next_word(stream, &word, 0);
      compile_cstr(&newob, word);
    }
    else if(! xstrcmp(word, "ZINT")) {
      stream = next_word(stream, &word, 0);
      compile_zint(&newob, word, /*force*/1);
    }
    else if(! xstrcmp(word, "#")) {
      stream = next_word(stream, &word, 0);
      compile_bint(&newob, word, /*force*/1);
    }
    else if(! xstrcmp(word, "%")) {
      stream = next_word(stream, &word, 0);
      compile_real(&newob, word, /*real*/ 1, /*force*/ 1);
    }
    else if(! xstrcmp(word, "C%")) {
      char *word1 = NULL;
      
      stream = next_word(stream, &word, 0);
      stream = next_word(stream, &word1, 0);
      
      compile_cplx(&newob, word, word1);
      
      free(word1);
    }
    else if(! xstrcmp(word, "{")) {
      stream = compile_cons(&newob, stream, DOLIST);
    }
    else if(! xstrcmp(word, "::")) {
      stream = compile_cons(&newob, stream, DOPROG);
    }
    else if(! xstrcmp(word, "SYMB")) {
      stream = next_word(stream, &word, /*math*/1);
      compile_symb(&newob, word);
    }
    else { /* implicit object definition */
      int handled = 0;
      
      /* code? */
      handled = compile_code(&newob, word, /*don't force*/ 0);
      
      /* lam? */
      if(! handled) {
	if(lambda_find(FOLDER_LAMBDA, word, 1))
	  handled = compile_lam(&newob, word);
      }
      
      /* cstr? */
      if(! handled)
	if(*word == '"')
	  handled = compile_cstr(&newob, word);
      
      /* real? */
      if(! handled)
	handled = compile_real(&newob, word, 
			       /*coerce*/ 0, /*don't force*/ 0);

      /* zint? */
      if(! handled)
	handled = compile_zint(&newob, word, /* don't force*/0);

      /* nothing, push it as id */
      if(! handled)
	handled = compile_id(&newob, word);
      
      /* handled = 1; */
    }

    if(word)
      free(word);
  }

  if(ob)
    *ob = newob;

  return stream;
}

const char *string_compile(T_CONS **cons, const char *stream) {
  T_ATOM *token = NULL;
  T_CONS *tail;

  tail = *cons = NULL;

  while(stream && *stream) {
    stream = next_token(&token, stream);

    if(token) {
      tail = cons_add(tail, token);

      if(*cons == NULL)
	*cons = tail;
    }
    else
      break; /* end of runstream OR composite object */
  }

  return stream;
}

void parse_string(const char *stream) {
  size_t N = 0;

  while(stream && *stream) {
    T_CONS *ptr, *cons = NULL;

    stream = string_compile(&cons, stream);

    for(ptr = cons; ptr; ptr = ptr->cdr) {
      DSTK_add(atom_link(ptr->car));
      N ++;
    }

    cons_unlink(cons);
  }

  DSTK_add(atom_make_bint(N));
}

void eval_string(const char *stream) {
  while(stream && *stream) {
    T_CONS *cons = NULL;

    stream = string_compile(&cons, stream);

    ISTK_add(cons);
    eval_runstream();
  }
}

/* main inner loop */

void eval_runstream(void) {
 runstream:

  while(IPTR) {
    T_ATOM *ob;

    ob = IPTR->car;
    IPTR = IPTR->cdr;
      
    /* might manipulate entire runstream */
    atom_eval(ob, NULL);
  }
  
  if(RSTK_LEVEL >= 0) {
    ISTK_kill();
    goto runstream;
  }

  /* nothing left to do, request input from user */
}

int main(int argc, char *argv[]) {
  int kernel_loaded;

  /* default system state */

  kernel_loaded   = 0;
  FLAGS.video     = 1;
  FLAGS.mod_run   = M_RUN;
  FLAGS.mod_angle = MOD_RAD;
  FLAGS.mod_bint  = MOD_DEC;
  FLAGS.mod_cplx  = MOD_RECT;
  FLAGS.mod_symb  = MOD_DAL;

  /* check user (initial) args */

  for(argc --, argv ++; argc; argc --, argv ++) {
    if(! xstrcmp(*argv, "--interactive") || ! xstrcmp(*argv, "-i"))
      FLAGS.video = 1;
    else if(! xstrcmp(*argv, "--batch") || ! xstrcmp(*argv, "-b"))
      FLAGS.video = 0;
    else if(! xstrcmp(*argv, "--nokernel") || ! xstrcmp(*argv, "-nk"))
      kernel_loaded = 1;
    else
      break;
  }

  /* initialize video & memory */

  video_initialize();
  xm_init();

  gstack();
  video_update(1);

  /* prepare for re(p)l */

  signal(SIGINT, sigCtrlC);

  RSTK_LEVEL = -1;
  LOOP_LEVEL = -1;

  /* re(p)l  */

  while(FLAGS.mod_run != M_OFF) {
    char *in;
    int tty;
    int interactive;

    if(setjmp(exception)) {
      /* jump to a sane state.. */

      while(LOOP_LEVEL >= 0)
	LOOP_drop();

      while(RSTK_LEVEL >= 0)
	ISTK_kill();

      /* stop processing command line.. */
      argc = 0;
    }

    in = NULL;
    tty = isatty(fileno(stdin));
    interactive = 0;

#define PROMPT "rpl> "

    if(! kernel_loaded) {
      in = file_read("kernel.mth");
      kernel_loaded = 1;
    }
    else if(argc) {
      char *word = *(argv ++);
      argc --;

      if(! xstrcmp(word, "-e")) {
	if(argc) {
	  in = file_read(*argv);
	  argc --, argv ++;
	} else
	  Exception(NULL, "No file name given in command line", 0);
      }
      else if(! xstrcmp(word, "--debug") || ! xstrcmp(word, "-d")) {
	FLAGS.mod_run = M_DBG;
      }
      else if(! xstrcmp(word, "--trace") || ! xstrcmp(word, "-t")) {
	FLAGS.mod_run = M_TRC;
      }
      else if(! xstrcmp(word, "--run") || ! xstrcmp(word, "-r")) {
	FLAGS.mod_run = M_RUN;
      }
      else if(! xstrcmp(word, "--off") || ! xstrcmp(word, "-o")) {
	FLAGS.mod_run = M_OFF;
      }
      else
	Exception(NULL, "Invalid command-line argument", 0);
    }

    if(in == NULL && argc == 0) {
      interactive = 1;

      if(tty) {
	gstack();
	in = console_read(PROMPT);
      } else
	in = stream_read(stdin);
    }

    if(in) {
      eval_string(in);
      free(in);
    } else
      break;
  }

  /* exit program */

  if(FLAGS.mod_run != M_OFF)
    xprintf(win_cmd, "(exiting..)\n");

  signal(SIGINT, SIG_DFL);

  /* shutdown memory system */

  xm_end();

  /* shutdown console system */

  video_abandon();

  return 0;
}
