#include "object.h"
#include "math.h"

T_CPLX cplx(T_REAL x, T_REAL y);
T_CPLX cplx_scale(T_CPLX z, T_REAL K);
T_CPLX cplx_add(T_CPLX z1, T_CPLX z2);
T_CPLX cplx_sub(T_CPLX z1, T_CPLX z2);
T_CPLX cplx_mul(T_CPLX z1, T_CPLX z2);
T_CPLX cplx_div(T_CPLX z1, T_CPLX z2);
T_CPLX cplx_pow(T_CPLX z1, T_CPLX z2);
T_CPLX cplx_sqrt(T_CPLX z);
T_CPLX cplx_neg(T_CPLX z);
T_CPLX cplx_conj(T_CPLX z);
T_REAL cplx_abs(T_CPLX z);
T_REAL cplx_angle(T_CPLX z);
T_CPLX cplx_inv(T_CPLX z);
T_CPLX cplx_sq(T_CPLX z);
T_CPLX cplx_exp(T_CPLX z);
T_CPLX cplx_log(T_CPLX z);
T_CPLX cplx_log10(T_CPLX z);
T_CPLX cplx_asinh(T_CPLX z);
T_CPLX cplx_acosh(T_CPLX z);
T_CPLX cplx_atanh(T_CPLX z);
T_CPLX cplx_acos(T_CPLX z);
T_CPLX cplx_asin(T_CPLX z);
T_CPLX cplx_atan(T_CPLX z);
T_CPLX cplx_cos(T_CPLX z);
T_CPLX cplx_cosh(T_CPLX z);
T_CPLX cplx_sin(T_CPLX z);
T_CPLX cplx_sinh(T_CPLX z);
T_CPLX cplx_tan(T_CPLX z);
T_CPLX cplx_tanh(T_CPLX z);

T_CPLX cplx(T_REAL x, T_REAL y) {
  T_CPLX z;

  z.x = x;
  z.y = y;

  return z;
}

T_CPLX cplx_scale(T_CPLX z, T_REAL K) {
  z.x *= K;
  z.y *= K;

  return z;
}

T_CPLX cplx_add(T_CPLX z1, T_CPLX z2) {
  z1.x += z2.x;
  z1.y += z2.y;

  return z1;
}

T_CPLX cplx_sub(T_CPLX z1, T_CPLX z2) {
  z1.x -= z2.x;
  z1.y -= z2.y;

  return z1;
}

T_CPLX cplx_mul(T_CPLX z1, T_CPLX z2) {
  T_CPLX z;

  z.x = (z1.x * z2.x) - (z1.y * z2.y);
  z.y = (z1.x * z2.y) + (z1.y * z2.x);

  return z;
}

T_CPLX cplx_div(T_CPLX z1, T_CPLX z2) {
  return cplx_mul(z1, cplx_inv(z2));
}

T_CPLX cplx_pow(T_CPLX z1, T_CPLX z2) {
  z1 = cplx_log(z1);
  z1 = cplx_mul(z1, z2);
  z1 = cplx_exp(z1);

  return z1;
}

T_CPLX cplx_sqrt(T_CPLX z) {
  return (cplx_exp(cplx_scale(cplx_log(z), 0.5)));
}

T_CPLX cplx_neg(T_CPLX z) {
  z.x *= -1.0;
  z.y *= -1.0;

  return z;
}

T_CPLX cplx_conj(T_CPLX z) {
  z.y *= -1.0;

  return z;
}

T_REAL cplx_abs(T_CPLX z) {
  return sqrt(z.x * z.x + z.y * z.y);
}

T_REAL cplx_angle(T_CPLX z) {
  return atan2(z.y, z.x);
}

T_CPLX cplx_inv(T_CPLX z) {
  T_REAL a;

  a = z.x * z.x + z.y * z.y;
  z.x /= a;
  z.y /= -a;

  return z;
}

T_CPLX cplx_sq(T_CPLX z) {
  T_CPLX r;

  r.x = z.x * z.x - z.y * z.y;
  r.y = 2 * z.x * z.y;

  return r;
}

T_CPLX cplx_exp(T_CPLX z) {
  T_CPLX r;
  double a;

  a = exp(z.x);
  r.x = a * cos(z.y);
  r.y = a * sin(z.y);

  return r;
}

T_CPLX cplx_log(T_CPLX z) {
  T_CPLX r;

  r.x = log(cplx_abs(z));
  r.y = atan2(z.y, z.x);

  return r;
}

T_CPLX cplx_log10(T_CPLX z) {
  z = cplx_log(z);
  z = cplx_scale(z, 1.0 / log(10));

  return z;
}

T_CPLX cplx_asinh(T_CPLX z) {
  T_CPLX r;

  r = cplx_sq(z);
  r.x += 1.0;
  r = cplx_sqrt(r);
  r = cplx_add(r, z);
  r = cplx_log(r);

  return r;
}

T_CPLX cplx_acosh(T_CPLX z) {
  T_CPLX r;

  r = cplx_sq(z);
  r.x -= 1.0;
  r = cplx_sqrt(r);
  r = cplx_add(r, z);
  r = cplx_log(r);

  return r;
}

T_CPLX cplx_atanh(T_CPLX z) {
  T_CPLX t, b, r;

  b = t = z;
  t.x += 1.0;
  b.x = 1 - b.x;
  b.y = -b.y;
  r = cplx_sqrt(cplx_div(t, b));
  r = cplx_log(r);

  return r;
}

T_CPLX cplx_acos(T_CPLX z) {
  T_CPLX a, i, r;

  a.x = 1; a.y = 0;
  i.x = 0; i.y = 1;

  r = cplx_sq(z);
  r = cplx_sub(a, r);
  r = cplx_sqrt(r);
  z = cplx_mul(i, z);
  r = cplx_add(r, z);
  r = cplx_log(r);
  r = cplx_mul(i, r);
  r.x += 0.5 * M_PI;

  return r;
}

T_CPLX cplx_asin(T_CPLX z) {
  T_CPLX a, i, r;

  a.x = 1; a.y = 0;
  i.x = 0; i.y = 1;

  r = cplx_sq(z);
  r = cplx_sub(a, r);
  r = cplx_sqrt(r);
  z = cplx_mul(i, z);
  r = cplx_add(r, z);
  r = cplx_log(r);
  r = cplx_mul(i, r);
  r = cplx_neg(r);

  return r;
}

T_CPLX cplx_atan(T_CPLX z) {
  T_CPLX i, a;

  a.x = 1; a.y = 0;
  i.x = 0; i.y = 1;

  z = cplx_add(cplx_mul(i, z), a);
  z = cplx_scale(z, 1.0 / cplx_abs(z));
  z = cplx_log(z);

  i.y = -1;
  z = cplx_mul(z, i);
  return z;
}

T_CPLX cplx_cos(T_CPLX z) {
  T_CPLX r;

  r.x = cos(z.x) * cosh(z.y);
  r.y = -sin(z.x) * sinh(z.y);

  return r;
}

T_CPLX cplx_cosh(T_CPLX z) {
  T_CPLX r;

  r.x = cosh(z.x) * cos(z.y);
  r.y = sinh(z.x) * sin(z.y);

  return r;
}
T_CPLX cplx_sin(T_CPLX z) {
  T_CPLX r;

  r.x = sin(z.x) * cosh(z.y);
  r.y = cos(z.x) * sinh(z.y);

  return r;
}

T_CPLX cplx_sinh(T_CPLX z) {
  T_CPLX r;

  r.x = sinh(z.x) * cos(z.y);
  r.y = cosh(z.x) * sin(z.y);

  return r;
}

T_CPLX cplx_tan(T_CPLX z) {
  T_CPLX r1, r2;

  r1 = cplx_sin(z);
  r2 = cplx_cos(z);

  r1 = cplx_div(r1, r2);

  return r1;
}

T_CPLX cplx_tanh(T_CPLX z) {
  T_CPLX r1, r2;

  r1 = cplx_sinh(z);
  r2 = cplx_cosh(z);

  r1 = cplx_div(r1, r2);

  return r1;
}

T_CPLX cplx_gamma(T_CPLX z) {
  T_CPLX y;
  double xr, xi, wr, wi, ur, ui, vr, vi, yr, yi, t;

  xr = z.x;
  xi = z.y;

  if (xr < 0) {
    wr = 1 - xr;
    wi = -xi;
  } else {
    wr = xr;
    wi = xi;
  }

  ur = wr + 6.00009857740312429;
  vr = ur * (wr + 4.99999857982434025) - wi * wi;
  vi = wi * (wr + 4.99999857982434025) + ur * wi;
  yr = ur * 13.2280130755055088 + vr * 66.2756400966213521 + 
    0.293729529320536228;
  yi = wi * 13.2280130755055088 + vi * 66.2756400966213521;
  ur = vr * (wr + 4.00000003016801681) - vi * wi;
  ui = vi * (wr + 4.00000003016801681) + vr * wi;
  vr = ur * (wr + 2.99999999944915534) - ui * wi;
  vi = ui * (wr + 2.99999999944915534) + ur * wi;
  yr += ur * 91.1395751189899762 + vr * 47.3821439163096063;
  yi += ui * 91.1395751189899762 + vi * 47.3821439163096063;
  ur = vr * (wr + 2.00000000000603851) - vi * wi;
  ui = vi * (wr + 2.00000000000603851) + vr * wi;
  vr = ur * (wr + 0.999999999999975753) - ui * wi;
  vi = ui * (wr + 0.999999999999975753) + ur * wi;
  yr += ur * 10.5400280458730808 + vr;
  yi += ui * 10.5400280458730808 + vi;
  ur = vr * wr - vi * wi;
  ui = vi * wr + vr * wi;
  t = ur * ur + ui * ui;
  vr = yr * ur + yi * ui + t * 0.0327673720261526849;
  vi = yi * ur - yr * ui;
  yr = wr + 7.31790632447016203;
  ur = log(yr * yr + wi * wi) * 0.5 - 1;
  ui = atan2(wi, yr);
  yr = exp(ur * (wr - 0.5) - ui * wi - 3.48064577727581257) / t;
  yi = ui * (wr - 0.5) + ur * wi;
  ur = yr * cos(yi);
  ui = yr * sin(yi);
  yr = ur * vr - ui * vi;
  yi = ui * vr + ur * vi;
  if (xr < 0) {
    wr = xr * 3.14159265358979324;
    wi = exp(xi * 3.14159265358979324);
    vi = 1 / wi;
    ur = (vi + wi) * sin(wr);
    ui = (vi - wi) * cos(wr);
    vr = ur * yr + ui * yi;
    vi = ui * yr - ur * yi;
    ur = 6.2831853071795862 / (vr * vr + vi * vi);
    yr = ur * vr;
    yi = ur * vi;
  }

  y.x = yr;
  y.y = yi;

  return y;
}
