#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <curses.h>
#include <menu.h>

#include "gstack.h"
#include "main.h"
#include "xlib.h"

int dbg(const char *format, ...);
int xprintf(WINDOW *win, const char *format, ...);

int do_menu(int words, const char *a_word[]);
int do_complete(char **pbuf, int *ppos, int *plen, int *pallocated);

WINDOW *win_header, *win_footer;
WINDOW *win_stk_border, *win_stk;
WINDOW *win_cmd_border, *win_cmd;

char footer[1024];

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))

void video_update(int force) {
  if(FLAGS.video) {
    if(force) {
      touchwin(win_header);
      touchwin(win_footer);
      touchwin(win_stk_border);
      touchwin(win_stk);
      touchwin(win_cmd_border);
      touchwin(win_cmd);
    }

    wnoutrefresh(win_header);
    wnoutrefresh(win_footer);

    wnoutrefresh(win_stk_border);
    wnoutrefresh(win_stk);

    wnoutrefresh(win_cmd_border);
    wnoutrefresh(win_cmd);
    doupdate();
  } else
    refresh();
}

void set_header(const char *s) {
  if(FLAGS.video) {
    int n, mx, my;

    mvwaddch(win_header, 0, 0, ' ');
    if(s) {
      n = strlen(s);
      getmaxyx(win_header, my, mx);

      if(n > mx - 2)
	n = mx - 2;
    
      waddnstr(win_header, s, n);
    }
    wclrtoeol(win_header);
    wrefresh(win_header);
  }
}

void draw_footer(void) {
  if(FLAGS.video) {
    int n, mx, my;

    mvwaddch(win_footer, 0, 0, ' ');
    n = strlen(footer);
    getmaxyx(win_footer, my, mx);

    if(n > mx - 2)
      n = mx - 2;
    
    waddnstr(win_footer, footer, n);
    wclrtoeol(win_footer);
    wrefresh(win_footer);
  } else
    printw("(dbg) %s\n", footer);
}

void set_footer(const char *s) {
  strncpy(footer, s, 1024);
  footer[1023] = '\0';

  draw_footer();
}

void add_footer(const char *s) {
  strncat(footer, s, 1024 - strlen(footer));
  footer[1023] = '\0';

  draw_footer();
}

int xwgetch(WINDOW *win) {
  int x;

  curs_set(1);
  x = wgetch(FLAGS.video ? win : stdscr);
  curs_set(0);

  return x;
}

void video_initialize(void) {
  initscr();

  if(FLAGS.video) {
    int h, w, stk_h, cmd_h;

    getmaxyx(stdscr, h, w);

    stk_h = (int)(0.6 * (h - 5));
    cmd_h = h - 6 - stk_h;

    win_header = newwin(1, w, 0, 0);
    win_footer = newwin(1, w, h - 1, 0);

    win_stk_border = newwin(stk_h+2, w, 1, 0);
    wborder(win_stk_border, 0, 0, 0, 0, 0, 0, 0, 0);
    win_stk = derwin(win_stk_border, stk_h, w-2, 1, 1);

    win_cmd_border = newwin(cmd_h+2, w, stk_h+3, 0);
    wborder(win_cmd_border, 0, 0, 0, 0, 0, 0, 0, 0);
    win_cmd = derwin(win_cmd_border, cmd_h, w-2, 1, 1);

    wbkgdset(win_header, A_REVERSE | A_BOLD);
    wbkgdset(win_footer, A_REVERSE);

    set_header("sysrpl");
    set_footer("OK");

    scrollok(win_header, 0);
    scrollok(win_footer, 0);
    scrollok(win_stk_border, 0);
    scrollok(win_stk, 0);
    scrollok(win_cmd_border, 0);
    scrollok(win_cmd, 1);

    keypad(win_cmd, TRUE);

    video_update(1);
  } else
    scrollok(stdscr, 1);

  curs_set(0);
  nonl();
  cbreak();
  noecho();

  keypad(stdscr, TRUE);
}

void video_abandon(void) {
  if(FLAGS.video) {
    delwin(win_cmd);
    delwin(win_cmd_border);

    delwin(win_stk);
    delwin(win_stk_border);

    delwin(win_header);

    wbkgdset(win_footer, A_NORMAL);
    wmove(win_footer, 0, 0);
    wclrtoeol(win_footer);
    wrefresh(win_footer);
  
    delwin(win_footer);
  }

  curs_set(1);
  endwin();
}

int dbg(const char *format, ...) {
  if(0) {
    int n, size = 1024;
    char *p;
    va_list ap;
  
    if((p = malloc(size)) == NULL)
      return -1;

    while(1) {
      va_start(ap, format);
      n = vsnprintf(p, size, format, ap);
      va_end(ap);

      if(n > -1 && n < size) {
	set_footer(p);
	free(p);

	return n;
      }

      if(n > -1)
	size = n + 1;
      else
	size *= 2;

      if((p = realloc(p, size)) == NULL)
	return -1;
    }
  }

  return 0;
}

int xprintf(WINDOW *win, const char *format, ...) {
  int n, size = 1024;
  char *p;
  va_list ap;

  if((p = malloc(size)) == NULL)
    return -1;

  while(1) {
    va_start(ap, format);
    n = vsnprintf(p, size, format, ap);
    va_end(ap);

    if(n > -1 && n < size) {
      waddnstr(FLAGS.video ? win : stdscr, p, n);
      free(p);
      return n;
    }

    if(n > -1)
      size = n + 1;
    else
      size *= 2;

    if((p = realloc(p, size)) == NULL)
      return -1;
  }
}

/* BUG: screen mixed-up if window needs to scroll */

char *console_read(const char *prompt) {
  WINDOW *edwin;
  char *buf;

  int allocated;
  int pos, len;

  int ref_x, ref_y;
  int h, w, offset;

  int ch;
  int stop;
  int completing;

  allocated = len = pos = 0;
  buf = NULL;

  edwin = FLAGS.video ? win_cmd : stdscr;
  getmaxyx(edwin, h, w);

  waddstr(edwin, prompt);

  offset = ref_x = ref_y = 0;
  getyx(edwin, ref_y, ref_x);

  w -= ref_x;

  for(completing = 0, stop = 0; !stop; ) {
    if (len + 1 >= allocated) {
      allocated += 256; /* 256-byte jumps */
      buf = realloc(buf, allocated);
    }

    buf[len] = '\0';

    /* adjust display offset */

    if(pos < offset)
      offset = pos;

    while(pos >= offset + w)
      offset ++;

    /* display current line */

    mvwaddnstr(edwin, ref_y, ref_x, buf + offset, min(w, len - offset));
    wclrtoeol(edwin);
    wmove(edwin, ref_y, ref_x + pos - offset);

    ch = 0;

    if(completing) {
      if(FLAGS.video)
	ch = do_complete(&buf, &pos, &len, &allocated);

      if(ch == 0) {
	completing = 0;
	continue;
      }
    }

    if(ch == 0)
      ch = xwgetch(edwin);

    switch(ch) {
    case '\0':
      break;

    case EOF:
      stop = 1;
      break;

    case 0x0c: /* CTRL-L */
      video_update(1);
      break;

    case '\b':
    case 127:
    case KEY_BACKSPACE:
      if(pos > 0) {
	memmove(buf + pos - 1, buf + pos, len - pos + 1);
	len --;
	pos --;
      }
      break;

    case KEY_ENTER:
    case '\r':
    case '\n':
      stop = 1;
      break;

    case KEY_RESIZE:
      gstack();
      break;

    case KEY_DOWN:
      break;

    case KEY_UP:
      break;

    case KEY_LEFT:
      if(pos > 0)
	pos --;
      break;

    case KEY_RIGHT:
      if(pos < len)
	pos ++;
      break;

    case KEY_HOME:
      pos = 0;
      break;

    case KEY_END:
      pos = len;
      break;

    case KEY_DC: /* DEL */
      if(pos < len) {
	memmove(buf + pos, buf + pos + 1, len - pos);
	len --;
      } 
     break;

    case KEY_IC: /* INS */
      break;

    case '\t': /* TAB */
      completing = 1;
      break;

    default:
      if(! iscntrl(ch) && (ch < KEY_MIN)) {
	memmove(buf + pos + 1, buf + pos, len - pos + 1);
	buf[pos ++] = (char) ch;
	len ++;
      }
      break;
    }
  }

  buf[len] = '\0';

  if(pos < offset)
    offset = pos;

  while(pos >= offset + w)
    offset ++;

  /* display final state of the edited line */

  mvwaddnstr(edwin, ref_y, ref_x, buf + offset, min(w, len - offset));
  wclrtoeol(edwin);
  waddch(edwin, '\n');

  /* trim excess memory */
  if(allocated > len + 1)
    buf = realloc(buf, len + 1);

  return buf;
}

int star_strcmp(const void *s1, const void *s2) {
  return xstrcmp(*(const char **)s1, *(const char **)s2);
}

int do_complete(char **pbuf, int *ppos, int *plen, int *pallocated) {
  const char *a_word[1024];
  const char *word;
  int words, ndx;
  char *buf, *token, *copy_token, *end;
  size_t token_ndx, end_ndx;

  buf = *pbuf;
  end = token = buf + *ppos;

  while((token > buf) && isspace(*token))
    token --;

  while((token > buf) && !isspace(*token))
    token --;

  if(isspace(*token))
    token ++;

  while(*end && !isspace(*end))
    end ++;

  word = NULL;
  ndx = words = 0;

  token_ndx = (size_t)(token - buf);
  end_ndx = (size_t)(end - buf);

  copy_token = malloc(end_ndx - token_ndx + 1);
  strncpy(copy_token, token, end_ndx - token_ndx);
  copy_token[end_ndx - token_ndx] = '\0';

  //set_footer(copy_token);

  do {
    word = next_command(copy_token, ndx ++);

    if(word) {
      a_word[words ++] = word;
    }
  } while(word);

  free(copy_token);

  if(words) {
    qsort(a_word, words, sizeof *a_word, star_strcmp);
    ndx = do_menu(words, a_word);

    if(ndx >= 0) { /* item selected */
      size_t old_len, new_len;

      old_len = end_ndx - token_ndx;
      new_len = strlen(word = a_word[ndx]);

      if(new_len > old_len) {
	if (*plen + new_len - old_len + 1 >= *pallocated) {
	  *pallocated = *plen + new_len - old_len + 1;
	  *pbuf = realloc(*pbuf, *pallocated);

	}
	/* make space for new string */
	memmove(*pbuf + token_ndx + new_len,
		*pbuf + end_ndx,
		*plen - old_len + 1);
      }
      /* now overwrite token with selected word */
      memcpy(*pbuf + token_ndx, word, new_len);

      /* update position and length */
      *ppos = token_ndx + new_len;
      *plen += new_len - old_len;
    }
    else if(ndx < -1) { /* pass event to caller */
      return -ndx -1;
    }
  }

  return 0;
}

int do_menu(int words, const char *a_word[]) {
  int which = -1;

  if(words == 1)
    which = 0;
  else if(words > 1) {
    WINDOW *border;
    WINDOW *wpick;
    ITEM **item;
    MENU *pick;

    int x0, y0, xref, yref;
    int abs_h, h, abs_w, w;
    int len, max_len;
    int ndx, key;
    int picked;
 
    max_len = 0;
    item = calloc(words + 1, sizeof *item);

    for(ndx = 0; ndx < words; ndx ++) {
      item[ndx] = new_item(a_word[ndx], NULL);
      len = strlen(a_word[ndx]);

      if(len > max_len)
	max_len = len;
    }

    item[words] = NULL;

    pick = new_menu(item);
    set_menu_mark(pick, "> ");

    getmaxyx(stdscr, abs_h, abs_w);

    w = min(max_len + 5, 0.7*abs_w);
    h = min(words + 2, 0.7*abs_h);

    getbegyx(win_cmd, yref, xref);
    getyx(win_cmd, y0, x0);

    for(y0 += yref - h; y0 < 4; y0 ++) ;
    for(x0 += xref + 2; x0 + w + 2 > abs_w; x0 --) ;

    h -= 2;
    w -= 2;
    
    border = newwin(h + 2, w + 2, y0, x0);
    wattrset(border, A_BOLD);
    werase(border);
    box(border, 0, 0);

    wpick = derwin(border, h, w, 1, 1);

    set_menu_format(pick, h, 1);
    set_menu_win(pick, border);
    set_menu_sub(pick, wpick);

    keypad(wpick, TRUE);

    post_menu(pick);
    wrefresh(border);

    for(picked = 0; ! picked; ) {
      wrefresh(wpick);
      key = xwgetch(wpick);

      switch(key) {
      case KEY_DOWN:
	menu_driver(pick, REQ_DOWN_ITEM);
	break;

      case KEY_UP:
	menu_driver(pick, REQ_UP_ITEM);
	break;

      case KEY_NPAGE:
	for(ndx = 0; ndx < h; ndx ++)
	  menu_driver(pick, REQ_DOWN_ITEM);
	break;

      case KEY_PPAGE:
	for(ndx = 0; ndx < h; ndx ++)
	  menu_driver(pick, REQ_UP_ITEM);
	break;

      case KEY_LEFT:
	menu_driver(pick, REQ_LEFT_ITEM);
	break;

      case KEY_RIGHT:
	menu_driver(pick, REQ_RIGHT_ITEM);
	break;

      case KEY_HOME:
	menu_driver(pick, REQ_FIRST_ITEM);
	break;

      case KEY_END:
	menu_driver(pick, REQ_LAST_ITEM);
	break;

      case 0x1b:
	picked = 1;
	which = -1;
	break;

      case '\r':
      case '\n':
      case KEY_ENTER:
	which = item_index(current_item(pick));
	picked = 1;
	break;

	/*
      case KEY_BACKSPACE:
	menu_driver(pick, REQ_BACK_PATTERN);
	break;
	*/

      default:
	// menu_driver(pick, key);
	which = -(key + 1);
	picked = 1;
	break;
      }
    }

    unpost_menu(pick);

    for(ndx = 0; ndx < words; ndx ++)
      free_item(item[ndx]);

    free_menu(pick);

    delwin(wpick);
    delwin(border);

    video_update(1);
  }

  return which;
}
