#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <math.h>

#include "console.h"
#include "gstack.h"

#include "main.h"
#include "object.h"
#include "stack.h"
#include "mem.h"
#include "xlib.h"
#include "symb.h"
#include "dump.h"
#include "lambda.h"
#include "cplx.h"
#include "special.h"
#include "cmd.h"

/* Helper (MACRO) */

#define builtin(id)       void CMD_##id(T_ATOM *self)

#define __NEXT     IPTR = IPTR->cdr

/* Helper (FUNCTION) */

/* convert level1 to real/(cplx)/symbolic/arry/list/prog, if needed */
void __coerce(int prog, 
	      int list, 
	      int arry,
	      int symb, 
	      int cplx) {
  if(DSTK_CKANY(NULL, 1)) {
    T_OBTYPE T = __peek_type(0);

    if(prog && T != PROG)
      DSTK_add(stk_meta2ob(&DSTK, 1, DOPROG));
    else if(list && T != LIST)
      DSTK_add(stk_meta2ob(&DSTK, 1, DOLIST));
    else if(arry && T != ARRY)
      CMD_ARRY1(NULL);
    else {
      int valid = 1;
      int symbolic = 0;

      switch(T) {
      case CPLX: /* OK */
	break;

      case REAL: /* OK */
	if(cplx)
	  __push_cplx2(__pop_real(NULL), 0);
	break;

      case BINT:
	if(cplx)
	  __push_cplx2((T_REAL)__pop_bint(NULL), 0);
	else
	  __push_real((T_REAL)__pop_bint(NULL));
	break;

      case ZINT:
	if(cplx) {
	  T_ZINT z;

	  mpz_init(z);
	  __pop_zint(&z);
	  __push_cplx2((T_REAL)mpz_get_d(z), 0.0);
	  mpz_clear(z);
	}
	else {
	  T_ZINT z;

	  mpz_init(z);
	  __pop_zint(&z);
	  __push_real((T_REAL)mpz_get_d(z));
	  mpz_clear(z);
	}
	break;

      case IDNT:
      case LAM:
	CMD_SYMB1(NULL);
	symbolic = 1;
	break;

      case SYMB:
	symbolic = 1;
	break;

      default: /* OK */
	valid = 0;
	break;
      }

      if(symb && valid && !symbolic)
	CMD_SYMB1(NULL);
    }
  }
}

/* convert level1 and level2 to suitable numeric/symbolic/cons type */
void __coerce2(void) {
  if(DSTK_CKANY(NULL, 2)) {
    int prog = 0;
    int list = 0;
    int arry = 0;
    int symb = 0;
    int cplx = 0;

    /* decide if we need symbolic and/or (complex) for level 1 */
    switch(__peek_type(1)) {
    case PROG:
      prog = 1;
      break;

    case LIST:
      list = 1;
      break;

    case ARRY:
      arry = 1;
      break;

    case IDNT:
    case LAM:
    case SYMB:
      symb = 1;
      break;

    case CPLX:
      cplx = 1;
      break;

    default:
      break;
    }

    /* decide if we need symbolic and/or complex for level 1 */
    switch(__peek_type(0)) {
    case PROG:
      prog = 1;
      break;

    case LIST:
      list = 1;
      break;

    case ARRY:
      arry = 1;
      break;

    case IDNT:
    case LAM:
    case SYMB:
      symb = 1;
      break;

    case CPLX:
      cplx = 1;
      break;

    default:
      break;
    }

    __coerce(prog, list, arry, symb, cplx); DSTK_swap();
    __coerce(prog, list, arry, symb, cplx); DSTK_swap();
  }
}

T_ATOM *stk_meta2ob(T_CONS **STK, size_t z, T_PROLOG *t) {
  T_ATOM *cons;
  T_CONS **fst;

  cons = atom_make_cons(t);
  fst = &(atom_get_cons(cons));
  *fst = NULL;

  /* append STK (starting from TAIL) to composite object */
  for( ; *STK && (z > 0); z --) {
    T_CONS *prog;
    T_ATOM *popped;

    prog = cons_alloc();
    popped = STK_pop(STK);

    prog->car = atom_link(popped);
    prog->cdr = *fst;

    *fst = prog;

    atom_unlink(popped);
  }

  return cons;
}

void explode(T_ATOM *comp) {
  T_CONS *ob;
  size_t z;

  ob = atom_get_cons(comp);

  for(z = 0; ob; z ++, ob = ob->cdr)
    DSTK_add(atom_link(ob->car));

  __push_bint(z);
}

T_ATOM *dovars(T_FLDR *folder) {
  T_ATOM *cons;
  T_CONS **fst, *end;

  cons = atom_make_cons(DOLIST);
  fst = &(atom_get_cons(cons));
  end = *fst = NULL;

  if(folder) {
    T_WORD *L = NULL;

    while((L = lambda_enum(folder, L)) != NULL) {
      T_CONS *prog = cons_alloc();

      prog->car = atom_make_idnt(L->id);
      prog->cdr = NULL;

      if(end)
	end->cdr = prog;
      else
	*fst = prog;

      end = prog;
    }
  }

  return cons;
}

/* mathematical */

const double M_GAMMA = 0.57721566490153286061;

static inline double _factorial(double x) {
  return _gamma(1.0 + x);
}

/* should really be defined somewhere else.. (eg. unix.c) */

void micro_delay(unsigned long usec) {
  struct timespec req, rem;

  req.tv_sec = usec / 1000000;
  req.tv_nsec = 1000 * (usec % 1000000);

  rem.tv_sec = req.tv_sec;
  rem.tv_nsec = req.tv_nsec;

  while(nanosleep(&req, &rem)) {
    req.tv_sec = rem.tv_sec;
    req.tv_nsec = rem.tv_nsec;
  }
}

/*
 * 
 * SysRPL Built-In Implementation
 *
 */

builtin(NOP) {
}

builtin(NIL) {
  DSTK_add(atom_make_cons(DOLIST));
}

builtin(DEPTH) {
  __push_bint(STK_depth(DSTK));
}

builtin(DSTACK) {
  gstack();
  DSTK_dump(win_cmd);
}

builtin(DISPLAY) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *popped = DSTK_pop();
    atom_dump(win_cmd, popped);
    xprintf(win_cmd, "\n");
    atom_unlink(popped);
  }
}

builtin(OFF) {
  FLAGS.mod_run = M_OFF;
}

builtin(DBG) {
  FLAGS.mod_run = M_DBG;
}

builtin(TRACE) {
  FLAGS.mod_run = M_TRC;
}

builtin(CONT) {
  FLAGS.mod_run = M_RUN;
}

builtin(MOD_DAL) {
  FLAGS.mod_symb = MOD_DAL;
}

builtin(MOD_LISP) {
  FLAGS.mod_symb = MOD_LISP;
}

builtin(MOD_RPN) {
  FLAGS.mod_symb = MOD_RPN;
}

builtin(MOD_DEG) {
  FLAGS.mod_angle = MOD_DEG;
}

builtin(MOD_RAD) {
  FLAGS.mod_angle = MOD_RAD;
}

builtin(MOD_POLAR) {
  FLAGS.mod_cplx = MOD_POLAR;
}

builtin(MOD_RECT) {
  FLAGS.mod_cplx = MOD_RECT;
}

builtin(MOD_DEC) {
  FLAGS.mod_bint = MOD_DEC;
}

builtin(MOD_HEX) {
  FLAGS.mod_bint = MOD_HEX;
}

builtin(MOD_OCT) {
  FLAGS.mod_bint = MOD_OCT;
}

builtin(MOD_BIN) {
  FLAGS.mod_bint = MOD_BIN;
}

builtin(MEM) {
  __push_bint(mem_avail());
}

builtin(SLOW) {
  micro_delay(1000);
}

builtin(VERYSLOW) {
  micro_delay(100000);
}

builtin(TIME) {
  __push_bint(time(NULL));
}

builtin(ROOM) {
  xm_statz(0);
}

builtin(CLOCK) {
  struct timeval tm;
  double clk;

  gettimeofday(&tm, NULL);

  clk = (double)tm.tv_sec * 1.0e6 + (double)tm.tv_usec;
  __push_real(clk);
}

builtin(CK1) {
  if(DSTK_CK1(self, BINT)) {
    T_BINT T1 = __pop_bint(NULL);
    int flag;

    flag = DSTK_CK1(self, T1);

    //__push_flag(flag);
  }
}

builtin(CK2) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT T2, T1;
    int flag;

    T1 = __pop_bint(NULL);
    T2 = __pop_bint(NULL);

    flag = DSTK_CK2(self, T2, T1);

    //__push_flag(flag);
  }
}

builtin(CK3) {
  if(DSTK_CK3(self, BINT, BINT, BINT)) {
    T_BINT T3, T2, T1;
    int flag;

    T1 = __pop_bint(NULL);
    T2 = __pop_bint(NULL);
    T3 = __pop_bint(NULL);

    flag = DSTK_CK3(self, T3, T2, T1);

    //__push_flag(flag);
  }
}

builtin(CKN) {
  DSTK_CKN(self);
}

builtin(NEWOB) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *newob = atom_copy(DSTK->car);

    DSTK_drop();
    DSTK_add(newob);
  }
}

builtin(CKREF) {
  if(DSTK_CKANY(self, 1) && DSTK->car && (atom_ref(DSTK->car) > 1))
    CMD_NEWOB(self);
}

builtin(LOAD) {
  if(DSTK_CK1(self, CSTR | IDNT | LAM)) {
    char *name;
    T_ATOM *ob;
    char *stream;

    ob = DSTK_pop();
    name = atom_get_name(ob);

    stream = file_read(name);
    atom_unlink(ob);

    if(stream) {
      /* embed stream in a list */
      size_t len = strlen(stream);
      char *list;

      list = malloc(len + 5);
      strcpy(list, "{ ");
      strcat(list, stream);
      strcat(list, " }");

      free(stream);

      eval_string(list);
      free(list);
    } else
      DSTK_add(atom_make_code(CMD_NIL));
  }
}

builtin(EXEC) {
  if(DSTK_CK1(self, CSTR | IDNT | LAM)) {
    char *name;
    T_ATOM *ob;
    char *stream;

    ob = DSTK_pop();
    name = atom_get_name(ob);
    
    stream = file_read(name);
    atom_unlink(ob);

    if(stream) {
      eval_string(stream);
      free(stream);
    } else
      Exception(self, "Unable to read file", 0);
  }
}

builtin(TYPE) {
  if(DSTK_CKANY(self, 1)) {
    T_OBTYPE T = __peek_type(0);

    DSTK_drop();
    __push_bint((T_BINT)T);
  }
}

builtin(DUPTYPE) {
  if(DSTK_CKANY(self, 1)) {
    T_OBTYPE T = __peek_type(0);

    __push_bint((T_BINT)T);
  }
}

builtin(SAME) {
  if(DSTK_CKANY(self, 2)) {
    T_ATOM *left, *right;
    int flag;

    right = DSTK_pop();
    left = DSTK_pop();

    flag = (left == right); /* both objects are the same object */

    DSTK_add(atom_make_flag(flag));

    atom_unlink(left);
    atom_unlink(right);
  }
}

builtin(EQ) {
  if(DSTK_CKANY(self, 2)) {
    T_ATOM *left, *right;
    int flag;

    right = DSTK_pop();
    left = DSTK_pop();

    flag = atom_equal(left, right);

    DSTK_add(atom_make_flag(flag));

    atom_unlink(left);
    atom_unlink(right);
  }
}

builtin(QUOTE) { /* prefix */
  DSTK_add(atom_link(IPTR->car)); 
  __NEXT;
}

builtin(SKIP) { /* prefix */
  __NEXT;
}

builtin(QUOTE_R) { /* prefix */
  if(RSTK_LEVEL >= 0) {
    T_CONS *parent = RSTK[RSTK_LEVEL];
    
    if(parent && parent->car) {
      DSTK_add(atom_link(parent->car));
      RSTK[RSTK_LEVEL] = parent->cdr;
    } else
      DSTK_add(atom_make_code(CMD_NIL));
  } else
    DSTK_add(atom_make_code(CMD_NIL));
}

builtin(TIC_R) { /* prefix */
  if(RSTK_LEVEL >= 0) {
    T_CONS *parent = RSTK[RSTK_LEVEL];

    if(parent && parent->car) {
      DSTK_add(atom_link(parent->car));
      RSTK[RSTK_LEVEL] = parent->cdr;
      DSTK_add(atom_make_flag(1));
    } else {
      ISTK_kill();
      DSTK_add(atom_make_flag(0));
    }
  } else {
    DSTK_add(atom_make_flag(0));
  }
}

builtin(TO_R) {
  if(DSTK_CK1(self, PROG | LIST)) {
    T_ATOM *ob = DSTK_pop();

    ISTK[ISTK_LEVEL + 1] = ISTK[ISTK_LEVEL];
    ISTK[ISTK_LEVEL] = RSTK[++ RSTK_LEVEL] = cons_link(atom_get_cons(ob));

    ISTK_LEVEL ++;
    atom_unlink(ob);
  }
}

builtin(TAIL_R) { /* push parent->IPTR tail to DSTK, remove from runstream */
  T_ATOM *prog;
  T_CONS *cons;

  cons = RSTK[RSTK_LEVEL];

  prog = atom_make_cons(DOPROG);
  atom_get_cons(prog) = cons_link(cons);

  RSTK_LEVEL --;
  cons_unlink(ISTK[ISTK_LEVEL --]);

  DSTK_add(prog); /* push runstream object (or null program) */
}

builtin(RCL_R) {
  T_ATOM *prog;
  T_CONS *cons;

  cons = cons_link(RSTK[RSTK_LEVEL]);

  prog = atom_make_cons(DOPROG);
  atom_get_cons(prog) = cons;

  DSTK_add(prog); /* push runstream object (or null program) */
}

builtin(SELF) {
  T_ATOM *prog;
  T_CONS *cons;

  cons = cons_link(ISTK[ISTK_LEVEL]);

  prog = atom_make_cons(DOPROG);
  atom_get_cons(prog) = cons;

  DSTK_add(prog); /* push runstream object */
}

builtin(RDROP) {
  ISTK_kill();
}

builtin(IDUP) {
  RSTK_dup();
}

builtin(BEGIN) {
  RSTK_dup();
}

builtin(AGAIN) {
  IPTR = RSTK_peek(); /* rewind */
}

builtin(REPEAT) {
  IPTR = RSTK_peek(); /* rewind */
}

builtin(WHILE) {
  if(DSTK_CKANY(self, 1)) {
    if(! __pop_flag(NULL)) {
      RSTK_pop();

      __NEXT;
      __NEXT;
    }
  }
}

builtin(UNTIL) {
  if(DSTK_CKANY(self, 1)) {
    if(__pop_flag(NULL)) {
      RSTK_pop();
    } else
      IPTR = RSTK[RSTK_LEVEL]; /* jump back to start of runstream */
  }
}

builtin(RECURSE) {
  IPTR = ISTK[ISTK_LEVEL]; /* rewind */
}

builtin(COLA) { /* prefix */
  T_ATOM *ob;

  ob = atom_link(IPTR->car);
  ISTK_kill();

  atom_eval(ob, NULL);
  atom_unlink(ob);
}

builtin(IT) { /* prefix if-then */
  if(DSTK_CKANY(self, 1)) {
    if(__pop_flag(NULL)) { 
      T_ATOM *ob = atom_link(IPTR->car);
      __NEXT;

      atom_eval(ob, NULL);
      atom_unlink(ob);
    } else
      __NEXT; /* skip next runstream object */
  }
}

builtin(QSKIP) { /* prefix if-not-then */
  if(DSTK_CKANY(self, 1)) {
    if(__pop_flag(NULL)) {
      __NEXT;
    } else {
      T_ATOM *ob = atom_link(IPTR->car);
      __NEXT;

      atom_eval(ob, NULL);
      atom_unlink(ob);
    }
  }
}

builtin(ITE) { /* prefix if-then-else */
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *ob;

    if(__pop_flag(NULL)) { 
      ob = atom_link(IPTR->car); /* ob if true */

      __NEXT; /* skip false object */
    } else {
      __NEXT; /* skip true object */
    
      ob = atom_link(IPTR->car); /* ob if false */
    }

    __NEXT;

    atom_eval(ob, NULL);
    atom_unlink(ob);
  }
}

builtin(CASE) { /* prefix case */
  if(DSTK_CKANY(self, 1)) {
    if(__pop_flag(NULL)) { /* if true, do COLA */
      T_ATOM *ob = atom_link(IPTR->car);

      ISTK_kill();

      atom_eval(ob, NULL);
      atom_unlink(ob);
    } else /* if not the case, SKIP and continue.. */
      __NEXT;
  }
}

builtin(QSEMI) { /* exit if true */
  if(DSTK_CKANY(self, 1)) {
    if(__pop_flag(NULL))
      ISTK_kill();
  }
}

builtin(FALSE) {
  DSTK_add(atom_link(self));
}

builtin(TRUE) {
  DSTK_add(atom_link(self));
}

builtin(NOT) {
  if(DSTK_CKANY(self, 1)) {
    DSTK_add(atom_make_flag(! __pop_flag(NULL)));
  }
}

builtin(OR) {
  if(DSTK_CKANY(self, 2)) {
    int a, b;

    b = __pop_flag(NULL);
    a = __pop_flag(NULL);

    DSTK_add(atom_make_flag(a || b));
  }
}

builtin(AND) {
  if(DSTK_CKANY(self, 2)) {
    int a, b;

    b = __pop_flag(NULL);
    a = __pop_flag(NULL);

    DSTK_add(atom_make_flag(a && b));
  }
}

builtin(XOR) {
  if(DSTK_CKANY(self, 2)) {
    int a, b;

    b = __pop_flag(NULL);
    a = __pop_flag(NULL);

    DSTK_add(atom_make_flag((a && !b) || (!a && b)));
  }
}

builtin(EVAL) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *ob;

    ob = DSTK_pop();
    atom_eval(ob, NULL);
    atom_unlink(ob);
  }
}

builtin(COMPEVAL) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *ob;

    ob = DSTK_pop();

    if(atom_q_cons(ob))
      atom_eval(ob, DOPROG);
    else if(ob->prolog == DOSYMB)
      atom_eval(ob, DOSYMBEVAL);
    else
      atom_eval(ob, NULL);

    atom_unlink(ob);
  }
}

builtin(EVAL_STR) {
  if(DSTK_CK1(self, CSTR)) {
    T_ATOM *ob;

    ob = DSTK_pop();
    eval_string(atom_get_name(ob));
    atom_unlink(ob);
  }
}

builtin(PARSE_STR) {
  if(DSTK_CK1(self, CSTR)) {
    T_ATOM *ob;

    ob = DSTK_pop();
    parse_string(atom_get_name(ob));
    atom_unlink(ob);
  }
}

builtin(NEXT_TOKEN) {
  if(DSTK_CK1(self, CSTR)) {
    T_ATOM *str;
    T_ATOM *ob;
    const char *stream;

    str = DSTK_pop();
    stream = atom_get_name(str);
    ob = NULL;

    if(stream && *stream) {
      stream = next_token(&ob, stream);

      DSTK_add(atom_make_cstr((T_NAME)stream));
      DSTK_add(ob);

      DSTK_add(atom_make_flag(1));
    } else
      DSTK_add(atom_make_flag(0));

    atom_unlink(str);
  }
}

builtin(DO_STR) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM *popped;
    char *text = NULL;

    popped = DSTK_pop();

    text = atom_string(popped, &text);
    DSTK_add(atom_make_cstr(text));

    if(text)
      free(text);

    atom_unlink(popped);
  }
}

builtin(CATCH) { /* prefix ( ob handler -- ) */
  T_ATOM *ob = atom_link(IPTR->car);

  __NEXT;             /* runstream pointing to handler */
  __NEXT;             /* runstream pointing to next object */

  atom_eval(ob, NULL);
  atom_unlink(ob);
}

builtin(THROW) { /* ERRJMP */
  Exception(NULL, NULL, 0);
}

builtin(THROW_MSG) { /* prefix ( msg -- ) */
  if(IPTR) {
    T_ATOM *arg1 = IPTR->car;

    if(atom_q_name(arg1)) {
      char *msg = atom_get_name(arg1);

      /* skip past message */
      __NEXT;

      /* generate (non-fatal) exception */
      Exception(NULL, msg, 0);
    } else
      Exception(self, "Expecting message", 0);
  } else
    Exception(self, "Null IPTR", 0);
}

/* Counted loop */

builtin(DO) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT Index, Stop;

    Index = __pop_bint(NULL);
    Stop = __pop_bint(NULL);

    LOOP_add(Index, Stop);
    RSTK_dup();
  }
}

builtin(ZERO_DO) {
  if(DSTK_CK1(self, BINT)) {
    T_BINT Stop;

    Stop = __pop_bint(NULL);

    LOOP_add(0, Stop);
    RSTK_dup();
  }
}

builtin(ONE_DO) {
  if(DSTK_CK1(self, BINT)) {
    T_BINT Stop;

    Stop = __pop_bint(NULL);

    LOOP_add(1, Stop);
    RSTK_dup();
  }
}

builtin(LOOP) {
  if(++ LOOP[LOOP_LEVEL].Index >= LOOP[LOOP_LEVEL].Stop) {
    LOOP_drop();
    RSTK_pop();
  } else
    IPTR = RSTK_peek(); /* rewind */
}

builtin(INDEX_RCL) { __push_bint(LOOP_Index(0)); }
builtin(ISTOP_RCL) { __push_bint(LOOP_Stop(0)); }

builtin(INDEX_STO) { 
  if(DSTK_CK1(self, BINT)) {
    LOOP[LOOP_LEVEL].Index = __pop_bint(NULL); 
  }
}

builtin(ISTOP_STO) { 
  if(DSTK_CK1(self, BINT)) {
     LOOP[LOOP_LEVEL].Stop = __pop_bint(NULL);
  }
}

/* Word/Lambda */

builtin(STO) {
  if(DSTK_CK1(self, LAM | IDNT | BINT)) {
    T_ATOM *ob = DSTK_pop();
    T_OBTYPE t = atom_q_type(ob);

    if(t == LAM || t == IDNT) {
      char *id;
      T_FLDR *path;

      id = xm_strdup(atom_get_name(ob));
      path = (atom_q_type(ob) == LAM) ? FOLDER_LAMBDA : FOLDER_CWD;
      atom_unlink(ob);

      ob = DSTK_pop();
      lambda_set(path, id, ob, 0);

      xm_free(id);
    }
    else if(t == BINT) { /* fetch (lam only) by index */
      size_t ndx;

      ndx = atom_get_bint(ob);
      atom_unlink(ob);

      ob = DSTK_pop();
      lambda_setn(FOLDER_LAMBDA, ndx, ob);
    }

    atom_unlink(ob);
  }
}

builtin(EXIST) {
  if(DSTK_CK1(self, LAM | IDNT | BINT)) {
    T_ATOM *ob = DSTK_pop();
    T_OBTYPE t = atom_q_type(ob);
    T_WORD *L = NULL;

    if(t == LAM || t == IDNT) {
      char *id;
      T_FLDR *path;

      id = atom_get_name(ob);

      path = (t == LAM) ? FOLDER_LAMBDA : FOLDER_CWD;
      L = lambda_find(path, id, 1);
    }
    else if(t == BINT) { /* fetch (lam only) by index */
      size_t ndx;

      ndx = atom_get_bint(ob);
      L = lambda_findn(FOLDER_LAMBDA, ndx);
    }

    if(L) {
      atom_unlink(ob);
      
      DSTK_add(atom_link(L->ob));
      DSTK_add(atom_make_flag(1));
    } else {
      DSTK_add(ob);
      DSTK_add(atom_make_flag(0));
    }
  }
}

builtin(RCL) {
  if(DSTK_CK1(self, LAM | IDNT | BINT)) {
    T_ATOM *ob = DSTK_pop();
    T_OBTYPE t = atom_q_type(ob);
    T_WORD *L = NULL;

    if(t == LAM || t == IDNT) {
      char *id;
      T_FLDR *path;

      id = atom_get_name(ob);

      path = (t == LAM) ? FOLDER_LAMBDA : FOLDER_CWD;
      L = lambda_find(path, id, 1);
    }
    else if(t == BINT) { /* fetch (lam only) by index */
      size_t ndx;

      ndx = atom_get_bint(ob);
      L = lambda_findn(FOLDER_LAMBDA, ndx);
    }

    if(L) {
      DSTK_add(atom_link(L->ob));
      atom_unlink(ob);
    }
    else
      DSTK_add(ob);
  }
}

builtin(PURGE) {
  if(DSTK_CK1(self, IDNT)) {
    T_ATOM *ob;
    char *id;

    ob = DSTK_pop();
    id = atom_get_name(ob);
    atom_unlink(ob);

    if(lambda_purge(FOLDER_CWD, id, 0))
      Exception(self, "Undefined word", 0);
  }
}

/* Type */

builtin(_BINT) { __push_bint(BINT); }
builtin(_REAL) { __push_bint(REAL); }

builtin(_CPLX) { __push_bint(CPLX); }

builtin(_ZINT) { __push_bint(ZINT); }
builtin(_CSTR) { __push_bint(CSTR); }
builtin(_CHAR) { __push_bint(CHAR); }
builtin(_IDNT) { __push_bint(IDNT);} 
builtin(_LAM)  { __push_bint(LAM ); }
builtin(_CODE) { __push_bint(CODE); }
builtin(_ARRY) { __push_bint(ARRY); }
builtin(_SYMB) { __push_bint(SYMB); }
builtin(_LIST) { __push_bint(LIST); }
builtin(_PROG) { __push_bint(PROG); }

/* Integer */

builtin(BINT_ZERO)  { __push_bint(0); }
builtin(BINT_ONE)   { __push_bint(1); }
builtin(BINT_TWO)   { __push_bint(2); }
builtin(BINT_THREE) { __push_bint(3); }
builtin(BINT_FOUR)  { __push_bint(4); }
builtin(BINT_FIVE)  { __push_bint(5); }

builtin(BINT_PLUS) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a + b);
  }
}

builtin(BINT_MINUS) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a - b);
  }
}

builtin(BINT_TIMES) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a * b);
  }
}

builtin(BINT_DIVIDE) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;
    ldiv_t q;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    q = ldiv(a, b);

    __push_bint(q.rem);
    __push_bint(q.quot);
  }
}

builtin(BINT_OR) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a | b);
  }
}

builtin(BINT_AND) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a & b);
  }
}

builtin(BINT_XOR) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    __push_bint(a ^ b);
  }
}

builtin(BINT_NOT) {
  if(DSTK_CK1(self, BINT)) {
    T_BINT a;

    a = __pop_bint(NULL);

    __push_bint(~ a);
  }
}

builtin(BINT_LESS) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a < b));
  }
}

builtin(BINT_LESS_EQ) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a <= b));
  }
}

builtin(BINT_MORE) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a > b));
  }
}

builtin(BINT_MORE_EQ) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a >= b));
  }
}

builtin(BINT_EQUAL) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a == b));
  }
}

builtin(BINT_NOT_EQUAL) {
  if(DSTK_CK2(self, BINT, BINT)) {
    T_BINT a, b;

    b = __pop_bint(NULL);
    a = __pop_bint(NULL);

    DSTK_add(atom_make_flag(a != b));
  }
}

builtin(BINT_Q_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) == 0));
}

builtin(BINT_Q_NOT_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) != 0));
}

builtin(BINT_MORE_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) > 0));
}

builtin(BINT_MORE_EQ_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) >= 0));
}

builtin(BINT_LESS_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) < 0));
}

builtin(BINT_LESS_EQ_ZERO) {
  if(DSTK_CK1(self, BINT))
    DSTK_add(atom_make_flag(__pop_bint(NULL) <= 0));
}

/* ZINT */

builtin(ZINT_PLUS) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, z;

    mpz_init(a);
    mpz_init(b);
    mpz_init(z);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_add(z, a, b);
    __push_zint(z);

    mpz_clear(z);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_MINUS) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, z;

    mpz_init(a);
    mpz_init(b);
    mpz_init(z);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_sub(z, a, b);
    __push_zint(z);

    mpz_clear(z);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_TIMES) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, z;

    mpz_init(a);
    mpz_init(b);
    mpz_init(z);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_mul(z, a, b);
    __push_zint(z);

    mpz_clear(z);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_DIVIDE) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, z;

    mpz_init(a);
    mpz_init(b);
    mpz_init(z);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_cdiv_q(z, a, b); /* cdiv, fdiv, tdiv */
    __push_zint(z);

    mpz_clear(z);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_DIV_QR) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, q, r;

    mpz_init(a);
    mpz_init(b);
    mpz_init(q);
    mpz_init(r);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_cdiv_qr(q, r, a, b); /* cdiv, fdiv, tdiv */

    __push_zint(q);
    __push_zint(r);

    mpz_clear(r);
    mpz_clear(q);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_MOD) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, z;

    mpz_init(a);
    mpz_init(b);
    mpz_init(z);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_cdiv_r(z, a, b); /* cdiv, fdiv, tdiv */
    __push_zint(z);

    mpz_clear(z);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_GCD) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, g;

    mpz_init(a);
    mpz_init(b);
    mpz_init(g);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_gcd(g, a, b);
    __push_zint(g);

    mpz_clear(g);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_EGCD) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b, g, s, t;

    mpz_init(a);
    mpz_init(b);
    mpz_init(g);
    mpz_init(s);
    mpz_init(t);

    __pop_zint(&b);
    __pop_zint(&a);

    mpz_gcdext(g, s, t, a, b);

    __push_zint(g);
    __push_zint(s);
    __push_zint(t);

    mpz_clear(t);
    mpz_clear(s);
    mpz_clear(g);
    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_ABS) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    mpz_abs(a, a);
    __push_zint(a);

    mpz_clear(a);
  }
}

builtin(ZINT_NEG) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    mpz_neg(a, a);
    __push_zint(a);

    mpz_clear(a);
  }
}

builtin(ZINT_SGN) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;
    int s;

    mpz_init(a);
    __pop_zint(&a);

    s = mpz_sgn(a);
    mpz_set_si(a, s);
    __push_zint(a);

    mpz_clear(a);
  }
}

builtin(ZINT_FACTORIAL) {
  if(DSTK_CK1(self, BINT | ZINT)) {
    T_ZINT z;
    T_BINT a = 0;

    mpz_init(z);

    switch(__peek_type(0)) {
    case BINT: a = __pop_bint(NULL); break;
    case ZINT: __pop_zint(&z); a = mpz_get_ui(z); break;
    default: break;
    }

    mpz_fac_ui(z, a);
    __push_zint(z);

    mpz_clear(z);
  }
}

builtin(ZINT_LESS) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) < 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_LESS_EQ) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) <= 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_MORE) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) > 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_MORE_EQ) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) >= 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_EQUAL) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) == 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_NOT_EQUAL) {
  if(DSTK_CK2(self, ZINT, ZINT)) {
    T_ZINT a, b;

    mpz_init(a);
    mpz_init(b);

    __pop_zint(&b);
    __pop_zint(&a);

    __push_flag(mpz_cmp(a, b) != 0);

    mpz_clear(b);
    mpz_clear(a);
  }
}

builtin(ZINT_Q_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) == 0);

    mpz_clear(a);
  }
}

builtin(ZINT_Q_NOT_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) != 0);

    mpz_clear(a);
  }
}

builtin(ZINT_MORE_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) > 0);

    mpz_clear(a);
  }
}

builtin(ZINT_LESS_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) < 0);

    mpz_clear(a);
  }
}

builtin(ZINT_MORE_EQ_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) >= 0);

    mpz_clear(a);
  }
}

builtin(ZINT_LESS_EQ_ZERO) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT a;

    mpz_init(a);
    __pop_zint(&a);

    __push_flag(mpz_cmp_ui(a, 0) <= 0);

    mpz_clear(a);
  }
}

builtin(REAL_ZERO)      { __push_real(0.0); }
builtin(REAL_ONE)       { __push_real(1.0); }
builtin(REAL_NEG_ONE)   { __push_real(-1.0); }
builtin(REAL_PI)        { __push_real(4.0 * atan(1.0)); }
builtin(REAL_E)         { __push_real(exp(1.0)); }
builtin(REAL_G)         { __push_real(M_GAMMA); }

builtin(REAL_PLUS) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    __push_real(a + b);
  }
}

builtin(REAL_MINUS) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    __push_real(a - b);
  }
}

builtin(REAL_TIMES) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    __push_real(a * b);
  }
}

builtin(REAL_DIVIDE) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    __push_real(a / b);
  }
}

builtin(REAL_POWER) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_CPLX a, b, z;

    a.x = __pop_real(NULL); a.y = 0;
    b.x = __pop_real(NULL); b.y = 0;

    z = cplx_pow(a, b);

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_NROOT) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_CPLX a, b, z;

    a.x = __pop_real(NULL); a.y = 0;
    b.x = __pop_real(NULL); b.y = 0;

    z = cplx_pow(a, cplx_inv(b));

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_MOD) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b, x;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    x = a - b*floor(a/b);

    __push_real(x);
  }
}

builtin(REAL_ANGLE) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    __push_real(atan2(b, a));
  }
}

builtin(REAL_COMB) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b, m = 0;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    if(b > 0.0 && a >= b) {
      if(a >= 0.0 && floor(a) == a && b >= 0.0 && floor(b) == b) {
	T_REAL b2 = a - b;
	for(m = 1; a > b2; a -= 1.0)
	  m *= a;
      } else
	m = _factorial(a) / _factorial(a - b);

      m /= _factorial(b);
    }

    __push_real(m);
  }
}

builtin(REAL_PERM) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b, m = 0;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    if(b > 0.0 && a >= b) {
      if(a >= 0.0 && floor(a) == a && b >= 0.0 && floor(b) == b) {
	for(b = a - b, m = 1; a > b; a -= 1.0)
	  m *= a;
      } else
	m = _factorial(a) / _factorial(a - b);
    }

    __push_real(m);
  }
}

builtin(REAL_ABS) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(fabs(a));
  }
}

builtin(REAL_SGN) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real((a > 0.0) ? 1.0 : ((a < 0.0) ? -1.0 : 0.0));
  }
}

builtin(REAL_NEG) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(-a);
  }
}

builtin(REAL_INV) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0 / a);
  }
}

builtin(REAL_SQRT) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    if(a >= 0.0)
      __push_real(sqrt(a));
    else {
      T_CPLX z;

      z.x = a;
      z.y = 0;

      __push_cplx(cplx_sqrt(z));
    }
  }
}

builtin(REAL_EXP) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(exp(a));
  }
}

builtin(REAL_EXPM1) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(expm1(a));
  }
}

builtin(REAL_LN) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(log(a));
  }
}

builtin(REAL_LNP1) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(log1p(a));
  }
}

builtin(REAL_LOG) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(log10(a));
  }
}

builtin(REAL_ALOG) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(pow(10.0, a));
  }
}

builtin(REAL_COS) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(cos(a));
  }
}

builtin(REAL_COSH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(cosh(a));
  }
}

builtin(REAL_ACOS) {
  if(DSTK_CK1(self, REAL)) {
    T_CPLX a, z;

    a.x = __pop_real(NULL);
    a.y = 0;

    z = cplx_acos(a);

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_ACOSH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(acosh(a));
  }
}

builtin(REAL_SEC) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/cos(a));
  }
}

builtin(REAL_SECH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/cosh(a));
  }
}

builtin(REAL_ASEC) {
  if(DSTK_CK1(self, REAL)) {
    T_CPLX a, z;

    a.x = 1.0 / __pop_real(NULL);
    a.y = 0;

    z = cplx_acos(a);

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_ASECH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(acosh(1.0/a));
  }
}

builtin(REAL_SIN) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(sin(a));
  }
}

builtin(REAL_SINH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(sinh(a));
  }
}

builtin(REAL_ASIN) {
  if(DSTK_CK1(self, REAL)) {
    T_CPLX a, z;

    a.x = __pop_real(NULL);
    a.y = 0;

    z = cplx_asin(a);

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_ASINH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(asinh(a));
  }
}

builtin(REAL_CSC) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/sin(a));
  }
}

builtin(REAL_CSCH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/sinh(a));
  }
}

builtin(REAL_ACSC) {
  if(DSTK_CK1(self, REAL)) {
    T_CPLX a, z;

    a.x = 1.0 / __pop_real(NULL);
    a.y = 0;

    z = cplx_asin(a);

    if(z.y)
      __push_cplx(z);
    else
      __push_real(z.x);
  }
}

builtin(REAL_ACSCH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(asinh(1.0/a));
  }
}

builtin(REAL_TAN) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(tan(a));
  }
}

builtin(REAL_TANH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(tanh(a));
  }
}

builtin(REAL_ATAN) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(atan(a));
  }
}

builtin(REAL_ATANH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(atanh(a));
  }
}

builtin(REAL_COT) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/tan(a));
  }
}

builtin(REAL_COTH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(1.0/tanh(a));
  }
}

builtin(REAL_ACOT) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(atan(1.0/a));
  }
}

builtin(REAL_ACOTH) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(atanh(1.0/a));
  }
}

builtin(REAL_IP) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);
    T_REAL n;

    (void) modf(a, &n);

    __push_real(n);
  }
}

builtin(REAL_FP) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);
    T_REAL n;

    __push_real(modf(a, &n));
  }
}

builtin(REAL_FLOOR) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(floor(a));
  }
}

builtin(REAL_CEIL) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(ceil(a));
  }
}

builtin(REAL_FACTORIAL) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(_factorial(a));
  }
}

builtin(REAL_GAMMA) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(_gamma(a));
  }
}

builtin(REAL_ERF) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(_erf(a));
  }
}

builtin(REAL_ERFC) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(_erfc(a));
  }
}

builtin(REAL_IERFC) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    __push_real(_ierfc(a));
  }
}

builtin(REAL_LESS) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a < b));
  }
}

builtin(REAL_LESS_EQ) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a <= b));
  }
}

builtin(REAL_MORE) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a > b));
  }
}

builtin(REAL_MORE_EQ) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a >= b));
  }
}

builtin(REAL_EQUAL) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a == b));
  }
}

builtin(REAL_NOT_EQUAL) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_REAL a, b;

    b = __pop_real(NULL);
    a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a != b));
  }
}

builtin(REAL_Q_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a == 0.0));
  }
}

builtin(REAL_Q_NOT_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a != 0.0));
  }
}

builtin(REAL_MORE_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a > 0.0));
  }
}

builtin(REAL_MORE_EQ_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a >= 0.0));
  }
}

builtin(REAL_LESS_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a < 0.0));
  }
}

builtin(REAL_LESS_EQ_ZERO) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL a = __pop_real(NULL);

    DSTK_add(atom_make_flag(a <= 0.0));
  }
}

/* CPLX */

builtin(CPLX_ZERO)     { __push_cplx2(0, 0); }
builtin(CPLX_ONE)      { __push_cplx2(1, 0); }
builtin(CPLX_NEG_ONE)  { __push_cplx2(-1, 0); }
builtin(CPLX_J)        { __push_cplx2(0, 1); }

builtin(CPLX_REAL) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX z = __pop_cplx(NULL);

    __push_real(z.x);
  }
}

builtin(CPLX_IMAG) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX z = __pop_cplx(NULL);

    __push_real(z.y);
  }
}

builtin(CPLX_PLUS) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_add(a, b);
    __push_cplx(z);
  }
}

builtin(CPLX_MINUS) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_sub(a, b);
    __push_cplx(z);
  }
}

builtin(CPLX_TIMES) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_mul(a, b);
    __push_cplx(z);
  }
}

builtin(CPLX_DIVIDE) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_div(a, b);
    __push_cplx(z);
  }
}

builtin(CPLX_MOD) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_div(a, b);
    z.x = floor(z.x);
    z.y = floor(z.y);

    z = cplx_sub(a, cplx_mul(b, z));
    __push_cplx(z);
  }
}

builtin(CPLX_POWER) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_pow(a, b);
    __push_cplx(z);
  }
}

builtin(CPLX_ANGLE) {
  if(DSTK_CK1(self, CPLX)) {
    __push_real(cplx_angle(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ABS) {
  if(DSTK_CK1(self, CPLX)) {
    __push_real(cplx_abs(__pop_cplx(NULL)));
  }
}

builtin(CPLX_NEG) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX z = __pop_cplx(NULL);

    __push_cplx(cplx_neg(z));
  }
}

builtin(CPLX_SGN) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX z = __pop_cplx(NULL);

    __push_cplx(cplx_scale(z, 1.0/cplx_abs(z)));
  }
}

builtin(CPLX_CONJ) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_conj(__pop_cplx(NULL)));
  }
}

builtin(CPLX_INV) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(__pop_cplx(NULL)));
  }
}

builtin(CPLX_SQRT) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_sqrt(__pop_cplx(NULL)));
  }
}

builtin(CPLX_NROOT) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, z;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    z = cplx_pow(a, cplx_inv(b));
    __push_cplx(z);
  }
}

builtin(CPLX_EXP) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_exp(__pop_cplx(NULL)));
  }
}

builtin(CPLX_LN) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_log(__pop_cplx(NULL)));
  }
}


builtin(CPLX_LOG) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_log10(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ALOG) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX ten = cplx(10.0, 0);

    __push_cplx(cplx_pow(ten, __pop_cplx(NULL)));
  }
}

builtin(CPLX_SIN) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_sin(__pop_cplx(NULL)));
  }
}

builtin(CPLX_SINH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_sinh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ASIN) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_asin(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ASINH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_asinh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_COS) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_cos(__pop_cplx(NULL)));
  }
}

builtin(CPLX_COSH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_cosh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ACOS) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_acos(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ACOSH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_acosh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_TAN) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_tan(__pop_cplx(NULL)));
  }
}

builtin(CPLX_TANH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_tanh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ATAN) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_atan(__pop_cplx(NULL)));
  }
}

builtin(CPLX_ATANH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_atanh(__pop_cplx(NULL)));
  }
}

builtin(CPLX_SEC) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_cos(__pop_cplx(NULL))));
  }
}

builtin(CPLX_SECH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_cosh(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ASEC) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_acos(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ASECH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_acosh(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_CSC) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_sin(__pop_cplx(NULL))));
  }
}

builtin(CPLX_CSCH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_sinh(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ACSC) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_asin(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ACSCH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_asinh(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_COT) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_tan(__pop_cplx(NULL))));
  }
}

builtin(CPLX_COTH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_inv(cplx_tanh(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ACOT) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_atan(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_ACOTH) {
  if(DSTK_CK1(self, CPLX)) {
    __push_cplx(cplx_atanh(cplx_inv(__pop_cplx(NULL))));
  }
}

builtin(CPLX_EQUAL) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    DSTK_add(atom_make_flag(a.x == b.x && a.y == b.y));
  }
}

builtin(CPLX_NOT_EQUAL) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b;

    b = __pop_cplx(NULL);
    a = __pop_cplx(NULL);

    DSTK_add(atom_make_flag(a.x != b.x || a.y != b.y));
  }
}

builtin(CPLX_Q_ZERO) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX a;

    a = __pop_cplx(NULL);

    DSTK_add(atom_make_flag(cplx_abs(a) == 0.0));
  }
}

builtin(CPLX_Q_NOT_ZERO) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX a;

    a = __pop_cplx(NULL);

    DSTK_add(atom_make_flag(cplx_abs(a) > 0.0));
  }
}

builtin(CPLX_FACTORIAL) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX a;

    a = __pop_cplx(NULL);
    a.x += 1.0;

    __push_cplx(cplx_gamma(a));
  }
}

builtin(CPLX_GAMMA) {
  if(DSTK_CK1(self, CPLX)) {
    T_CPLX a;

    a = __pop_cplx(NULL);
    __push_cplx(cplx_gamma(a));
  }
}

builtin(CPLX_COMB) {
  if(DSTK_CK2(self, CPLX, CPLX)) {
    T_CPLX a, b, m;

    b = __pop_cplx(NULL); b.x += 1.0;
    a = __pop_cplx(NULL); a.x += 1.0;

    m = cplx_div(cplx_gamma(a), cplx_gamma(cplx_sub(a, b)));
    m = cplx_div(m, cplx_gamma(b));
    
    __push_cplx(m);
  }
}

builtin(CPLX_PERM) {
  if(DSTK_CK2(self, REAL, REAL)) {
    T_CPLX a, b, m;

    b = __pop_cplx(NULL); b.x += 1.0;
    a = __pop_cplx(NULL); a.x += 1.0;

    m = cplx_div(cplx_gamma(a), cplx_gamma(cplx_sub(a, b)));
    
    __push_cplx(m);
  }
}


/* STACK */

builtin(DUP) {
  if(DSTK_CKANY(self, 1))
    DSTK_dup();
}

builtin(DROP) {
  if(DSTK_CKANY(self, 1))
    DSTK_drop();
}

builtin(SWAP) {
  if(DSTK_CKANY(self, 2))
    DSTK_swap();
}

builtin(OVER) {
  if(DSTK_CKANY(self, 2))
    DSTK_add(atom_link(DSTK->cdr->car));
}

builtin(ROT) {
  if(DSTK_CKANY(self, 3))
    DSTK_roll(2);
}

builtin(UNROT) {
  if(DSTK_CKANY(self, 3))
    DSTK_unroll(2);
}

builtin(PICK) {
  if(DSTK_CKN(self)) {
    T_BINT level = __pop_bint(NULL);

    if(level > 0)
      DSTK_add(atom_link(DSTK_peek(level - 1)));
  }
}

builtin(ROLL) {
  if(DSTK_CKN(self)) {
    T_BINT level = __pop_bint(NULL);

    if(level > 0)
      DSTK_roll(level - 1);
  }
}

builtin(UNROLL) {
  if(DSTK_CKN(self)) {
    T_BINT level = __pop_bint(NULL);

    if(level > 0)
      DSTK_unroll(level - 1);
  }
}

builtin(BIND) {
  if(DSTK_CK1(self, LIST)) {
    T_ATOM *map = DSTK_pop();
    T_CONS *ob;
    size_t z;

    lam_call();

    ob = atom_get_cons(map);

    for(z = 0; ob; ob = ob->cdr)
      z ++;

    if(z) {
      size_t j;

      ob = atom_get_cons(map);

      for(j = z; ob; ob = ob->cdr) {
	T_ATOM *level;
	char *id;

	id = atom_get_name(ob->car);
	level = DSTK_peek(-- j);

	lambda_set(FOLDER_LAMBDA, id, level, 1);
      }

      for(j = z; j > 0; j --)
	DSTK_drop();
    }

    atom_unlink(map);
  }
}

builtin(ABND) {
  lam_kill();
}

builtin(CACHE) {
  if(DSTK_CKN(self)) {
    T_BINT j, z;

    z = atom_get_bint(DSTK->car);

    lam_call();

    for(j = z + 1; j > 0; ) {
      T_ATOM *level;
      
      level = DSTK_peek(-- j);
      lambda_set(FOLDER_LAMBDA, NULL, level, 1);
    }

    for(j = z + 1; j > 0; j --)
      DSTK_drop();
  }
}

builtin(DUMP) {
  if(FOLDER_LAMBDA) {
    T_WORD *L = NULL;
    size_t z;

    for(z = 0; (L = lambda_enum(FOLDER_LAMBDA, L)) != NULL; z ++) ;
    
    for(; z > 0; z --) {
      L = lambda_findn(FOLDER_LAMBDA, z - 1);
      DSTK_add(atom_link(L->ob));
    }

    lam_kill();
  }
}

builtin(VARS) {
  T_ATOM *vars = dovars(FOLDER_CWD);

  DSTK_add(vars);
}

builtin(INNER) {
  if(DSTK_CK1(self, PROG | LIST | SYMB)) {
    T_ATOM *popped = DSTK_pop();

    if(atom_q_cons(popped)) {
      explode(popped);
    }
    else if(popped->prolog == DOSYMB) {
      T_SYMB *symb = atom_get_symb(popped);
      T_ATOM *op;
      T_BINT narg = 0;

      op = symb->ob;

      if(op == NULL) {
	if(symb->car) {
	  DSTK_add(atom_link(symb->car->ob));
	}
      }
      else if(! atom_q_exec(op)) {
	DSTK_add(atom_link(op));
      }
      else if(atom_q_function(op)) {
	T_SYMB *arg;

	for(arg = symb; arg; narg ++, arg = arg->cdr) {
	  T_ATOM *ob = atom_make_symb();
	  atom_get_symb(ob) = symb_link(arg->car);
	  DSTK_add(ob);
	}
      } else {
	T_ATOM *ob;

	ob = atom_make_symb();
	atom_get_symb(ob) = symb_link(symb->car);
	DSTK_add(ob);

	ob = atom_make_symb();
	atom_get_symb(ob) = symb_link(symb->cdr);
	DSTK_add(ob);
	
	narg = 2;
      }

      if(narg) {
	__push_bint(narg);
	DSTK_add(atom_link(op));
      }
    }

    atom_unlink(popped);
  }
}

builtin(CAR) {
  if(DSTK_CK1(self, PROG | LIST)) {
    T_ATOM *popped = DSTK_pop();

    if(atom_q_cons(popped)) {
      T_CONS *ob;
      int flag;

      ob = atom_get_cons(popped);

      for(flag = 0; !flag && ob; flag = 1, ob = ob->cdr)
	DSTK_add(atom_link(ob->car));
    }

    atom_unlink(popped);
  }
}

builtin(CDR) {
  if(DSTK_CK1(self, PROG | LIST)) {
    T_ATOM *popped = DSTK_pop();
    T_PROLOG *t = popped->prolog;

    if(atom_q_cons(popped)) {
      T_CONS *ob;
      T_ATOM *newob;

      newob = atom_make_cons(t);
      ob = atom_get_cons(popped);

      if(ob)
	atom_get_cons(newob) = cons_link(ob->cdr);

      DSTK_add(newob);
    }

    atom_unlink(popped);
  }
}

builtin(NEXT) {
  if(DSTK_CK1(self, PROG | LIST)) {
    T_ATOM *popped = DSTK_pop();
    T_PROLOG *t = popped->prolog;

    if(atom_q_cons(popped)) {
      T_CONS *ob;
      T_ATOM *newob;
      int flag;

      newob = atom_make_cons(t);
      ob = atom_get_cons(popped);

      if(ob)
	atom_get_cons(newob) = cons_link(ob->cdr);

      DSTK_add(newob);


      for(flag = 0; !flag && ob; flag = 1, ob = ob->cdr)
	DSTK_add(atom_link(ob->car));
    }

    atom_unlink(popped);
  }
}


builtin(META_PROG) {
  if(DSTK_CKN(self))
    DSTK_add(stk_meta2ob(&DSTK, __pop_bint(NULL), DOPROG));
}

builtin(META_LIST) {
  if(DSTK_CKN(self))
    DSTK_add(stk_meta2ob(&DSTK, __pop_bint(NULL), DOLIST));
}

builtin(SYMBN) {
  if(DSTK_CKN(self)) {
    T_ATOM *rpn = stk_meta2ob(&DSTK, __pop_bint(NULL), DOPROG);
    T_SYMB *symb = symb_gen(atom_get_cons(rpn));
    T_ATOM *ob = atom_make_symb();

    atom_unlink(rpn);
    atom_get_symb(ob) = symb;

    DSTK_add(ob);
  }
}

builtin(SYMB1) {
  if(DSTK_CK1(self, SYMB | BINT | ZINT | REAL | CPLX | IDNT | LAM)) {
    if(__peek_type(0) != SYMB) {
      T_ATOM *rpn = stk_meta2ob(&DSTK, 1, DOPROG);
      T_SYMB *symb = symb_gen(atom_get_cons(rpn));
      T_ATOM *ob = atom_make_symb();

      atom_unlink(rpn);
      atom_get_symb(ob) = symb;

      DSTK_add(ob);
    }
  }
}

builtin(SIMPLIFY) {
  if(DSTK_CK1(self, SYMB)) {
    T_ATOM *symb_in = DSTK_pop();
    T_ATOM *symb_out = symb_simplify(symb_in);

    atom_unlink(symb_in);
    DSTK_add(symb_out);    
  }
}

builtin(ARRY1) {
  if(DSTK_CKANY(self, 1)) {
    T_ATOM **data = NULL;
    T_ATOM *arry;

    unsigned char dims = 0;
    unsigned long *dim = NULL;

    dims = 1;
    dim = xm_calloc(dims, sizeof(*dim));
    dim[0] = 1;

    arry = atom_make_arry(dims, dim, NULL, &data);
    xm_free(dim);

    data[0] = DSTK_pop();

    DSTK_add(arry);
  }
}

builtin(MAKEARRY) {
  if(DSTK_CK1(self, BINT | LIST)) {
    unsigned long total = 0;
    T_ATOM **data = NULL;
    T_ATOM *arry;

    unsigned char dims = 0;
    unsigned long *dim = NULL;

    if(__peek_type(0) == BINT) {
      dims = 1;
      dim = xm_calloc(dims, sizeof(*dim));
      total = dim[0] = __pop_bint(NULL);
    } else {
      T_ATOM *ob = DSTK_pop();
      explode(ob);
      atom_unlink(ob);

      dims = __pop_bint(NULL);
      if(dims) {
	size_t j;

	dim = xm_calloc(dims, sizeof(*dim));

	for(j = dims; j > 0; j --)
	  dim[j - 1] = __pop_bint(NULL);
      }
    }

    arry = atom_make_arry(dims, dim, &total, &data);
    xm_free(dim);

    while(total > 0) {
      data[-- total] = DSTK_pop();
    }

    DSTK_add(arry);
  }
}

builtin(ARSIZE) {
  if(DSTK_CK1(self, ARRY)) {
    T_ATOM *ob = DSTK_pop();
    T_ARRY *arry = atom_get_arry(ob);
    unsigned long dims = 0;

    dims = arry->dims;
    
    if(dims> 0) {
      unsigned long j;

      for(j = 0; j < dims; j ++)
	__push_bint(arry->dim[j]);
    }

    DSTK_add(stk_meta2ob(&DSTK, dims, DOLIST));
    atom_unlink(ob);
  }
}

builtin(ALEN) {
  if(DSTK_CK1(self, ARRY)) {
    T_ATOM *ob = DSTK_pop();
    T_ARRY *arry = atom_get_arry(ob);
    unsigned long dims = 0;
    unsigned long alen = 0;

    dims = arry->dims;
    
    if(dims> 0) {
      unsigned long j;

      for(alen = 1, j = 0; j < dims; j ++)
	alen *= arry->dim[j];
    }

    __push_bint(alen);
    atom_unlink(ob);
  }
}

builtin(RANK) {
  if(DSTK_CK1(self, ARRY)) {
    T_ATOM *ob = DSTK_pop();
    T_ARRY *arry = atom_get_arry(ob);

    __push_bint(arry->dims);
    atom_unlink(ob);
  }
}

builtin(AREF) {
  if(DSTK_CK2(self, ARRY, BINT | LIST)) {
    T_OBTYPE t;
    T_ATOM *ob, *which;
    T_ARRY *arry;
    unsigned long dims;
    unsigned long total;

    which = DSTK_pop();
    t = atom_q_type(which);

    ob = DSTK_pop();
    arry = atom_get_arry(ob);
    dims = arry->dims;

    if(dims > 0) {
      unsigned long dim_ndx;
      unsigned long dim_base = 0;

      signed long atom_ndx = 0;
      
      if(t == BINT) {
	atom_ndx = atom_get_bint(which);
	dim_base = dims;
      } else {
	T_CONS *el = atom_get_cons(which);
	T_BINT val;

	for(dim_base = 0; el; el = el->cdr) {
	  val = atom_get_bint(el->car);

	  for(dim_ndx = ++ dim_base; dim_ndx < dims; dim_ndx ++)
	    val *= arry->dim[dim_ndx];

	  atom_ndx += val;
	}
      }

      atom_unlink(which);

      for(total = 1, dim_ndx = 0; dim_ndx < dims; dim_ndx ++)
	total *= arry->dim[dim_ndx];

      while(atom_ndx < 0)
	atom_ndx += total;

      while(atom_ndx >= total)
	atom_ndx -= total;

      DSTK_add(atom_link(arry->data[atom_ndx]));
    } else
      DSTK_add(atom_make_code(CMD_NIL));

    atom_unlink(ob);
  }
}

builtin(GETATELN) {
  if(DSTK_CK2(self, BINT | LIST, ARRY)) {
    T_OBTYPE t = __peek_type(1);
    T_ATOM *ob = DSTK_pop();
    T_ARRY *arry = atom_get_arry(ob);

    unsigned long dims = arry->dims;
    unsigned long total;

    if(dims > 0) {
      T_ATOM *which = DSTK_pop();
      unsigned long atom_ndx = 0;
      unsigned long dim_ndx;
      unsigned long dim_base = 0;
      
      if(t == BINT) {
	atom_ndx = atom_get_bint(which) - 1;
	dim_base = dims;
      } else {
	T_CONS *el = atom_get_cons(which);
	T_BINT val;

	for(dim_base = 0; el; el = el->cdr) {
	  val = atom_get_bint(el->car) - 1;

	  /* 
	     FIXME: ARRY atom should contain precomputed 
	     multiplier table 
	  */

	  for(dim_ndx = ++ dim_base; dim_ndx < dims; dim_ndx ++)
	    val *= arry->dim[dim_ndx];

	  atom_ndx += val;
	}
      }

      atom_unlink(which);

      for(total = 1, dim_ndx = 0; dim_ndx < dims; dim_ndx ++)
	total *= arry->dim[dim_ndx];

      if(atom_ndx >= 0 && atom_ndx < total && dim_base == dims) {
	DSTK_add(atom_link(arry->data[atom_ndx]));
	DSTK_add(atom_make_flag(1));
      } else
	DSTK_add(atom_make_flag(0));
    } else
      DSTK_add(atom_make_flag(0));

    atom_unlink(ob);
  }
}

/* BINT <-> ZINT <-> REAL */

builtin(BINT2ZINT) {
  if(DSTK_CK1(self, BINT)) {
    T_BINT b;
    T_ZINT z;

    b = __pop_bint(NULL);
    mpz_init_set_si(z, b);

    __push_zint(z);
    mpz_clear(z);
  }
}

builtin(REAL2ZINT) {
  if(DSTK_CK1(self, REAL)) {
    T_REAL x;
    T_ZINT z;

    x = __pop_real(NULL);
    mpz_init_set_d(z, x);

    __push_zint(z);
    mpz_clear(z);
  }
}

builtin(ZINT2BINT) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT z;
    T_BINT b;

    mpz_init(z);
    __pop_zint(&z);
    /* if mpz_fits_slong(z) ... */

    b = mpz_get_si(z);
    mpz_clear(z);

    __push_bint(b);
  }
}

builtin(ZINT2REAL) {
  if(DSTK_CK1(self, ZINT)) {
    T_ZINT z;
    T_REAL x;

    mpz_init(z);
    __pop_zint(&z);

    x = mpz_get_d(z);
    mpz_clear(z);

    __push_real(x);
  }
}

builtin(REAL2BINT) {
  if(DSTK_CK1(self, REAL)) {
    __push_bint((T_BINT)__pop_real(NULL));
  }
}

builtin(BINT2REAL) {
  if(DSTK_CK1(self, BINT)) {
    __push_bint((T_REAL)__pop_bint(NULL));
  }
}

/* symbolic algebra */

builtin(COERCE1) {
  __coerce(0, 0, 0, 0, 0);
}

builtin(COERCE2) {
  __coerce2();
}

void symbAPPLY1(T_ATOM *op) {
  T_ATOM *arg, *newob;
  T_SYMB *newsymb;

  arg = DSTK_pop();

  newob = atom_make_symb();
  newsymb = symb_alloc();

  newsymb->ob = atom_link(op);
  newsymb->car = symb_link(atom_get_symb(arg));
  newsymb->cdr = NULL;

  atom_get_symb(newob) = newsymb;

  DSTK_add(newob);

  atom_unlink(arg);
}

void symbAPPLY2(T_ATOM *op) {
  T_ATOM *level2, *level1, *newob;
  T_SYMB *newsymb;

  level1 = DSTK_pop();
  level2 = DSTK_pop();

  newob = atom_make_symb();
  newsymb = symb_alloc();

  newsymb->ob = atom_link(op);
  newsymb->car = symb_link(atom_get_symb(level2));
  newsymb->cdr = symb_link(atom_get_symb(level1));

  atom_get_symb(newob) = newsymb;

  DSTK_add(newob);

  atom_unlink(level2);
  atom_unlink(level1);
}

/* apply operation to a program (append) */

void progAPPLY1(T_ATOM *op) {
  T_ATOM *popped;
  T_CONS *newtail, **tail;

  CMD_CKREF(op); /* newob if ref'd */

  popped = DSTK->car; /* it'd better be a composite! */

  for(tail = &(atom_get_cons(popped)); *tail; tail = &((*tail)->cdr)) ;

  newtail = cons_alloc();

  newtail->car = atom_link(op);
  newtail->cdr = *tail;

  *tail = newtail;
}

/* prog1 prog2 op -- prog */

void progAPPLY2(T_ATOM *op) {
  T_ATOM *level2, *level1, *newob;
  T_CONS *s2, *s1, **tail;

  level1 = DSTK_pop();
  level2 = DSTK_pop();

  newob = atom_make_cons(DOPROG);
  tail = &(atom_get_cons(newob));
  
  for(s2 = atom_get_cons(level2); s2; s2 = s2->cdr) {
    T_CONS *s = cons_alloc();

    s->car = atom_link(s2->car);
    s->cdr = NULL;

    *tail = s;
    tail = &(s->cdr);
  }

  for(s1 = atom_get_cons(level1); s1; s1 = s1->cdr) {
    T_CONS *s = cons_alloc();

    s->car = atom_link(s1->car);
    s->cdr = NULL;

    *tail = s;
    tail = &(s->cdr);
  }

  DSTK_add(newob);

  atom_unlink(level2);
  atom_unlink(level1);
}

/* apply operation to all elements of a list */

void listAPPLY1(T_ATOM *op) {
  T_ATOM *newob, *popped;
  T_CONS *sptr, **dptr;

  popped = DSTK_pop();

  newob = atom_make_cons(DOLIST);
  dptr = &(atom_get_cons(newob));
  sptr = atom_get_cons(popped);

  while(sptr) {
    DSTK_add(atom_link(sptr->car));
    atom_eval(op, NULL);

    (*dptr) = cons_alloc();
    (*dptr)->car = DSTK_pop();

    dptr = &((*dptr)->cdr);
    sptr = sptr->cdr;
  }

  atom_unlink(popped);
  DSTK_add(newob);
}

/* apply (binary) operation element-wise to two lists */

void listAPPLY2(T_ATOM *op) {
  T_ATOM *newob, *level1, *level2;
  T_CONS **dptr, *s1, *s2;
  size_t len1, len2, len;

  level1 = DSTK_pop();
  level2 = DSTK_pop();

  newob = atom_make_cons(DOLIST);
  dptr = &(atom_get_cons(newob));

  for(s1 = atom_get_cons(level1), len1 = 0; s1; s1 = s1->cdr)
    len1 ++;

  for(s2 = atom_get_cons(level2), len2 = 0; s2; s2 = s2->cdr)
    len2 ++;

  len = (len1 < len2) ? len2 : len1;

  for(s1 = s2 = NULL; len > 0; len --) {
    if(s2 == NULL)
      s2 = atom_get_cons(level2);

    if(s1 == NULL)
      s1 = atom_get_cons(level1);

    DSTK_add(atom_link(s2->car));
    DSTK_add(atom_link(s1->car));
    atom_eval(op, NULL);

    (*dptr) = cons_alloc();
    (*dptr)->car = DSTK_pop();

    dptr = &((*dptr)->cdr);

    s2 = s2->cdr;
    s1 = s1->cdr;
  }

  atom_unlink(level2);
  atom_unlink(level1);

  DSTK_add(newob);
}

/* apply unary operation to every element of matrix */

void arryAPPLY1(T_ATOM *op) {
  T_ATOM *newob, *popped;
  T_ARRY *arry, *s_arry;

  popped = DSTK_pop();

  newob = atom_alloc();
  newob->prolog = popped->prolog;

  s_arry = atom_get_arry(popped);
  arry = atom_get_arry(newob);

  if(s_arry) {
    unsigned long dims, j, total;
    T_ATOM **s_data, **d_data;

    arry->dims = dims = s_arry->dims;
    arry->data = NULL;

    if(dims > 0) {
      arry->dim = xm_calloc(dims, sizeof(arry->dim[0]));

      for(total = 1, j = 0; j < dims; j ++)
	total *= (arry->dim[j] = s_arry->dim[j]);

      s_data = s_arry->data;
      d_data = arry->data = xm_calloc(total, sizeof(T_ATOM *));

      for(j = 0; j < total; j ++) {
	DSTK_add(atom_link(s_data[j]));
	atom_eval(op, NULL);
	d_data[j] = DSTK_pop();
      }
    } else {
      arry->dim = NULL;
      total = 0;
    }
  }

  atom_unlink(popped);
  DSTK_add(newob);
}

/* apply (binary) operation element-wise to two matrices */

void arryAPPLY2(T_ATOM *op) {
  T_ATOM *newob, *level2, *level1;
  T_ARRY *arry, *a2, *a1;

  level1 = DSTK_pop();
  level2 = DSTK_pop();

  newob = atom_alloc();
  newob->prolog = DOARRY;

  a2 = atom_get_arry(level2);
  a1 = atom_get_arry(level1);

  arry = atom_get_arry(newob);

  if(a1 && a2) {
    unsigned long dims1, dims2, dims;

    dims1 = a1->dims;
    dims2 = a2->dims;

    dims = (dims1 < dims2) ? dims2 : dims1;

    arry->dims = dims;
    arry->dim = NULL;
    arry->data = NULL;

    if(dims > 0) {
      unsigned long j1, j2, j;
      unsigned long total1, total2, total;
      T_ATOM **data;

      arry->dim = xm_calloc(dims, sizeof(arry->dim[0]));

      total = total1 = total2 = 1;

      for(j = 0; j < dims; j ++) {
	unsigned long m1 = 0, m2 = 0, m;

	if(j < dims1) {
	  m1 = a1->dim[j];
	  total1 *= m1;
	}

	if(j < dims2) {
	  m2 = a2->dim[j];
	  total2 *= m2;
	}

	m = (m1 < m2) ? m2 : m1;
	total *= (arry->dim[j] = m);
      }

      data = arry->data = xm_calloc(total, sizeof(T_ATOM *));

      for(j = j1 = j2 = 0; j < total; j ++, j1 ++, j2 ++) {
	if(j1 >= total1)
	  j1 = 0;

	if(j2 >= total2)
	  j2 = 0;

	DSTK_add(atom_link(a2->data[j2]));
	DSTK_add(atom_link(a1->data[j1]));

	atom_eval(op, NULL);

	data[j] = DSTK_pop();
      }
    }
  }

  atom_unlink(level2);
  atom_unlink(level1);

  DSTK_add(newob);
}

void xAPPLY1(T_ATOM *cmd, void *real_call, void *cplx_call) {
  if(DSTK_CK1(cmd, 
	      BINT | ZINT | REAL | CPLX | 
	      IDNT | LAM | SYMB | 
	      LIST | PROG | ARRY)) {
    __coerce(0, 0, 0, 0, 0);

    switch(__peek_type(0)) {
    case REAL:
      if(real_call) (*(T_PROLOG *)real_call)(cmd); 
      break;

    case CPLX:
      if(cplx_call) (*(T_PROLOG *)cplx_call)(cmd);
      break;

    case SYMB:
      symbAPPLY1(cmd);
      break;

    case LIST:
      listAPPLY1(cmd);
      break;

    case PROG:
      progAPPLY1(cmd);
      break;

    case ARRY:
      arryAPPLY1(cmd);
      break;

    default:
      Exception(cmd, "Bad object type", 0);
    }
  }
}

void xAPPLY2(T_ATOM *cmd, void *real_call, void *cplx_call) {
  if(DSTK_CK2(cmd,
	      BINT | ZINT | REAL | CPLX | 
	      IDNT | LAM | SYMB | 
	      LIST | PROG | ARRY,
	      BINT | ZINT | REAL | CPLX | 
	      IDNT | LAM | SYMB | 
	      LIST | PROG | ARRY)) {
    /* coerce both operands to prog, list, symb and/or cplx, as needed */
    __coerce2();

    switch(__peek_type(1)) {
    case PROG:
      progAPPLY2(cmd);
      break;

    case LIST:
      listAPPLY2(cmd);
      break;

    case ARRY:
      arryAPPLY2(cmd);
      break;
      
    case SYMB:
      symbAPPLY2(cmd);
      break;

    case CPLX:
      if(cplx_call) 
	(*(T_PROLOG *)cplx_call)(cmd); 
      break;

    case REAL:
      if(real_call) 
	(*(T_PROLOG *)real_call)(cmd); 
      break;

    default:
      Exception(cmd, "Bad object type", 0);
    }
  }
}

builtin(xOPAREN) { Exception(NULL, "don't call me!", 0); }
builtin(xCPAREN) { Exception(NULL, "don't call me!", 0); }

builtin(xASSIGN) { }
builtin(xCOMMA) { }

/* 1-ary */
builtin(xNEGATIVE) { xAPPLY1(self, CMD_REAL_NEG, CMD_CPLX_NEG); }

builtin(xABS){ xAPPLY1(self, CMD_REAL_ABS, CMD_CPLX_ABS); }
builtin(xSGN){ xAPPLY1(self, CMD_REAL_SGN, CMD_CPLX_SGN); }
builtin(xINV){ xAPPLY1(self, CMD_REAL_INV, CMD_CPLX_INV); }
builtin(xSQRT){ xAPPLY1(self, CMD_REAL_SQRT, CMD_CPLX_SQRT); }
builtin(xEXP){ xAPPLY1(self, CMD_REAL_EXP, CMD_CPLX_EXP); }
builtin(xLN){ xAPPLY1(self, CMD_REAL_LN, CMD_CPLX_LN); }
builtin(xLOG){ xAPPLY1(self, CMD_REAL_LOG, CMD_CPLX_LOG); }
builtin(xALOG){ xAPPLY1(self, CMD_REAL_ALOG, CMD_CPLX_ALOG); }
builtin(xSIN){ xAPPLY1(self, CMD_REAL_SIN, CMD_CPLX_SIN); }
builtin(xCOS){ xAPPLY1(self, CMD_REAL_COS, CMD_CPLX_COS); }
builtin(xSINH){ xAPPLY1(self, CMD_REAL_SINH, CMD_CPLX_SINH); }
builtin(xCOSH){ xAPPLY1(self, CMD_REAL_COSH, CMD_CPLX_COSH); }
builtin(xASIN){ xAPPLY1(self, CMD_REAL_ASIN, CMD_CPLX_ASIN); }
builtin(xACOS){ xAPPLY1(self, CMD_REAL_ACOS, CMD_CPLX_ACOS); }
builtin(xASINH){ xAPPLY1(self, CMD_REAL_ASINH, CMD_CPLX_ASINH); }
builtin(xACOSH){ xAPPLY1(self, CMD_REAL_ACOSH, CMD_CPLX_ACOSH); }
builtin(xTAN){ xAPPLY1(self, CMD_REAL_TAN, CMD_CPLX_TAN); }
builtin(xATAN){ xAPPLY1(self, CMD_REAL_ATAN, CMD_CPLX_ATAN); }
builtin(xTANH){ xAPPLY1(self, CMD_REAL_TANH, CMD_CPLX_TANH); }
builtin(xATANH){ xAPPLY1(self, CMD_REAL_ATANH, CMD_CPLX_ATANH); }

builtin(xFACTORIAL){ xAPPLY1(self, CMD_REAL_FACTORIAL, CMD_CPLX_FACTORIAL); }
builtin(xGAMMA){ xAPPLY1(self, CMD_REAL_GAMMA, CMD_CPLX_GAMMA); }

/* 2-ary */
builtin(xPOW) { xAPPLY2(self, CMD_REAL_POWER, CMD_CPLX_POWER); }
builtin(xTIMES) { xAPPLY2(self, CMD_REAL_TIMES, CMD_CPLX_TIMES); }
builtin(xDIVIDE) { xAPPLY2(self, CMD_REAL_DIVIDE, CMD_CPLX_DIVIDE); }
builtin(xMOD) { xAPPLY2(self, CMD_REAL_MOD, CMD_CPLX_MOD); }
builtin(xPLUS) { xAPPLY2(self, CMD_REAL_PLUS, CMD_CPLX_PLUS); }
builtin(xMINUS) { xAPPLY2(self, CMD_REAL_MINUS, CMD_CPLX_MINUS); }

builtin(xNROOT) { xAPPLY2(self, CMD_REAL_NROOT, CMD_CPLX_NROOT); }
builtin(xANGLE) { xAPPLY2(self, CMD_REAL_ANGLE, CMD_CPLX_ANGLE); }
builtin(xCOMB) { xAPPLY2(self, CMD_REAL_COMB, CMD_CPLX_COMB); }
builtin(xPERM) { xAPPLY2(self, CMD_REAL_PERM, CMD_CPLX_PERM); }
