#!/usr/bin/python2.5
import sys, os, fcntl, gtk, gobject, hildon, dbus, signal, alarm, time
import pyconic; conic=pyconic # ideally, import conic, but that's broken
import syncevolution

GUI_VERBOSITY = 0

THUMB_SIZE = gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_THUMB_HEIGHT
FINGER_SIZE = gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT

TIMEFMT = "%x %X" # should perhaps make this configurable

default_sources = [
  ("addressbook", "Contacts"),
  ("calendar", "Calendar"),
  ("todo", "Tasks"),
  ("memo", "Notes"),
]

backend_aliases = {
  # syncevolution backends used on Maemo
  "evolution-contacts": "addressbook",
  "maemo-events": "calendar",
  "maemo-tasks": "todo",
  "maemo-notes": "memo",
}

calendar_names = {
  "cal_ti_calendar_private": "Private",
  "cal_ti_smart_birthdays": "Birthdays",
}

default_modes = [
  # When the user selects Normal Sync, the mode configured in the sync sources will be used.
  # (Each sync source can have its own mode, independent of the other sources.)
  # Could perhaps have a separate "Default" for that, but that might be confusing,
  # and a "Normal Sync" option that enforces two-way sync even for sync sources where only
  # one-way is configured seems like a bad idea anyway.
  (None, "Normal Sync"),
  ("slow", "Slow Sync"),
  ("refresh-from-client", "Refresh From Client"),
  ("refresh-from-server", "Refresh From Server"),
  ("one-way-from-client", "One-Way From Client"),
  ("one-way-from-server", "One-Way From Server"),
]

source_modes = [
  ("disabled", "Disabled"),
  ("two-way", "Normal Sync"),
  ("slow", "Slow Sync"),
  ("refresh-from-client", "Refresh From Client"),
  ("refresh-from-server", "Refresh From Server"),
  ("one-way-from-client", "One-Way From Client"),
  ("one-way-from-server", "One-Way From Server"),
]

RESULT_NO_CONNECTION = -2
RESULT_LOCKED = -3

def get_method(url):
  pos = url.find(":")
  if pos < 0:
    return None
  return url[:pos]

class SyncRunner(object):
  def __init__(self, auto, sync, server, dir, config, mode=None, sources=None):
    self.auto = auto
    self.sync = sync
    self.server = server
    self.dir = dir
    self.config = config
    self.mode = mode
    self.sources = sources
    self.state = None
    self.sync_watches = None
    self.connection = None
    self.conn_status = None
    self.poll_cb = None
    self.tick_cb = None
    # determine connection method
    url = config.get("syncURL")
    self.method = get_method(url)
    # grab exclusive lock
    self.acquire_lock()
    # connect to network
    self.connection = conic.Connection()
    flags = conic.CONNECT_FLAG_NONE
    if self.auto:
      flags |= conic.CONNECT_FLAG_AUTOMATICALLY_TRIGGERED
    self.connection.connect("connection-event", self.connected)
    self.connection.request_connection(flags)
  def acquire_lock(self):
    lock_name = self.dir + "/.sync_lock"
    tries = 0
    while True:
      try:
        fd = os.open(lock_name, os.O_RDWR|os.O_CREAT|os.O_EXCL)
        try:
          fcntl.flock(fd, fcntl.LOCK_EX)
        except:
          # since we're blocking, this shouldn't happen, in theory
          os.close(fd)
          continue
        break
      except:
        # check if this might be a stale lock
        try:
          fd = os.open(lock_name, os.O_RDWR)
        except:
          # file vanished, try to lock again
          continue
        try:
          fcntl.flock(fd, fcntl.LOCK_EX)
        except:
          # since we're blocking, this shouldn't happen, in theory
          os.close(fd)
          continue
        buf = os.read(fd, 256)
        if buf == "":
          # another process might be in the process of locking
          # (opened the file but didn't lock it yet)
          if tries > 10:
            # perhaps not, guess we'll take the lock instead
            break
          else:
            tries += 1
          os.close(fd)
          continue
        pid = int(buf)
        # since the file isn't empty, and we have the lock,
        # the other process must have terminated prematurely.
        # Take over the lock.
        os.lseek(fd, 0, 0)
        os.ftruncate(fd, 0)
        break
    print "lock acquired"
    self.lock_fd = fd
    pid = os.getpid()
    os.write(fd, "%d" % pid)
    print "locked"
  def release_lock(self):
    os.unlink(self.dir + "/.sync_lock")
    os.close(self.lock_fd)
  def connected(self, connection, event):
    self.conn_status = event.get_status()
    if self.conn_status == conic.STATUS_CONNECTED:
      # we're connected, see if the current connection is using proxies
      proxy_mode = self.connection.get_proxy_mode()
      if proxy_mode == conic.PROXY_MODE_NONE:
        proxy = None
      elif proxy_mode == conic.PROXY_MODE_MANUAL:
        if self.method == "http":
          proto = conic.PROXY_PROTOCOL_HTTP
        elif self.method == "https":
          proto = conic.PROXY_PROTOCOL_HTTPS
        else: # hmm... just default to http?
          proto = conic.PROXY_PROTOCOL_HTTP
        proxy_host = self.connection.get_proxy_host(proto)
        proxy_port = self.connection.get_proxy_port(proto)
        proxy = self.method + "://%s:%d" % (proxy_host, proxy_port)
      elif proxy_mode == conic.PROXY_MODE_AUTO:
        print "Argh, can't handle automatic proxy yet!"
        proxy = None
      # all set, launch SyncEvolution
      self.state = self.sync.synchronize_start(self.server, self.config, self.mode, self.sources, proxy)
      if not (self.poll_cb is None and self.tick_cb is None):
        self.activate_watches()
    elif not self.poll_cb is None:
      self.poll_cb(None, None)
  def set_callbacks(self, poll_cb, tick_cb=None):
    self.poll_cb = poll_cb
    self.tick_cb = tick_cb
    if self.connect_error():
      self.poll_cb(None, None)
      return
    if not self.state is None:
      self.activate_watches()
  def connect_error(self):
    if self.connection is None:
      return True
    if self.conn_status is None or self.conn_status == conic.STATUS_CONNECTED:
      return False
    return True
  def activate_watches(self):
    if not self.sync_watches is None:
      self.deactivate_watches()
    self.sync_watches = [gobject.io_add_watch(w, gobject.IO_IN|gobject.IO_HUP, self.poll_cb) for w in self.state.watches()]
    if not self.tick_cb is None:
      self.sync_watches.append(gobject.timeout_add(100, self.tick_cb))
  def deactivate_watches(self):
    if self.sync_watches is None:
      return
    for w in self.sync_watches:
      gobject.source_remove(w)
    self.sync_watches = None
  def __del__(self):
    if not self.state is None:
      self.finish()
  def poll(self):
    if self.state is None:
      return self.connect_error()
    return self.state.poll()
  def progress(self):
    if self.state is None:
      return None
    return self.state.progress()
  def abort(self):
    if self.state is None:
      return
    self.state.abort()
  def finish(self):
    # avoid recursive calls from cleanup
    self.poll_cb = None
    self.tick_cb = None
    if self.state is None:
      status = {}
      if self.connection is None:
        status["result"] = RESULT_LOCKED
      else:
        status["result"] = RESULT_NO_CONNECTION
        self.release_lock()
      return status
    self.deactivate_watches()
    status = self.sync.synchronize_status(self.state)
    self.connection.disconnect()
    self.state = None
    self.release_lock()
    return status

class SyncGUI(object):
  def __init__(self, quiet=False):
    gtk.set_application_name("SyncEvolution")
    self.program = hildon.Program.get_instance()
    self.sync = syncevolution.SyncEvolution(quiet)
    self.dbus = dbus.SessionBus()
    self.server = None
    self.server_dir = None
    self.server_config = None
    self.server_cookie = None
    self.server_sessions = None
    self.main_window = None
    self.server_window = None
    self.history_window = None
    self.wizard_state = None
    self.sync_dialog = None
    self.sync_mode = None
    self.sync_sources = None
    self.sync_progress = None
    self.sync_bar = None
    self.sync_state = None
    self.sync_runner = None
    self.servers = None
    self.selector = None
    self.init_main()
  def main(self):
    gtk.main()

### MAIN WINDOW
  def get_main_title(self):
    return "Sync services"
  def init_main(self):
    self.main_window = hildon.StackableWindow()
    self.main_window.set_title(self.get_main_title())
    self.main_window.connect("destroy", self.destroy_main)
    self.program.add_window(self.main_window)

    menu = hildon.AppMenu()
    button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
    button.set_label("Add new service")
    button.connect("clicked", self.create_server)
    menu.append(button)
    button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
    button.set_label("Delete service")
    button.connect("clicked", self.delete_servers)
    menu.append(button)
    menu.show_all()
    self.main_window.set_app_menu(menu)

    self.init_main_selector()
    self.main_window.show()
  def init_main_selector(self):
    if not self.selector is None:
      self.selector.destroy()
    self.servers = self.sync.get_servers()
    self.selector = hildon.TouchSelector(text = True)
    self.selector.set_hildon_ui_mode("normal")
    self.selector.connect("changed", self.server_selected)
    for server in self.servers:
      self.selector.append_text(server)
    self.selector.show()
    self.main_window.add(self.selector)
  def destroy_main(self, window):
    gtk.main_quit()
  def server_selected(self, selector, column):
    row = selector.get_last_activated_row(column)[0]
    self.server = self.servers[row]
    self.server_dir = self.sync.get_server_dir(self.server)
    self.server_config = self.sync.get_server_config(self.server)
    self.load_cookie()
    self.init_server_config()
  def create_server(self, button):
    self.init_server_create()
  def delete_servers(self, button):
    self.init_server_delete()
  def load_server_cookie(self, server_dir):
    try:
      return int(open(server_dir + "/.alarmcookie", "r").read())
    except:
      return None
  def load_cookie(self):
    self.server_cookie = self.load_server_cookie(self.server_dir)
### SERVER DELETION
  def init_server_delete(self):
    window = hildon.StackableWindow()
    toolbar = hildon.EditToolbar("Delete services", "Delete")
    window.set_edit_toolbar(toolbar)
    selector = hildon.TouchSelector(text = True)
    window.add(selector)
    for server in self.servers:
      selector.append_text(server)
    selector.set_column_selection_mode(hildon.TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE)
    selector.unselect_all(0)
    toolbar.connect("button-clicked", self.server_delete_clicked, window, selector)
    toolbar.connect("arrow-clicked", self.server_delete_close, window)
    window.show_all()
    window.fullscreen()
  def server_delete_clicked(self, button, window, selector):
    # for some reason, the python bindings don't seem to have get_selected_rows,
    # so I'll have to use this hack, parsing the string
    server_list = selector.get_current_text()[1:-1]
    if server_list == "":
      servers = []
    else:
      servers = server_list.split(",")
    # ask for confirmation
    if len(servers) == 0:
      banner = hildon.hildon_banner_show_information(window, "", "No services selected")
      response = gtk.RESPONSE_DELETE_EVENT
    elif len(servers) == 1:
      note = hildon.hildon_note_new_confirmation(window, "Delete service %s?" % servers[0])
      response = gtk.Dialog.run(note)
      note.destroy()
    else:
      note = hildon.hildon_note_new_confirmation(window, "Delete selected services?")
      response = gtk.Dialog.run(note)
      note.destroy()
    if response == gtk.RESPONSE_OK:
      for server in servers:
        self.do_delete_server(server)
      self.server_delete_close(button, window)
      self.init_main_selector()
  def server_delete_close(self, button, window):
    window.destroy()
  def do_delete_server(self, server):
    server_dir = self.sync.get_server_dir(server)
    server_cookie = self.load_server_cookie(server_dir)
    if not server_cookie is None:
      os.unlink(server_dir + "/.alarmcookie")
      alarm.delete_event(server_cookie)
    self.sync.delete_server(server)

### SERVER CREATION
  def init_server_create(self):
    self.wizard_state = {}

    templates = self.sync.get_templates()
    templates.sort()

    self.wizard_state["templates"] = templates

    notebook = gtk.Notebook()

    label = gtk.Label("This wizard will walk you through configuring a new SyncML synchronization service.\n"
                      "Tap 'Next' to continue.")
    label.set_line_wrap(True)
    notebook.append_page(label, None)

    # Page 2
    vbox = gtk.VBox(False, 0)

    servername = hildon.Entry(gtk.HILDON_SIZE_AUTO)
#    servername.set_placeholder("Service Name")
    self.wizard_state["name"] = servername
    caption = hildon.Caption(None, "Service Name", servername)
    caption.set_status(hildon.CAPTION_MANDATORY)
    vbox.pack_start(caption, False, False, 0)

    selector = hildon.TouchSelector(text = True)
    for label in templates:
      selector.append_text(label)
    selector.set_active(0, 0)
    self.wizard_state["template"] = selector

    button = hildon.PickerButton(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL)
    button.set_title("Template")
    button.set_selector(selector)
    vbox.pack_start(button, False, False, 0)

    notebook.append_page(vbox, gtk.Label("Basics"))

    # Page 3
    vbox = gtk.VBox(False, 0)

    username = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    self.wizard_state["user"] = username
    caption = hildon.Caption(None, "Username", username)
    vbox.pack_start(caption, False, False, 0)

    password = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    password.set_visibility(False)
    self.wizard_state["pass"] = password
    caption = hildon.Caption(None, "Password", password)
    vbox.pack_start(caption, False, False, 0)

    notebook.append_page(vbox, gtk.Label("Credentials"))

    # Page 4
    vbox = gtk.VBox(False, 0)

    label = gtk.Label("Service setup is complete. Tap 'Finish' to save or discard the settings by tapping outside the wizard.\n")
#                      "Tap 'Advanced settings' to modify the advanced service settings.")
    label.set_line_wrap(True)
    vbox.pack_start(label, False, False, 0)

    notebook.append_page(vbox, gtk.Label("Complete"))

    # Done

    dialog = hildon.WizardDialog(self.main_window, "Add new service", notebook)
    dialog.set_forward_page_func(self.server_create_page_func)
    notebook.connect("switch-page", self.server_create_page_switch, dialog)
    dialog.show_all()
    response = dialog.run()

    if response == hildon.WIZARD_DIALOG_FINISH:
      template_idx = selector.get_active(0)
      self.sync.create_server(servername.get_text(),
        templates[template_idx],
        username.get_text(),
        password.get_text(),
        {"printChanges": "0"})
      self.init_main_selector()

    dialog.destroy()
    self.wizard_state = None

  def server_create_page_switch(self, notebook, page, page_num, dialog):
    dialog.set_response_sensitive(hildon.WIZARD_DIALOG_FINISH, page_num >= 3)

  def server_create_page_func(self, notebook, page_num, user_data):
    if page_num == 1:
      servername = self.wizard_state["name"]
      return len(servername.get_text()) != 0
    return True

### SERVER WINDOW
  def get_server_title(self):
    return "Sync with " + self.server
  def get_source_label(self, source):
    return source_names.get(source, source)
  def build_source_list(self, all=False):
    sources = self.sync.get_sources_from_config(self.server_config, all=all)
    # build source list with labels for the UI
    source_list = []
    # add known sources
    for name, label in default_sources:
      if name in sources:
        source_list.append((name, label))
        sources.remove(name)
    # add unknown sources
    for name in sources:
      source_list.append((name, name))
    return source_list
  def init_server_config(self):
    self.server_window = hildon.StackableWindow()
    self.server_window.set_title(self.get_server_title())
    self.server_window.connect("destroy", self.destroy_server_config)
    self.program.add_window(self.server_window)

    area = hildon.PannableArea()
    area.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
    self.server_window.add(area)
    vbox = gtk.VBox(False, 0)
    area.add_with_viewport(vbox)

    button = hildon.Button(THUMB_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Synchronize!")
    button.connect("clicked", self.sync_clicked)
    vbox.pack_start(button, False, False, 0)

    if self.server_config.has_key("WebURL"):
      button = hildon.Button(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Open Service Provider Website", self.server_config["WebURL"])
      button.connect("clicked", self.weburl_clicked)
      vbox.pack_start(button, False, False, 0)

    if GUI_VERBOSITY > 0:
      self.sync_mode = hildon.TouchSelector(text = True)
      for name, label in default_modes:
        self.sync_mode.append_text(label)
      self.sync_mode.set_active(0, 0)

      button = hildon.PickerButton(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      button.set_title("Synchronization Mode")
      button.set_selector(self.sync_mode)
      vbox.pack_start(button, False, False, 0)

    source_list = self.build_source_list()

    if GUI_VERBOSITY > 1:
      # create checkboxes for them
      self.sync_sources = []
      for name, label in source_list:
        button = hildon.CheckButton(FINGER_SIZE)
        button.set_label("Synchronize " + label)
        button.set_active(True)
        self.sync_sources.append((name, button))
        vbox.pack_start(button, False, False, 0)

    view_button = hildon.Button(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL, "View Synchronization History")
    view_button.connect("clicked", self.history_clicked)
    vbox.pack_start(view_button, False, False, 0)

    auto_button = hildon.CheckButton(FINGER_SIZE)
    auto_button.set_label("Automatically synchronize daily")
    auto_button.set_active(not self.server_cookie is None)
    vbox.pack_start(auto_button, False, False, 0)

    time_button = hildon.TimeButton(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
    time_button.set_title("Daily synchronization time")
    time_button.set_alignment(0.25, 0.5, 0.5, 0.5)
    vbox.pack_start(time_button, False, False, 0)

    self.load_sync_time(time_button)
    auto_button.connect("toggled", self.auto_sync_toggled, time_button)
    time_button.connect("value-changed", self.sync_time_changed)

    menu = hildon.AppMenu()
    button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
    button.set_label("Edit service")
    button.connect("clicked", self.edit_server)
    menu.append(button)
    button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
    button.set_label("Delete service")
    button.connect("clicked", self.delete_server)
    menu.append(button)
    menu.show_all()
    self.server_window.set_app_menu(menu)

    self.server_window.show_all()
  def destroy_server_config(self, window):
    self.server_window = None
    self.sync_sources = None
    self.sync_mode = None
  def edit_server(self, button):
    self.init_server_edit()
  def delete_server(self, button):
    note = hildon.hildon_note_new_confirmation(self.server_window, "Delete service %s?" % self.server)
    response = gtk.Dialog.run(note)
    note.destroy()
    if response == gtk.RESPONSE_OK:
      self.do_delete_server(self.server)
      self.server_window.destroy()
      self.init_main_selector()
  def sync_clicked(self, button):
    self.synchronize_start()
  def weburl_clicked(self, button):
    url = self.server_config["WebURL"]
    self.launch_browser(url)
  def history_clicked(self, button):
    self.init_server_history()
  def load_sync_time(self, time_button):
    if not self.server_cookie is None:
      event = alarm.get_event(self.server_cookie)
      recur = event.get_recurrence(0)
      hours = 0
      mask = recur.mask_hour
      while mask > 1:
        hours = hours + 1
        mask = mask >> 1
      minutes = 0
      mask = recur.mask_min
      while mask > 1:
        minutes = minutes + 1
        mask = mask >> 1
      time_button.set_time(hours, minutes)
  def auto_sync_toggled(self, auto_button, time_button):
    active = auto_button.get_active()
    if active:
      (hours, minutes) = time_button.get_time()
      event = alarm.Event()
      event.appid = "syncevolution"
      event.title = "Synchronization with " + self.server
#      event.flags |= alarm.EVENT_RUN_DELAYED
      action = event.add_actions(1)[0]
      action.flags |= alarm.ACTION_WHEN_TRIGGERED | alarm.ACTION_WHEN_DELAYED | alarm.ACTION_TYPE_EXEC
      action.command = os.path.abspath(sys.argv[0]) + " --quiet " + self.server
      recur = event.add_recurrences(1)[0]
      # let's see what this does...
      recur.mask_min = 1 << minutes
      recur.mask_hour = 1 << hours
      # initialize alarm time to somewhere in the future
      event.alarm_time = time.time() + 5
#      lt = time.localtime(time.time() + 5)
#      tz = time.tzname[lt.tm_isdst]
#      event.alarm_time = time.mktime(recur.next(lt, tz))
      event.recurrences_left = -1
      f = open(self.server_dir + "/.alarmcookie", "w")
      self.server_cookie = alarm.add_event(event)
      f.write("%d" % self.server_cookie)
      f.close()
    else:
      if not self.server_cookie is None:
        os.unlink(self.server_dir + "/.alarmcookie")
        alarm.delete_event(self.server_cookie)
        self.server_cookie = None
  def sync_time_changed(self, time_button):
    if not self.server_cookie is None:
      (hours, minutes) = time_button.get_time()
      event = alarm.get_event(self.server_cookie)
      f = open(self.server_dir + "/.alarmcookie", "w")
      recur = event.get_recurrence(0)
      recur.mask_min = 1 << minutes
      recur.mask_hour = 1 << hours
      self.server_cookie = alarm.update_event(event)
      f.write("%d" % self.server_cookie)
      f.close()
  def launch_browser(self, url):
    proxy_obj = self.dbus.get_object('com.nokia.osso_browser', '/com/nokia/osso_browser')
    dbus_iface = dbus.Interface(proxy_obj, 'com.nokia.osso_browser')
    dbus_iface.open_new_window(url)

### SERVER HISTORY
  def get_history_title(self):
    return "History for " + self.server
  def init_server_history(self):
    self.history_window = hildon.StackableWindow()
    self.history_window.set_title(self.get_history_title())
    self.history_window.connect("destroy", self.destroy_server_history)
    self.program.add_window(self.history_window)

    self.server_sessions = self.sync.get_sessions(self.server)
    self.server_sessions.reverse() # show newest first

    selector = hildon.TouchSelector(text = True)
    selector.set_hildon_ui_mode("normal")
    selector.connect("changed", self.server_history_selected)
    self.history_window.add(selector)

    # FIXME: left align the selector entries (can't figure out how)

    for (start_time, code, logf) in self.server_sessions:
      ts = time.strftime(TIMEFMT, time.localtime(start_time))
      if code is None:
        rs = "Aborted"
      elif code == 200:
        rs = "Success"
      else:
        rs = "Error %d" % code
      selector.append_text("%s [%s]" % (ts, rs))

    self.history_window.show_all()

  def server_history_selected(self, selector, column):
    row = selector.get_last_activated_row(column)[0]
    (start_time, code, logf) = self.server_sessions[row]
    # The first HTML anchor I found in the logs is named H1.
    # Although it's not the first line of the interesting part of the log,
    # using it might help the user avoid scrolling past a lot of config cruft.
    url = "file://" + logf + "#H1"
    self.launch_browser(url)

  def destroy_server_history(self, window):
    self.history_window = None
    self.server_sessions = None

### SERVER EDITING
  def init_server_edit(self):
    dialog = gtk.Dialog()
    dialog.set_transient_for(self.server_window)
    dialog.set_title("Edit " + self.server)
    dialog.add_button("Done", gtk.RESPONSE_OK)

    area = hildon.PannableArea()
    area.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
    vbox = gtk.VBox(False, 0)
    area.add_with_viewport(vbox)
    dialog.vbox.add(area)

    sync_url = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    sync_url.set_text(self.server_config.get("syncURL", ""))
    caption = hildon.Caption(None, "Sync URL", sync_url)
    vbox.pack_start(caption, False, False, 0)

    web_url = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    web_url.set_text(self.server_config.get("WebURL", ""))
    caption = hildon.Caption(None, "Web URL", web_url)
    vbox.pack_start(caption, False, False, 0)

    username = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    username.set_text(self.server_config.get("username", ""))
    caption = hildon.Caption(None, "Username", username)
    vbox.pack_start(caption, False, False, 0)

    password = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    password.set_text(self.server_config.get("password", ""))
    password.set_visibility(False)
    caption = hildon.Caption(None, "Password", password)
    vbox.pack_start(caption, False, False, 0)

    source_list = self.build_source_list(True)

    # server-specific source configuration
    sources_remote = {}
    for name, label in source_list:
      config = self.sync.get_source_config(self.server_config, name)
      source_url = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      source_url.set_text(config.get("uri", ""))
      caption = hildon.Caption(None, label + " database", source_url)
      vbox.pack_start(caption, False, False, 0)
      sources_remote[name] = source_url

    # client-specific source configuration
    backends = self.sync.get_backends()
    sources_local = {}
    for name, label in source_list:
      config = self.sync.get_source_config(self.server_config, name)

      mode_selector = hildon.TouchSelector(text = True)
      mode = config.get("sync", "normal")
      idx = 0
      count = 0
      for mode_name, mode_label in source_modes:
        if mode_name == mode:
          idx = count
        mode_selector.append_text(mode_label)
        count += 1
      mode_selector.set_active(0, idx)

      button = hildon.PickerButton(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      button.set_alignment(0, 0.5, 0.5, 0.5)
      button.set_title(label + " synchronization")
      button.set_selector(mode_selector)
      vbox.pack_start(button, False, False, 0)

      base_selector = hildon.TouchSelector(text = True)
      base_list = []
      base = config.get("evolutionsource")
      idx = 0
      count = 0
      for backend_id, bases in backends.items():
        backend = backend_aliases.get(backend_id, backend_id)
        if backend != name:
          continue
        for base_name, base_uri, default in bases:
          if base is None:
            if default:
              idx = count
              base_uri = None # if the value was unset, leave unset if the user keeps the setting...
          elif base == base_uri:
            idx = count
          base_list.append((backend, base_uri))
          base_selector.append_text(calendar_names.get(base_name, base_name))
          count += 1
      base_selector.set_active(0, idx)

      # there's only one contacts database, so we don't have to show a button for that
      if count > 1:
        button = hildon.PickerButton(FINGER_SIZE, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        button.set_alignment(0, 0.5, 0.5, 0.5)
        button.set_title(label + " source")
        button.set_selector(base_selector)
        vbox.pack_start(button, False, False, 0)

      sources_local[name] = (mode_selector, base_selector, base_list)

    dialog.show_all()
    response = dialog.run()
    if response == gtk.RESPONSE_OK:
      config = {}
      config["syncURL"] = sync_url.get_text()
      config["WebURL"] = web_url.get_text()
      config["username"] = username.get_text()
      config["password"] = password.get_text()
      for name, label in source_list:
        source_config = {}
        source_url = sources_remote[name]
        mode_selector, base_selector, base_list = sources_local[name]
        source_config["uri"] = source_url.get_text()
        idx = mode_selector.get_active(0)
        source_config["sync"] = source_modes[idx][0]
        idx = base_selector.get_active(0)
        source_config["evolutionsource"] = base_list[idx][1]
        self.sync.set_source_config(config, name, source_config)
      self.sync.configure_server(self.server, config)

    dialog.destroy()

    if response == gtk.RESPONSE_OK:
      self.server_config = self.sync.get_server_config(self.server)
      # Rather than trying to update the existing server window
      # to match the new configuration, just recreate it from scratch,
      # at least for now
      self.server_window.destroy()
      self.init_server_config()

### SYNCHRONIZATION PROGRESS
  def synchronize_start(self):
    # get parameters
    mode = None
    if not self.sync_mode is None:
      idx = self.sync_mode.get_active(0)
      mode = default_modes[idx][0]
    sources = None
    if not self.sync_sources is None:
      sources = []
      for name, button in self.sync_sources:
        if button.get_active():
          sources.append(name)
      if len(sources) == 0:
        banner = hildon.hildon_banner_show_information(self.server_window, "", "Nothing to synchronize")
        return

    # build progress dialog
    self.sync_progress = None
    self.sync_bar = gtk.ProgressBar()
    self.sync_bar.set_text("Synchronizing...")

    self.sync_dialog = hildon.hildon_note_new_cancel_with_progress_bar(self.server_window, "Synchronizing with " + self.server, self.sync_bar)
    self.sync_dialog.connect("response", self.synchronize_cancel)
    # I don't seem able to trigger "close" or "delete_event", but just in case anyone can...
    self.sync_dialog.connect("close", self.synchronize_close)
    self.sync_dialog.connect("delete_event", self.synchronize_quit)

    self.sync_dialog.show_all()

    # start synchronization
    self.sync_state = SyncRunner(False, self.sync, self.server, self.server_dir, self.server_config, mode, sources)
    self.sync_state.set_callbacks(self.synchronize_poll, self.synchronize_tick)

  def synchronize_cleanup(self):
    if self.sync_dialog is None:
      return
    self.sync_dialog.destroy()
    self.sync_bar = None
    self.sync_progress = None
    self.sync_dialog = None
    status = self.sync_state.finish()
    self.sync_state = None
    if status["result"] == 0:
      banner = hildon.hildon_banner_show_information(self.server_window, "", "Synchronization successful")
    elif status["result"] == RESULT_LOCKED:
      banner = hildon.hildon_banner_show_information(self.server_window, "", "Synchronization already in progress")
    elif status["result"] == RESULT_NO_CONNECTION:
      banner = hildon.hildon_banner_show_information(self.server_window, "", "No network connection")
    else:
#      banner = hildon.hildon_banner_show_information(self.server_window, "", "Synchronization failed")
      note = hildon.hildon_note_new_information(self.server_window, "Synchronization failed")
      response = gtk.Dialog.run(note)

  def synchronize_cancel(self, dialog, id):
    self.sync_state.abort()

  def synchronize_close(self, dialog):
    self.sync_state.abort()

  def synchronize_quit(self, dialog, event):
    self.sync_state.abort()
    return True

  def synchronize_poll(self, source, condition):
    if self.sync_state is None:
      return False
    if self.sync_state.poll():
      self.synchronize_cleanup()
      return False
    else:
      self.sync_progress = self.sync_state.progress()
      if not self.sync_progress is None:
        self.sync_bar.set_fraction(self.sync_progress)
      return True
  def synchronize_tick(self):
    if self.sync_progress is None:
      self.sync_bar.pulse()
    return True

class SyncCLI(object):
  def __init__(self, server, quiet=False):
    gtk.set_application_name("SyncEvolution")
    self.sync = syncevolution.SyncEvolution(quiet)
    self.server = server
    self.server_dir = self.sync.get_server_dir(self.server)
    self.server_config = self.sync.get_server_config(self.server)
  def synchronize(self):
    self.sync_state = SyncRunner(True, self.sync, self.server, self.server_dir, self.server_config)
    self.sync_state.set_callbacks(self.synchronize_poll)
    if self.sync_state is None:
      return
    self.old_sigint = signal.signal(signal.SIGINT, self.synchronize_cancel)
    gtk.main()
    signal.signal(signal.SIGINT, self.old_sigint)
  def synchronize_cleanup(self):
    status = self.sync_state.finish()
    self.sync_state = None
    # FIXME: not sure how to show banner without window... so currently not displaying anything...
    if gtk.main_level() > 0:
      gtk.main_quit()
  def synchronize_cancel(self, sig, frame):
    self.sync_state.abort()
  def synchronize_poll(self, source, condition):
    if self.sync_state.poll():
      self.synchronize_cleanup()
      return False
    else:
      return True
