#include <QDir>
#include <QScrollArea>
#include <QDesktopServices>
#include <QVariant>
#include <phonon>
#include "symfoniemediaplayer.h"
#include "symfonieplaylist.h"
#include "symfonieplaylistitem.h"
#include "symfonieplaylistitemfolder.h"
#include "symfoniesong.h"
#include "symfoniedb.h"
#include "symfoniefiletypesdialog.h"

SymfonieMediaPlayer::SymfonieMediaPlayer(QObject *parent) :
        QObject(parent),
        playList(NULL),
        timer(new QTimer()),
        timerUpdater(new QTimer()),
        graduateVolumeTimer(new QTimer()),
        equalizerEffect(NULL),
        speedEffect(NULL),
        currentPlaybackIndex(-1),
        queuedIndex(-1),
        seekToBuffer(-1),
        timerRemainCounter(0),
        hasEqualizerEffect(false),
        isEqualizerEnabled(false),


        graduateVolumeResumeTo(0)
{   
    this->initPlayer();
}


void SymfonieMediaPlayer::initPlayer(){
    this->timer->setSingleShot(true);
    connect(timer, SIGNAL(timeout()), this, SLOT(timerEndSlot()));




    this->player = new Phonon::MediaObject;
    this->player->setTickInterval(1000);


    this->audioOutput = new Phonon::AudioOutput;


    this->path = Phonon::createPath(player, audioOutput);


    this->setFileFilter(SymfonieFileTypesDialog::getSavedFilterList());


    connect(this->player, SIGNAL(stateChanged(Phonon::State,Phonon::State)),
            this, SLOT(mediaStateChangedSlot(Phonon::State,Phonon::State)));

    connect(this->player, SIGNAL(aboutToFinish()), this, SLOT(aboutToFinishSlot()));
    connect(this->player, SIGNAL(currentSourceChanged(const Phonon::MediaSource &)),
            this, SLOT(currentSourceChangedSlot(const Phonon::MediaSource &)));

    connect(this, SIGNAL(playListUpdatedSignal(SymfoniePlayList*)),
            this, SLOT(playListUpdatedSlot(SymfoniePlayList*)));





    this->timerUpdater->setInterval(SymfonieMediaPlayer::TIMER_VALUE_UPDATE_RATE);
    connect(this->timerUpdater, SIGNAL(timeout()), this, SLOT(updateTimerValueSlot()));


    connect(this->graduateVolumeTimer, SIGNAL(timeout()), this, SLOT(doGraduateVolumeResuming()));

   //this->audioOutput->setVolume(0);
}

SymfonieMediaPlayer::~SymfonieMediaPlayer()
{
    this->saveState();
    delete player;
    delete audioOutput;
    delete this->timer;
    delete this->timerUpdater;
    delete this->graduateVolumeTimer;
    if(this->equalizerEffect != NULL){
        delete this->equalizerEffect;
    }

    if(this->speedEffect != NULL){
        delete this->speedEffect;
    }

}




void SymfonieMediaPlayer::startGraduateVolumeResuming(qreal value){
    this->graduateVolumeTimer->start(50);
    this->graduateVolumeResumeTo = value;
}


void SymfonieMediaPlayer::doGraduateVolumeResuming(){
   // qDebug() << "--------------doGraduateVolumeResuming()";
   // qreal lower = 0;
    qreal cur = this->audioOutput->volume();
    qreal value = this->graduateVolumeResumeTo;
    qreal inc = SymfonieMediaPlayer::GRADUATE_VOLUME_INCREMENT;

/*
    if(this->graduateVolumeResumeTo > lower && cur < lower){
        cur = lower;
    }
    */

    if(cur + inc > value){
        this->audioOutput->setVolume(value);
        this->graduateVolumeTimer->stop();
    }else{
        this->audioOutput->setVolume(cur + inc);
    }
}


Phonon::Effect* SymfonieMediaPlayer::getEffectDescriptionByName(QString effectName){

    QList<Phonon::EffectDescription> effectDescriptions =
            Phonon::BackendCapabilities::availableAudioEffects();

    foreach( Phonon::EffectDescription ef, effectDescriptions){

        if(QString::compare(ef.name(), effectName) == 0){
            return new Phonon::Effect(ef);
        }

    }
    return NULL;

}



bool SymfonieMediaPlayer::insertEqualizerEffect(){

    if(this->isEqualizerEnabled){
        return false;
    }

    QString efName = "equalizer-10bands";
    this->equalizerEffect = this->getEffectDescriptionByName(efName);

    if(this->equalizerEffect == NULL){
        return false;
    }

    if(path.insertEffect(this->equalizerEffect )){
        this->hasEqualizerEffect = true;
        this->isEqualizerEnabled = true;
        emit equalizerEffectInsertedSignal();
        qDebug() << "Effect inserted: " << efName;
        return true;
    }else{
        qDebug() << "Effect insertion failed: " << efName;
        return false;
    }

}


void SymfonieMediaPlayer::removeEqualizerEffect(){
    if(this->isEqualizerEnabled){
        this->path.removeEffect(this->equalizerEffect);
        this->isEqualizerEnabled = false;

        emit equalizerEffectRemovedSignal();
    }
}



bool SymfonieMediaPlayer::insertSpeedEffect(){
    QString efName = "speed";
    this->speedEffect = this->getEffectDescriptionByName(efName);

    if(this->speedEffect == NULL){
        return false;
    }

    if(path.insertEffect(this->speedEffect )){

        qDebug() << "Effect inserted: " << efName;
        return true;
    }else{
        qDebug() << "Effect insertion failed: " << efName;
        return false;
    }

}

/*
void SymfonieMediaPlayer::removeSpeedEffect(){
    if(this->speedEffect != NULL){
        this->path.removeEffect(this->speedEffect);
        delete this->speedEffect;
    }
}*/

void SymfonieMediaPlayer::setSpeed(float speed){
    //this->removeSpeedEffect();

    if(this->speedEffect == NULL){
        this->insertSpeedEffect();
    }

    if(this->speedEffect == NULL){
       return;
    }


    QList<Phonon::EffectParameter> parameters = this->speedEffect->parameters();

    foreach(Phonon::EffectParameter parameter, parameters) {
        if(QString::compare(parameter.name(), "speed") == 0){
            qDebug() << "Speed Value set to: " << speed;
            this->speedEffect->setParameterValue(parameter, speed);
        }
    }

}


void SymfonieMediaPlayer::doTimer(int time){
    if(time < 0){
        time = 0;
    }


    this->timerRemainCounter = time;
    this->timerUpdater->start();
    this->timer->start(time);

}

void SymfonieMediaPlayer::updateTimerValueSlot(){
    qDebug() << "updateTimerValueSlot: " << this->timerRemainCounter;
    if(this->timer->isActive()){
        if(this->timerRemainCounter > 0){
            this->timerRemainCounter = this->timerRemainCounter - SymfonieMediaPlayer::TIMER_VALUE_UPDATE_RATE;
        }
    }else{
        this->timerRemainCounter = 0;
        this->timerUpdater->stop();
    }
    emit timerRemainTimeChangedSignal(this->timerRemainCounter);
}

int SymfonieMediaPlayer::getTimerRemain(){
    return this->timerRemainCounter;
}

void SymfonieMediaPlayer::stopTimer(){
    this->timer->stop();
    this->timerUpdater->stop();
    this->timerRemainCounter = 0;
    emit timerStoppedSignal();
}

void SymfonieMediaPlayer::timerEndSlot(){
    this->pause();
    emit timerStoppedSignal();
}

Phonon::AudioOutput* SymfonieMediaPlayer::getAudioOutput(){
    return this->audioOutput;
}


void SymfonieMediaPlayer::saveState(){
    int index = this->getCurrentPlaybackIndex();

    qDebug() << "trying to save current played file index: " << index << " path:" << this->getFilePathAt(index);

    SymfonieDb::getInstance()->saveSetting(
            "last_played_file",
            (index != -1) ? this->getFilePathAt(index) : QVariant("")
            );

    SymfonieDb::getInstance()->saveSetting(
            "last_played_index",
            index);


    SymfonieDb::getInstance()->saveSetting(
            "last_played_volume", this->audioOutput->volume());



    if(this->getCurrentPlaybackIndex() != -1){
        SymfonieDb::getInstance()->saveSetting("last_played_position", QVariant(this->getMediaObject()->currentTime()));
    }else{
        SymfonieDb::getInstance()->saveSetting("last_played_position", QVariant(0));
    }

    SymfonieDb::getInstance()->saveSetting("is_equalizer_enabled", QVariant(this->isEqualizerEnabled == true ? 1: 0));
}



Phonon::Path SymfonieMediaPlayer::getPath(){
    return this->path;
}

Phonon::Effect *SymfonieMediaPlayer::getEqualizerEffect(){
    return this->equalizerEffect;
}


void SymfonieMediaPlayer::setCurrentDirectory(QString dir){
    this->currentDirectory = dir;
    emit directoryChangedSignal(dir);
    qDebug() <<  "directoryChangedSignal emitted: dir: " << dir;
}

QString SymfonieMediaPlayer::getCurrentDirectory(){
    return this->currentDirectory;
}

Phonon::MediaObject *SymfonieMediaPlayer::getMediaObject(){
    return player;
}


void SymfonieMediaPlayer::setFileFilter(QStringList filterList){
    this->fileFilterList = filterList;
}

//remove any entry in the list
SymfoniePlayList* SymfonieMediaPlayer::scanDir(QString path){//bool isUpdatePlayList){

    QDir directory(path);

    SymfoniePlayList *playList = new SymfoniePlayList(path);

    QStringList fileList = directory.entryList(this->fileFilterList);
    QStringList dirList = directory.entryList(
            QStringList(),
            QDir::Dirs|QDir::NoDotAndDotDot|QDir::Drives|QDir::AllDirs
            );


    // playList->clear();

    SymfoniePlayListItemFolder *folder =  new SymfoniePlayListItemFolder(path, SymfoniePlayListItem::DotDot);
    QFileInfo fileInfo(path + "/..");
    folder->setDisplayName(path);
    folder->setName(fileInfo.canonicalFilePath());
    playList->addFolder(folder);

    qDebug() << "first dir = "  << folder->getName() << " | " << path + "/.." << " can name: " + fileInfo.canonicalFilePath();

    for (int i = 0; i < dirList.size(); i++) {
        QFileInfo fileInfo(path + "/" +dirList.at(i));
        if(!fileInfo.isDir()){
            continue;
        }

        SymfoniePlayListItemFolder *folder =  new SymfoniePlayListItemFolder(fileInfo.canonicalFilePath());
        folder->setDisplayName(fileInfo.fileName());
        playList->addFolder(folder);

    }

    if(this->fileFilterList.size() > 0){



        for (int i = 0; i < fileList.size(); i++) {
            QFileInfo fileInfo(path + "/" +fileList.at(i));
            QString filePath = fileInfo.canonicalFilePath();

            SymfoniePlayListItemFile *file =  new SymfoniePlayListItemFile(filePath);
            file->setDisplayName(fileInfo.fileName());

            /*
        SymfonieSong *song = SymfonieSong::findByFilePath(filePath);
        if(song != NULL){

        }else{
            SymfonieSong songToSave;
            songToSave.setFilePath(file->getName());
            songToSave.setFileSize(fileInfo.size());
            songToSave.setLength(-1);
            songToSave.save();
            song = &songToSave;
            qDebug() << "Inserted to db: " << songToSave.getFileDir() << " + " << songToSave.getFileName();

        }
        file->setSong(song);
*/

            playList->addFile(file);
        }
    }

    qDebug() << playList->getSize() << "size";

    /** @todo temporary disabled until this is optimized */
    /*
    qDebug() << "is update playList: " << isUpdatePlayList;
    if(isUpdatePlayList){
        this->setPlayList(playList);
        qDebug() << "Set currentPlayback dir: " << this->getCurrentDirectory();
    }
    */

    emit scanDirCompletedSignal(playList, path);

    return playList;
}

SymfoniePlayList *SymfonieMediaPlayer::getPlayList(){
    return this->playList;
}

void SymfonieMediaPlayer::setPlayList(SymfoniePlayList* playList){
    if(this->playList != NULL){
        delete this->playList;
    }
    this->playList = playList;
    this->setCurrentDirectory(playList->getCurrentDirectory());
    emit playListUpdatedSignal(playList);
}

void SymfonieMediaPlayer::playListUpdatedSlot(SymfoniePlayList* playList){
    this->queuedIndex = playList->getFirstPlaybackIndex();
}

bool SymfonieMediaPlayer::isCanPlay(){
    this->getPlayList()->getFileIndexList();

    if(this->getPlayList()->getFileIndexList()->size() > 0){
        return TRUE;
    }else{
        return FALSE;
    }
}


void SymfonieMediaPlayer::mediaStateChangedSlot(Phonon::State newState, Phonon::State oldState)
{

    if(
            (newState == Phonon::PlayingState ||
             newState == Phonon::BufferingState ||
             newState == Phonon::PausedState)
            ){

        if(newState == Phonon::PlayingState){
           // Timer::singleShot( 10*60*1000, &a, SLOT(quit()) );
          //  qDebug() << "------------- playing stae triggere";

        }

        if(this->seekToBuffer > 0){
            //grudate volume resuming
            qreal savedVolume = SymfonieDb::getInstance()->getSetting("last_played_volume", QVariant(0)).toDouble();

            if(savedVolume == 0){
                savedVolume = .5;
            }

            this->audioOutput->setVolume(0);
            qDebug() << "Resume volume to " << savedVolume;
            this->startGraduateVolumeResuming(savedVolume);
            //


            qDebug() <<"state: " << newState << "player seek to " << this->seekToBuffer << " this->player->isSeekable(): " <<  this->player->isSeekable();

            //QTimer::singleShot( 10*60*1000, &this->aboutToFinishSlot(), SLOT(quit()) );
            this->player->seek(this->seekToBuffer);
            this->seekToBuffer = -1;

        }
    }
    emit this->mediaStateChangedSignal(newState, oldState);
}

void SymfonieMediaPlayer::oneCompletedSlot(){
    qDebug() << "reading completed";
}


void SymfonieMediaPlayer::fetchSongInfo(QString path){
    SymfonieSong *song = SymfonieSong::findByFilePath(path);
    if(song != NULL){
        qDebug() << "in db";
        return;
    }else{
        qDebug() << "not in db";
        delete song;
    }
}



QString SymfonieMediaPlayer::getFilePathAt(int index){
    SymfoniePlayListItem *item = this->getPlayList()->get(index);
    if(item == NULL){
        qWarning() << "Invalid playback index.";
        return QString();
    }

    if(item->getItemType() != SymfoniePlayListItem::File){
        qWarning() << "invalid file type: " << item->getName();
        return QString();
    }

    return item->getName();
}

void SymfonieMediaPlayer::play(qint64 seekTo){
    
    if(this->isCanPlay() != TRUE){
        qWarning() << "Playlist is empty.";
        return;
    }

    if(this->queuedIndex == -1){
        qWarning() << "Empty queue, nothing to play.";
        return;
    }

    QString path = this->getFilePathAt(this->queuedIndex);


    if(path.length() > 0){
        this->seekToBuffer = seekTo;
        this->player->stop();
        this->player->clearQueue();
        this->player->setCurrentSource(Phonon::MediaSource(path));
        this->currentPlaybackIndex = this->queuedIndex;
        this->player->play();
        this->queuedIndex = -1;
    }

}

void SymfonieMediaPlayer::playOrPause(){
    if(this->player->state() == Phonon::PlayingState){
        this->player->pause();
    }else{
        this->player->play();
    }
}

void SymfonieMediaPlayer::pause(){
    if(this->player->state() == Phonon::PlayingState){
        this->player->pause();
    }
}

void SymfonieMediaPlayer::enqueue(int index){
    this->queuedIndex = index;
}

void SymfonieMediaPlayer::playPrevious(){
    this->enqueue(this->getPreviousPlaybackIndex());
    this->play();
}

void SymfonieMediaPlayer::playNext(){
    this->enqueue(this->getNextPlaybackIndex());
    this->play();
}


int SymfonieMediaPlayer::getCurrentPlaybackIndex(){
    return this->currentPlaybackIndex;
}

int SymfonieMediaPlayer::getNextPlaybackIndex(){
    return this->getPlayList()->getNextPlaybackIndex(this->currentPlaybackIndex);
}

int SymfonieMediaPlayer::getPreviousPlaybackIndex(){
    return this->getPlayList()->getPreviousPlaybackIndex(this->currentPlaybackIndex);
}

void SymfonieMediaPlayer::aboutToFinishSlot(){
    this->queuedIndex = this->getNextPlaybackIndex();
    this->player->clearQueue();
    this->player->enqueue(Phonon::MediaSource(this->getFilePathAt(this->queuedIndex)));
}


void SymfonieMediaPlayer::currentSourceChangedSlot ( const Phonon::MediaSource &newSource ) {
    this->currentPlaybackIndex = this->queuedIndex;
}


