﻿#!/usr/bin/env python
#coding=utf-8

# Stroke Order Chinese Input Method 筆劃輸入法
# Author: Amanda Lam
# E-mail: amanda.hoic@gmail.com

import osso
import sqlite3
import sys
import gtk
import hildon
import os
import gettext
import i18n
import gobject
import string
import ctypes
import time
import pygobject
import sendSMS
import locale
import dbus
import pango
import xml.dom.minidom
import xml.parsers.expat
from portrait import FremantleRotation
from virtualKeyboard import vKeyboard

# By using apt-cache search, the following packages are required:
# python2.5-gtk2, python2.5-hildon, python-gobject

_ = i18n.language.gettext	# Internationization support

class StrokeOrder():
	_PORTRAIT, _LANDSCAPE = ('portrait', 'landscape')
	
	strokesInputted = gtk.Label()	# Define label for displaying strokes
	strokesInputted.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#002090"))	# Modify label font color
	strokesCodeInputted = ""	# Define variable for stroke codes being inputted
	eb = gtk.EventBox()
	eb.add(strokesInputted)
	eb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#AADDFF"))	# Modify label background color
	eb.set_size_request(800,50)
	
	candidateChars = hildon.TouchSelector(text = True)	# Define TouchSelector control for displaying candidate characters
	smsTextLabel = gtk.Label()	
	pannableArea = hildon.PannableArea()
	pannableArea.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
	
	textArea = hildon.TextView()	# Define the text area for input
	textArea.set_placeholder(_("StrokeOrder_textAreaPlaceHolder"))	# Set the info text for display when text area is empty
	textArea.set_wrap_mode(gtk.WRAP_CHAR)		
	textArea.set_cursor_visible(True)	
	
	vkb = vKeyboard(textArea)
	vkbVBox = vkb.getKeyboard()
	vnpVBox = vkb.getNumpad()
	
	charLeft = gtk.Label()	# Define label for displaying number of characters left and number of messages being sent
	charLeft.set_text("70(1)")
	textAreaBuffer = textArea.get_buffer()	
	sendTo = hildon.TextView()	# Define the text field for phone number(s)
	sendToBuffer = sendTo.get_buffer()
	sendTo.set_placeholder(_("StrokeOrder_sendToPlaceHolder"))	# Set the info text for display when phone number text field is empty
	textboxInFocus = "textArea"	# Set input focus to the text area initially

	sendToNumbers = ""	# Variable for remembering the contact numbers when user switches back and forth the pages
	
	osso_ctx = osso.Context("test_abook", "0.one")	# Define variable for Address Book
	capi = pygobject.PyGObjectCPAI()
	
	vKB = False
	vNP = False
	orientation = ""
	
	def __init__(self):	
		self.main()
			
	def _on_orientation_signal(self, orientation, stand, face, x, y, z):
		self.AppWidth, self.AppHeight = self.currentwindow.get_size()		
		self.orientation = orientation
		print "Width: " + str(self.AppWidth) + ", Height: " + str(self.AppHeight)		
		if orientation == "portrait":
			self.resizeControls(480,744)
		else:
			self.resizeControls(800,424)

	# This method returns a default setting XML string
	def getDefaultSettingsXML(self):
		defaultSettingsXML = "<?xml version=\"1.0\"?><Settings><UseVirtualKB>" + str(True) + "</UseVirtualKB><UseCustIMDB>" + str(True) + "</UseCustIMDB><CandCustOrder>" + str(True) +"</CandCustOrder><AssocCustOrder>" + str(True) + "</AssocCustOrder></Settings>" 
		return defaultSettingsXML

	# This method writes the default settings to the actual XML file
	def writeDefaultSettingsXML(self):
		self.settingsDOM = xml.dom.minidom.parseString(self.getDefaultSettingsXML())
		self.settingsFile = open("/home/user/MyDocs/.StrokeOrder/settings.xml", "w")
		self.settingsFile.write(self.settingsDOM.toxml("UTF-8"))
		self.settingsFile.close()

	# This method writes the currently selected settings to the actual XML file 
	def writeSettingsXML(self,UseVirtualKB,UseCustIMDB,CandCustOrder,AssocCustOrder):
		self.SettingsXML = "<?xml version=\"1.0\"?><Settings><UseVirtualKB>" + str(UseVirtualKB) + "</UseVirtualKB><UseCustIMDB>" + str(UseCustIMDB) + "</UseCustIMDB><CandCustOrder>" + str(CandCustOrder) +"</CandCustOrder><AssocCustOrder>" + str(AssocCustOrder) + "</AssocCustOrder></Settings>" 
		self.settingsFile = open("/home/user/MyDocs/.StrokeOrder/settings.xml", "w")
		self.settingsFile.write(self.SettingsXML)
		self.settingsFile.close() 

	# This method takes in the setting XML DOM object and the name of a node, and then returns the value of that node.
	def getKeyValue(self, settingsDOM, key):
		returnValue = ""
		for node in self.settingsDOM.getElementsByTagName(key)[0].childNodes:
			if node.nodeType == node.TEXT_NODE:
				returnValue = returnValue + node.data
		return returnValue

	# This method creates a directory for the settings file under MyDocs/ , or returns true if it is already created
	def makedir(self,dirname):
		if os.path.isdir(dirname):
			return True
		elif os.path.isfile(dirname):
			return False
		else:
			head, tail = os.path.split(dirname)
			if head and not os.path.isdir(head):
				self.makedir(head)
			if tail:
				os.mkdir(dirname)
		return True
  
	def resizeControls(self, AppWidth, AppHeight):
		self.eb.set_size_request(AppWidth,50)
		if self.vKB == False and self.vNP == False:
			self.buttonHBox1.set_size_request(AppWidth,int(AppHeight*0.165))
			self.buttonHBox2.set_size_request(AppWidth,int(AppHeight*0.165))		
			self.EngButton.set_size_request(int(AppWidth/3),int(AppHeight*0.165))
			self.SymButton.set_size_request(int(AppWidth/3),int(AppHeight*0.165))
			self.SMSPageButton.set_size_request(int(AppWidth/3),int(AppHeight*0.165))
			self.textAreaButtonsHBox.set_size_request(AppWidth,int(AppHeight*0.165))
		elif self.vKB == True and self.vNP == False:
			self.EngButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.SymButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.SMSPageButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.textAreaButtonsHBox.set_size_request(AppWidth,int(AppHeight*0.1))
			self.vkbVBox.set_size_request(AppWidth,int(AppHeight*0.45))
		elif self.vKB == False and self.vNP == True:
			self.EngButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.SymButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.SMSPageButton.set_size_request(int(AppWidth/3),int(AppHeight*0.1))
			self.textAreaButtonsHBox.set_size_request(AppWidth,int(AppHeight*0.1))
			self.vnpVBox.set_size_request(AppWidth,int(AppHeight*0.45))
		
		self.candidateChars.set_size_request(150,int(AppHeight*0.2830))
		self.candidateCharsAndtextAreaHBox.set_size_request(int(AppWidth),int(AppHeight*0.2830))			
	
	def getCurrentLocale(self):
		# Return the current system locale
		lc, encoding = locale.getdefaultlocale()
		CurrentLocale = lc
		return CurrentLocale
		
	def queryString(self, stroke):
		# Takes in the inputted stroke sequence and display the candidate characters for selection
		if stroke!="":
			if (len(self.strokesCodeInputted)<=4 and "6" not in self.strokesCodeInputted and "7" not in self.strokesCodeInputted) or ("%" in self.strokesCodeInputted):	# For <=4 strokes, since there would be a huge amount of characters having strokes beginning with the same stroke sequence, limit the number of candidate characters to 75 for the sake of performance
				self.cur.execute("SELECT DISTINCT Character FROM StrokeOrder WHERE Strokes LIKE '" + stroke + "%' ORDER BY Frequency DESC, Strokes ASC, Character ASC LIMIT 75")
			else:
				self.cur.execute("SELECT DISTINCT Character FROM StrokeOrder WHERE Strokes LIKE '" + stroke + "%' ORDER BY Frequency DESC, Strokes ASC, Character ASC")	# For >4 strokes, display all candidate characters
			self.con.commit()
		
		tm = gtk.ListStore(gobject.TYPE_STRING)
		self.candidateChars.set_model(0,tm)	# Clear the TouchSelector

		if stroke!="":
			for row in self.cur.fetchall():
				self.candidateChars.append_text(row[0])	# Generate candidate character selection list

	def queryStrokes(self, queryStr):
		# Takes in a Chinese character and returns its stroke orders
		querychar = queryStr[0:1]
		print querychar
		sQuery = "SELECT DISTINCT Strokes FROM StrokeOrder WHERE Character='" + querychar + "' LIMIT 1"
		print sQuery
		self.cur.execute(sQuery)
		self.con.commit()
		result=""
		for row in self.cur.fetchall():
			result = row[0]
		return result
	
	def queryAssoc(self, charKey):		
		# Takes in a Chinese character and display its associated phrases for selection
		if charKey!="":
			if self.getCurrentLocale() == "zh_HK" or self.getCurrentLocale() == "zh_Yue" or self.getCurrentLocale() == "zh_TW":	# Display Traditional Chinese associated phrases first if system locale is Chinese (Hong Kong), Cantonese or Chinese (Taiwan)
				self.cur.execute("SELECT DISTINCT Phrase FROM Assoc WHERE CharKey='" + charKey + "' ORDER BY Frequency DESC, TradSim DESC, Phrase ASC")		
			else: # Display Simplifeid Chinese associated phrases first if system locale is Chinese (PRC), Chinese, or English
				self.cur.execute("SELECT DISTINCT Phrase FROM Assoc WHERE CharKey='" + charKey + "' ORDER BY Frequency DESC, TradSim ASC, Phrase ASC")
			self.con.commit()
		
		tm = gtk.ListStore(gobject.TYPE_STRING)
		self.candidateChars.set_model(0,tm)	# Clear the TouchSelector
		
		if charKey!="":
			for row in self.cur.fetchall():		
				self.candidateChars.append_text(row[0])	# Generate associated phrases selection list
		
	def connectIMDB(self): # Connect to user's customized database or system's default database, based on the value stored in the settings file
		try:
			if self.makedir('/home/user/MyDocs/.StrokeOrder') == True:
				self.settingsDOM = xml.dom.minidom.parse('/home/user/MyDocs/.StrokeOrder/settings.xml') # parse the settings XML and returns the XML DOM object	
		except xml.parsers.expat.ExpatError: # displays error message in case of parse error
				self.writeDefaultSettingsXML()
				self.settingsDOM = xml.dom.minidom.parseString(self.getDefaultSettingsXML())
		except IOError: # displays error message in case of file I/O error
				self.writeDefaultSettingsXML()
				self.settingsDOM = xml.dom.minidom.parseString(self.getDefaultSettingsXML())
				
		if self.getKeyValue(self.settingsDOM,"UseCustIMDB") == "True":
			if os.path.exists("/home/user/MyDocs/.StrokeOrder/IM.db") == False:
				try:
					if self.makedir('/home/user/MyDocs/.StrokeOrder') == True:	# Copy default IM.db to user's folder if it doesn't exist
						copycmd = "cp /opt/StrokeOrder/IM.db /home/user/MyDocs/.StrokeOrder/"
						os.system(copycmd)
						print "IM.db copied to /home/user/MyDocs/.StrokeOrder/"
				except IOError: # displays error message in case of file I/O error
					dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window, _("StrokeOrder_ResetIMDBFailed")) 
					response = dialog.run()
					dialog.destroy()										
			self.con=sqlite3.connect("/home/user/MyDocs/.StrokeOrder/IM.db")	# Connect to the customized stroke lookup database			
			print "User's customized IM.db is being used."
		else:
			self.con=sqlite3.connect("/opt/StrokeOrder/IM.db")	# Connect to default stroke lookup database	
			print "System's default IM.db is being used."
			
		self.con.text_factory=str
		self.cur = self.con.cursor()			
		
	def main(self):	
		self.connectIMDB()		
				
		system_bus = dbus.Bus.get_system()
		system_bus.add_signal_receiver(self._on_orientation_signal, \
			signal_name='sig_device_orientation_ind', \
			dbus_interface='com.nokia.mce.signal', \
			path='/com/nokia/mce/signal')
			
		program = hildon.Program.get_instance()
		window = hildon.StackableWindow()
				
		self.currentwindow = window		

		self.AppWidth, self.AppHeight = self.currentwindow.get_size()
		# scrolledWindow = gtk.ScrolledWindow(hadjustment=None, vadjustment=None)
		# scrolledWindow.show()
		# scrolledWindow.set_policy('automatic','automatic')
		
		main_window = window
		app_name = _("StrokeOrder_AppName")
		app_version = "1.0"	
		# Support Portrait Mode, adopting gPodder's code
		initial_mode = FremantleRotation.AUTOMATIC
		rotation_object = FremantleRotation(app_name, main_window, app_version, initial_mode) 
	 
		program.add_window(window)		
		self.candidateChars.set_column_selection_mode(hildon.TOUCH_SELECTOR_SELECTION_MODE_SINGLE)
		
		# Creating user interface
		
		self.buttonHBox1 = gtk.HBox(False, 0)
		self.buttonHBox2 = gtk.HBox(False, 0)
		self.SymButtonVBox = gtk.VBox(False, 0)
		self.candidateCharsAndtextAreaHBox = gtk.HBox(False, 0)		
		
		self.textAreaButtonsVBox = gtk.VBox(False, 0)	
		self.textAreaButtonsHBox = gtk.HBox(False, 0)	
		 
		self.mainVBox = gtk.VBox(False, 0)
		
		OneButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		OneButton.set_label("一")   
		OneButton.connect("clicked", self.StrokeClicked, 1)   

		TwoButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		TwoButton.set_label("丨")   
		TwoButton.connect("clicked", self.StrokeClicked, 2)   

		ThreeButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		ThreeButton.set_label("丿")   
		ThreeButton.connect("clicked", self.StrokeClicked, 3)

		FourButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		FourButton.set_label("丶")   
		FourButton.connect("clicked", self.StrokeClicked, 4)

		FiveButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		FiveButton.set_label("乛")   
		FiveButton.connect("clicked", self.StrokeClicked, 5)
		
		if self.getKeyValue(self.settingsDOM,"UseVirtualKB") == "True":
			self.EngButton = gtk.ToggleButton(_("StrokeOrder_EngMode"))
			self.EngButton.connect("clicked", self.showHideVirtualKeyboard, window)
			self.SymButton = gtk.ToggleButton(_("StrokeOrder_SymMode"))
			self.SymButton.connect("clicked", self.showHideVirtualNumpad, window)
		else:
			self.EngButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			self.EngButton.set_label(_("StrokeOrder_EngMode"))   
			self.EngButton.connect("clicked", self.StrokeClicked, 6)
			
			self.SymButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			self.SymButton.set_label(_("StrokeOrder_SymMode"))   
			self.SymButton.connect("clicked", self.StrokeClicked, 7)	

		
		WildcardButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		WildcardButton.set_label("？")   
		WildcardButton.connect("clicked", self.StrokeClicked, 8)		

		CopyButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		CopyButton.set_label(_("StrokeOrder_Copy"))   
		CopyButton.connect("clicked", self.CopyClicked, window)

		DelButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		DelButton.set_label("←")   
		DelButton.connect("clicked", self.DelClicked, window)

		self.SMSPageButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		self.SMSPageButton.set_label(_("StrokeOrder_Send"))   
		self.SMSPageButton.connect("clicked", self.SMSPageClicked)
		
		NewButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		NewButton.set_label(_("StrokeOrder_New"))   
		NewButton.connect("clicked", self.NewClicked)
		
		self.buttonHBox1.pack_start(OneButton, True, True, 0)
		self.buttonHBox1.pack_start(TwoButton, True, True, 0)
		self.buttonHBox1.pack_start(ThreeButton, True, True, 0)

		self.buttonHBox2.pack_start(FourButton, True, True, 0)
		self.buttonHBox2.pack_start(FiveButton, True, True, 0)
		self.buttonHBox2.pack_start(WildcardButton, True, True, 0)				
		
		print "Width: " + str(self.AppWidth) + ", Height: " + str(self.AppHeight)
	
		self.textArea.connect("grab-focus", self.setCurrentTextboxInFocus, "textArea")
		self.textAreaBuffer.connect("changed", self.refreshCharLeft)

		self.textAreaButtonsHBox.pack_start(self.EngButton, True, True, 0)	
		self.textAreaButtonsHBox.pack_start(self.SymButton, True, True, 0)	
		# self.textAreaButtonsHBox.pack_start(CopyButton, True, True, 0)	
		self.textAreaButtonsHBox.pack_start(self.SMSPageButton, True, True, 0)						
						
		self.pannableArea.add_with_viewport(self.textArea)
		self.candidateCharsAndtextAreaHBox.pack_start(self.pannableArea, True, True, 0)
		self.candidateCharsAndtextAreaHBox.pack_start(self.candidateChars, False, True, 0)		

		self.textAreaButtonsVBox.pack_start(self.charLeft, False, False, 0)
		self.textAreaButtonsVBox.pack_start(DelButton, True, True, 0)	
		self.textAreaButtonsVBox.pack_start(CopyButton, True, True, 0)

		self.candidateCharsAndtextAreaHBox.pack_start(self.textAreaButtonsVBox, False, False, 0)	

		self.candidateChars.connect("changed", self.CharSelected)
		self.strokesCodeInputted = "6"
		self.queryString(self.strokesCodeInputted)
		self.strokesInputted.set_text( self.ConvertCodeToChar(self.strokesCodeInputted) )	# Display the inputted strokes
		
		self.mainVBox.pack_start(self.eb, False, False ,0)				
		self.mainVBox.pack_start(self.candidateCharsAndtextAreaHBox, True, True, 0)		
		self.mainVBox.pack_start(self.buttonHBox1, False, True ,0)
		self.mainVBox.pack_start(self.buttonHBox2, False, True ,0)		
		self.mainVBox.pack_start(self.textAreaButtonsHBox, False, True, 0)				
		self.resizeControls(800,424)
		
		window.set_title(_("StrokeOrder_AppName")) 
		window.connect("delete_event",self.app_preclose)
		menu = self.create_menu(window, "MainPage") 
		window.set_app_menu(menu)	
		window.add(self.mainVBox)	
		window.show_all()		
		self.textArea.grab_focus()
		gtk.main()			
		
	def showHideVirtualKeyboard(self, EngButton, window):			
		if self.EngButton.get_active() == False:
			self.vKB = False			
			self.mainVBox.remove(self.vkbVBox)
		else: 
			if self.SymButton.get_active() == True:
				self.SymButton.set_active(False)
				self.showHideVirtualNumpad(self.SymButton,window)
				
			self.buttonHBox1.set_size_request(0,0)
			self.buttonHBox2.set_size_request(0,0)
			self.mainVBox.pack_start(self.vkbVBox,False,True,0)
			self.vKB = True
			self.StrokeClicked(self.EngButton,6)
			window.show_all()	

		if self.orientation == "portrait":
			self.resizeControls(480,744)
		else:
			self.resizeControls(800,424)			
			
	def showHideVirtualNumpad(self, SymButton, window):			
		if self.SymButton.get_active() == False:
			self.vNP = False
			self.mainVBox.remove(self.vnpVBox)
		else: 
			if self.EngButton.get_active() == True:
				self.EngButton.set_active(False)
				self.showHideVirtualKeyboard(self.EngButton,window)
			self.buttonHBox1.set_size_request(0,0)
			self.buttonHBox2.set_size_request(0,0)
			self.mainVBox.pack_start(self.vnpVBox,False,True,0)
			self.vNP = True
			self.StrokeClicked(self.SymButton,7)
			window.show_all()				

		if self.orientation == "portrait":
			self.resizeControls(480,744)
		else:
			self.resizeControls(800,424)			
			
	def aboutProduct(self, AboutProductButton, window):
		# Show the About this Product dialog box
		
		text = _("StrokeOrder_AboutProductText")
		dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window,text) 
		response = dialog.run()
		dialog.destroy()

	def settingsClicked(self, QueryStrokesButton, window):
		# Create the UI for the Settings page
		settingsPage = hildon.StackableWindow()
		settingsPage.set_title(_("StrokeOrder_AppName") + " - " + _("StrokeOrder_Settings"))
		
		try:
			if self.makedir('/home/user/MyDocs/.StrokeOrder') == True:
				self.settingsDOM = xml.dom.minidom.parse('/home/user/MyDocs/.StrokeOrder/settings.xml') # parse the settings XML and returns the XML DOM object	
		except xml.parsers.expat.ExpatError: # displays error message in case of parse error
				self.writeDefaultSettingsXML()
				self.settingsDOM = xml.dom.minidom.parseString(self.getDefaultSettingsXML())
		except IOError: # displays error message in case of file I/O error
				self.writeDefaultSettingsXML()
				self.settingsDOM = xml.dom.minidom.parseString(self.getDefaultSettingsXML())

		UseVirtualKBLabel = gtk.Label()
		UseVirtualKBLabel.set_text(_("StrokeOrder_UseVirtualKBLabel"))
		
		if self.getKeyValue(self.settingsDOM,"UseVirtualKB") == "True":
			self.UseVirtualKBButton = gtk.ToggleButton(_("StrokeOrder_UseVirtualKB_On"))
			self.UseVirtualKBButton.set_active(True)
		else:
			self.UseVirtualKBButton = gtk.ToggleButton(_("StrokeOrder_UseVirtualKB_Off"))
			self.UseVirtualKBButton.set_active(False)
			
		self.UseVirtualKBButton.connect("clicked",self.UseVirtualKBButtonClicked, window)		

		UseVirtualKBHBox = gtk.HBox()		
		UseVirtualKBHBox.pack_start(UseVirtualKBLabel,True,True,0)
		UseVirtualKBHBox.pack_start(self.UseVirtualKBButton,True,True,0)
		
		CandCustOrderLabel = gtk.Label()
		CandCustOrderLabel.set_text(_("StrokeOrder_CandCustOrderLabel"))
		
		if self.getKeyValue(self.settingsDOM,"CandCustOrder") == "True":
			self.CandCustOrderButton = gtk.ToggleButton(_("StrokeOrder_CandCustOrder_On"))
			self.CandCustOrderButton.set_active(True)
		else:
			self.CandCustOrderButton = gtk.ToggleButton(_("StrokeOrder_CandCustOrder_Off"))
			self.CandCustOrderButton.set_active(False)
			
		self.CandCustOrderButton.connect("clicked",self.CandCustOrderButtonClicked)		

		CandCustOrderHBox = gtk.HBox()		
		CandCustOrderHBox.pack_start(CandCustOrderLabel,True,True,0)
		CandCustOrderHBox.pack_start(self.CandCustOrderButton,True,True,0)
		
		AssocCustOrderLabel = gtk.Label()
		AssocCustOrderLabel.set_text(_("StrokeOrder_AssocCustOrderLabel"))
		
		if self.getKeyValue(self.settingsDOM,"AssocCustOrder") == "True":
			self.AssocCustOrderButton = gtk.ToggleButton(_("StrokeOrder_AssocCustOrder_On"))
			self.AssocCustOrderButton.set_active(True)
		else:
			self.AssocCustOrderButton = gtk.ToggleButton(_("StrokeOrder_AssocCustOrder_Off"))
			self.AssocCustOrderButton.set_active(False)
			
		self.AssocCustOrderButton.connect("clicked",self.AssocCustOrderButtonClicked)		
		
		AssocCustOrderHBox = gtk.HBox()		
		AssocCustOrderHBox.pack_start(AssocCustOrderLabel,True,True,0)
		AssocCustOrderHBox.pack_start(self.AssocCustOrderButton,True,True,0)

		UseCustIMDBLabel = gtk.Label()
		UseCustIMDBLabel.set_text(_("StrokeOrder_UseCustIMDBLabel"))
		
		if self.getKeyValue(self.settingsDOM,"UseCustIMDB") == "True":
			self.UseCustIMDBButton = gtk.ToggleButton(_("StrokeOrder_UseCustIMDB_On"))
			self.UseCustIMDBButton.set_active(True)			
			self.CandCustOrderButton.set_sensitive(True)
			self.AssocCustOrderButton.set_sensitive(True)			
		else:
			self.UseCustIMDBButton = gtk.ToggleButton(_("StrokeOrder_UseCustIMDB_Off"))
			self.UseCustIMDBButton.set_active(False)
			self.CandCustOrderButton.set_sensitive(False)
			self.AssocCustOrderButton.set_sensitive(False)			
			
		self.UseCustIMDBButton.connect("clicked",self.UseCustIMDBButtonClicked)		
		
		UseCustIMDBHBox = gtk.HBox()		
		UseCustIMDBHBox.pack_start(UseCustIMDBLabel,True,True,0)
		UseCustIMDBHBox.pack_start(self.UseCustIMDBButton,True,True,0)			
		self.ResetIMDBButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		self.ResetIMDBButton.set_label(_("StrokeOrder_ResetIMDB"))   
		self.ResetIMDBButton.connect("clicked", self.ResetIMDBButtonClicked, window)
		
		vbox = gtk.VBox()
		vbox.pack_start(UseVirtualKBHBox, True, True, 0)								
		vbox.pack_start(UseCustIMDBHBox, True, True, 0)								
		vbox.pack_start(CandCustOrderHBox, True, True, 0)								
		vbox.pack_start(AssocCustOrderHBox, True, True, 0)								
		vbox.pack_start(self.ResetIMDBButton, True, True, 0)
		settingsPage.connect("delete_event",self.settingsPageClosed)
		settingsPage.add(vbox)
		settingsPage.show_all()					

	def UseVirtualKBButtonClicked(self, UseVirtualKBButton, window):
		if self.UseVirtualKBButton.get_active() == True:		
			self.UseVirtualKBButton.set_label(_("StrokeOrder_UseVirtualKB_On"))
		else:
			self.UseVirtualKBButton.set_label(_("StrokeOrder_UseVirtualKB_Off"))
		dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window, _("StrokeOrder_UseVirtualKBChange")) 
		response = dialog.run()
		dialog.destroy()					
			
	def UseCustIMDBButtonClicked(self, UseCustIMDBButton):
		if self.UseCustIMDBButton.get_active() == True:
			self.UseCustIMDBButton.set_label(_("StrokeOrder_UseCustIMDB_On"))
			self.CandCustOrderButton.set_sensitive(True)
			self.AssocCustOrderButton.set_sensitive(True)
		else:
			self.UseCustIMDBButton.set_label(_("StrokeOrder_UseCustIMDB_Off"))
			self.CandCustOrderButton.set_sensitive(False)
			self.AssocCustOrderButton.set_sensitive(False)
			
	def CandCustOrderButtonClicked(self, CandCustOrderButton):
		if self.CandCustOrderButton.get_active() == True:
			self.CandCustOrderButton.set_label(_("StrokeOrder_CandCustOrder_On"))
		else:
			self.CandCustOrderButton.set_label(_("StrokeOrder_CandCustOrder_Off"))

	def AssocCustOrderButtonClicked(self, AssocCustOrderButton):
		if self.AssocCustOrderButton.get_active() == True:
			self.AssocCustOrderButton.set_label(_("StrokeOrder_AssocCustOrder_On"))
		else:
			self.AssocCustOrderButton.set_label(_("StrokeOrder_AssocCustOrder_Off"))

	def ResetIMDBButtonClicked(self, ResetIMDBButton, window):
		confirm = hildon.hildon_note_new_confirmation(window, _("StrokeOrder_Confirm_ResetIMDB"))
		response = gtk.Dialog.run(confirm)
		gtk.Dialog.hide(confirm)
		if response == gtk.RESPONSE_DELETE_EVENT:
			print "Reset IM.db cancelled"
		elif response == gtk.RESPONSE_OK:	
			try:
				if self.makedir('/home/user/MyDocs/.StrokeOrder') == True:
					copycmd = "cp /opt/StrokeOrder/IM.db /home/user/MyDocs/.StrokeOrder/"
					os.system(copycmd)						
			except IOError: # displays error message in case of file I/O error
				dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window, _("StrokeOrder_ResetIMDBFailed")) 
				response = dialog.run()
				dialog.destroy()			
			
			dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window, _("StrokeOrder_ResetIMDBSuccess")) 
			response = dialog.run()
			dialog.destroy()			
		
	def settingsPageClosed(self, window, widget):
		self.writeSettingsXML(self.UseVirtualKBButton.get_active(),self.UseCustIMDBButton.get_active(),self.CandCustOrderButton.get_active(),self.AssocCustOrderButton.get_active())
		self.connectIMDB()
		# Set the input focus back to the text area once the Settings page is closed		
		self.textArea.grab_focus()
		
	def queryStrokesClicked(self, QueryStrokesButton, window):
		# Create the UI for the stroke query module
		
		notebook = gtk.Notebook()
		queryCharLabel = gtk.Label()
		queryCharLabel.set_justify(gtk.JUSTIFY_LEFT)
		queryCharLabel.set_width_chars(25)
		queryCharLabel.set_line_wrap(True)		
		queryCharLabel.set_text(_("StrokeOrder_QueryStrokesInstruction"))
		
		self.queryCharEntry = hildon.Entry(gtk.HILDON_SIZE_AUTO)				
		self.queryCharEntry.set_max_length(1)
		
		queryCharEntryFont=pango.FontDescription('normal 60')
		self.queryCharEntry.modify_font(queryCharEntryFont)
		self.queryCharEntry.set_size_request(120,120)
		
		PasteButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		PasteButton.set_label(_("StrokeOrder_Paste"))
		PasteButton.connect("clicked", self.queryStrokesPasteClicked)
		
		queryCharHBox = gtk.HBox()
		queryCharHBox.pack_start(queryCharLabel, True, True, 0)		
		queryCharHBox.pack_start(self.queryCharEntry, False, True, 0)		
		
		queryCharVBox = gtk.VBox()
		queryCharVBox.pack_start(queryCharHBox, False, True, 0)		
		queryCharVBox.pack_start(PasteButton, True, True, 0)		
			
		self.queryStrokesResultLabel = gtk.Label()		
		self.queryStrokesResultExplainLabel = gtk.Label()		

		self.queryStrokesCopyButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		self.queryStrokesCopyButton.set_label(_("StrokeOrder_Copy"))
		self.queryStrokesCopyButton.connect("clicked", self.queryStrokesCopyClicked, window)
		resultVBox = gtk.VBox()
		resultVBox.pack_start(self.queryStrokesResultExplainLabel, True, True, 0)
		
		queryStrokesPannableArea = hildon.PannableArea()
		queryStrokesPannableArea.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
		queryStrokesPannableArea.add_with_viewport(self.queryStrokesResultLabel)
		
		resultVBox.pack_start(queryStrokesPannableArea, True, True, 0)
		resultVBox.pack_start(self.queryStrokesCopyButton, True, True, 0)
						
		notebook.append_page(queryCharVBox, None)
		notebook.append_page(resultVBox, None)
		dialog = hildon.WizardDialog(None, _("StrokeOrder_QueryStrokes"), notebook)
		notebook.connect("switch-page", self.queryStrokesStartQuery, dialog)
		dialog.set_forward_page_func(self.queryStrokesWizardNextFunc)
		dialog.show_all()
		response = dialog.run()
		if response == hildon.WIZARD_DIALOG_FINISH: dialog.destroy()

	def queryStrokesWizardNextFunc(self, page, num, dialog):
		# Define the criteria for proceeding to the next step
		
		if self.queryCharEntry.get_text() == "": 
			return False
		else:
			return True

	def queryStrokesCopyClicked(self, CopyButton, window):
		# Copy the strokes to the clipboard
		
		textToBeCopied = self.queryStrokesResultLabel.get_text()
		if textToBeCopied != _("StrokeOrder_QueryStrokesResultNotFound") and textToBeCopied != _("StrokeOrder_SymMode") and textToBeCopied != _("StrokeOrder_EngMode"):
			clip = gtk.Clipboard()
			clip.set_text( textToBeCopied )
			clip.store()		

	def queryStrokesPasteClicked(self, window):
		# Triggered when user clicks the Paste button from the menu
		clip = gtk.Clipboard()
		data = clip.wait_for_contents('UTF8_STRING')
		text = unicode( str(data.get_text()), "utf-8")
		text = text[0:1]
		self.queryCharEntry.set_text(text)

			
	def queryStrokesStartQuery(self, notebook, page, num, dialog):
		# Performed the stroke query operation
		
		result = self.queryStrokes( unicode(self.queryCharEntry.get_text(),"utf-8") )
		print result;
		if result != "":
			if result=="6" or result=="7":
				self.queryStrokesCopyButton.set_sensitive(False)
			else:
				self.queryStrokesCopyButton.set_sensitive(True)
			self.queryStrokesResultExplainLabel.set_text("「" + self.queryCharEntry.get_text() + _("StrokeOrder_QueryStrokesResultExplain"))				
			self.queryStrokesResultLabel.set_text( self.ConvertCodeToChar(result) )
		else:
			self.queryStrokesResultLabel.set_text( _("StrokeOrder_QueryStrokesResultNotFound") )
			self.queryStrokesCopyButton.set_sensitive(False)
		return True
	
	def SMSPageClicked(self, window):
		# Proceed to the SMS Preview page
		
		smsText = self.textAreaBuffer.get_text(self.textAreaBuffer.get_start_iter(), self.textAreaBuffer.get_end_iter(), False)
		
		if (smsText!=""):
			sendSMSPage = hildon.StackableWindow()
			sendSMSPage.set_title(_("StrokeOrder_Send"))
			
			vbox = gtk.VBox(False, 0)		
			
			smsText = unicode(smsText,"utf-8")
			msglength = len(smsText)
			textToBeSent = ""
			for i in range(1, msglength/70+2 ):
				textToBeSent = textToBeSent + "＊＊＊ " + _("StrokeOrder_MessageCount") + str(i) + " ＊＊＊\n" + smsText[(i*70-70):(i*70)] + "\n\n"
			
			self.smsTextLabel.set_text(_("StrokeOrder_MessagePreview") + "\n\n" + textToBeSent)
			self.smsTextLabel.set_alignment(0,0)						
			# self.smsTextLabel.set_line_wrap(True)
		
			self.smsTextLabel.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#000000"))	# Modify label font color	
			evtb = gtk.EventBox()
			evtb.add(self.smsTextLabel)
			evtb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#999999"))	# Modify label background color		

			pannableAreaForSMSText = hildon.PannableArea()
			pannableAreaForSMSText.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
			pannableAreaForSMSText.add_with_viewport(evtb)
			pannableAreaForSMSText.set_size_request(800,100)
			
			sendToAndSelectContactButtonHBox = gtk.HBox(False, 0)
						
			self.sendToBuffer.set_text(self.sendToNumbers)			# Retrieve remembered contact numbers, if user switches pages back and forth
			self.sendToBuffer.connect("changed",self.rememberContactNumbers)	# In case sendTo text box content has been changed, call the rememberContactNumbers function
			self.sendTo.set_buffer(self.sendToBuffer)			
			self.sendTo.connect("grab-focus", self.setCurrentTextboxInFocus, "sendTo")

			sendSMSButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			sendSMSButton.set_label(_("StrokeOrder_Send"))
			sendSMSButton.connect("clicked", self.sendSMSPreCheck, self.currentwindow)		
		
			selectContactButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			selectContactButton.set_label(_("StrokeOrder_SelectContact"))
			selectContactButton.connect("clicked", self.selectContactClicked)
			
			OneButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			OneButton.set_label("１")
			OneButton.connect("clicked", self.numpadButtonClicked, 1)
			
			TwoButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			TwoButton.set_label("２")
			TwoButton.connect("clicked", self.numpadButtonClicked, 2)
			
			ThreeButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			ThreeButton.set_label("３")
			ThreeButton.connect("clicked", self.numpadButtonClicked, 3)		
			
			FourButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			FourButton.set_label("４")
			FourButton.connect("clicked", self.numpadButtonClicked, 4)		
			
			FiveButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			FiveButton.set_label("５")
			FiveButton.connect("clicked", self.numpadButtonClicked, 5)

			SixButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			SixButton.set_label("６")
			SixButton.connect("clicked", self.numpadButtonClicked, 6)

			SevenButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			SevenButton.set_label("７")
			SevenButton.connect("clicked", self.numpadButtonClicked, 7)
			
			EightButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			EightButton.set_label("８")
			EightButton.connect("clicked", self.numpadButtonClicked, 8)

			NineButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			NineButton.set_label("９")
			NineButton.connect("clicked", self.numpadButtonClicked, 9)
			
			PlusButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			PlusButton.set_label("＋")
			PlusButton.connect("clicked", self.numpadButtonClicked, "+")
			
			ZeroButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			ZeroButton.set_label("０")
			ZeroButton.connect("clicked", self.numpadButtonClicked, "0")
			
			SemicolonButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			SemicolonButton.set_label("；")
			SemicolonButton.connect("clicked", self.numpadButtonClicked, ";")

			DelButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
			DelButton.set_label("←")   
			DelButton.connect("clicked", self.DelClicked, window)
			
			sendToAndSelectContactButtonHBox.pack_start(self.sendTo, True, True, 0)						
			sendToAndSelectContactButtonHBox.pack_start(selectContactButton, False, False ,0)
			
			numpadHBox1 = gtk.HBox(False, 0)
			numpadHBox1.pack_start(OneButton,True,True,0)
			numpadHBox1.pack_start(TwoButton,True,True,0)
			numpadHBox1.pack_start(ThreeButton,True,True,0)
					
			numpadHBox2 = gtk.HBox(False, 0)
			numpadHBox2.pack_start(FourButton,True,True,0)
			numpadHBox2.pack_start(FiveButton,True,True,0)
			numpadHBox2.pack_start(SixButton,True,True,0)
			
			numpadHBox3 = gtk.HBox(False, 0)
			numpadHBox3.pack_start(SevenButton,True,True,0)
			numpadHBox3.pack_start(EightButton,True,True,0)
			numpadHBox3.pack_start(NineButton,True,True,0)
			
			numpadHBox4 = gtk.HBox(False, 0)
			numpadHBox4.pack_start(PlusButton,True,True,0)
			numpadHBox4.pack_start(ZeroButton,True,True,0)
			numpadHBox4.pack_start(DelButton,True,True,0)		
			
			numpadHBox5 = gtk.HBox(False, 0)
			numpadHBox5.pack_start(SemicolonButton,True,True,0)
			numpadHBox5.pack_start(sendSMSButton,True,True,0)
			
			numpad = gtk.VBox(False, 0)
			numpad.pack_start(numpadHBox1,True,True,0)
			numpad.pack_start(numpadHBox2,True,True,0)
			numpad.pack_start(numpadHBox3,True,True,0)
			numpad.pack_start(numpadHBox4,True,True,0)
			numpad.pack_start(numpadHBox5,True,True,0)
			
			textAndnumpadVBox = gtk.VBox(False, 0)
			textAndnumpadVBox.pack_start(pannableAreaForSMSText, True, True, 0)
			textAndnumpadVBox.pack_start(numpad, True, True, 0)				
			
			vbox.pack_start(sendToAndSelectContactButtonHBox, False, False, 0)						
			vbox.pack_start(textAndnumpadVBox, True, True, 0)		

			sendSMSPage.connect("delete_event",self.sendSMSPageClosed)
			sendSMSPage.add(vbox)
			menu = self.create_menu(sendSMSPage, "SMSPage") 
			sendSMSPage.set_app_menu(menu)				
			sendSMSPage.show_all()
		else:					
			banner = hildon.hildon_banner_show_information(window,"" , _("StrokeOrder_Error_BlankMessage"))
			
	def sendSMSPageClosed(self, widnow, widget):
		# Set the input focus back to the text area once the SMS Preview page is closed
		self.textArea.grab_focus()
	
	def rememberContactNumbers(self, widget):
		# For remembering the contact numbers when user switches back and forth the pages
		self.sendToNumbers = self.sendToBuffer.get_text(self.sendToBuffer.get_start_iter(), self.sendToBuffer.get_end_iter())	# For remembering the contact numbers when user switches back and forth the pages
	
	def numpadButtonClicked(self, window, number):
		buffer = self.sendTo.get_buffer() 
		buffer.set_text(buffer.get_text(buffer.get_start_iter(), buffer.get_iter_at_mark(buffer.get_insert()), False) + str(number) + buffer.get_text(buffer.get_iter_at_mark(buffer.get_insert()), buffer.get_end_iter(), False))
		self.sendTo.set_buffer(buffer)			
			
	def create_menu(self, window, page):
		# Create the application menu
		
		menu = hildon.AppMenu()   
		
		CopyButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		CopyButton.set_label(_("StrokeOrder_Copy"))   
		CopyButton.connect("clicked", self.CopyClicked, window)
		
		PasteButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		PasteButton.set_label(_("StrokeOrder_Paste"))   
		PasteButton.connect("clicked", self.pasteClicked, window)    
		
		AboutButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		AboutButton.set_label(_("StrokeOrder_AboutProduct"))   
		AboutButton.connect("clicked", self.aboutProduct, window)    
		
		NewButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		NewButton.set_label(_("StrokeOrder_New"))   
		NewButton.connect("clicked", self.NewClicked)			

		QueryStrokesButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		QueryStrokesButton.set_label(_("StrokeOrder_QueryStrokes"))   
		QueryStrokesButton.connect("clicked", self.queryStrokesClicked, window)			

		SettingsButton = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		SettingsButton.set_label(_("StrokeOrder_Settings"))
		SettingsButton.connect("clicked", self.settingsClicked, window)			
		
		if page == "MainPage":
			NewButton.set_sensitive(True)
		elif page == "SMSPage":
			NewButton.set_sensitive(False)	# Disable button if user is using SMS page
		menu.append(NewButton)							
		menu.append(QueryStrokesButton)
		menu.append(CopyButton)
		menu.append(PasteButton)
		menu.append(SettingsButton)
		menu.append(AboutButton)   		

		menu.show_all()
		return menu

		
	def ConvertCodeToChar(self, strokes):
		# Convert stroke input code into actual strokes for display
		s = strokes
		s = s.replace("1","一")
		s = s.replace("2","丨")
		s = s.replace("3","丿")
		s = s.replace("4","丶")
		s = s.replace("5","乛")
		s = s.replace("6",_("StrokeOrder_EngModeFull"))
		s = s.replace("7",_("StrokeOrder_SymModeFull"))
		s = s.replace("%","？")
		return s

	def ConvertCharToCode(self, strokesInChar):
		# Convert actual strokes to stroke input code in numbers
		s = strokesInChar
		s = s.replace("一","1")
		s = s.replace("丨","2")
		s = s.replace("丿","3")
		s = s.replace("丶","4")
		s = s.replace("乛","5")
		s = s.replace(_("StrokeOrder_EngModeFull"),"6")
		s = s.replace(_("StrokeOrder_SymModeFull"),"7")
		s = s.replace("？","%")
		return s
		
	def setCurrentTextboxInFocus(self, widget, textbox):
		# Set the input focus to the given text field
		self.textboxInFocus = textbox
		
	def StrokeClicked(self, StrokeButton, stroke):
		# Triggered once a stroke (1 to 5, and ? ) button is pressed.
		# Build up the stroke sequence string and submit it to queryString() to lookup the candidate characters
		
		if stroke==6 or stroke==7:	# Do not build up stroke sequence if English or Symbol button is pressed
			self.strokesCodeInputted = str(stroke)				
		elif ("6" in self.strokesCodeInputted or "7" in self.strokesCodeInputted) and stroke!=8:
			self.strokesCodeInputted = str(stroke)
		elif stroke==8:
			if ("6" in self.strokesCodeInputted or "7" in self.strokesCodeInputted):
				self.strokesCodeInputted = "%"
			else:
				self.strokesCodeInputted = self.strokesCodeInputted + "%"
		else:
			self.strokesCodeInputted = self.strokesCodeInputted + str(stroke)
		self.strokesInputted.set_text( self.ConvertCodeToChar(self.strokesCodeInputted) )	# Display the inputted strokes
		
		self.queryString(self.strokesCodeInputted)
		
	def pasteClicked(self, PasteButton, window):
		# Triggered when user clicks the Paste button from the menu
		clip = gtk.Clipboard()
		data = clip.wait_for_contents('UTF8_STRING')
		text = str(data.get_text())
		if self.textboxInFocus == "textArea":
			buffer = self.textArea.get_buffer()
		elif self.textboxInFocus == "sendTo":
			buffer = self.sendTo.get_buffer() 
			
		selectedText = buffer.get_selection_bounds()
		if selectedText!=(): # If there is selected text, delete the selected text
			buffer.delete_selection(True,True)
		buffer.insert(buffer.get_iter_at_mark(buffer.get_insert()), text)		
 
		if self.textboxInFocus == "textArea":
			self.textArea.set_buffer(buffer)
			self.textArea.grab_focus()
		elif self.textboxInFocus == "sendTo":
			self.sendTo.set_buffer(buffer)
			self.sendTo.grab_focus()
	
	def CharSelected(self, Selector, column):
		# Triggered once a candidate / English / Numeric / Symbol character is selected from the TouchSelector control.
		# Send the selected text to the text area.
		
		text = Selector.get_current_text()
		
#		if self.textboxInFocus == "textArea":	# If text area is having input focus, insert text to the current cursor position
		buffer = self.textArea.get_buffer()
		buffer.insert(buffer.get_iter_at_mark(buffer.get_insert()), text)
		self.textArea.grab_focus()
		
#		elif self.textboxInFocus == "sendTo":	# If phone number field is having input focus, insert text to the current cursor position
#			buffer = self.sendTo.get_buffer() 
#			buffer.set_text(buffer.get_text(buffer.get_start_iter(), buffer.get_iter_at_mark(buffer.get_insert()), False) + text + buffer.get_text(buffer.get_iter_at_mark(buffer.get_insert()), buffer.get_end_iter(), False))
#			self.sendTo.set_buffer(buffer)							
				
		if not "6" in self.strokesCodeInputted and not "7" in self.strokesCodeInputted:	# If any of the stroke 1-5 is pressed, once text is inserted into the text area, clear the stroke sequence variable and associated phrases
			text = unicode(text, "utf-8")		
		
			if self.strokesInputted.get_text().find(_("StrokeOrder_SelectAssoc1")) > -1 and self.strokesInputted.get_text().find(_("StrokeOrder_SelectAssoc2")) > -1:
				if self.getKeyValue(self.settingsDOM,"UseCustIMDB") == "True" and self.getKeyValue(self.settingsDOM,"AssocCustOrder") == "True":	# If selected phrase was from an associated phrase...
					OriginalCharKey = self.strokesInputted.get_text().replace(_("StrokeOrder_SelectAssoc1"),"").replace(_("StrokeOrder_SelectAssoc2"),"") # Obtain the original Character Key
					self.cur.execute("UPDATE Assoc SET Frequency=(SELECT Frequency FROM Assoc WHERE CharKey='" + OriginalCharKey + "' AND Phrase='" + text + "' LIMIT 1)+1 WHERE CharKey='" + OriginalCharKey + "' AND Phrase='" + text + "';")	# Increment the frequency by 1 so that next time the same phrase can be shown earlier
					self.con.commit()
					print "Frequency of Associated Phrase CharKey=" + OriginalCharKey + " and text = " + text + " is incremented by 1."
			else:	# type strokes and then select single character
				if self.getKeyValue(self.settingsDOM,"UseCustIMDB") == "True" and self.getKeyValue(self.settingsDOM,"CandCustOrder") == "True":
					OriginalStrokes = self.ConvertCharToCode(self.strokesInputted.get_text())
					self.cur.execute("UPDATE StrokeOrder SET Frequency=(SELECT Frequency FROM StrokeOrder WHERE Strokes LIKE '" + OriginalStrokes + "%' AND Character='" + text + "' LIMIT 1)+1 WHERE Strokes LIKE '" + OriginalStrokes + "%' AND Character='" + text + "';")	# Increment the frequency by 1 so that next time the same character can be shown earlier
					self.con.commit()					
					print "Frequency of character " + text + " with Strokes beginning with " + OriginalStrokes + " is incremented by 1."
			if len(text)>1:	# Use the last character of the phrase if the length of the selected phrase is > 1
				text = text[-1]
				
			self.strokesInputted.set_text(_("StrokeOrder_SelectAssoc1") + text + _("StrokeOrder_SelectAssoc2"))	# Clear the stroke display area once the text is inserted
			self.strokesCodeInputted = ""
			self.queryString("")			
			self.queryAssoc(text)
	
	def CopyClicked(self, CopyButton, window):
		# Copy the text area content to the clipboard
		clipboard = gtk.Clipboard()
		if self.textboxInFocus == "textArea":
			buffer = self.textArea.get_buffer()
		elif self.textboxInFocus == "sendTo":
			buffer = self.sendTo.get_buffer()
			
		selectedText = buffer.get_selection_bounds()
		if selectedText==():	# If some text is selected, copy the selected text to clipboard only
			textToBeCopied = buffer.get_text(buffer.get_start_iter(), buffer.get_end_iter(), False)
		else:	# If no text is selected, copy all the contents in the text area
			textToBeCopied = buffer.get_text(selectedText[0], selectedText[1], False)
		clipboard.set_text(textToBeCopied)
		clipboard.store()
		dialog = hildon.Note(hildon.NOTE_TYPE_INFORMATION_THEME, window, "「" + textToBeCopied + "」 " + _("StrokeOrder_HasBeenCopied")) 
		response = dialog.run()
		dialog.destroy()	

	def DelClicked(self, LeftButton, window):		
		if self.textboxInFocus == "sendTo":		# Set buffer according to current input focus
			buffer = self.sendTo.get_buffer()			
		elif self.textboxInFocus == "textArea":
			buffer = self.textArea.get_buffer()
			
		if len(self.strokesCodeInputted) > 0:	# If strokes sequence is not blank, delete stroke from the sequence first
			self.strokesCodeInputted = self.strokesCodeInputted[0:len(self.strokesCodeInputted)-1]
			self.queryString(self.strokesCodeInputted)			
			self.strokesInputted.set_text( self.ConvertCodeToChar(self.strokesCodeInputted) )
		else:						
			selectedText = buffer.get_selection_bounds()
			if selectedText==():	# If there is no selected text, issue a Backspace keystroke
				offset = buffer.get_iter_at_mark(buffer.get_insert())
				buffer.backspace(offset,True,True)
			else:	# If there is selected text, delete the selected text
				buffer.delete_selection(True,True)

	def refreshCharLeft(self, widget):
		# Recalculates the number of characters left in the current message, and the number of messages to be sent out
		msg = unicode(self.textAreaBuffer.get_text(self.textAreaBuffer.get_start_iter(), self.textAreaBuffer.get_end_iter(), False),"utf-8")
		charactersLeft = ( 70 * ( len(msg) / 70 + 1 ) ) - len(msg)
		msgCount = len(msg) / 70 + 1
		self.charLeft.set_text( str( charactersLeft ) + "(" + str(msgCount) + ")" )
							
	def NewClicked(self, widget): 
		# Reset everything back to the very beginning
		text = ""
		buffer = self.textArea.get_buffer()
		buffer.set_text(text)
		self.textArea.set_buffer(buffer)
		
		buffer = self.sendTo.get_buffer()
		buffer.set_text("")
		
		self.smsTextLabel.set_text("")
		
		self.strokesCodeInputted = "6"
		self.queryString(self.strokesCodeInputted)
		self.strokesInputted.set_text( self.ConvertCodeToChar(self.strokesCodeInputted) )	# Display the inputted strokes
				
	def selectContactClicked(self, widget):
		# Pops up the Address Book, select a contact and phone number will be inserted to the phone number text field
		start, end = self.sendToBuffer.get_bounds()
		oldbuffer = self.sendToBuffer.get_text(start, end)
		osso_abook = ctypes.CDLL('libosso-abook-1.0.so.0')
		argv_type = ctypes.c_char_p * len(sys.argv)
		argv = argv_type(*sys.argv)
		argc = ctypes.c_int(len(sys.argv))
		osso_abook.osso_abook_init(ctypes.byref(argc), ctypes.byref(argv),
								   hash(self.osso_ctx))
		c_chooser = osso_abook.osso_abook_contact_chooser_new(None, _("StrokeOrder_SelectContact"))
		chooser = self.capi.pygobject_new(c_chooser)
		hildon.hildon_gtk_window_set_portrait_flags(chooser, hildon.PORTRAIT_MODE_SUPPORT)

		chooser.run()
		chooser.hide()

		contacts = osso_abook.osso_abook_contact_chooser_get_selection(c_chooser)
		glib = ctypes.CDLL('libglib-2.0.so.0')

		def glist(addr):
			class _GList(ctypes.Structure):
				_fields_ = [('data', ctypes.c_void_p),
							('next', ctypes.c_void_p)]
			l = addr
			while l:
				l = _GList.from_address(l)
				yield l.data
				l = l.next

		for i in glist(contacts):
			c_selector = osso_abook.osso_abook_contact_detail_selector_new_for_contact(c_chooser, i, 2) 	

			selector = self.capi.pygobject_new(c_selector)
			hildon.hildon_gtk_window_set_portrait_flags(selector, hildon.PORTRAIT_MODE_SUPPORT)
			selector.run()
			selector.hide()

			c_field = osso_abook.osso_abook_contact_detail_selector_get_selected_field(c_selector)
			get_display_value = osso_abook.osso_abook_contact_field_get_display_value
			get_display_value.restype = ctypes.c_char_p
			finalval = osso_abook.osso_abook_contact_field_get_display_value(c_field)
	#            print oldbuffer
			if oldbuffer == "":
				if finalval != None:
					self.sendToBuffer.set_text(str(finalval))
			else:
				if finalval != None:
					newbuffer = oldbuffer +";" + str(finalval)
					self.sendToBuffer.set_text(newbuffer)
		
	def sendSMSPreCheck(self, widget, window):
		# Before sending out SMS, detect long message and break it down into multiple ones.
		start, end = self.textAreaBuffer.get_bounds()
		text = unicode(self.textAreaBuffer.get_text(start,end), "utf-8")
		start, end = self.sendToBuffer.get_bounds()
		number = self.sendToBuffer.get_text(start,end)
		
		if text=="":			  
			banner = hildon.hildon_banner_show_information(window,"" , _("StrokeOrder_Error_BlankMessage"))
		elif number == "":	# If phone number is not specified...
			banner = hildon.hildon_banner_show_information(window,"" , _("StrokeOrder_Error_NoContact"))
			print "error, no number specified!"
		else:
			msglength = len(text)
			for i in range(1, msglength/70+2 ):
				textToBeSent = text[(i*70-70):(i*70)]
				self.sendSMS(widget, window, textToBeSent )
				print "Loop: " + str(i)
				print "Message length: " + str(msglength)
				print "Total no. of messages to be sent: " + str(msglength/70+1)
				print "Text to be sent:\n" + textToBeSent
			os.system("sudo pkill -f rtcom-messaging-ui")

	def sendSMS(self, widget, window, text):
		# Send SMS to the specified recipient(s)
		
		start2, end2 = self.sendToBuffer.get_bounds()
		fullnumber = self.sendToBuffer.get_text(start2,end2)
		number = fullnumber.replace("+","").replace(" ","") 		
		if number == "":	# If phone number is not specified...
			  note2 = _("StrokeOrder_Error_NoContact")
			  banner = hildon.hildon_banner_show_information(window,"" , note2)
			  print "error, no number specified!"
		elif text == "":	# If text area is blank...
			  note3 = _("StrokeOrder_Error_BlankMessage")
			  banner = hildon.hildon_banner_show_information(window,"" , note3)
			  print "error, message is empty!"
		else:	# If message is valid....
			confirm = hildon.hildon_note_new_confirmation(window, _("StrokeOrder_Confirm_SendMessage") + "「" + text + "」？")
			response = gtk.Dialog.run(confirm)
			gtk.Dialog.hide(confirm)
			print "response was: " +str(response)

			if response == gtk.RESPONSE_DELETE_EVENT:
				print "sending cancelled"
			elif response == gtk.RESPONSE_OK:
				
				if number.isdigit():                
					note = _("StrokeOrder_SendingSMSTo") + number
					banner = hildon.hildon_banner_show_information(window,"" , note)
					sendSMScmd = "python /opt/StrokeOrder/sendSMS.py " + str(number) + " \"" + text.replace("\"","”").replace("'","’").replace("\\","＼").replace("`","‘") + "\""
					print sendSMScmd
					os.system(sendSMScmd)
				else:	# If there are more than one recipients, loop through the recipient list and send the message to all of them.
					numb = number.split(";")
					i=0
					while i < len(numb):
						note = _("StrokeOrder_SendingSMSTo") + numb[i]
						banner = hildon.hildon_banner_show_information(window,"" , note)
						sendSMScmd = "python /opt/StrokeOrder/sendSMS.py " + str(numb[i]).replace("+","").replace(" ","") + " \"" + text.replace("\"","”").replace("'","’").replace("\\","＼").replace("`","‘") + "\""
						print sendSMScmd
						os.system(sendSMScmd)						
						i = i+1
						time.sleep(5)
	
	def app_preclose(self, widnow, widget):
		# Close database connection before closing the application.
		self.con.close()
		gtk.main_quit(None)
		
if __name__ == "__main__":	
#	checkInstanceCMD = "ps | grep \"python /opt/StrokeOrder/StrokeOrder.py\" | wc -l"
#	fin,fout = os.popen4(checkInstanceCMD)
#	checkInstanceResult = str(fout.read().replace("\n",""))	
#	if checkInstanceResult=="2":	# If only two matches found, i.e. one for the grep command and another one the program itself, continue program execution
#		StrokeOrder()
	StrokeOrder()