/*
** Copyright (c) 2010  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "volumeview.h"

#define COLOR_LIGHT QColor(48, 48, 48)
#define COLOR_DARK  QColor(0, 0, 0)
#define COLOR_BACK  QColor(32, 32, 32)

//-----------------------------------------------------------------------------
/**
** Constructor
**
** \param pParent The parent for the widget.
*/
CVolumeView::CVolumeView(QWidget* pParent) : QWidget(pParent)
{
    m_ColorTextLight = QColor(255, 255, 255);
    m_ColorTextDark = QColor(200, 200, 200);
    m_ColorBack = QColor(64, 64, 64);
    m_ColorFront = QColor(255, 0, 0);
    m_Background = COLOR_BACK;
    m_Angle = 0;
    m_pVolume = NULL;
    m_bMouseDown = false;
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CVolumeView::~CVolumeView()
{
}

//-----------------------------------------------------------------------------
/**
** Sets the colors used in the widget.
**
** \param back The color for the unused space
** \param front The color for the used space
** \param textLight The light text
** \param textDark The dark text
*/
void CVolumeView::setColors(QColor back, QColor front, QColor textLight, QColor textDark)
{
    m_ColorTextLight = textLight;
    m_ColorTextDark = textDark;
    m_ColorBack = back;
    m_ColorFront = front;
}

//-----------------------------------------------------------------------------
/**
** Sets the volume information. The ownership of the parameter doesn't change!
**
** \param pVolume Pointer to the volume information. Do not delete.
*/
void CVolumeView::setVolume(Volume* pVolume)
{
    m_pVolume = pVolume;
    m_ColorFront = CMainWindow::getColor(pVolume->color);

    if (pVolume && pVolume->total > 0)
    {
        m_Angle = 360 * 16 * (pVolume->total - pVolume->free) / pVolume->total;
    }
}

///////////////////////////////////////////////////////////////////////////////
/// OVERRIDES
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Paint handler. Draws the usage pie and the texts to the window.
**
** \param pEvent The paint event.
*/
void CVolumeView::paintEvent(QPaintEvent* /*pEvent*/)
{
    if (m_pVolume)
    {
        QPainter painter(this);
        QPen pen;

        QRect area = rect();
        area.adjust(1, 1, 0, 0);

        QColor color1(COLOR_LIGHT);
        QColor color2(COLOR_DARK);
        if (m_bMouseDown)
        {
            QColor colorTmp = color1;
            color1 = color2;
            color2 = colorTmp;
        }

        painter.fillRect(rect(), m_Background);
        pen.setColor(color1);
        painter.setPen(pen);
        painter.drawLine(area.bottomLeft(), area.topLeft());
        painter.drawLine(area.topLeft(), area.topRight());
        pen.setColor(color2);
        painter.setPen(pen);
        painter.drawLine(area.topRight(), area.bottomRight());
        painter.drawLine(area.bottomRight(), area.bottomLeft());

        area.adjust(10, 10, -10, -10);

        int radiusLarge = qMin(area.height(), area.width()) - 4;
        radiusLarge = qMin(radiusLarge, area.width() / 3);
        int radiusSmall = radiusLarge * 0.5;

        QRect r(area.left() + area.width() - 4 - radiusLarge, area.top() + (area.height() - radiusLarge) / 2, radiusLarge, radiusLarge);
        QRect r2(r.left() + (radiusLarge - radiusSmall) / 2, r.top() + (radiusLarge - radiusSmall) / 2, radiusSmall, radiusSmall);

        painter.setRenderHint(QPainter::Antialiasing, true);

        pen.setWidth(2);
        pen.setColor(m_ColorBack);
        painter.setBrush(m_ColorBack.darker());
        painter.setPen(pen);
        painter.drawEllipse(r);

        pen.setColor(m_ColorFront);
        painter.setBrush(m_ColorFront.darker());
        painter.setPen(pen);
        painter.drawPie(r, 90 * 16, -m_Angle);

        pen.setColor(m_ColorBack);
        painter.setBrush(palette().background());
        painter.setPen(pen);
        painter.drawEllipse(r2);

        pen.setColor(m_ColorFront);
        painter.setPen(pen);
        painter.drawArc(r2, 90 * 16, -m_Angle);

        pen.setColor(m_ColorTextLight);
        painter.setPen(pen);
        r2.adjust(0, r2.height() / 10, 0, -r2.height() / 10);
        QFont fontNumber("Arial");
        QFont fontSmall("Arial");
        fontNumber.setPixelSize(r2.height() / 2);
        fontSmall.setPixelSize(r2.height() / 4);

        QString strValue = formatSize(m_pVolume->free);
        QStringList listValues = strValue.split(' ');

        if (listValues.size() == 2)
        {
            painter.setFont(fontSmall);
            painter.drawText(r2, Qt::AlignTop | Qt::AlignHCenter, tr("FREE"));
            painter.drawText(r2, Qt::AlignBottom | Qt::AlignHCenter, listValues[1]);
            painter.setFont(fontNumber);
            painter.drawText(r2, Qt::AlignCenter, listValues[0]);
        }

        // Draw the volume name, path and total size
        painter.setFont(fontNumber);

        QRect r3(area);
        r3.setHeight(r3.height() / 2);
        r3.setWidth(r3.width() - radiusLarge - 20);
        r3.adjust(0, 0, 0, 0);
        painter.drawText(r3, Qt::AlignBottom, m_pVolume->name);

        strValue = formatSize(m_pVolume->total);
        QFontMetrics metrics(fontSmall);
        int widthSize = metrics.width(strValue);

        painter.setFont(fontSmall);
        pen.setColor(m_ColorTextDark);
        painter.setPen(pen);

        r3.adjust(0, r3.height(), 0, r3.height());
        painter.drawText(r3, Qt::AlignTop | Qt::AlignRight, strValue);
        r3.adjust(0, 0, -(widthSize + 10), 0);
        painter.drawText(r3, Qt::AlignTop | Qt::AlignLeft, m_pVolume->path);
    }
}

//-----------------------------------------------------------------------------
/**
** Mouse press handler.
**
** \param pEvent The paint event.
*/
void CVolumeView::mousePressEvent(QMouseEvent* /*pEvent*/)
{
    m_bMouseDown = true;
    m_Background = COLOR_DARK;
    update();
}

//-----------------------------------------------------------------------------
/**
** Mouse release handler. Emits clicked signal.
**
** \param pEvent The paint event.
*/
void CVolumeView::mouseReleaseEvent(QMouseEvent* pEvent)
{
    if (m_pVolume && m_bMouseDown && rect().contains(pEvent->pos()))
    {
        emit clicked(m_pVolume->path);
    }

    m_bMouseDown = false;
    m_Background = COLOR_BACK;
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// PRIVATE
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Formats the size as a string.
**
** \param size The size to be formatted.
** \return The size as a string.
*/
QString CVolumeView::formatSize(quint64 size)
{
    if (size > 1000)
    {
        if (size > 1000000)
        {
            if (size > 1000000000)
            {
                return QString("%1 GB").arg(size / 1024.0 / 1024.0 / 1024.0, 0, 'f', 1);
            }
            return QString("%1 MB").arg(size / 1024.0 / 1024.0, 0, 'f', 1);
        }
        return QString("%1 KB").arg(size / 1024.0, 0, 'f', 1);
    }
    return QString("%1 B").arg(size);
}

// EOF
