/*
** Copyright (c) 2010  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "mainwindow.h"
#include "volumeview.h"
#include "scanwindow.h"
#include <sys/statvfs.h>

QStringList CMainWindow::c_MountPoints;
QList<Volume*> CMainWindow::c_Volumes;
QMap<QString, QString> CMainWindow::c_Binds;

//-----------------------------------------------------------------------------
/**
** Constructor
**
** \param pParent The parent for the main window (usually NULL)
*/
CMainWindow::CMainWindow(QWidget* pParent)
    : QMainWindow(pParent)
{
    setWindowTitle("Storage Usage");

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CMainWindow::~CMainWindow()
{
    for (int i = 0; i < c_Volumes.size(); i++)
    {
        delete c_Volumes[i];
    }
    c_Volumes.clear();
}

//-----------------------------------------------------------------------------
/**
** Initializes the widgets.
*/
void CMainWindow::initialize()
{
    readVolumes();

    QWidget* pRoot = new QWidget(this);

    QGridLayout* pLayout = new QGridLayout();
    pLayout->setContentsMargins(20, 20, 20, 20);
    pLayout->setSpacing(20);

    int count = c_Volumes.size();
    int cols = sqrt(count + 1);
    for (int i = 0; i < count; i++)
    {
        CVolumeView* pVolume = new CVolumeView(pRoot);
        pVolume->setVolume(c_Volumes[i]);
        pLayout->addWidget(pVolume, i / cols, i % cols);

        bool bOk = false;
        bOk = connect(pVolume, SIGNAL(clicked(QString)), this, SLOT(onScanVolume(QString)));
        Q_ASSERT(bOk);
    }

    pRoot->setLayout(pLayout);
    setCentralWidget(pRoot);

    // Create menu items
    QAction* aboutAction = new QAction(tr("About"), this);
    QAction* scanFolderAction = new QAction(tr("Scan Folder..."), this);
    QAction* scanAppsAction = new QAction(tr("Scan Packages"), this);

    // Add item into menu
#if defined(Q_WS_MAEMO_5)
    menuBar()->addAction(aboutAction);
    menuBar()->addAction(scanFolderAction);
    menuBar()->addAction(scanAppsAction);
#else
    // else File menu
    QMenu* menu = new QMenu(tr("File"), this);
    menu->addAction(aboutAction);
    menu->addAction(scanFolderAction);
    menu->addAction(scanAppsAction);

    QAction* exitAction = new QAction(tr("Exit"), this);
    menu->addAction(exitAction);
    connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));

    menuBar()->addMenu(menu);
#endif

    connect(aboutAction, SIGNAL(triggered()), this, SLOT(onAbout()));
    connect(scanAppsAction, SIGNAL(triggered()), this, SLOT(onScanApps()));
    connect(scanFolderAction, SIGNAL(triggered()), this, SLOT(onScanFolder()));
}

//-----------------------------------------------------------------------------
/**
** Scans all the volumes and finds out their total and free space. The /opt
** folder is handled as a special case since Maemo links it to a different
** volume.
*/
void CMainWindow::readVolumes()
{
    QProcess process;
#ifdef Q_WS_MAC
    process.start("/sbin/mount");
    QString strMatch = "(.+) on (.+) \\((.+),";
#else
    process.start("/bin/mount");
    QString strMatch = "(.+) on (.+) type (.+) ";
#endif
    process.waitForFinished();

    QByteArray arrayOutput = process.readAllStandardOutput();
    QString strMountOutput(arrayOutput);
    QStringList listOutput = strMountOutput.split(QChar('\n'));

    int index = 0;
    QRegExp rx(strMatch);
    rx.setMinimal(true);
    foreach (QString strLine, listOutput)
    {
        if (rx.indexIn(strLine, 0) != -1)
        {
            QString strMountPoint = rx.cap(2);
            if (!c_MountPoints.contains(strMountPoint))
            {
                c_MountPoints.append(strMountPoint);
                //qDebug() << strMountPoint;
            }

            QString strType = rx.cap(3);
            if (strType == "hfs" ||
                strType == "ext" ||
                strType == "ext2" ||
                strType == "ext3" ||
                strType == "ext4" ||
                strType == "nfs" ||
                strType == "smbfs" ||
                strType == "vfat" ||
                strType == "rootfs")
            {
                struct statvfs vfs = {0};
                if (statvfs(strMountPoint.toUtf8(), &vfs) == 0)
                {
                    Volume* pVolume = new Volume;
                    pVolume->path = rx.cap(2);
                    pVolume->name = rx.cap(2).split('/').last();
                    if (pVolume->path == "/")
                    {
                        pVolume->name = "RootFS";
                    }
                    else if (pVolume->path == "/media/mmc1")
                    {
                        pVolume->name = "Memory Card";
                    }
                    pVolume->total = (quint64)vfs.f_blocks * vfs.f_frsize;
                    pVolume->free = (quint64)vfs.f_bavail * vfs.f_frsize;
                    pVolume->color = (Colors)index++;
                    c_Volumes.append(pVolume);
                }
            }
            else if ((strType == "bind") || (strType == "none"))
            {
                QString strTarget = rx.cap(1);
                c_Binds[rx.cap(2)] = strTarget;
            }
        }
    }

/*    // DEBUG
    qDebug() << "Volumes:";
    foreach (Volume* pVol, c_Volumes) qDebug() << pVol->name << pVol->path;
    qDebug() << "Binds:";
    QMap<QString, QString>::iterator iter = c_Binds.begin();
    for ( ; iter != c_Binds.end(); iter++)
    {
        qDebug() << iter.key() << iter.value();
    }
    // ~DEBUG
*/
//    // TEST
//    Volume* pVolume = new Volume;
//    pVolume->path = "/usr/share";
//    pVolume->name = "Test";
//    pVolume->total = 10000000;
//    pVolume->free = 1000000;
//    pVolume->color = Color_Magenta;
//    c_Volumes.append(pVolume);
//    // ~TEST
}

//-----------------------------------------------------------------------------
/**
** Returns the correspoinding color for the parameter. If the given parameter
** does not have a color defined a white color is returned.
**
** \param color The color value
** \return The color
*/
QColor CMainWindow::getColor(Colors color)
{
    switch (color)
    {
    case Color_Red:
        return  QColor(255, 0, 0);
    case Color_Green:
        return  QColor(0, 200, 0);
    case Color_Blue:
        return  QColor(0, 0, 255);
    case Color_Magenta:
        return  QColor(200, 0, 200);
    case Color_Yellow:
        return  QColor(200, 200, 0);
    case Color_Cyan:
        return  QColor(0, 200, 200);
    case Color_White:
    default:
        return  QColor(200, 200, 200);
    }
}

//-----------------------------------------------------------------------------
/**
** Finds out the correct color for the given path. The colors are the same
** as in the summary page for the volumes.
**
** \param strPath The file path
** \return The color value
*/
Colors CMainWindow::getColorForPath(QString strPath)
{
    int len = 0;
    Colors color = Color_White;

    // Check the binds first and replace the path. Do it twice so that binds to binds (/opt) work.
    strPath = replaceBind(strPath);
    strPath = replaceBind(strPath);

    // Check the volumes and pick the same color as in the front page
    for (int i = 0; i < c_Volumes.size(); i++)
    {
        if (strPath.startsWith(c_Volumes[i]->path))
        {
            // There might be mount points further in the path so we need to check all of them
            if (len < c_Volumes[i]->path.size())
            {
                len = c_Volumes[i]->path.size();
                color = c_Volumes[i]->color;
            }
        }
    }

    return color;
}

//-----------------------------------------------------------------------------
/**
** Replaces the start of the path with the bind target if there is one.
**
** \param strPath The file path
** \return The modified path
*/
QString CMainWindow::replaceBind(QString strPath)
{
    QMap<QString, QString>::iterator iter = c_Binds.begin();
    for ( ; iter != c_Binds.end(); iter++)
    {
        if (strPath.startsWith(iter.key()))
        {
            strPath.replace(iter.key(), iter.value());
            break;
        }
    }
    return strPath;
}

///////////////////////////////////////////////////////////////////////////////
/// SLOTS
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens the about dialog.
*/
void CMainWindow::onAbout()
{
    QString strText = tr("Storage Usage by Kimmo Pekkola");
    QMessageBox::information(this, tr("Storage Usage"), strText);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the installed apps.
*/
void CMainWindow::onScanApps()
{
    CScanWindow* pScanWindow = new CScanWindow(this);
    pScanWindow->initialize("", true);

#ifndef Q_WS_MAEMO_5
    pScanWindow->resize(800, 480);
    pScanWindow->show();
#else
    pScanWindow->showMaximized();
#endif

    pScanWindow->startScan();
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the selected folder.
*/
void CMainWindow::onScanFolder()
{
    QString strFolder = QFileDialog::getExistingDirectory(this, tr("Choose a folder"));
    if (!strFolder.isEmpty())
    {
        CScanWindow* pScanWindow = new CScanWindow(this);
        pScanWindow->initialize(strFolder, false);

#ifndef Q_WS_MAEMO_5
        pScanWindow->resize(800, 480);
        pScanWindow->show();
#else
        pScanWindow->showMaximized();
#endif

        pScanWindow->startScan();
   }
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the given volume.
*/
void CMainWindow::onScanVolume(QString strPath)
{
    CScanWindow* pScanWindow = new CScanWindow(this);
    pScanWindow->initialize(strPath, false);

#ifndef Q_WS_MAEMO_5
    pScanWindow->resize(800, 480);
    pScanWindow->show();
#else
    pScanWindow->showMaximized();
#endif

    pScanWindow->startScan();
}

// EOF
