/*
** Copyright (c) 2010  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "mainwindow.h"

//-----------------------------------------------------------------------------
/**
** Constructor
**
** \param pParent The parent for the main window (usually NULL)
*/
CMainWindow::CMainWindow(QWidget* pParent)
    : QMainWindow(pParent)
{
    m_pTreeWidget = NULL;
    m_pDelegate = NULL;
    m_pModel = NULL;
    m_pSortModel = NULL;
    m_pUsageGrid = NULL;
    m_pScanThread = NULL;
    m_pCurrentItem = NULL;
    m_pSelectedItem = NULL;
    m_pPathLabel = NULL;
    m_pProgress = NULL;
    m_pCancelButton = NULL;
    m_bScanApps = false;
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CMainWindow::~CMainWindow()
{
    if (m_pScanThread)
    {
        m_pScanThread->cancel();
        m_pScanThread->wait(5000);      // Wait max 5 secs for the thread to quit
        delete m_pScanThread;
    }
}

//-----------------------------------------------------------------------------
/**
** Initializes the widgets.
*/
void CMainWindow::intialize()
{
    QWidget* pRoot = new QWidget(this);
    QSplitter* pSplitter = new QSplitter(pRoot);

    // The folder tree
    m_pTreeWidget = new QTreeView(pSplitter);
    m_pTreeWidget->setProperty("FingerScrollable", true);
    m_pTreeWidget->setHeaderHidden(true);
    m_pTreeWidget->setRootIsDecorated(false);
    m_pTreeWidget->setAllColumnsShowFocus(true);
    m_pTreeWidget->header()->setMovable(false);
    m_pDelegate = new CDelegate(this);
    m_pTreeWidget->setItemDelegate(m_pDelegate);
    m_pTreeWidget->setEditTriggers(QAbstractItemView::NoEditTriggers);
    pSplitter->addWidget(m_pTreeWidget);

    m_pModel = new QStandardItemModel(this);
    m_pSortModel = new QSortFilterProxyModel(this);
    m_pSortModel->setSourceModel(m_pModel);
    m_pSortModel->setDynamicSortFilter(true);
    m_pSortModel->setSortRole(SizeRole);
    m_pSortModel->sort(0, Qt::DescendingOrder);
    m_pTreeWidget->setModel(m_pSortModel);

    // The usage grid
    m_pUsageGrid = new CUsageGrid(pSplitter, m_pTreeWidget);
    pSplitter->addWidget(m_pUsageGrid);

    QVBoxLayout* pLayout = new QVBoxLayout();
    pLayout->addWidget(pSplitter, 1);

    QList<int> listSizes;
    listSizes << 100 << 100;
    pSplitter->setSizes(listSizes);

    QHBoxLayout* bBottomLayout = new QHBoxLayout();
    m_pPathLabel = new QLabel(pRoot);
    bBottomLayout->addWidget(m_pPathLabel, 1);
    m_pProgress = new QProgressBar(pRoot);
    bBottomLayout->addWidget(m_pProgress, 1);
    m_pCancelButton = new QPushButton("Cancel", pRoot);
    bBottomLayout->addWidget(m_pCancelButton, 0);
    pLayout->addLayout(bBottomLayout, 0);

    m_pProgress->hide();
    m_pCancelButton->hide();

    pRoot->setLayout(pLayout);
    setCentralWidget(pRoot);

    // Create menu items
    QAction* aboutAction = new QAction(tr("About"), this);
    QAction* scanRootAction = new QAction(tr("Scan Root"), this);
    QAction* scanHomeAction = new QAction(tr("Scan Home"), this);
    QAction* scanMemCardAction = new QAction(tr("Scan Mem Card"), this);
    QAction* scanFolderAction = new QAction(tr("Scan Folder..."), this);
    QAction* scanAppsAction = new QAction(tr("Scan Packages"), this);

    // Add item into menu
#if defined(Q_WS_HILDON)
    menuBar()->addAction(scanRootAction);
    menuBar()->addAction(scanHomeAction);
    QDir dir("/media/mmc1");
    if (dir.exists() && !dir.entryList(QDir::Files | QDir::Dirs | QDir::Hidden | QDir::NoSymLinks | QDir::NoDotAndDotDot).isEmpty())
    {
        menuBar()->addAction(scanMemCardAction);
    }
    menuBar()->addAction(scanFolderAction);
    menuBar()->addAction(scanAppsAction);
#else
    // else File menu
    QMenu* menu = new QMenu(tr("File"), this);
    menu->addAction(aboutAction);
    menu->addAction(scanRootAction);
    menu->addAction(scanHomeAction);
    menu->addAction(scanFolderAction);
    menu->addAction(scanAppsAction);

    QAction* exitAction = new QAction(tr("Exit"), this);
    menu->addAction(exitAction);
    connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));

    menuBar()->addMenu(menu);
#endif

    connect(aboutAction, SIGNAL(triggered()), this, SLOT(onAbout()));
    connect(scanRootAction, SIGNAL(triggered()), this, SLOT(onScanRoot()));
    connect(scanHomeAction, SIGNAL(triggered()), this, SLOT(onScanHome()));
    connect(scanAppsAction, SIGNAL(triggered()), this, SLOT(onScanApps()));
    connect(scanMemCardAction, SIGNAL(triggered()), this, SLOT(onScanMemCard()));
    connect(scanFolderAction, SIGNAL(triggered()), this, SLOT(onScanFolder()));

    bool bOk;
    bOk = connect(m_pTreeWidget->selectionModel(), SIGNAL(currentChanged(const QModelIndex&, const QModelIndex&)), this, SLOT(onItemSelectionChanged(const QModelIndex&, const QModelIndex&)));
    Q_ASSERT(bOk);
    bOk = connect(m_pTreeWidget, SIGNAL(doubleClicked(const QModelIndex&)), this, SLOT(onItemDoubleClicked(const QModelIndex&)));
    Q_ASSERT(bOk);

    onScanHome();
}

//-----------------------------------------------------------------------------
/**
** Starts the scan thread on the given folder
*/
void CMainWindow::startScan(QString strFolder)
{
    if (m_pScanThread)
    {
        m_pScanThread->cancel();
        m_pScanThread->wait(5000);      // Wait max 5 secs for the thread to quit
        delete m_pScanThread;
        m_pScanThread = NULL;
    }

    m_pModel->clear();
    m_pUsageGrid->clear();
    m_pCurrentItem = NULL;

    m_pScanThread = new CScanThread(this, strFolder);

    bool bOk = false;
    bOk = connect(m_pScanThread, SIGNAL(folderFound(QString)), this, SLOT(onFolderFound(QString)));
    Q_ASSERT(bOk);
    bOk = connect(m_pScanThread, SIGNAL(folderScanned()), this, SLOT(onFolderScanned()));
    Q_ASSERT(bOk);
    bOk = connect(m_pScanThread, SIGNAL(fileFound(QString, qint64, bool)), this, SLOT(onFileFound(QString, qint64, bool)));
    Q_ASSERT(bOk);
    bOk = connect(m_pScanThread, SIGNAL(progressMaximum(int)), this, SLOT(onProgressMaximum(int)));
    Q_ASSERT(bOk);
    bOk = connect(m_pScanThread, SIGNAL(progressValue(QString, int)), this, SLOT(onProgressValue(QString, int)));
    Q_ASSERT(bOk);

    bOk = connect(m_pCancelButton, SIGNAL(clicked()), m_pScanThread, SLOT(cancel()));
    Q_ASSERT(bOk);

    m_pScanThread->start();
}

///////////////////////////////////////////////////////////////////////////////
/// SLOTS
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens the about dialog.
*/
void CMainWindow::onAbout()
{
    QString strText = tr("Disk Usage by Kimmo Pekkola");
    QMessageBox::information(this, tr("Disk Usage"), strText);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the root.
*/
void CMainWindow::onScanRoot()
{
    startScan("/");
    m_bScanApps = false;
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the home folder.
*/
void CMainWindow::onScanHome()
{
    startScan(QDesktopServices::storageLocation(QDesktopServices::HomeLocation));
    m_bScanApps = false;
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the installed apps.
*/
void CMainWindow::onScanApps()
{
    startScan(QString());
    m_bScanApps = true;
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the memory card.
*/
void CMainWindow::onScanMemCard()
{
    startScan("/media/mmc1");
    m_bScanApps = false;
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Starts to scan the selected folder.
*/
void CMainWindow::onScanFolder()
{
    QString strFileName = QFileDialog::getExistingDirectory(this, tr("Choose a folder"));
    if (!strFileName.isEmpty())
    {
        startScan(strFileName);
        m_bScanApps = false;
   }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onFolderFound(QString strFolder)
{
    if (m_pModel)
    {
        QStandardItem* pItem = new QStandardItem();
        pItem->setData(TYPE_FOLDER, TypeRole);
        pItem->setData(strFolder, Qt::DisplayRole);

        if (m_pCurrentItem)
        {
            m_pCurrentItem->appendRow(pItem);
        }
        else
        {
            m_pModel->appendRow(pItem);
            m_pTreeWidget->expandToDepth(0);
        }
        m_pCurrentItem = pItem;
    }
}


//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onFolderScanned()
{
    if (m_pCurrentItem)
    {
        // Calculate the total for the folder
        qint64 total = 0;
        qint64 totalOpt = 0;
        for (int i = 0; i < m_pCurrentItem->rowCount(); i++)
        {
            total += m_pCurrentItem->child(i)->data(SizeRole).toLongLong();
            totalOpt += m_pCurrentItem->child(i)->data(OptSizeRole).toLongLong();
        }
        m_pCurrentItem->setData(total, SizeRole);
        m_pCurrentItem->setData(totalOpt, OptSizeRole);

        m_pCurrentItem = m_pCurrentItem->parent();
        if (m_pCurrentItem == NULL && m_pUsageGrid && m_pDelegate)
        {
            // This was the last item -> Create the usage grid
            m_pUsageGrid->initialize(m_pModel->item(0, 0), !m_bScanApps);
        }
    }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onFileFound(QString strFilename, qint64 size, bool bInOpt)
{
    if (m_pCurrentItem)
    {
        QStandardItem* pItem = new QStandardItem();
        pItem->setData(TYPE_FILE, TypeRole);
        pItem->setData(strFilename, Qt::DisplayRole);
        pItem->setData(size, SizeRole);
        if (bInOpt)
        {
            pItem->setData(size, OptSizeRole);
        }
        m_pCurrentItem->appendRow(pItem);
    }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onItemSelectionChanged(const QModelIndex& current, const QModelIndex& /*previous*/)
{
    if (m_pTreeWidget && m_pDelegate)
    {
        if (current.isValid())
        {
            // Make sure the current is either the selected index or a child of it
            QModelIndex index = current;
            while (index.isValid() && m_pDelegate->selectedIndex().isValid())
            {
                if (index == m_pDelegate->selectedIndex())
                {
                    break;
                }
                index = index.parent();
            }

            if (index.isValid())
            {
                m_pUsageGrid->setSelectionRect(current.data(AreaRole).toRect());
            }
            else
            {
                m_pUsageGrid->setSelectionRect(QRect());
            }

            QString strPath;
            QStandardItem* pItem = m_pModel->itemFromIndex(m_pSortModel->mapToSource(current));

            if (m_bScanApps)
            {
                strPath = pItem->data(Qt::DisplayRole).toString();
            }
            else
            {
                while (pItem)
                {
                    if (!strPath.isEmpty())
                    {
                        strPath.insert(0, "/");
                    }
                    strPath = pItem->data(Qt::DisplayRole).toString() + strPath;
                    pItem = pItem->parent();
                }
            }
            m_pPathLabel->setText(strPath);
        }
        else
        {
            m_pUsageGrid->setSelectionRect(QRect());
            m_pPathLabel->setText(QString());
        }
    }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onItemDoubleClicked(const QModelIndex& index)
{
    if (index.data(TypeRole).toInt() == TYPE_FOLDER)
    {
        if (!index.parent().isValid())
        {
            // Root item -> Keep it expanded
            m_pTreeWidget->collapse(index);     // Collapse so it gets re-expanded on the double click
        }

        m_pDelegate->setSelectedIndex(index);

        if (m_pUsageGrid)
        {
            QStandardItem* pItem = m_pModel->itemFromIndex(m_pSortModel->mapToSource(index));
            bool bRecursive = !m_bScanApps;
            if (m_bScanApps && pItem->parent())    // If an app is selected as shown in the graph we need to do the image recursively
            {
                bRecursive = true;
            }

            m_pUsageGrid->initialize(pItem, bRecursive);
        }
    }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onProgressMaximum(int maxValue)
{
    if (m_pProgress && m_pPathLabel && m_pCancelButton)
    {
        m_pProgress->show();
        m_pCancelButton->show();
        m_pPathLabel->hide();
        m_pProgress->setMaximum(maxValue);
    }
}

//-----------------------------------------------------------------------------
/**
**
*/
void CMainWindow::onProgressValue(QString strText, int value)
{
    if (m_pProgress && m_pPathLabel && m_pCancelButton)
    {
        if (value == -1)    // -1 means end of operation
        {
            m_pProgress->hide();
            m_pCancelButton->hide();
            m_pPathLabel->show();
        }
        else
        {
            m_pProgress->setValue(value);
            m_pProgress->setFormat(strText + " (%p%)");
        }
    }
}

// EOF
