//      stopish-stopwatch.c
//
//      Copyright 2010 Michael Cronenworth <mike@cchtml.com>
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <stdlib.h>

#include "stopish.h"

struct timerData {
    GtkWidget *vBox;
    GtkWidget *label;
    GtkWidget *labelHour;
    GtkWidget *labelMinute;
    GtkWidget *labelSecond;
};

static struct timerData timerdata;
static GtkWidget *timerHistoryLabel[4];
static GtkWidget *resetButton;
static GSList *historyList = NULL;
static int timerHandle = -1;


static gint timeout_cb( gpointer data );
static void start_cb( GtkButton* button, gpointer data );
static void reset_cb( GtkButton* button, gpointer data );


GtkWindow *stopish_stopwatch_new( void )
{
    GtkWidget *window, *button, *label;
    GtkWidget *vBoxMain, *vBox0, *hBox0;

    window = hildon_stackable_window_new(  );

    gtk_container_set_border_width( GTK_CONTAINER( window ), 20 );

    gtk_window_set_title( GTK_WINDOW( window ), "Stopish" );

    vBoxMain = gtk_vbox_new( FALSE, 10 );

    // separator
    label = gtk_label_new( NULL );
    gtk_container_add( GTK_CONTAINER( vBoxMain ), label );

    vBox0 = gtk_vbox_new( FALSE, 5 );

    // stop watch area
    timerdata.vBox = gtk_vbox_new( FALSE, 0 );
    gtk_container_add( GTK_CONTAINER( vBox0 ), timerdata.vBox );
    stopish_stopwatch_label_timer_landscape(  );

    // history area
    timerHistoryLabel[0] = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[0] ),
                          "<span size=\"large\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel[0], FALSE, FALSE, 0 );
    timerHistoryLabel[1] = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel[1], FALSE, FALSE, 0 );
    timerHistoryLabel[2] = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[2] ),
                          "<span size=\"small\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel[2], FALSE, FALSE, 0 );
    timerHistoryLabel[3] = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[3] ),
                          "<span size=\"x-small\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel[3], FALSE, FALSE, 0 );

    gtk_container_add( GTK_CONTAINER( vBoxMain ), vBox0 );

    // separator
    label = gtk_label_new( NULL );
    gtk_container_add( GTK_CONTAINER( vBoxMain ), label );

    // button area
    hBox0 = gtk_hbox_new( FALSE, 15 );
    gtk_widget_set_size_request( hBox0, -1, 80 );

    // start/pause stopwatch button
    button = hildon_button_new_with_text( HILDON_SIZE_HALFSCREEN_WIDTH,
                                          HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
                                          "Start", NULL );
    resetButton = hildon_button_new_with_text( HILDON_SIZE_HALFSCREEN_WIDTH,
                                               HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
                                               "Reset", NULL );
    g_signal_connect( G_OBJECT( button ), "clicked",
                      G_CALLBACK( start_cb ), resetButton );
    gtk_container_add( GTK_CONTAINER( hBox0 ), button );

    // reset button
    gtk_widget_set_sensitive( resetButton, FALSE );
    g_signal_connect( G_OBJECT( resetButton ), "clicked",
                      G_CALLBACK( reset_cb ), button );
    gtk_container_add( GTK_CONTAINER( hBox0 ), resetButton );

    gtk_box_pack_start( GTK_BOX( vBoxMain ), hBox0, FALSE, FALSE, 0 );

    gtk_container_add( GTK_CONTAINER( window ), vBoxMain );

    gtk_widget_show_all( window );

    return GTK_WINDOW( window );
}


void stopish_stopwatch_reset( void )
{
    if ( timerHandle == -1 &&
         stopish_get_mode(  ) != STOPISH_MODE_RESUME )
        return;
    gtk_button_clicked( GTK_BUTTON( resetButton ) );
}


void stopish_stopwatch_perf_timer_hour( GtkRadioButton* radio, GtkLabel *label )
{
    stopish_timer_set_precision( TIMER_PRECISION_HOUR );
    if ( stopish_get_orientation(  ) == STOPISH_LANDSCAPE )
        stopish_stopwatch_label_timer_landscape(  );
    else
        stopish_stopwatch_label_timer_portrait(  );
}


void stopish_stopwatch_perf_timer_minute( GtkRadioButton* radio, GtkLabel *label )
{
    stopish_timer_set_precision( TIMER_PRECISION_MINUTE );
    if ( stopish_get_orientation(  ) == STOPISH_LANDSCAPE )
        stopish_stopwatch_label_timer_landscape(  );
    else
        stopish_stopwatch_label_timer_portrait(  );
}


void stopish_stopwatch_label_timer_landscape( void )
{
    gtk_widget_set_size_request( timerdata.vBox, 800, -1 );

    gtk_widget_destroy( timerdata.label );
    timerdata.label = gtk_label_new( NULL );
    if ( stopish_timer_get_precision(  ) == TIMER_PRECISION_MINUTE )
        gtk_label_set_markup( GTK_LABEL( timerdata.label ),
                              "<span font_family=\"monospace\" "
                              "size=\"80000\" weight=\"ultrabold\">"
                              "00:00.0</span>" );
    else
        gtk_label_set_markup( GTK_LABEL( timerdata.label ),
                              "<span font_family=\"monospace\" "
                              "size=\"80000\" weight=\"ultrabold\">"
                              "00:00:00.0</span>" );
    gtk_misc_set_alignment( GTK_MISC( timerdata.label ), 0.5f, 0.5f );
    gtk_container_add( GTK_CONTAINER( timerdata.vBox ), timerdata.label );
    gtk_widget_show( timerdata.label );
}


void stopish_stopwatch_label_timer_portrait( void )
{
    GtkWidget *vBox, *hBox, *label;

    gtk_widget_set_size_request( timerdata.vBox, 480, -1 );

    gtk_widget_destroy( timerdata.label );
    vBox = gtk_vbox_new( FALSE, 10 );

    if ( stopish_timer_get_precision(  ) == TIMER_PRECISION_HOUR ) {
        hBox = gtk_hbox_new( FALSE, 10 );
        label = gtk_label_new( "Hours" );
        gtk_widget_set_size_request( label, 100, -1 );
        gtk_misc_set_alignment( GTK_MISC( label ), 1.0f, 0.5f );
        gtk_container_add( GTK_CONTAINER( hBox ), label );
        timerdata.labelHour = gtk_label_new( NULL );
        gtk_widget_set_size_request( timerdata.labelHour, 350, -1 );
        gtk_misc_set_alignment( GTK_MISC( timerdata.labelHour ), 0.0f, 0.5f );
        gtk_label_set_markup( GTK_LABEL( timerdata.labelHour ),
                              "<span font_family=\"monospace\" "
                              "size=\"90000\" weight=\"ultrabold\">"
                              "00</span>" );
        gtk_container_add( GTK_CONTAINER( hBox ), timerdata.labelHour );
        gtk_container_add( GTK_CONTAINER( vBox ), hBox );
    }

    hBox = gtk_hbox_new( FALSE, 10 );
    label = gtk_label_new( "Minutes" );
    gtk_widget_set_size_request( label, 100, -1 );
    gtk_misc_set_alignment( GTK_MISC( label ), 1.0f, 0.5f );
    gtk_container_add( GTK_CONTAINER( hBox ), label );
    timerdata.labelMinute = gtk_label_new( NULL );
    gtk_widget_set_size_request( timerdata.labelMinute, 350, -1 );
    gtk_misc_set_alignment( GTK_MISC( timerdata.labelMinute ), 0.0f, 0.5f );
    gtk_label_set_markup( GTK_LABEL( timerdata.labelMinute ),
                          "<span font_family=\"monospace\" "
                          "size=\"90000\" weight=\"ultrabold\">"
                          "00</span>" );
    gtk_container_add( GTK_CONTAINER( hBox ), timerdata.labelMinute );
    gtk_container_add( GTK_CONTAINER( vBox ), hBox );

    hBox = gtk_hbox_new( FALSE, 10 );
    label = gtk_label_new( "Seconds" );
    gtk_widget_set_size_request( label, 100, -1 );
    gtk_misc_set_alignment( GTK_MISC( label ), 1.0f, 0.5f );
    gtk_container_add( GTK_CONTAINER( hBox ), label );
    timerdata.labelSecond = gtk_label_new( NULL );
    gtk_widget_set_size_request( timerdata.labelSecond, 350, -1 );
    gtk_misc_set_alignment( GTK_MISC( timerdata.labelSecond ), 0.0f, 0.5f );
    gtk_label_set_markup( GTK_LABEL( timerdata.labelSecond ),
                          "<span font_family=\"monospace\" "
                          "size=\"90000\" weight=\"ultrabold\">"
                          "00.0</span>" );
    gtk_container_add( GTK_CONTAINER( hBox ), timerdata.labelSecond );
    gtk_container_add( GTK_CONTAINER( vBox ), hBox );

    timerdata.label = vBox;
    gtk_container_add( GTK_CONTAINER( timerdata.vBox ), vBox );
    gtk_widget_show_all( vBox );
}


//
// Timer callback
//
static gint timeout_cb( gpointer data )
{
    char formatBuffer[128], tempBuffer[8];
    char *tempString;

    // print to screen
    tempString = stopish_get_time_string(  );
    if ( stopish_get_orientation(  ) == STOPISH_LANDSCAPE ) {
        sprintf( formatBuffer, "<span font_family=\"monospace\" "
                               "size=\"80000\" weight=\"ultrabold\">"
                               "%s</span>", tempString );
        gtk_label_set_markup( GTK_LABEL( timerdata.label ), formatBuffer );
    }
    else {
        if ( stopish_timer_get_precision(  ) == TIMER_PRECISION_HOUR ) {
            sprintf( tempBuffer, "%.2s", tempString );
            sprintf( formatBuffer, "<span font_family=\"monospace\" "
                                   "size=\"90000\" weight=\"ultrabold\">"
                                   "%s</span>", tempBuffer );
            gtk_label_set_markup( GTK_LABEL( timerdata.labelHour ),
                                  formatBuffer );
        }
        if ( stopish_timer_get_precision(  ) == TIMER_PRECISION_HOUR )
            sprintf( tempBuffer, "%.2s", tempString + 3 );
        else
            sprintf( tempBuffer, "%.2s", tempString );
        sprintf( formatBuffer, "<span font_family=\"monospace\" "
                               "size=\"90000\" weight=\"ultrabold\">"
                               "%s</span>", tempBuffer );
        gtk_label_set_markup( GTK_LABEL( timerdata.labelMinute ),
                              formatBuffer );
        if ( stopish_timer_get_precision(  ) == TIMER_PRECISION_HOUR )
            sprintf( tempBuffer, "%.4s", tempString + 6 );
        else
            sprintf( tempBuffer, "%.4s", tempString + 3 );
        sprintf( formatBuffer, "<span font_family=\"monospace\" "
                               "size=\"90000\" weight=\"ultrabold\">"
                               "%s</span>", tempBuffer );
        gtk_label_set_markup( GTK_LABEL( timerdata.labelSecond ),
                              formatBuffer );
    }
    free( tempString );

    return TRUE;
}


static void start_cb( GtkButton* button, gpointer data )
{
    if ( stopish_get_mode(  ) == STOPISH_MODE_START ) {
        // set label text and add timer handle
        gtk_button_set_label( button, "Pause" );
        stopish_set_mode( STOPISH_MODE_PAUSE );
        stopish_set_time_start( stopish_current_time(  ) );
        timerHandle = g_timeout_add( 100, timeout_cb, NULL );
    }
    else if ( stopish_get_mode(  ) == STOPISH_MODE_RESUME ) {
        // resume timer
        gtk_button_set_label( button, "Pause" );
        stopish_set_mode( STOPISH_MODE_PAUSE );
        stopish_timer_resume(  );
        timerHandle = g_timeout_add( 100, timeout_cb, NULL );
    }
    else {
        // pause timer, remove timeout
        gtk_button_set_label( button, "Resume" );
        stopish_set_mode( STOPISH_MODE_RESUME );
        g_source_remove( timerHandle );
        stopish_timer_save(  );
        timerHandle = -1;
    }

    // allow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( data ), TRUE );
}


static void reset_cb( GtkButton* button, gpointer data )
{
    GSList *tempList;
    char *tempString;
    char formatString[128];

    if ( stopish_get_mode(  ) == STOPISH_MODE_RESUME )
        stopish_timer_resume(  );

    // set label text and remove timer handle
    gtk_button_set_label( GTK_BUTTON( data ), "Start" );
    stopish_set_mode( STOPISH_MODE_START );
    if ( stopish_get_orientation(  ) == STOPISH_LANDSCAPE )
        stopish_stopwatch_label_timer_landscape(  );
    else
        stopish_stopwatch_label_timer_portrait(  );
    g_source_remove( timerHandle );
    timerHandle = -1;

    // add current time to history
    historyList = g_slist_prepend( historyList,
                                   ( gpointer ) stopish_get_time_string(  ) );
    sprintf( formatString, "<span size=\"large\">%s</span>",
             ( char * ) historyList->data );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[0] ),
                          formatString );
    tempList = historyList;
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        gtk_label_set_text( GTK_LABEL( timerHistoryLabel[1] ),
                            ( char * ) tempList->data );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[2] ),
                              formatString );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"x-small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel[3] ),
                              formatString );
    }

    // remove the history time after the 4th
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        tempString = tempList->data;
        historyList = g_slist_remove( historyList, tempList->data );
        free( tempString );
    }

    // reset start time
    stopish_set_time_start( 0 );

    // disallow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( button ), FALSE );
}
