//      stopish.c
//
//      Copyright 2009 Michael Cronenworth <mike@cchtml.com>
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.

#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include <libosso.h>
#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#define MCE_SIGNAL_MATCH "type='signal'," \
        "interface='" MCE_SIGNAL_IF   "'," \
        "member='" MCE_DEVICE_ORIENTATION_SIG "'"

#include "stopish.h"

// Application data struct
typedef struct _AppData AppData;
struct _AppData {
    GtkWindow *main_window;
    osso_context_t *osso_context;
    DBusConnection *system_bus;
};

static AppData appdata;
static GtkWidget *timerLabel = NULL;
static GtkWidget *timerHistoryLabel1 = NULL;
static GtkWidget *timerHistoryLabel2 = NULL;
static GtkWidget *timerHistoryLabel3 = NULL;
static GtkWidget *timerHistoryLabel4 = NULL;
static GSList *historyList = NULL;
static int stopishMode = STOPISH_MODE_START;
static int timerHandle = -1;

//Prototypes
gint dbus_callback( const gchar *interface, const gchar *method,
	                GArray *arguments, gpointer data, osso_rpc_t *retval );
static GtkWindow *stopish_new( void );
static void start_cb( GtkButton* button, gpointer data );
static void reset_cb( GtkButton* button, gpointer data );
static void close_cb( GtkButton* button, gpointer data );
static gboolean focus_in_cb( GtkWidget *widget, GdkEventFocus *event,
                             gpointer data );
static gboolean focus_out_cb( GtkWidget *widget, GdkEventFocus *event,
                              gpointer data );
static void accelerometer_enable( void );
static void accelerometer_disable( void );
static DBusHandlerResult mce_filter_func( DBusConnection * connection,
                                          DBusMessage * message,
                                          void *data );


int main( int argc, char *argv[] )
{
    osso_return_t ret;
    HildonProgram *program;

    appdata.osso_context = osso_initialize( "com.nokia.stopish",
                                            PACKAGE_VERSION, TRUE, NULL );
    if ( appdata.osso_context == NULL ) {
        fprintf( stderr, "osso_initialize failed.\n" );
        exit( 1 );
    }

    // initialize Hildonized GTK libraries
    hildon_gtk_init( &argc, &argv );
    program = hildon_program_get_instance(  );

    // create main window
    appdata.main_window = stopish_new(  );
    hildon_program_add_window( program, HILDON_WINDOW( appdata.main_window ) );

    // Connect to session bus, add a match rule, install filter callback
    appdata.system_bus = osso_get_sys_dbus_connection( appdata.osso_context );
    if ( appdata.system_bus ) {
        dbus_bus_add_match( appdata.system_bus, MCE_SIGNAL_MATCH, NULL );
        dbus_connection_add_filter( appdata.system_bus,
                                    mce_filter_func,
                                    NULL, NULL );
    }
    else
        g_printerr( "ERROR: Cannot connect to system dbus.\n" );

    ret = osso_rpc_set_default_cb_f( appdata.osso_context,
                                     dbus_callback, appdata.main_window );
    if ( ret != OSSO_OK ) {
        fprintf( stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret );
        exit( 1 );
    }

    gtk_main(  );

    return 0;
}


gint dbus_callback( const gchar *interface, const gchar *method,
    	            GArray *arguments, gpointer data, osso_rpc_t *retval )
{
    //printf( "stopish dbus: %s, %s\n", interface, method );

    if ( !strcmp( method, "top_application" ) )
        gtk_window_present( GTK_WINDOW( data ) );

    retval->type = DBUS_TYPE_INVALID;

    return OSSO_OK;
}


int stopish_get_mode( void )
{
    return stopishMode;
}


static GtkWindow *stopish_new( void )
{
    GtkWidget *window, *button, *button0, *label;
    GtkWidget *vBoxMain, *vBox0, *hBox0;

    window = hildon_stackable_window_new(  );

    gtk_container_set_border_width( GTK_CONTAINER( window ), 20 );

    gtk_window_set_title( GTK_WINDOW( window ), "Stopish" );

    // attach signals to main window
    g_signal_connect( G_OBJECT( window ), "destroy",
                      G_CALLBACK( close_cb ), window );
    g_signal_connect( G_OBJECT( window ), "focus-in-event",
                      G_CALLBACK( focus_in_cb ), NULL );
    g_signal_connect( G_OBJECT( window ), "focus-out-event",
                      G_CALLBACK( focus_out_cb ), NULL );

    vBoxMain = gtk_vbox_new( FALSE, 10 );

    // separator
    label = gtk_label_new( NULL );
    gtk_container_add( GTK_CONTAINER( vBoxMain ), label );

    // stop watch area
    vBox0 = gtk_vbox_new( FALSE, 5 );

    // main timer
    timerLabel = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerLabel ),
                          "<span font_family=\"monospace\" "
                          "size=\"70000\" weight=\"ultrabold\">"
                          "00:00:00.0</span>" );
    gtk_container_add( GTK_CONTAINER( vBox0 ), timerLabel );

    // history area
    timerHistoryLabel1 = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel1 ),
                          "<span size=\"large\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel1, FALSE, FALSE, 0 );
    timerHistoryLabel2 = gtk_label_new( NULL );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel2, FALSE, FALSE, 0 );
    timerHistoryLabel3 = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel3 ),
                          "<span size=\"small\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel3, FALSE, FALSE, 0 );
    timerHistoryLabel4 = gtk_label_new( NULL );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel4 ),
                          "<span size=\"x-small\"> </span>" );
    gtk_box_pack_start( GTK_BOX( vBox0 ), timerHistoryLabel4, FALSE, FALSE, 0 );

    gtk_container_add( GTK_CONTAINER( vBoxMain ), vBox0 );

    // separator
    label = gtk_label_new( NULL );
    gtk_container_add( GTK_CONTAINER( vBoxMain ), label );

    // button area
    hBox0 = gtk_hbox_new( FALSE, 15 );
    gtk_widget_set_size_request( hBox0, -1, 80 );

    // start/pause stopwatch button
    button = hildon_button_new_with_text( HILDON_SIZE_HALFSCREEN_WIDTH,
                                          HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
                                          "Start", NULL );
    button0 = hildon_button_new_with_text( HILDON_SIZE_HALFSCREEN_WIDTH,
                                           HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
                                           "Reset", NULL );
    g_signal_connect( G_OBJECT( button ), "clicked",
                      G_CALLBACK( start_cb ), button0 );
    gtk_container_add( GTK_CONTAINER( hBox0 ), button );

    // reset button
    gtk_widget_set_sensitive( button0, FALSE );
    g_signal_connect( G_OBJECT( button0 ), "clicked",
                      G_CALLBACK( reset_cb ), button );
    gtk_container_add( GTK_CONTAINER( hBox0 ), button0 );

    gtk_box_pack_start( GTK_BOX( vBoxMain ), hBox0, FALSE, FALSE, 0 );

    gtk_container_add( GTK_CONTAINER( window ), vBoxMain );

    gtk_widget_show_all( window );

    return GTK_WINDOW( window );
}


static void start_cb( GtkButton* button, gpointer data )
{
    if ( stopishMode == STOPISH_MODE_START ) {
        // set label text and add timer handle
        gtk_button_set_label( button, "Pause" );
        stopishMode = STOPISH_MODE_PAUSE;
        stopish_set_time_start( stopish_current_time(  ) );
        timerHandle = g_timeout_add( 100, stopish_timeout_cb, timerLabel );
    }
    else if ( stopishMode == STOPISH_MODE_RESUME ) {
        // resume timer
        gtk_button_set_label( button, "Pause" );
        stopishMode = STOPISH_MODE_PAUSE;
        stopish_timer_resume(  );
        timerHandle = g_timeout_add( 100, stopish_timeout_cb, timerLabel );
    }
    else {
        // pause timer, remove timeout
        gtk_button_set_label( button, "Resume" );
        stopishMode = STOPISH_MODE_RESUME;
        g_source_remove( timerHandle );
        stopish_timer_save(  );
    }

    // allow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( data ), TRUE );
}


static void reset_cb( GtkButton* button, gpointer data )
{
    GSList *tempList;
    char *tempString;
    char formatString[128];

    if ( stopishMode == STOPISH_MODE_RESUME )
        stopish_timer_resume(  );

    // set label text and remove timer handle
    gtk_button_set_label( GTK_BUTTON( data ), "Start" );
    stopishMode = STOPISH_MODE_START;
    gtk_label_set_markup( GTK_LABEL( timerLabel ),
                          "<span font_family=\"monospace\" "
                          "size=\"70000\" weight=\"ultrabold\">"
                          "00:00:00.0</span>" );
    g_source_remove( timerHandle );

    // add current time to history
    historyList = g_slist_prepend( historyList,
                                   ( gpointer ) stopish_get_time_string(  ) );
    sprintf( formatString, "<span size=\"large\">%s</span>",
             ( char * ) historyList->data );
    gtk_label_set_markup( GTK_LABEL( timerHistoryLabel1 ),
                          formatString );
    tempList = historyList;
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        gtk_label_set_text( GTK_LABEL( timerHistoryLabel2 ),
                            ( char * ) tempList->data );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel3 ),
                              formatString );
    }
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        sprintf( formatString, "<span size=\"x-small\">%s</span>",
                 ( char * ) tempList->data );
        gtk_label_set_markup( GTK_LABEL( timerHistoryLabel4 ),
                              formatString );
    }

    // remove the history time after the 4th
    tempList = g_slist_next( tempList );
    if ( tempList ) {
        tempString = tempList->data;
        historyList = g_slist_remove( historyList, tempList->data );
        free( tempString );
    }

    // reset start time
    stopish_set_time_start( 0 );

    // disallow user to reset timer
    gtk_widget_set_sensitive( GTK_WIDGET( button ), FALSE );
}


static void close_cb( GtkButton* button, gpointer data )
{
    // disable accelerometer for battery savings
    accelerometer_disable(  );

    // destroy main window and exit gtk main loop
    gtk_widget_destroy( GTK_WIDGET( data ) );
    gtk_main_quit(  );
}


static gboolean focus_in_cb( GtkWidget *widget, GdkEventFocus *event,
                             gpointer data )
{
    // enable accelerometer hardware for portrait mode support
    accelerometer_enable(  );

    return FALSE;
}


static gboolean focus_out_cb( GtkWidget *widget, GdkEventFocus *event,
                              gpointer data )
{
    // disable accelerometer for battery savings
    accelerometer_disable(  );

    return FALSE;
}


static void accelerometer_enable( void )
{
    if ( osso_rpc_run_system( appdata.osso_context, MCE_SERVICE,
                              MCE_REQUEST_PATH, MCE_REQUEST_IF,
                              "req_accelerometer_enable", NULL,
                              DBUS_TYPE_INVALID ) != OSSO_OK ) {
        g_printerr("WARN: Cannot enable accelerometers\n");
    }
}


static void accelerometer_disable( void )
{
    if ( osso_rpc_run_system( appdata.osso_context, MCE_SERVICE,
                              MCE_REQUEST_PATH, MCE_REQUEST_IF,
                              "req_accelerometer_disable", NULL,
                              DBUS_TYPE_INVALID ) != OSSO_OK ) {
        g_printerr("WARN: Cannot disable accelerometers\n");
    }
}


static DBusHandlerResult mce_filter_func( DBusConnection * connection,
                                          DBusMessage * message,
                                          void *data )
{
    DBusMessageIter iter;
    char *rotation = NULL;

    if ( dbus_message_is_signal( message, MCE_SIGNAL_IF,
                                 MCE_DEVICE_ORIENTATION_SIG ) ) {
        // here if we received an orientation dbus signal
        if ( dbus_message_iter_init( message, &iter ) ) {
            dbus_message_iter_get_basic( &iter, &rotation );

            // Rotate main window
            if ( !strcmp( rotation, MCE_ORIENTATION_PORTRAIT ) )
                hildon_gtk_window_set_portrait_flags( GTK_WINDOW( appdata.main_window ),
                                                      HILDON_PORTRAIT_MODE_REQUEST );
            else
                hildon_gtk_window_set_portrait_flags( GTK_WINDOW( appdata.main_window ),
                                                      ~HILDON_PORTRAIT_MODE_REQUEST );
        }
        else
            g_printerr( "ERROR: dbus_message_iter_init() failed.\n" );
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
