import QtQuick 1.0
import QtWebKit 1.0

Item {
    id: container
    width: 640
    height: 480

    property bool txtSmooth: true
    property color statColor: "#595454"
    property int statTopMargin: 1
    property int listFontSize : 48
    property int fontSize1: listFontSize - 18
    property int fontSize2: listFontSize - 28
    property int fontSize3: listFontSize - 20

    property string myDescription: ""
    property string myNewsLink: ""

    property bool webViewEnb: true

    signal close

    Loader {
        id: webviewLoader
        anchors.fill: parent
        onLoaded: { webviewLoader.item.url = myNewsLink; }
    }

    Component{
        id: newsWebviewComp
        Flickable {
            id: newsWebview
            anchors.fill: parent

            contentWidth: webView.width
            contentHeight: webView.height

            opacity: 0

            property string url: myNewsLink
            property alias progress: webView.progress

            WebView {
                id: webView

                preferredWidth: mainWindow.width
                contentsScale: 1

                settings.minimumFontSize: fontSize3 - 2
                settings.privateBrowsingEnabled: true
                settings.zoomTextOnly: true
                settings.javascriptEnabled: false
                url: newsWebview.url

                onAlert: console.log(message)

                onLoadFinished: {
                    console.log("Load webview done.");
                    newsErrorDialog.state = "";
                    if (container.state=="webview") { newsWebview.opacity = 1; }
                }
                onLoadFailed: {
                    console.log("Load webview failed");
                    newsErrorDialog.state = "showWebviewError";
                    newsErrorDialog.message = "Fail to load news page";
                }

                // This is un-documented
                onUrlChanged: {
                    // got to topleft
                    console.log("url="+webView.url);
                    //newsWebview.contentX = 0;
                    //newsWebview.contentY = 0;
                }

                /*
            onDoubleClick: {
                if (!heuristicZoom(clickX,clickY,2.5)) {
                    var zf = newsWebview.width / contentsSize.width
                    if (zf >= contentsScale)
                        zf = 2.0*contentsScale // zoom in (else zooming out)
                    doZoom(zf,clickX*zf,clickY*zf)
                }
            }
            function doZoom(zoom,centerX,centerY)
            {
                if (centerX) {
                    var sc = zoom*contentsScale;

                    newsWebview.contentX = Math.max(0,Math.min(centerX-newsWebview.width/2,webView.width*sc-newsWebview.width));
                    newsWebview.contentY = Math.max(0,Math.min(centerY-newsWebview.height/2,webView.height*sc-newsWebview.height));
                }
            }*/
            }
            PropertyAnimation { target: newsWebview; properties: "opacity"; duration: 130}
        }
    }

    ProgressBar3 {
            id: newsModelProgress
            anchors.centerIn: parent

            duration: 200
            width: 320
            height: 34

            // rssModel.progress is un-reliable
            progress: (container.state=="webview") ? webviewLoader.item.progress : rssModel.progress
    }

    BusyDialog {
        id: newsErrorDialog
        anchors.centerIn: parent
        message: "RSS feed unavailable"
        duration: 300
        opacity: 0

        states: [
            State {
                name: "showFeedError"
                when: (rssModel.status==XmlListModel.Error)
                PropertyChanges { target: newsErrorDialog; opacity: 1 }
                PropertyChanges { target: newsErrorDialog; message: "RSS feed unavailable" }
            },
            State {
                name: "showWebviewError"
                PropertyChanges { target: newsErrorDialog; opacity: 1 }
                PropertyChanges { target: newsErrorDialog; message: "News unavailable" }
            }
        ]
    }

    // newsWebview area to display detail info
    Rectangle {
        id: descRect
        anchors.fill: parent
        color: "ivory"
        opacity: 0

        Flickable {
            id: descFlick
            anchors.fill: parent
            anchors { topMargin: 10; bottomMargin: 10; leftMargin: 10; rightMargin: 10 }

            contentHeight: descText.height + 80
            clip: true

            Text {
                id: descText;
                wrapMode: Text.WordWrap;
                anchors { left: parent.left; right: parent.right; }
                font.pixelSize: fontSize3
                text: myDescription
            }

            SimpleButton{
                id: readMoreText
                width: parent.width
                anchors { left: descText.left; right: parent.right; top: descText.bottom; topMargin: 20 }
                text: "Read More"
                textSize: fontSize3

                onClicked: {
                    if (webViewEnb)
                        container.state = "webview";
                    else
                        Qt.openUrlExternally(myNewsLink);
                }
            }

            MouseArea {
                id: detailMouseArea
                anchors.fill: descText
                onClicked: container.state='';
            }
        }
    }

    ListView {
        id: newsView
        anchors.fill: parent
        model: rssModel
        delegate: newsDelegate
        snapMode: ListView.SnapToItem
        section.property: "date"
        section.criteria: ViewSection.FullString
        section.delegate: sectionComp
        visible: rssModel.status==XmlListModel.Ready
     }

    Component {
        id: sectionComp

        Rectangle {
            width: container.width
            height: childrenRect.height
            color: "#E4F5FF"

            Text {
                id: sectionText

                smooth: txtSmooth
                color: statColor
                font.pixelSize: fontSize2
                horizontalAlignment: Text.AlignRight
                text: "   " + section
            }
        }
    }

    Component {
        id: newsDelegate

        Item {
            id: newsDelegateItem
            width: newsView.width
            height: titleBox.height + 45 //80

            property real detailOpacity: 0
            property real titleOpacity: 1

            Rectangle {
                id: background
                x: 2; y: 2; width: parent.width - x*2; height: parent.height - y*2
                color: "ivory"
                border.color: "#3577B9"
                radius: 5
            }

            Text {
                id: titleBox
                opacity: titleOpacity
                anchors { top: background.top; left: background.left; right: background.right; leftMargin: 10; rightMargin: 10; topMargin: 5 }

                smooth: txtSmooth
                font.pixelSize: fontSize1
                wrapMode: Text.WordWrap;
                text: title
            }

            Image {
                id: upDownArrow
                anchors { top: titleBox.bottom; right: background.right; rightMargin: 10; }

                width: 30
                height: 30
                source: "gfx/rnd_br_down.png"
                smooth: txtSmooth
            }

            MouseArea {
                id: newsMouseArea
                anchors.fill: background
                onClicked: {
                    // TODO: Context menu to open link or stats view
                    newsDelegateItem.ListView.view.currentIndex = index
                    myDescription = newsView.model.get(newsView.currentIndex).desc
                    myNewsLink = newsView.model.get(newsView.currentIndex).link

                    if (container.state == "details") { container.state=''; }
                    else                              { container.state='details'; }
                }
            }
         }
    }

    states: [
        State {
            name: "details"

            // Pre-load webview
            PropertyChanges { target: webviewLoader; sourceComponent: newsWebviewComp }

            PropertyChanges { target: descRect; opacity: 1  }
            PropertyChanges { target: newsView; opacity: 0  }

            StateChangeScript { name: "detailScript";  script: console.log("State.newsView.details"); }
        },
        State {
            name: "webview"

            PropertyChanges { target: webviewLoader; sourceComponent: newsWebviewComp }
            PropertyChanges { target: newsView; opacity: 0  }
            StateChangeScript { name: "webviewScript";  script: console.log("State.newsView.webview"); }
        }
    ]

    transitions: [
        Transition {
            ParallelAnimation {
                PropertyAnimation { target: descRect; properties: "opacity"; from:0; duration: 130}
                PropertyAnimation { target: newsView; properties: "opacity"; from:0; duration: 130}
            }
        }
    ]

    BackButton {
        id: backButton
        anchors { bottom: container.bottom; left: container.left; leftMargin: 20; bottomMargin: 20; }

        width: 65
        height: 65
        onClicked: {
            if (container.state == "webview") {
                //newsWebview.contentX = 0;
                //newsWebview.contentY = 0;
                container.state = "details";
            }
            else if (container.state == "details") {
                // myNewsLink is set from default to details so clear here.
                myNewsLink = "";
                container.state='';
            }
            else {
                if (rssModel.status==XmlListModel.Error) {
                    rssModel.source = "";
                    rssModel.reload();
                }

                container.close();
            }
        }
    }
}


