
#include "qemaemo5dynamichomescreenwidget.h"
#include "qemaemo5dynamicwidgethelper.h"
#include "qesettingsmanager.h"

#include <QtCore>
#include <QtGui>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

bool _atomsInitialized = false;

static Atom winTypeAtom;
static Atom homeAppletAtom;
static Atom appletIDAtom;
static Atom utf8Atom;
static Atom appletSettingAtom;
static Atom appletShowSettingAtom;
static Atom onCurrentHomescreenAtom;

static unsigned long get_card_prop(Display *dpy, Window w, Atom atom)
{
    Atom type;
    int format, rc;
    unsigned long items;
    unsigned long left;
    unsigned char *value; // will point to 'unsigned long', actually

    rc = XGetWindowProperty(dpy, w, atom, 0, 1, False, XA_CARDINAL, &type, &format, &items, &left, &value);

    if (type == None || rc != Success)
        return 0;
    else
        return *value;
}

QeMaemo5DynamicHomescreenWidget::QeMaemo5DynamicHomescreenWidget(bool SettingsAvailable, QWidget *parent)
    : QWidget(parent),
      _isPositionLoaded(false), _isPressed(false), _settingsDialogAvailable(SettingsAvailable), _shouldSavePositionInMoveEvent(true),
      _homescreenNumber(QeMaemo5DynamicWidgetHelper::globalInstance()->currentHomeScreen())
{
    setAttribute(Qt::WA_TranslucentBackground);
    _appletId = WIDGET_SETTINGS_BASE_APPLETID + QUuid::createUuid().toString();
    _shouldRegister = true;

    if (!_atomsInitialized)
    {
        winTypeAtom = XInternAtom(QX11Info::display(), "_NET_WM_WINDOW_TYPE", false);
        homeAppletAtom = XInternAtom(QX11Info::display(), "_HILDON_WM_WINDOW_TYPE_HOME_APPLET", false);
        appletIDAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ID", false);
        utf8Atom = XInternAtom(QX11Info::display(), "UTF8_STRING", false);
        appletSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SETTINGS", false);
        appletShowSettingAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_SHOW_SETTINGS", false);
        onCurrentHomescreenAtom = XInternAtom(QX11Info::display(), "_HILDON_APPLET_ON_CURRENT_DESKTOP", false);
        _atomsInitialized = true;
        //qDebug() << "initialized atoms";
    }

    // Set correct window type
    XChangeProperty(QX11Info::display(), winId(), winTypeAtom, XA_ATOM, 32, PropModeReplace, (unsigned char *) &homeAppletAtom, 1);

    // Fill AppletID
    QByteArray id(appletId().remove(' ').toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8, PropModeReplace, (unsigned char *)id.constData(), id.length());

    if (_settingsDialogAvailable)
    {
        // Add setting button. This button is shown when hildon-desktop is in edit mode.
        int settings = 0;
        XChangeProperty(QX11Info::display(), winId(), appletSettingAtom, XA_CARDINAL, 32, PropModeReplace, (unsigned char*)&settings, 1);
    }

    // Hacks for the helper class
    _hackSequenceForSavingFromCallback = 0;
    _hackSequenceCount = 0;
}

QeMaemo5DynamicHomescreenWidget::~QeMaemo5DynamicHomescreenWidget()
{
    //qDebug() << "widget destructor called for:" << _appletId;
}

void QeMaemo5DynamicHomescreenWidget::showSettingsDialog()
{
    QMessageBox::information(NULL, "No settings!", "Sorry, this widget has no settings!");
}

bool QeMaemo5DynamicHomescreenWidget::settingsDialogAvailable()
{
    return _settingsDialogAvailable;
}

QString QeMaemo5DynamicHomescreenWidget::appletId()
{
    return _appletId;
}

bool QeMaemo5DynamicHomescreenWidget::x11Event(XEvent *event)
{
    //qDebug() << XGetAtomName(QX11Info::display(), event->xclient.message_type);
    if (event->xclient.message_type == appletShowSettingAtom)
    {
        showSettingsDialog();
        return true;
    }
    else if (event->xclient.message_type == onCurrentHomescreenAtom)
    {
        // Setting whether we are on the current desktop or not
        _isVisibleOnCurrentHomescreen = get_card_prop(QX11Info::display(), winId(), onCurrentHomescreenAtom) == 1;

        // Hack needed because of Maemo.org bug 10725: https://bugs.maemo.org/show_bug.cgi?id=10725
        // This stuff will make sure that the gconf callback will also save the widget position in the required corner case.
        // Setting the value of the hacks - if these values go above 2, the helper should also save the position of this widget
        if (_isVisibleOnCurrentHomescreen && _hackSequenceForSavingFromCallback > 0)
            _hackSequenceForSavingFromCallback++;
        else
            _hackSequenceForSavingFromCallback = 0;

        if (_hackSequenceForSavingFromCallback == 2)
            _hackSequenceCount++;

        // Determining whether we should save the position from here
        if (_isVisibleOnCurrentHomescreen && _homescreenNumber != QeMaemo5DynamicWidgetHelper::globalInstance()->currentHomeScreen())
            savePosition();

        //qDebug() << "on current desktop atom changed for:" << _appletId << "value:" << _isVisibleOnCurrentHomescreen << "hack:" << _hackSequenceForSavingFromCallback;
        return true;
    }
    return false;
}

void QeMaemo5DynamicHomescreenWidget::showEvent(QShowEvent *event)
{
    QWidget::showEvent(event);
    _homescreenNumber = QeMaemo5DynamicWidgetHelper::globalInstance()->currentHomeScreen();

    if (_shouldRegister && !QeMaemo5DynamicWidgetHelper::globalInstance()->isWidgetRegistered(this))
    {
        QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(this);
    }
    //qDebug() << "widget shown on homescreen" << _homescreenNumber << "id=" << _appletId;
}

void QeMaemo5DynamicHomescreenWidget::closeEvent(QCloseEvent *event)
{
    if (QeMaemo5DynamicWidgetHelper::globalInstance()->isWidgetRegistered(this))
        QeMaemo5DynamicWidgetHelper::globalInstance()->unregisterWidget(this);

    QWidget::closeEvent(event);
    QeSettingsManager::removeSettingsStartingWith(_appletId);

    deleteLater();
}

void QeMaemo5DynamicHomescreenWidget::paintEvent(QPaintEvent *event)
{
    //qDebug() << "paint event, called for:" << _appletId;

    QPainter p(this);
    QColor bgColor(0, 0, 0, 128);
    if (_isPressed)
        bgColor.setRgb(101,150,254,128);
    p.setBrush(bgColor);
    p.setPen(Qt::NoPen);
    p.drawRoundedRect(rect(), 10, 10);

    QWidget::paintEvent(event);
}

void QeMaemo5DynamicHomescreenWidget::mousePressEvent(QMouseEvent *event)
{
    _isPressed = true;
    update();

    QWidget::mousePressEvent(event);
}

void QeMaemo5DynamicHomescreenWidget::mouseReleaseEvent(QMouseEvent *event)
{
    _isPressed = false;
    update();

    QWidget::mouseReleaseEvent(event);
}

void QeMaemo5DynamicHomescreenWidget::moveEvent(QMoveEvent *event)
{
    //qDebug() << "Move event for: " << _appletId;

    if (!_isPositionLoaded && isVisible())
    {
        loadPosition();
    }
    else if (_isPositionLoaded && _shouldSavePositionInMoveEvent)
    {
        QWidget::moveEvent(event);
        savePosition();
    }
    _hackSequenceForSavingFromCallback = 1;
}

bool QeMaemo5DynamicHomescreenWidget::isVisibleOnCurrentHomescreen()
{
    return _isVisibleOnCurrentHomescreen;
}

int QeMaemo5DynamicHomescreenWidget::homescreenNumber()
{
    return _homescreenNumber;
}

void QeMaemo5DynamicHomescreenWidget::saveSetting(const QString& key, const QVariant& data)
{
    QeSettingsManager::storeSetting(appletId() + key, data);
}

QVariant QeMaemo5DynamicHomescreenWidget::loadSetting(const QString& key, const QVariant &defaultValue)
{
    return QeSettingsManager::retrieveSetting(appletId() + key, defaultValue);
}

bool QeMaemo5DynamicHomescreenWidget::restoreAppletId(const QString &restoredAppletId)
{
    _appletId = restoredAppletId;
    _homescreenNumber = loadSetting(WIDGET_SETTINGS_HOMESCREEN_N).toInt();

    if (_homescreenNumber == 0)
        _homescreenNumber = 1;

    // Fill the AppletID atom
    QByteArray id(appletId().remove(' ').toUtf8());
    XChangeProperty(QX11Info::display(), winId(), appletIDAtom, utf8Atom, 8, PropModeReplace, (unsigned char *)id.constData(), id.length());

    qDebug() << "restoring widget state for:" << _appletId << "homescreen:" << _homescreenNumber;
    return restoreWidgetState();
}

bool QeMaemo5DynamicHomescreenWidget::restoreWidgetState()
{
    return true;
}

void QeMaemo5DynamicHomescreenWidget::savePosition()
{
    if (_isVisibleOnCurrentHomescreen)
        _homescreenNumber = QeMaemo5DynamicWidgetHelper::globalInstance()->currentHomeScreen();

    saveSetting(WIDGET_SETTINGS_POSITION_X, pos().x());
    saveSetting(WIDGET_SETTINGS_POSITION_Y, pos().y());
    saveSetting(WIDGET_SETTINGS_HOMESCREEN_N, _homescreenNumber);
    qDebug() << "saved position for:" << _appletId << "x=" << pos().x() << "y=" << pos().y() << "homescreen=" << _homescreenNumber;
}

bool QeMaemo5DynamicHomescreenWidget::isPositionLoaded()
{
    return _isPositionLoaded;
}

void QeMaemo5DynamicHomescreenWidget::loadPosition()
{
    _isPositionLoaded = true;
    int x = loadSetting(WIDGET_SETTINGS_POSITION_X, 0).toInt();
    int y = loadSetting(WIDGET_SETTINGS_POSITION_Y, 0).toInt();

    if (x == 0 && y == 0) // If the widget is new, we save its position for the first time
    {
        savePosition();
    }
    else
    {
        qDebug() << "loaded position for:" << _appletId << "x=" << x << "y=" << y;
        _shouldSavePositionInMoveEvent = false;
        move(x, y);
        _shouldSavePositionInMoveEvent = true;
    }
}

bool QeMaemo5DynamicHomescreenWidget::shouldRegister()
{
    return _shouldRegister;
}

void QeMaemo5DynamicHomescreenWidget::setShouldRegister(bool value)
{
    _shouldRegister = value;

    if (_shouldRegister && !QeMaemo5DynamicWidgetHelper::globalInstance()->isWidgetRegistered(this))
    {
        QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(this);
    }
    else if (QeMaemo5DynamicWidgetHelper::globalInstance()->isWidgetRegistered(this))
    {
        QeMaemo5DynamicWidgetHelper::globalInstance()->unregisterWidget(this);
    }
}
