/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "SearchDialog.hpp"

#include "StelApp.hpp"
#include "StelCore.hpp"
#include "StelObjectMgr.hpp"
#include "StelModuleMgr.hpp"
#include "StelNavigator.hpp"
#include "StelMovementMgr.hpp"

//#include "SimbadSearcher.hpp"

#include <QKeyEvent>

SearchDialog::SearchDialog(QWidget* parent) : QDialog(parent)//, simbadReply(NULL)
{
	setupUi(this);

//	simbadSearcher = new SimbadSearcher(this);
	objectMgr = GETSTELMODULE(StelObjectMgr);
	Q_ASSERT(objectMgr);

	greekLetters["alpha"] = QString(QChar(0x03B1));
	greekLetters["beta"] = QString(QChar(0x03B2));
	greekLetters["gamma"] = QString(QChar(0x03B3));
	greekLetters["delta"] = QString(QChar(0x03B4));
	greekLetters["epsilon"] = QString(QChar(0x03B5));
	greekLetters["zeta"] = QString(QChar(0x03B6));
	greekLetters["eta"] = QString(QChar(0x03B7));
	greekLetters["theta"] = QString(QChar(0x03B8));
	greekLetters["iota"] = QString(QChar(0x03B9));
	greekLetters["kappa"] = QString(QChar(0x03BA));
	greekLetters["lambda"] = QString(QChar(0x03BB));
	greekLetters["mu"] = QString(QChar(0x03BC));
	greekLetters["nu"] = QString(QChar(0x03BD));
	greekLetters["xi"] = QString(QChar(0x03BE));
	greekLetters["omicron"] = QString(QChar(0x03BF));
	greekLetters["pi"] = QString(QChar(0x03C0));
	greekLetters["rho"] = QString(QChar(0x03C1));
	greekLetters["sigma"] = QString(QChar(0x03C3)); // second lower-case sigma shouldn't affect anything
	greekLetters["tau"] = QString(QChar(0x03C4));
	greekLetters["upsilon"] = QString(QChar(0x03C5));
	greekLetters["phi"] = QString(QChar(0x03C6));
	greekLetters["chi"] = QString(QChar(0x03C7));
	greekLetters["psi"] = QString(QChar(0x03C8));
	greekLetters["omega"] = QString(QChar(0x03C9));

	connect(lineEditSearchSkyObject, SIGNAL(textChanged(const QString&)), this, SLOT(onTextChanged(const QString&)));
	connect(pushButtonGotoSearchSkyObject, SIGNAL(clicked()), this, SLOT(gotoObject()));
	onTextChanged(lineEditSearchSkyObject->text());
	connect(lineEditSearchSkyObject, SIGNAL(returnPressed()), this, SLOT(gotoObject()));
	installEventFilter(this);
}

SearchDialog::~SearchDialog()
{
//	if (simbadReply)
//	{
//		simbadReply->deleteLater();
//		simbadReply = NULL;
//	}
}


//// Called when the current simbad query status changes
//void SearchDialog::onSimbadStatusChanged()
//{
//	Q_ASSERT(simbadReply);
//	if (simbadReply->getCurrentStatus()==SimbadLookupReply::SimbadLookupErrorOccured)
//	{
//		simbadStatusLabel->setText(QString("Simbad Lookup Error: ")+simbadReply->getErrorString());
//		if (completionLabel->isEmpty())
//			pushButtonGotoSearchSkyObject->setEnabled(false);
//	}
//	else
//	{
//		simbadStatusLabel->setText(QString("Simbad Lookup: ")+simbadReply->getCurrentStatusString());
//		// Query not over, don't disable button
//		pushButtonGotoSearchSkyObject->setEnabled(true);
//	}

//	if (simbadReply->getCurrentStatus()==SimbadLookupReply::SimbadLookupFinished)
//	{
//		simbadResults = simbadReply->getResults();
//		completionLabel->appendValues(simbadResults.keys());
//		// Update push button enabled state
//		pushButtonGotoSearchSkyObject->setEnabled(!completionLabel->isEmpty());
//	}

//	if (simbadReply->getCurrentStatus()!=SimbadLookupReply::SimbadLookupQuerying)
//	{
//		disconnect(simbadReply, SIGNAL(statusChanged()), this, SLOT(onSimbadStatusChanged()));
//		delete simbadReply;
//		simbadReply=NULL;

//		// Update push button enabled state
//		pushButtonGotoSearchSkyObject->setEnabled(!completionLabel->isEmpty());
//	}
//}


void SearchDialog::onTextChanged(const QString& text)
{
//	if (simbadReply)
//	{
//		disconnect(simbadReply, SIGNAL(statusChanged()), this, SLOT(onSimbadStatusChanged()));
//		delete simbadReply;
//		simbadReply=NULL;
//	}
//	simbadResults.clear();

	QString trimmedText = text.trimmed().toLower();
	if (trimmedText.isEmpty())
	{
		completionLabel->clearValues();
		completionLabel->selectFirst();
		//simbadStatusLabel->setText("");
		pushButtonGotoSearchSkyObject->setEnabled(false);
	}
	else
	{
//		simbadReply = simbadSearcher->lookup(trimmedText, 3);
//		onSimbadStatusChanged();
//		connect(simbadReply, SIGNAL(statusChanged()), this, SLOT(onSimbadStatusChanged()));

		QString greekText = substituteGreek(trimmedText);
		QStringList matches;
		if(greekText != trimmedText)
		{
			matches = objectMgr->listMatchingObjectsI18n(trimmedText, 3);
			matches += objectMgr->listMatchingObjectsI18n(greekText, (5 - matches.size()));
		}
		else
			matches = objectMgr->listMatchingObjectsI18n(trimmedText, 5);

		completionLabel->setValues(matches);
		completionLabel->selectFirst();

		// Update push button enabled state
		pushButtonGotoSearchSkyObject->setEnabled(true);
	}
}

void SearchDialog::gotoObject()
{
	QString name = completionLabel->getSelected();

	if (name.isEmpty())
		return;

	StelMovementMgr* mvmgr = GETSTELMODULE(StelMovementMgr);
//	if (simbadResults.contains(name))
//	{
//		close();
//		Vec3d pos = simbadResults[name];
//		objectMgr->unSelect();
//		pos = StelApp::getInstance().getCore()->getNavigator()->j2000ToEquinoxEqu(pos);
//		mvmgr->moveToJ2000(pos, mvmgr->getAutoMoveDuration());
//		lineEditSearchSkyObject->clear();
//		completionLabel->clearValues();
//	}
//	else
	if (objectMgr->findAndSelectI18n(name))
	{
		const QList<StelObjectP> newSelected = objectMgr->getSelectedObject();
		if (!newSelected.empty())
		{
			close();
			lineEditSearchSkyObject->clear();
			completionLabel->clearValues();
			// Can't point to home planet
			if (newSelected[0]->getEnglishName()!=StelApp::getInstance().getCore()->getNavigator()->getCurrentLocation().name)
			{
				mvmgr->moveToJ2000(newSelected[0]->getEquinoxEquatorialPos(StelApp::getInstance().getCore()->getNavigator()),mvmgr->getAutoMoveDuration());
				mvmgr->setFlagTracking(true);
			}
			else
			{
				GETSTELMODULE(StelObjectMgr)->unSelect();
			}
		}
	}
	//simbadResults.clear();
}

bool SearchDialog::eventFilter(QObject*, QEvent* event)
{
	if (event->type() == QEvent::KeyRelease)
	{
		QKeyEvent* keyEvent = static_cast<QKeyEvent*>(event);

		if (keyEvent->key() == Qt::Key_Tab || keyEvent->key() == Qt::Key_Down)
		{
			completionLabel->selectNext();
			event->accept();
			return true;
		}
		if (keyEvent->key() == Qt::Key_Up)
		{
			completionLabel->selectPrevious();
			event->accept();
			return true;
		}
	}

	return false;
}

QString SearchDialog::substituteGreek(const QString& keyString)
{
	if (!keyString.contains(' '))
		return getGreekLetterByName(keyString);
	else
	{
		QStringList nameComponents = keyString.split(" ", QString::SkipEmptyParts);
		if(!nameComponents.empty())
			nameComponents[0] = getGreekLetterByName(nameComponents[0]);
		return nameComponents.join(" ");
	}
}

QString SearchDialog::getGreekLetterByName(const QString& potentialGreekLetterName)
{
	if(greekLetters.contains(potentialGreekLetterName))
		return greekLetters[potentialGreekLetterName.toLower()];

	// There can be indices (e.g. "α1 Cen" instead of "α Cen A"), so strip
	// any trailing digit.
	int lastCharacterIndex = potentialGreekLetterName.length()-1;
	if(potentialGreekLetterName.at(lastCharacterIndex).isDigit())
	{
		QChar digit = potentialGreekLetterName.at(lastCharacterIndex);
		QString name = potentialGreekLetterName.left(lastCharacterIndex);
		if(greekLetters.contains(name))
			return greekLetters[name.toLower()] + digit;
	}

	return potentialGreekLetterName;
}
