/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "LocationDialog.hpp"

#include "StelApp.hpp"
#include "StelCore.hpp"
#include "StelNavigator.hpp"
#include "StelLocation.hpp"
#include "StelLocationMgr.hpp"
#include "StelTranslator.hpp"
#include "StelLocaleMgr.hpp"

#include "MobileGui.hpp"
#include "ListPicker.hpp"
#include "MobileLocationMgr.hpp"

#include <QStandardItemModel>
#include <QSortFilterProxyModel>
#include <QLayout>
#include <QSpinBox>
#include <QLabel>
#include <QStringListModel>
#include <QListView>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5>
#endif

class ManualSetting : public QDialog
{
public:
	ManualSetting(double longitude, double latitude, QWidget* parent=NULL);
	double getLongitude() const;
	double getLatitude() const;
protected:
	void setWidgets(double longitude, double latitude);
private:
	QSpinBox* latitudeDegree;
	QSpinBox* latitudeMinute;
	QSpinBox* latitudeSecond;
	QSpinBox* longitudeDegree;
	QSpinBox* longitudeMinute;
	QSpinBox* longitudeSecond;
};

ManualSetting::ManualSetting(double latitude, double longitude, QWidget* parent) : QDialog(parent)
{
	setWindowTitle(q_("Set coordinates"));
	QGridLayout* layout = new QGridLayout(this);

	layout->addWidget(new QLabel(q_("Latitude")), 0, 0);
	latitudeDegree = new QSpinBox;
	latitudeDegree->setSuffix(QChar(176));
	latitudeDegree->setRange(-90, 89);
	layout->addWidget(latitudeDegree, 1, 0);
	latitudeMinute = new QSpinBox;
	latitudeMinute->setSuffix("'");
	latitudeMinute->setMaximum(59);
	layout->addWidget(latitudeMinute, 2, 0);
	latitudeSecond = new QSpinBox;
	latitudeSecond->setSuffix("\"");
	latitudeSecond->setMaximum(59);
	layout->addWidget(latitudeSecond, 3, 0);

	layout->addWidget(new QLabel(q_("Longitude")), 0, 1);
	longitudeDegree = new QSpinBox;
	longitudeDegree->setSuffix(QChar(176));
	longitudeDegree->setRange(-180, 179);
	longitudeDegree->setWrapping(true);
	layout->addWidget(longitudeDegree, 1, 1);
	longitudeMinute = new QSpinBox;
	longitudeMinute->setSuffix("'");
	longitudeMinute->setMaximum(59);
	layout->addWidget(longitudeMinute, 2, 1);
	longitudeSecond = new QSpinBox;
	longitudeSecond->setSuffix("\"");
	longitudeSecond->setMaximum(59);
	layout->addWidget(longitudeSecond, 3, 1);

	QPushButton* okButton = new QPushButton("Save");
	connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
	layout->addWidget(okButton, 0, 2, 4, 1);

	setWidgets(latitude, longitude);
}

void ManualSetting::setWidgets(double latitude, double longitude)
{
	int sign, degree, minutes, seconds;

	sign = (latitude >= 0) ? 1 : -1;
	latitude = std::abs(latitude);
	degree = (int)latitude;
	latitude = (latitude - degree) * 60;
	minutes = (int)latitude;
	latitude = (latitude - minutes) * 60;
	seconds = (int)latitude;
	degree *= sign;

	latitudeDegree->setValue(degree);
	latitudeMinute->setValue(minutes);
	latitudeSecond->setValue(seconds);

	sign = (longitude >= 0) ? 1 : -1;
	longitude = std::abs(longitude);
	degree = (int)longitude;
	longitude = (longitude - degree) * 60;
	minutes = (int)longitude;
	longitude = (longitude - minutes) * 60;
	seconds = (int)longitude;
	degree *= sign;

	longitudeDegree->setValue(degree);
	longitudeMinute->setValue(minutes);
	longitudeSecond->setValue(seconds);
}

double ManualSetting::getLatitude() const
{
	return (double)latitudeDegree->value() +
		   (double)latitudeMinute->value() / 60 +
		   (double)latitudeSecond->value() / 3600;
}

double ManualSetting::getLongitude() const
{
	return (double)longitudeDegree->value() +
		   (double)longitudeMinute->value() / 60 +
		   (double)longitudeSecond->value() / 3600;
}

PickCity::PickCity(const StelLocation &loc) : location(loc)
{
	setWindowTitle(q_("Pick a city"));

	// Create the two buttons
	QGridLayout* layout = new QGridLayout(this);
#ifdef Q_WS_MAEMO_5
	countryButton = new QMaemo5ValueButton("Country");
	((QMaemo5ValueButton*)countryButton)->setValueLayout(QMaemo5ValueButton::ValueBesideText);
	((QMaemo5ValueButton*)countryButton)->setValueText(location.country);
	cityButton = new QMaemo5ValueButton("City");
	((QMaemo5ValueButton*)cityButton)->setValueLayout(QMaemo5ValueButton::ValueBesideText);
	((QMaemo5ValueButton*)cityButton)->setValueText(location.name);
#else
	countryButton = new QPushButton("Country");
	cityButton = new QPushButton("City");
#endif
	connect(countryButton, SIGNAL(clicked()), this, SLOT(onPickCountry()));
	layout->addWidget(countryButton, 0,0);
	connect(cityButton, SIGNAL(clicked()), this, SLOT(onPickCity()));
	layout->addWidget(cityButton, 1, 0);

	QPushButton* okButton = new QPushButton("Save");
	okButton->setMaximumWidth(250);
	layout->addWidget(okButton, 0, 1, 2, 1);
	connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
}

void PickCity::onPickCountry()
{
	QStringListModel countriesModel;
	countriesModel.setStringList(StelLocaleMgr::getAllCountryNames());
	ListPicker picker("Country", &countriesModel, this);
	picker.select(location.country);
	picker.exec();
	if (picker.getValue() == location.country || picker.getValue() == "")
		return;
	location.country = picker.getValue();

	// set the city to the first city in this country
	location.name = "";
	QList<StelLocation> allLocations = StelApp::getInstance().getLocationMgr().getAll();
	foreach(const StelLocation& loc, allLocations)
	{
		if (loc.country == location.country)
		{
			location.name = loc.name;
			break;
		}
	}

#ifdef Q_WS_MAEMO_5
	((QMaemo5ValueButton*)countryButton)->setValueText(location.country);
	((QMaemo5ValueButton*)cityButton)->setValueText(location.name);
#endif
}

void PickCity::onPickCity()
{
	QList<StelLocation> allLocations = StelApp::getInstance().getLocationMgr().getAll();
	QStringList list;
	foreach(const StelLocation& loc, allLocations)
	{
		if (loc.country == location.country)
			list << loc.name;
	}
	QStringListModel citiesModel;
	citiesModel.setStringList(list);

	ListPicker picker("City", &citiesModel, this);
	picker.select(location.name);
	picker.exec();
	if (picker.getValue() == location.name || picker.getValue() == "")
		return;
	location.name = picker.getValue();
#ifdef Q_WS_MAEMO_5
	((QMaemo5ValueButton*)cityButton)->setValueText(location.name);
#endif
}

StelLocation PickCity::getLocation() const
{
	// Since the location attribute only has its country and city set, we get the actual location from the location
	// manager.
	return StelApp::getInstance().getLocationMgr().locationForSmallString(location.getID());
}

LocationDialog::LocationDialog(QWidget *parent) : QDialog(parent)
{
	setupUi(this);
	
	locMgr = ((MobileGui*)StelApp::getInstance().getGui())->getGuiBaseItem()->getLocationMgr();
			
	
	updateAll();
	
	connect(mapLabel, SIGNAL(positionChanged(double, double)), locMgr, SLOT(moveToManualLocation(double, double)));
	connect(pickCityButton, SIGNAL(clicked()), this, SLOT(showPickCity()));
	connect(manualButton, SIGNAL(clicked()), this, SLOT(showManual()));
	connect(fromGpsButton, SIGNAL(toggled(bool)), locMgr, SLOT(setUseGps(bool)));
	connect(locMgr, SIGNAL(changed()), this, SLOT(updateAll()));
}

void LocationDialog::pickCity()
{
#ifdef Q_WS_MAEMO_5
	QWidget* window = citySelector->widget(this);
	window->setWindowTitle("Pick a city");
	window->show();
#endif
}

void LocationDialog::updateAll()
{
	const StelLocation& location = StelApp::getInstance().getCore()->getNavigator()->getCurrentLocation();
	fromGpsButton->setEnabled(locMgr->hasGps());
	fromGpsButton->setChecked(locMgr->getUseGps());
	manualButton->setEnabled(!locMgr->getUseGps());
	pickCityButton->setEnabled(!locMgr->getUseGps());
	
	mapLabel->setCursorPos(location.latitude, location.longitude);
	QString latitude = latitudeRepr(location.latitude);
	QString longitude = longitudeRepr(location.longitude);
	manualButton->setText(QString("%1\n%2").arg(latitude).arg(longitude));
	if (locMgr->getUseGps() || location.name=="Manual Location")
		pickCityButton->setText(q_("Pick City"));
	else
		pickCityButton->setText(location.name);
}

QString LocationDialog::latitudeRepr(double latitude) const
{
	bool sign = latitude >= 0;
	latitude = std::abs(latitude);
	int degree = (int)latitude;
	latitude = (latitude - degree) * 60;
	int minutes = (int)latitude;
	latitude = (latitude - minutes) * 60;
	int seconds = (int)latitude;
	return QString("%1 %2%3 %4' %5\"").arg(sign ? "N" : "S").arg(degree).arg(QChar(176)).arg(minutes).arg(seconds);
}

QString LocationDialog::longitudeRepr(double longitude) const
{
	bool sign = longitude >= 0;
	longitude = std::abs(longitude);
	int degree = (int)longitude;
	longitude = (longitude - degree) * 60;
	int minutes = (int)longitude;
	longitude = (longitude - minutes) * 60;
	int seconds = (int)longitude;
	return QString("%1 %2%3 %4' %5\"").arg(sign ? "W" : "E").arg(degree).arg(QChar(176)).arg(minutes).arg(seconds);
}

void LocationDialog::showManual()
{
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	double latitude = nav->getCurrentLocation().latitude;
	double longitude = nav->getCurrentLocation().longitude;
	ManualSetting manualSetting(latitude, longitude);
	if (manualSetting.exec() != QDialog::Accepted)
		return;
	locMgr->moveToManualLocation(manualSetting.getLatitude(), manualSetting.getLongitude());
}

void LocationDialog::showPickCity()
{
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	PickCity pickCity(nav->getCurrentLocation());
	if (pickCity.exec() == QDialog::Rejected || pickCity.getLocation().name == "")
		return;
	locMgr->moveObserverTo(pickCity.getLocation());
}

