/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "DateTimeDialog.hpp"

#include "StelApp.hpp"
#include "StelCore.hpp"
#include "StelNavigator.hpp"
#include "StelLocaleMgr.hpp"
#include "StelTranslator.hpp"

#include <QLayout>
#include <QDate>
#include <QTime>
#include <QCheckBox>
#include <QSettings>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5>
#endif


DateTimeDialog::DateTimeDialog(QWidget* parent) : QDialog(parent)
{
	setWindowTitle(q_("Date and Time"));
	QVBoxLayout* vertLayout = new QVBoxLayout(this);
	
	QCheckBox* box = new QCheckBox(q_("Fast forward time to night at startup"), this);
	box->setChecked(StelApp::getInstance().getSettings()->value("mobileGui/startAtNight", true).toBool());
	connect(box, SIGNAL(toggled(bool)), this, SLOT(setForwardTimeAtStartup(bool)));
	vertLayout->addWidget(box);
	
#ifdef Q_WS_MAEMO_5
	datePickSelector = new QMaemo5DatePickSelector;
	timePickSelector = new QMaemo5TimePickSelector;

	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	double jd = nav->getJDay();
	jd += StelApp::getInstance().getLocaleMgr().getGMTShift(jd) / 24.0; // UTC -> local tz
	setDateTime(jd); // UTC -> local tz

	QHBoxLayout* layout = new QHBoxLayout();
	QMaemo5ValueButton* dateButton = new QMaemo5ValueButton();
	dateButton->setPickSelector(datePickSelector);
	connect(datePickSelector, SIGNAL(selected(QString)), this, SLOT(valueChanged()));
	layout->addWidget(dateButton);

	QMaemo5ValueButton* timeButton = new QMaemo5ValueButton();
	timeButton->setPickSelector(timePickSelector);
	connect(timePickSelector, SIGNAL(selected(QString)), this, SLOT(valueChanged()));
	layout->addWidget(timeButton);
	
	vertLayout->addLayout(layout);
#endif
}

void DateTimeDialog::setDateTime(double newJd)
{
	int year, month, day, hour, minute, second;
	StelUtils::getDateFromJulianDay(newJd, &year, &month, &day);
	StelUtils::getTimeFromJulianDay(newJd, &hour, &minute, &second);

	QDate date(year, month, day);
	QTime time(day, hour, second);
#ifdef Q_WS_MAEMO_5
	datePickSelector->setCurrentDate(date);
	timePickSelector->setCurrentTime(time);
#endif
}

void DateTimeDialog::valueChanged()
{
#ifdef Q_WS_MAEMO_5
	QDate date = datePickSelector->currentDate();
	QTime time = timePickSelector->currentTime();

	double jd;
	StelUtils::getJDFromDate(&jd, date.year(), date.month(), date.day(),
							 time.hour(), time.minute(), time.second());
	jd -= (StelApp::getInstance().getLocaleMgr().getGMTShift(jd)/24.0); // local tz -> UTC
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	nav->setJDay(jd);
#endif
}

void DateTimeDialog::setForwardTimeAtStartup(bool b)
{
	StelApp::getInstance().getSettings()->setValue("mobileGui/startAtNight", b);
}
