/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "TimeWidget.hpp"

#include <QString>
#include <QSlider>
#include <QPushButton>
#include <QHBoxLayout>
#include <QLabel>

#include "StelCore.hpp"
#include "StelNavigator.hpp"
#include "StelApp.hpp"

#include <cmath>

TimeWidget::TimeWidget()
{
	//setMinimumHeight(40);
	QLayout* layout = new QHBoxLayout(this);
	label = new QLabel;
	label->setMinimumWidth(80);
	label->setMaximumWidth(80);
	layout->addWidget(label);
	QPushButton* timeNow = new QPushButton("reset");
	connect(timeNow, SIGNAL(clicked()), this, SLOT(timeNow()));
	layout->addWidget(timeNow);
	backButton = new QPushButton("backward");
	backButton->setCheckable(true);
	layout->addWidget(backButton);
	slider = new QSlider(Qt::Horizontal);
	slider->setMinimum(0);
	slider->setMaximum(8);
	connect(slider, SIGNAL(valueChanged(int)), this, SLOT(computeRate()));
	connect(backButton, SIGNAL(toggled(bool)), this, SLOT(computeRate()));
	layout->addWidget(slider);

	// This could be done outside this class for better genericity
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	Q_ASSERT(nav);
	connect(this, SIGNAL(rateChanged(double)), nav, SLOT(setTimeRate(double)));
	connect(nav, SIGNAL(timeRateChanged(double)), this, SLOT(setRate(double)));
	setRate(nav->getTimeRate());
}

void TimeWidget::timeNow()
{
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	Q_ASSERT(nav);
	nav->setTimeNow();
	nav->setRealTimeSpeed();
}

void TimeWidget::computeRate()
{
	if (slider->value() == 0)
	{
		emit rateChanged(0);
		return;
	}
	double rate = std::pow(10., slider->value() - 1) * JD_SECOND;
	if (backButton->isChecked())
		rate = -rate;
	emit rateChanged(rate);
}

void TimeWidget::setRate(double rate)
{
	blockSignals(true); // this can only be called from StelNavigator
	double coef = rate / JD_SECOND;
	label->setText(QString("x%1").arg(coef, 5));
	bool back = coef < 0;
	coef = std::abs(coef);
	backButton->setChecked(back);
	int value = (coef == 0)? 0 : std::log10(coef) + 1;
	slider->setValue(value);
	blockSignals(false);
}
