/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "TimeBar.hpp"
#include "ImageButton.hpp"
#include "StelApp.hpp"
#include "StelGuiBase.hpp"
#include "ButtonGroup.hpp"
#include "GraphicItemWrapper.hpp"

#include "StelCore.hpp"
#include "StelApp.hpp"
#include "StelNavigator.hpp"
#include "StelLocaleMgr.hpp"
#include "StelMovementMgr.hpp"
#include "StelModuleMgr.hpp"

#include "HelpMessage.hpp"
#include <QGraphicsAnchorLayout>
#include <QGraphicsLinearLayout>
#include <QGraphicsSceneMouseEvent>
#include <QDebug>
#include <QGraphicsDropShadowEffect>

TimeBar::TimeBar(QGraphicsItem* parent, HelpMessage* msg): QGraphicsWidget(parent), isTimeDragging(false)
{
	QGraphicsAnchorLayout* layout = new QGraphicsAnchorLayout(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0, 0, 0, 3);
	setContentsMargins(0, 0, 0, 0);
	StelGuiBase* gui = StelApp::getInstance().getGui();

	ButtonGroup* timeGroup = new ButtonGroup(this);
	timeGroup->layout->setSpacing(0);
	timeGroup->add(new ImageButton(":/mobile/timeRewind.png", gui->getGuiActions("actionDecrease_Time_Speed"), NULL, "", msg));
	btTimeReal = new ImageButton(":/mobile/timeReal.png", gui->getGuiActions("actionSet_Real_Time_Speed"), NULL, "", msg);
	timeGroup->add(btTimeReal);
	timeGroup->add(new ImageButton(":/mobile/timeForward.png", gui->getGuiActions("actionIncrease_Time_Speed"), NULL, "", msg));
	layout->addCornerAnchors(timeGroup, Qt::BottomLeftCorner, layout, Qt::BottomLeftCorner);

	ButtonGroup* timeNowGroup = new ButtonGroup(this);
	btTimeNow = new ImageButton(":/mobile/timeNow.png", gui->getGuiActions("actionReturn_To_Current_Time"), NULL, "", msg);
	timeNowGroup->add(btTimeNow);
	layout->addCornerAnchors(timeNowGroup, Qt::BottomRightCorner, layout, Qt::BottomRightCorner);
	
	textItem = new QGraphicsSimpleTextItem(this);
	textItem->setBrush(QColor(Qt::white));
	QFont font;
	font.setPixelSize(28);
	font.setBold(true);
	textItem->setFont(font);
	textItem->setText("2008-02-06  17:33:00");
	GraphicItemWrapper* wrap = new GraphicItemWrapper(this, textItem);
	layout->addAnchor(wrap, Qt::AnchorVerticalCenter, timeGroup, Qt::AnchorVerticalCenter);
	QGraphicsAnchor* anch = layout->addAnchor(wrap, Qt::AnchorLeft, timeGroup, Qt::AnchorRight);
	anch->setSpacing(50);

	timeSpeedTextItem = new QGraphicsSimpleTextItem(this);
	timeSpeedTextItem->setBrush(QColor(Qt::white));
	timeSpeedTextItem->setFont(font);
	timeSpeedTextItem->setText("x6666");
	wrap = new GraphicItemWrapper(this, timeSpeedTextItem);
	anch = layout->addAnchor(wrap, Qt::AnchorBottom, timeGroup, Qt::AnchorTop);
	anch->setSpacing(5);
	anch = layout->addAnchor(wrap, Qt::AnchorLeft, layout, Qt::AnchorLeft);
	anch->setSpacing(30);
	
	QGraphicsDropShadowEffect* shadow = new QGraphicsDropShadowEffect(this);
	shadow->setOffset(1.);
	shadow->setColor(QColor(0, 0, 0, 255));
	textItem->setGraphicsEffect(shadow);
	shadow = new QGraphicsDropShadowEffect(this);
	shadow->setOffset(1.);
	shadow->setColor(QColor(0, 0, 0, 255));
	timeSpeedTextItem->setGraphicsEffect(shadow);
	
	setCacheMode(QGraphicsItem::ItemCoordinateCache);
}

void TimeBar::updateText()
{
	StelCore* core = StelApp::getInstance().getCore();
	double jd = core->getNavigator()->getJDay();
	QString newDate = StelApp::getInstance().getLocaleMgr().getPrintableDateLocal(jd) + " "
			+ StelApp::getInstance().getLocaleMgr().getPrintableTimeLocal(jd);
	if (textItem->text()!=newDate)
	{
		textItem->setText(newDate);
	}
	
	QString newTimeSpeed;
	if (core->getNavigator()->getRealTimeSpeed())
		newTimeSpeed = " ";
	else
		newTimeSpeed = "x" + QString::number(core->getNavigator()->getTimeRate()/JD_SECOND, 'f', 0);
	if (isTimeDragging && GETSTELMODULE(StelMovementMgr)->getHasDragged())
		newTimeSpeed = "----";
	if (timeSpeedTextItem->text()!=newTimeSpeed)
	{
		timeSpeedTextItem->setText(newTimeSpeed);
	}
	
	btTimeNow->setChecked(!StelApp::getInstance().getCore()->getNavigator()->getIsTimeNow());
	btTimeReal->setChecked(!StelApp::getInstance().getCore()->getNavigator()->getRealTimeSpeed());
}

void TimeBar::mousePressEvent(QGraphicsSceneMouseEvent* event)
{
	event->accept();
}
