/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "StatusBar.hpp"
#include "GraphicItemWrapper.hpp"
#include "GraphicsTextLabel.hpp"
#include "StelGuiBase.hpp"
#include "MobileLocationMgr.hpp"
#include "MobileGui.hpp"

#include <QFont>
#include <QGraphicsSimpleTextItem>
#include <QGraphicsAnchorLayout>
#include <QBrush>
#include <QStateMachine>
#include <QPropertyAnimation>
#include <QSignalTransition>

#include <StelCore.hpp>
#include <StelApp.hpp>
#include <StelNavigator.hpp>
#include <StelLocaleMgr.hpp>
#include <StelMovementMgr.hpp>

StatusBar::StatusBar(QGraphicsItem *parent) : QGraphicsWidget(parent)
{
	QGraphicsAnchorLayout* layout = new QGraphicsAnchorLayout(this);
	layout->setSpacing(1);
	layout->setContentsMargins(3,0,10,0);
	//setContentsMargins(0,0,0,10);

	datetime = new GraphicsTextLabel("2008-02-06\n17:33:00", this, 18);
	datetime->setOffset(4, 0);
	layout->addCornerAnchors(datetime, Qt::BottomLeftCorner, layout, Qt::BottomLeftCorner);

	fov = new GraphicsTextLabel("43.45d", NULL, 28);
	layout->addAnchor(fov, Qt::AnchorBottom, layout, Qt::AnchorBottom);
	QGraphicsAnchor* anch = layout->addAnchor(fov, Qt::AnchorHorizontalCenter, layout, Qt::AnchorHorizontalCenter);
	anch->setSpacing(50);

	location = new GraphicsTextLabel("Munich", NULL, 28);
	layout->addCornerAnchors(location, Qt::BottomRightCorner, layout, Qt::BottomRightCorner);
	
	QStateMachine* fovStateMachine = new QStateMachine(this);
	QState* stateOn = new QState();
	stateOn->assignProperty(fov, "opacity", 2.0);
	QState* stateOff = new QState();
	stateOff->assignProperty(fov, "opacity", 0);
	QPropertyAnimation* an1 = new QPropertyAnimation(fov, "opacity");
	an1->setDuration(200);
	QPropertyAnimation* an2 = new QPropertyAnimation(fov, "opacity");
	an2->setDuration(1500);
	QSignalTransition* trHelpMessage = stateOff->addTransition(this, SIGNAL(showFovLabel()), stateOn);
	QSignalTransition* trHelpMessage2 = stateOn->addTransition(this, SIGNAL(hideFovLabel()), stateOff);
	trHelpMessage->addAnimation(an1);
	trHelpMessage2->addAnimation(an2);
	fovStateMachine->addState(stateOn);
	fovStateMachine->addState(stateOff);
	fovStateMachine->setInitialState(stateOn);
	fovStateMachine->start();
	
	QStateMachine* locationStateMachine = new QStateMachine(this);
	stateOn = new QState();
	stateOn->assignProperty(location, "opacity", 1.0);
	stateOff = new QState();
	stateOff->assignProperty(location, "opacity", 0.5);
	an1 = new QPropertyAnimation(location, "opacity");
	an1->setDuration(200);
	an2 = new QPropertyAnimation(location, "opacity");
	an2->setDuration(1500);
	trHelpMessage = stateOff->addTransition(this, SIGNAL(showLocationLabel()), stateOn);
	trHelpMessage2 = stateOn->addTransition(this, SIGNAL(hideLocationLabel()), stateOff);
	trHelpMessage->addAnimation(an1);
	trHelpMessage2->addAnimation(an2);
	locationStateMachine->addState(stateOn);
	locationStateMachine->addState(stateOff);
	locationStateMachine->setInitialState(stateOn);
	locationStateMachine->start();
	
	lastTimeFovLabelChanged = StelApp::getInstance().getTotalRunTime();
	lastTimeLocationLabelChanged = lastTimeFovLabelChanged;
}

void StatusBar::updateText()
{
	StelCore* core = StelApp::getInstance().getCore();
	double jd = core->getNavigator()->getJDay();
	const float now = StelApp::getInstance().getTotalRunTime();

	QString newDate = StelApp::getInstance().getLocaleMgr().getPrintableTimeLocal(jd) + "\n"
			+ StelApp::getInstance().getLocaleMgr().getPrintableDateLocal(jd);
	if (datetime->text()!=newDate)
	{
		datetime->setText(newDate);
	}

	QString newLocation;
	MobileLocationMgr* locMgr = ((MobileGui*)StelApp::getInstance().getGui())->getGuiBaseItem()->getLocationMgr();
	if (locMgr->getUseGps())
	{
		switch (locMgr->getStatus())
		{
			case MobileLocationMgr::StatusStopped:
				newLocation="GPS (stopped)";
				break;
			case MobileLocationMgr::StatusSearching:
				newLocation="GPS (searching)";
				break;
			case MobileLocationMgr::StatusRefining:
				newLocation="GPS (refining)";
				break;
			case MobileLocationMgr::StatusFound:
				newLocation="GPS";
				break;
			case MobileLocationMgr::StatusError:
				newLocation="GPS (error)";
		}
	}
	else
	{
		newLocation = core->getNavigator()->getCurrentLocation().name;
	}

	if (location->text()!=newLocation)
	{
		lastTimeLocationLabelChanged = now;
		emit showLocationLabel();
		location->setText(newLocation);
	}
	else
	{
		if (now-lastTimeLocationLabelChanged>5.0)
			emit hideLocationLabel();
	}

	QString str;
	QTextStream wos(&str);
	wos << qSetRealNumberPrecision(4) << forcepoint << core->getMovementMgr()->getCurrentFov() << QChar(0x00B0);
	if (fov->text()!=str)
	{
		lastTimeFovLabelChanged = now;
		emit showFovLabel();
		fov->setText(str);
	}
	else
	{
		if (now-lastTimeFovLabelChanged>3.0)
			emit hideFovLabel();
	}
}
