/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "MobileLocationMgr.hpp"
#include "StelApp.hpp"
#include "StelCore.hpp"
#include "StelNavigator.hpp"
#include "StelLocation.hpp"
#include "StelLocationMgr.hpp"
#include "Gps.hpp"

MobileLocationMgr::MobileLocationMgr(QObject *parent) :
    QObject(parent), currentStatus(StatusStopped)
{
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	
	
	bool ok;
	// Check if a Manual user location was already saved
	manualLocation= StelApp::getInstance().getLocationMgr().locationForSmallString("Manual Location, World", &ok);
	if (!ok)
	{
		qDebug() << "Create Manual Location user location";
		manualLocation = nav->getCurrentLocation();
		manualLocation.country = "World";
		manualLocation.name = "Manual Location";
		if (!StelApp::getInstance().getLocationMgr().saveUserLocation(manualLocation))
		{
			qWarning() << "Cannot save manual user location";
		}
	}
	
	// Check if a GPS user location was already saved
	lastGpsLocation = StelApp::getInstance().getLocationMgr().locationForSmallString("GPS, World", &ok);
	if (!ok)
	{
		qDebug() << "Create GPS user location";
		lastGpsLocation = nav->getCurrentLocation();
		lastGpsLocation.country = "World";
		lastGpsLocation.name = "GPS";
		if (!StelApp::getInstance().getLocationMgr().saveUserLocation(lastGpsLocation))
		{
			qWarning() << "Cannot save GPS user location";
		}
		
		// It's apparently the first time we start stellarium-mobile: use GPS by default.
		if (hasGps())
			moveObserverTo(lastGpsLocation);
	}
	useGps = hasGps() && (nav->getCurrentLocation().getID()==lastGpsLocation.getID());
	if (hasGps())
	{
		QObject::connect(Gps::getInstance(), SIGNAL(positionUpdate()), this, SLOT(gpsFixReceived()));
	}
	if (useGps)
	{
		// Fool cache
		useGps=false;
		setUseGps(true);
	}
}

bool MobileLocationMgr::hasGps() const
{
	return Gps::getInstance()->isAvailable();
}

void MobileLocationMgr::moveObserverTo(const StelLocation& location)
{
	StelNavigator* nav = StelApp::getInstance().getCore()->getNavigator();
	nav->moveObserverTo(location, 0, 0);
	nav->setDefaultLocationID(location.getID());
	emit changed();
}

void MobileLocationMgr::moveToManualLocation(double latitude, double longitude)
{
	if (useGps)
		setUseGps(false);
	manualLocation.latitude = latitude;
	manualLocation.longitude = longitude;
	StelApp::getInstance().getLocationMgr().deleteUserLocation(manualLocation.getID());
	StelApp::getInstance().getLocationMgr().saveUserLocation(manualLocation);
	moveObserverTo(manualLocation);
}

void MobileLocationMgr::setUseGps(bool b)
{
	if (useGps==b)
		return;
	useGps = b;
	if (!hasGps())
	{
		currentStatus=StatusError;
		return;
	}
	if (useGps)
	{
		Gps::getInstance()->start();
		currentStatus = StatusSearching;
		moveObserverTo(lastGpsLocation);
	}
	else
	{
		Gps::getInstance()->stop();
		currentStatus = StatusStopped;
		moveToManualLocation(lastGpsLocation.latitude, lastGpsLocation.longitude);
	}
}

void MobileLocationMgr::gpsFixReceived()
{
	if (!useGps)
		return;
	double lon, lat, alt, precision;
	if (!Gps::getInstance()->lastKnownPosition(&lon, &lat, &alt, &precision))
		return;
	lastGpsLocation.longitude=lon;
	lastGpsLocation.latitude=lat;
	if (alt>-1000)
		lastGpsLocation.altitude=alt;
	if (precision<50000.f)
	{
		// If accuracy better than 50km stop GPS
		Gps::getInstance()->stop();
		currentStatus = StatusFound;
	}
	else
	{
		currentStatus = StatusRefining;
	}
	StelApp::getInstance().getLocationMgr().deleteUserLocation(lastGpsLocation.getID());
	StelApp::getInstance().getLocationMgr().saveUserLocation(lastGpsLocation);
	moveObserverTo(lastGpsLocation);
}
