/*
 * Stellarium
 * Copyright (C) 2010 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ImageButton.hpp"
#include "HelpMessage.hpp"
#include "StelTranslator.hpp"
#include <QDebug>
#include <QAction>
#include <QGraphicsBlurEffect>
#include <QStateMachine>
#include <QParallelAnimationGroup>
#include <QPropertyAnimation>
#include <QSignalTransition>
#include <QGraphicsSceneMouseEvent>

ImageButton::ImageButton(const QString& image, QAction* aaction, QGraphicsItem* parent,
						 const QString& haloImage, HelpMessage* ahelpMsg) :
		QGraphicsWidget(parent), action(aaction), checkable(false), checked(false), helpMsg(ahelpMsg)
{
	//setCacheMode(QGraphicsItem::ItemCoordinateCache);
	icon = new QGraphicsPixmapItem(image, this);
	icon->setParentItem(this);
	icon->setZValue(1);

	setContentsMargins(13,13,13,13);

	hoverHalo = new QGraphicsPixmapItem(QPixmap(haloImage.isEmpty() ? ":/mobile/hoverHalo.png" : haloImage), this);
	hoverHalo->setParentItem(this);
	hoverHalo->setOpacity(0);
	hoverHalo->setZValue(0);
	hoverHalo->setFlag(QGraphicsItem::ItemIgnoresParentOpacity);

	QGraphicsDropShadowEffect* shadow = new QGraphicsDropShadowEffect(this);
	shadow->setOffset(3.);
	shadow->setColor(QColor(0, 0, 0, 255));
	icon->setGraphicsEffect(shadow);

	setAcceptHoverEvents(true);

	if (action!=NULL)
	{
		checkable = action->isCheckable();
		checked=checkable ? !action->isChecked() : false;
		setChecked(checkable ? action->isChecked() : true);
		if (checkable)
		{
			setChecked(action->isChecked());
			QObject::connect(this, SIGNAL(toggled(bool)), action, SLOT(setChecked(bool)));
			QObject::connect(action, SIGNAL(toggled(bool)), this, SLOT(setChecked(bool)));
		}
		else
			QObject::connect(this, SIGNAL(triggered()), action, SLOT(trigger()));
	}

	QStateMachine* stateMachine = new QStateMachine();
	QState *s1 = new QState();
	s1->assignProperty(this, "opacity", 1);
	s1->assignProperty(shadow, "blurRadius", 8);

	QState *s2 = new QState();
	s2->assignProperty(this, "opacity", 0.5);
	s2->assignProperty(shadow, "blurRadius", 15);

	QPropertyAnimation* a1 = new QPropertyAnimation(this, "opacity");
	a1->setDuration(200);
	QPropertyAnimation* a2 = new QPropertyAnimation(shadow, "blurRadius");
	a2->setDuration(200);
	QParallelAnimationGroup* modeSwitchAnimation = new QParallelAnimationGroup(this);
	modeSwitchAnimation->addAnimation(a1);
	modeSwitchAnimation->addAnimation(a2);

	QSignalTransition* tr1 = s1->addTransition(this, SIGNAL(toggledFalse()), s2);
	tr1->addAnimation(modeSwitchAnimation);

	QSignalTransition* tr2 = s2->addTransition(this, SIGNAL(toggledTrue()), s1);
	tr2->addAnimation(modeSwitchAnimation);

	stateMachine->addState(s1);
	stateMachine->addState(s2);

	stateMachine->setInitialState(checked ? s1 : s2);
	stateMachine->start();

}

QSizeF ImageButton::sizeHint(Qt::SizeHint, const QSizeF&) const
{
	return icon->pixmap().size();
}

void ImageButton::setChecked(bool b)
{
	if (b==checked)
		return;

	checked=b;
	emit toggled(checked);
	if (checked)
		emit toggledTrue();
	else
		emit toggledFalse();
}

void ImageButton::mousePressEvent(QGraphicsSceneMouseEvent*)
{
	hoverHalo->setOpacity(1);
	if (helpMsg && action)
	{
		helpMsg->displayMessage(q_(action->property("englishText").toString()));
	}
}

void ImageButton::mouseReleaseEvent(QGraphicsSceneMouseEvent* e)
{
	hoverHalo->setOpacity(0);
	if (contains(e->pos()))
	{
		if (checkable)
		{
			setChecked(!checked);
		}
		else
		{
			emit triggered();
		}
	}
	if (helpMsg)
	{
		helpMsg->hideHelpMessage();
	}
}

void ImageButton::hoverLeaveEvent(QGraphicsSceneHoverEvent*)
{
	hoverHalo->setOpacity(0);
}
