#!/usr/bin/python

"""
	Squeezecontrol: Control your Squeezeboxes from your N900

	Copyright 2011 Mikkel Munch Mortensen <3xm@detfalskested.dk> 

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import sys, os, sqlite3
from PySide.QtCore import *
from PySide.QtGui import *
from PySide.QtMaemo5 import *

from pysqueezecenter.server import Server
from pysqueezecenter.player import Player

app_path = os.environ['HOME'] + '/.squeezecontrol/'

class PlayerWindow(QMainWindow):
	def __init__(self, parent, player):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle(player.get_name())
		self.player = player
		self.render()
	
	def render(self):
		self.main_widget = QWidget(self)
		vbox = QVBoxLayout(self.main_widget)
		
		# Main controls
		controls = QWidget(self.main_widget)
		hbox = QHBoxLayout(controls)
		
		prev = QPushButton('<<')
		self.connect(prev, SIGNAL('clicked()'), self.prev)
		hbox.addWidget(prev)
		
		self.play_button = QPushButton('Playing' if self.player.get_mode() == 'play' else 'Paused')
		self.connect(self.play_button, SIGNAL('clicked()'), self.play_toggle)
		hbox.addWidget(self.play_button)
		
		next = QPushButton('>>')
		self.connect(next, SIGNAL('clicked()'), self.next)
		hbox.addWidget(next)
		vbox.addWidget(controls)
		
		# Volume slider
		vol = QSlider(Qt.Horizontal)
		vol.setValue(self.player.get_volume())
		vbox.addWidget(vol)
		vol.valueChanged.connect(self.change_vol)
		
		self.setCentralWidget(self.main_widget)
	
	def prev(self):
		self.player.prev()
	
	def next(self):
		self.player.next()
	
	def play_toggle(self):
		self.player.toggle()
		self.play_button.setText('Playing' if self.player.get_mode() == 'play' else 'Paused')
	
	def change_vol(self):
		self.player.set_volume(self.sender().value())

class StartWindow(QMainWindow):
	def __init__(self):
		QMainWindow.__init__(self)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('Squeezecontrol')
		
		# Make sure the app folder exists.
		if not os.path.exists(app_path):
			os.mkdir(app_path)
		
		# Create SQLite database and structure.
		self.db = sqlite3.connect(app_path + 'db.sqlite')
		self.cursor = self.db.cursor()
		self.cursor.execute('''
		create table if not exists settings
		(key, value)
		''')
		
	def get_setting(self, key):
		params = (key,)
		self.cursor.execute('select value from settings where key=?', params)
		result = self.cursor.fetchone()
		return result[0] if result else None
	
	def save_setting(self, key, value):
		params = (key,)
		self.cursor.execute('delete from settings where key=?', params)
		params = (key,value)
		self.cursor.execute('insert into settings values (?, ?)', params)
		self.db.commit()
	
	def render(self):
		container = QWidget(self)
		vbox = QVBoxLayout(container)
		
		for text, action in [
			('Select player', self.open_player_selection,),
			('Settings', self.open_settings,),
			('About', self.open_about,),
		]:
			button = QPushButton(text)
			self.connect(button, SIGNAL('clicked()'), action)
			vbox.addWidget(button)
		self.setCentralWidget(container)
	
	def open_player_selection(self):
		select_window = SelectPlayerWindow(self)
		select_window.show()
	
	def open_settings(self):
		settings_window = SettingsWindow(self)
		settings_window.show()
	
	def open_about(self):
		about_window = AboutWindow(self)
		about_window.show()

class AboutWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('About Squeezecontrol 0.0.1')
		self.render()
	
	def render(self):
		container = QWidget()
		vbox = QVBoxLayout(container)
		label = QLabel('This version of Squeezecontrol was made while listening to Mux Mool, DJ Shadow and Looptroop. A huge shoutout to JingleManSweep who made the pysqueezecenter wrapper library for Python.\n\nCopyright 2010, 2011 JingleManSweep and Mikkel Munch Mortensen. Released under the terms of the GNU General Public License version 3.')
		label.setWordWrap(True)
		vbox.addWidget(label)
		
		self.setCentralWidget(container)

class SettingsWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.setWindowTitle('Settings')
		self.main = parent
		self.render()
	
	def render(self):
		container = QWidget(self)
		vbox = QVBoxLayout(container)
		
		label = QLabel('Set the Squeezebox Server name. E.g. an IP address or something like myservername.com')
		label.setWordWrap(True)
		#label.setAlignment(Qt.AlignCenter)
		vbox.addWidget(label)
		
		fields = QWidget()
		hbox = QHBoxLayout(fields)
		vbox.addWidget(fields)
		
		label = QLabel('Server name:')
		hbox.addWidget(label)
		
		self.server_field = QLineEdit()
		server = self.main.get_setting('server')
		server = server if server else ''
		self.server_field.setText(server)
		hbox.addWidget(self.server_field)
		
		button = QPushButton('Save')
		self.connect(button, SIGNAL('clicked()'), self.save_settings)
		hbox.addWidget(button)
		
		label = QLabel('Is your server running on a non-default port? Are you using username/password to autheticate to your server? Have patience. These options will be added in a future version (coming soon).')
		label.setWordWrap(True)
		#label.setAlignment(Qt.AlignCenter)
		vbox.addWidget(label)
		
		self.setCentralWidget(container)
	
	def save_settings(self):
		self.main.save_setting('server', self.server_field.text())
		info = QMaemo5InformationBox()
		info.information(self, 'Server name saved.', 2500)

class SelectPlayerWindow(QMainWindow):
	def __init__(self, parent):
		QMainWindow.__init__(self, parent)
		self.setAttribute(Qt.WA_Maemo5StackedWindow)
		self.main = parent
		self.render()
	
	def render(self):
		server = self.main.get_setting('server')
		if server:
			self.render_player_list(server)
		else:
			self.setWindowTitle('No server set')
			container = QWidget(self)
			vbox = QVBoxLayout(container)
			label = QLabel('Please go back to the main screen, go to "Settings" and set what server to use.')
			label.setWordWrap(True)
			#label.setAlignment(Qt.AlignCenter)
			vbox.addWidget(label)
			self.setCentralWidget(container)
		
	def render_player_list(self, server):
		try:
			self.sc = Server(hostname=server)
			self.sc.connect()
		except Exception:
			self.setWindowTitle('Connection error')
			label = QLabel('Unable to connect to Squeezebox server at %s. Is the server accessible via your current network? Did you enter the right server settings?' % server)
			label.setWordWrap(True)
			#label.setAlignment(Qt.AlignCenter)
			self.setCentralWidget(label)
		else:
			self.setWindowTitle('Select player')
			self.main_widget = QWidget(self)
			self.main_widget.setMinimumSize(792,0)
			vbox = QVBoxLayout(self.main_widget)
			
			plist = {}
			for p in self.sc.get_players():
				name = p.get_name()
				plist[name] = QPushButton(name)
				plist[name].player = p
				self.connect(plist[name], SIGNAL('clicked()'), self.select_player)
				vbox.addWidget(plist[name])
	
			vbox.setSizeConstraint(QLayout.SetMinAndMaxSize)
			self.scroller = QScrollArea(self)
			self.scroller.setWidget(self.main_widget)
			self.setCentralWidget(self.scroller)
	
	def select_player(self):
		player = self.sender().player
		player_window = PlayerWindow(self, player)
		player_window.show()

if __name__ == '__main__':
	app = QApplication(sys.argv)
	w = StartWindow()
	w.render()
	w.show()
	app.exec_()
	sys.exit()

