#include "mainwindow.h"

#include <QtCore/QCoreApplication>

#if defined(Q_OS_SYMBIAN) && defined(ORIENTATIONLOCK)
#include <eikenv.h>
#include <eikappui.h>
#include <aknenv.h>
#include <aknappui.h>
#endif // Q_OS_SYMBIAN && ORIENTATIONLOCK

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{    
    central = new QWidget(this);
    glay = new QGridLayout(central);

    webView = new QWebView(this);

    glay->addWidget(webView);
    central->setLayout(glay);
    setCentralWidget(central);

    // Init toolbar
    tb = new QToolBar(this);

    m_fullscreen = new QAction(QIcon::fromTheme("general_fullsize"), QString(), this);
    m_fullscreen->setCheckable(true);
    connect(m_fullscreen, SIGNAL(toggled(bool)), this, SLOT(toggleFullScreen(bool)));

    m_spotme = new QAction(QIcon::fromTheme("act_spotme", QIcon(QLatin1String(":/spotme.png"))), QString(), this);
    m_spotme->setCheckable(true);
    QObject::connect(m_spotme, SIGNAL(toggled(bool)), this, SLOT(on_m_spotme_toggled(bool)));

    m_info = new QAction(QIcon::fromTheme("act_info", QIcon(QLatin1String(":/info.png"))), QString(), this);
    m_info->setCheckable(true);
    connect(m_info, SIGNAL(toggled(bool)), this, SLOT(on_m_info_toggled(bool)));

    //m_route = new QAction(QIcon::fromTheme("act_route", QIcon(QLatin1String(":/googleearth.png"))), QString(), this);
    //m_route->setCheckable(true);
    //connect(m_route, SIGNAL(triggered(bool)), this, SLOT(on_m_route_clicked(bool)));

    tb->addAction(m_fullscreen);
    tb->addSeparator();
    tb->addAction(m_spotme);
    tb->addSeparator();
    tb->addAction(m_info);
    tb->addSeparator();
    //tb->addAction(m_route);
    addToolBar(tb);


    // Init general variables and QNetworkConfiguration
    newCooordinateCount = 0;
    hideTbTimeout = 4000;
    updateCount = 0;
    LastLat.clear();
    LastLng.clear();
    activeConfigs = mgr.allConfigurations(QNetworkConfiguration::Active);
    qDebug() << "activeConfigs.count" << activeConfigs.count();

    // Load index.html to QWebView
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();
    webView->load(QUrl("qrc:/index.html"));

    // Create  filter
    keyEventListener = new KeyEventListener(webView);
    QObject::connect(keyEventListener,SIGNAL(screenTouched(bool)),this,SLOT(screenSlot(bool)));

    // Install it into your application for catching events
    webView->installEventFilter(keyEventListener);


    //********************
    // Setup GPS
    //********************
    LastLat.clear(); LastLng.clear();
    source = QGeoPositionInfoSource::createDefaultSource(this);
    if (source)
    {
        source->setUpdateInterval(1500); // time in milliseconds
        source->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
    }

    QTimer::singleShot(5000, this, SLOT(hideToolBar()));
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "closeEvent";
    event->accept();
    disable();
}

void MainWindow::hideToolBar()
{
    if(tb->isVisible())
        tb->setVisible(false);
}

void MainWindow::screenSlot(bool b)
{
    if(!tb->isVisible())
    {
        tb->setVisible(true);
        QTimer::singleShot(hideTbTimeout, this, SLOT(hideToolBar()));
    }
}

void MainWindow::positionUpdated(const QGeoPositionInfo &info)
{
    if (info.isValid())
    {
        qDebug() << ++updateCount << info;
        QGeoCoordinate co = info.coordinate();

        QString lat = QString::number(co.latitude());
        QString lng = QString::number(co.longitude());

        if( (lat != LastLat) or (lng != LastLng) )
        {
            qDebug() << "newCooordinateCount:" << ++newCooordinateCount;
            LastLat = lat;
            LastLng = lng;

            source->stopUpdates();
            updateGoogleMap(LastLat, LastLng);
            source->startUpdates();
        }
    }
    else {
        QMessageBox::information(this, tr("Error"),
                tr("Invalid GPS details."),
                QMessageBox::Ok);
    }
}

/* info */
void MainWindow::on_m_info_toggled(bool checked)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();
    //qDebug()<< "LastLat" << LastLat;
    //qDebug()<< "LastLng" << LastLng;
    //qDebug()<< "activeConfigs.count()" << activeConfigs.count();
    //qDebug()<< "mgr.isOnline()" << mgr.isOnline();

    if(checked)
    {
        if( !LastLat.isEmpty() && !LastLng.isEmpty() && (activeConfigs.count() > 0) && mgr.isOnline() )
            webView->page()->mainFrame()->evaluateJavaScript("getInfo(\""+LastLat+","+LastLng+"\")");
        else
            m_info->setChecked(false);
    }
    else
        webView->page()->mainFrame()->evaluateJavaScript("closeInfoWindow()");
}

void MainWindow::updateGoogleMap(QString pLat, QString pLng)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QCoreApplication::processEvents();

    qDebug() << "updateGoogleMap";
    webView->page()->mainFrame()->evaluateJavaScript("place_marker("+pLat+","+pLng+")");

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

/* start spotme button */
void MainWindow::on_m_spotme_toggled(bool checked)
{
    if (activeConfigs.count() > 0)
        Q_ASSERT(mgr.isOnline());   //Prints a warning message containing the source code file name and line number if test is false.
    else
    {
        Q_ASSERT(!mgr.isOnline());
        if(!mgr.isOnline())
        {
            QMessageBox::information(this, tr("Error"),
                    tr("Can not detect internet connection.\nMake sure you connect to the internet before proceeding."),
                    QMessageBox::Ok);
        }
    }

    if(checked)
    {
        enable();
        qDebug() << "GPS enabled";
    } else {
        m_info->setChecked(false);
        disable();
        qDebug() << "GPS disabled";
    }
}

void MainWindow::toggleFullScreen(bool b)
{
    setWindowState(b ? windowState() | Qt::WindowFullScreen
                     : windowState() & ~Qt::WindowFullScreen);

    if(b)
    {
        //tb->setVisible(false);
        QTimer::singleShot(hideTbTimeout, this, SLOT(hideToolBar()));
    }
    else
        tb->setVisible(true);
}

QString MainWindow::getCoordinates(QGeoPositionInfo info)
{
    QGeoCoordinate co = info.coordinate();
    QString lat = QString::number(co.latitude());
    QString lng = QString::number(co.longitude());
    return (lat + ","+ lng);
}

void MainWindow::disable()
{
    if (source)
    {
        source->stopUpdates();
        disconnect(source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                   this, SLOT(positionUpdated(QGeoPositionInfo)));
    }
}

void MainWindow::enable()
{
    if (source)
    {
        connect(source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                this, SLOT(positionUpdated(QGeoPositionInfo)));
        source->startUpdates();
    }
}

MainWindow::~MainWindow()
{
    //delete ui;
}

void MainWindow::setOrientation(ScreenOrientation orientation)
{
#ifdef Q_OS_SYMBIAN
    if (orientation != ScreenOrientationAuto) {
#if defined(ORIENTATIONLOCK)
        const CAknAppUiBase::TAppUiOrientation uiOrientation =
                (orientation == ScreenOrientationLockPortrait) ? CAknAppUi::EAppUiOrientationPortrait
                    : CAknAppUi::EAppUiOrientationLandscape;
        CAknAppUi* appUi = dynamic_cast<CAknAppUi*> (CEikonEnv::Static()->AppUi());
        TRAPD(error,
            if (appUi)
                appUi->SetOrientationL(uiOrientation);
        );
        Q_UNUSED(error)
#else // ORIENTATIONLOCK
        qWarning("'ORIENTATIONLOCK' needs to be defined on Symbian when locking the orientation.");
#endif // ORIENTATIONLOCK
    }
#elif defined(Q_WS_MAEMO_5)
    Qt::WidgetAttribute attribute;
    switch (orientation) {
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_Maemo5PortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_Maemo5LandscapeOrientation;
        break;
    case ScreenOrientationAuto:
    default:
        attribute = Qt::WA_Maemo5AutoOrientation;
        break;
    }
    setAttribute(attribute, true);
#else // Q_OS_SYMBIAN
    Q_UNUSED(orientation);
#endif // Q_OS_SYMBIAN
}

void MainWindow::showExpanded()
{
#ifdef Q_OS_SYMBIAN
    showFullScreen();
#elif defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
    showMaximized();
#else
    show();
#endif
}
