import csv
import re
import system_functions as sf

import dev_bluetooth

ID_LENGTH = 4
DIST_LENGTH = 3
PEER_LENGTH = ID_LENGTH + DIST_LENGTH

BT_TABLE_FILENAME = 'rssi_table_bt.csv'
WIFI_TABLE_FILENAME = 'rssi_table_wifi.csv'

import time
PROCESS_RETRY_WAIT_TIME = .200


class DeviceNotFoundError(Exception):

    def __init__(self, device):
        self.device = device


def ID_from_MAC(MAC):
    """converts MAC addr to acceptable ID_LENGTH

    Examples:
    >>> ID_from_MAC('12:34:56:78:90:AB')
    '90AB'
    """

    matches = re.search("(([0-9A-F]{2}):?([0-9A-F]{2}):?){3}", MAC)
    if matches is None:
        print "Invalid MAC"
        return ""
    ID = matches.group(2) + matches.group(3)
    return ID


def parse_ident(n, ng):
    """n = name, ng = neighbor graph

    Examples:
    >>> n = 'abcdEFGH090IJKL123MNOP456QRST789'
    >>> ng = {}
    >>> parse_ident(n, ng)
    'abcd'
    >>> ng
    {'abcd': {'EFGH': 90, 'IJKL': 123, 'MNOP': 456, 'QRST': 789}}

    >>> ng = {}
    >>> parse_ident('Illinoisnet', ng)
    'Illi'
    >>> ng
    {'Illi': {}}

    >>> parse_ident('', {})
    '0000'
    """

    if not (len(n) - ID_LENGTH) % PEER_LENGTH == 0:
        return sanitize_peer_id(n)

    peer = n[:ID_LENGTH]
    n = n[ID_LENGTH:] #slice ID off
    ng[peer] = {}
    try:
        while len(n) >= PEER_LENGTH:
            ng[peer][n[:ID_LENGTH]] = int(n[ID_LENGTH:PEER_LENGTH])
            n = n[PEER_LENGTH:] #slice peer off
    except ValueError:
        pass
    return peer


def sanitize_peer_id(peer_id):
    """ Takes long name and creates ID of proper length

    Examples:
    >>> sanitize_peer_id('IllinoisNet')
    'Illi'
    >>> sanitize_peer_id('')
    '    '
    >>> sanitize_peer_id('123')
    '123 '
    """
    peer_id = str(peer_id)
    if len(peer_id) >= ID_LENGTH:
        return peer_id[:ID_LENGTH]
    else:
        while len(peer_id) < ID_LENGTH:
            peer_id += " "
    return peer_id


def sanitize_peer_dist(peer_dist):
    """ Takes integer distance and produces proper length value

    Examples:
    >>> sanitize_peer_dist(123)
    '123'
    >>> sanitize_peer_dist(1)
    '001'
    >>> sanitize_peer_dist(10000)
    '999'
    """
    if peer_dist >= 999:
        return '999'
    return '%03d' % peer_dist


class TranslateTable(object):
    """Data structure for looking up RSSI data"""

    def _parse(self, csvFilename):
        rssiTable = {}
        rssiReader = csv.reader(open(csvFilename, 'rb'), delimiter=' ')
        for row in rssiReader:
            rssiTable[int(row[0])] = (float(row[1]), int(row[3]), int(row[5]))
        return rssiTable

    def lookup(self, rssiValue):
        if rssiValue in self.table:
            return self.table[rssiValue]
        else:
            print "Value", rssiValue, "missing from table", self.filename
            return (0, 0, 0)

    def __init__(self, filename):
        self.filename = filename
        self.table = self._parse(filename)


class Radio_Module(object):
    """ Generic Class for Radio Components """

    class stats:
        turned_on_count=0

    def getIdentifier(self):
        return self._ident

    def scan():
        assert "Method not implemented by derived class"

    def setIdentifier(name):
        assert "Method not implemented by derived class"

    def turnOn():
        assert "Method not implemented by derived class"

    def turnOff():
        assert "Method not implemented by derived class"

    def __init__(self, ID):
        pass


class GPS_Module(Radio_Module):
    pass


class BT_Module(Radio_Module):
    """ BT-specific Radio Component"""

    def _translate(self, value):
        return self.translateTable.lookup(value)

    def scan(self):
        """Provides a mapping between neighbors and distance"""

        results = {}
        parsed_results = {}

        #returns Name -> (addr,rssi)
        devices = self._device.scan()

        for device in devices:
            (MAC, entry) = (devices[device][0], str(device))
            entry = parse_ident(entry, parsed_results)
            results[entry] = self._translate(devices[device][1])

        return results, parsed_results

    def setIdentifier(self, pairs):
        """sets name which is broadcast to network
        """
        self._ident = self._ID
        for k, v in pairs.iteritems():
            self._ident += k+sanitize_peer_dist(v)

        self._device.setName(self._ident)
        print "Set BT ident to " + self._ident

    def turnOn(self):
        self._device.setPowerOn()
        self._power_state = True

    def turnOff(self):
        self._device.setPowerOff()
        self._power_state = False

    def __init__(self, ID):
        super(BT_Module, self).__init__(ID)
        self._device = dev_bluetooth.Device()
        if not self._device.exists():
            raise DeviceNotFoundError("Bluetooth")
        self._ID = sanitize_peer_id(ID)
        self.setIdentifier({})
        self.translateTable = TranslateTable(BT_TABLE_FILENAME)


class WIFI_Module(Radio_Module):
    """
    Wifi-specific Radio Component
    """

    def _translate(self, value):
        return self.translateTable.lookup(value)

    def device_reset(self):
        self.turnOff()
        try:
            sf.run("iwconfig wlan0 mode Ad-Hoc")
        except sf.RunProcessError:
            print "Setting WIFI to AdHoc failed"
            raise OSError
        self.turnOn()

    def scan(self):
        """
        Provides a mapping between neighbors and distance
        """

        results = {}
        parsed_results = {}
        error = True
        while error:
            try:
                output = sf.run("iwlist wlan0 scan")
                error = False
            except sf.RunProcessError:
                self.device_reset()

        for line in output:
            matches = re.search("Address: (.*)", line)
            if matches:
                MAC = matches.group(1)
                continue

            matches = re.search("ESSID:\"(.*)\"", line)
            if matches:
                ESSID = matches.group(1)
                if ESSID == "":
                    ID = ID_from_MAC(MAC)
                else:
                    ID = parse_ident(ESSID, parsed_results)
                continue

            matches = re.search("Signal level:(-?\d+) dBm", line)
            if matches:
                results[ID] = self._translate(int(matches.group(1)))
                continue
        return results, parsed_results

    def setIdentifier(self, pairs):
        """sets name which is broadcast to network
        """
        WIFI_MAX_IDENT_LENGTH = 32
        self._ident = self._ID
        for k, v in pairs.iteritems():
            if len(self._ident)+len(k)+len(str(v)) < WIFI_MAX_IDENT_LENGTH:
                self._ident += k+sanitize_peer_dist(v)
        try:
            sf.run("iwconfig wlan0 essid \"" + self._ident + "\"")
            print "Set Wifi ident to " + self._ident
        except sf.RunProcessError:
            print "Wifi setESSID failed"
            raise OSError()

    def turnOn(self, numRetries=1):
        try:
            sf.run("ifconfig wlan0 up")
            self._power_state = True
        except sf.RunProcessError, e:
            if numRetries > 0:
                time.sleep(PROCESS_RETRY_WAIT_TIME)
                self.turnOn(numRetries = numRetries - 1)
            else:
                print "Wifi turnOn failed"
                raise OSError(e)

    def turnOff(self, numRetries=1):
        try:
            sf.run("ifconfig wlan0 down")
            self._power_state = False
        except sf.RunProcessError, e:
            if numRetries > 0:
                time.sleep(PROCESS_RETRY_WAIT_TIME)
                self.turnOn(numRetries = numRetries - 1)
            else:
                print "Wifi turnOff failed"
                raise OSError(e)

    def __init__(self, ID):
        try:
            sf.run("ifconfig wlan0")
        except sf.RunProcessError:
            raise DeviceNotFoundError("Wifi")

        super(WIFI_Module, self).__init__(ID)
        self._ID = sanitize_peer_id(ID)
        self.setIdentifier({})
        self.device_reset()
        self.translateTable = TranslateTable(WIFI_TABLE_FILENAME)


class GSM_Module(Radio_Module):
    pass


class ZIGBEE_Module(Radio_Module):
    """Component to run Zigbee attached dongle"""

    pass

if __name__ == "__main__":
    import doctest
    doctest.testmod(verbose=True)
