#include "notificationsdialog.h"
#include "ui_notificationsdialog.h"
#include "systemnotification.h"

SystemNotification *m_notifier = 0;

NotificationsDialog::NotificationsDialog(QWidget *parent, QString token) :
    QDialog(parent),
    ui(new Ui::NotificationsDialog),
    accessToken(token),
    nam(new QNetworkAccessManager(this)),
    qfacebook(new QFacebook(token, this))
{
    ui->setupUi(this);
    isFetching = false;
    m_notifier = new SystemNotification(this);
    this->orientationChanged();
    this->updateNotifications();
    this->getNoficiationCount();
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

NotificationsDialog::~NotificationsDialog()
{
    m_notifier->clearNotifications();
    delete ui;
}

void NotificationsDialog::orientationChanged()
{
#ifdef Q_WS_MAEMO_5
    setMinimumHeight(QApplication::desktop()->screenGeometry().height());
#endif
}

void NotificationsDialog::getNoficiationCount()
{
    QString url = QString("https://api.facebook.com/method/notifications.get&access_token=%1").arg(accessToken);
    countReply = nam->get(QNetworkRequest(url));
    if (countReply)
        connect(countReply, SIGNAL(finished()), this, SLOT(onGotNotificationsCount()));
}

void NotificationsDialog::onGotNotificationsCount()
{
    if (countReply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch notifications!";
        countReply->deleteLater();
        countReply = 0;
        return;
    }

    QXmlStreamReader xml(countReply->readAll());
    while (!xml.atEnd() && !xml.hasError()) {
        QXmlStreamReader::TokenType token = xml.readNext();
        /* If token is just StartDocument, we'll go to next.*/
        if (token == QXmlStreamReader::StartDocument) {
                continue;
        }
        /* If token is StartElement, we'll see if we can read it.*/
        if (token == QXmlStreamReader::StartElement) {
            //if (xml.name() == "notification_counts")

        }
    }
    /* Error handling. */
    if (xml.hasError()) {
        QMessageBox::critical(this,
                              "QXSRExample::parseXML",
                              xml.errorString(),
                              QMessageBox::Ok);
    }
    /* Removes any device() or data from the reader
     * and resets its internal state to the initial state. */
    xml.clear();
}

void NotificationsDialog::updateNotifications()
{
    if (isFetching)
        return;

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif

    QString url;
    url = QString("https://api.facebook.com/method/notifications.getList&include_read=true&access_token=%1").arg(accessToken);
    reply = nam->get(QNetworkRequest(url));
    if (reply) {
        connect(reply, SIGNAL(finished()), this, SLOT(onNotificationsListReceived()));
        isFetching = true;
    }
}

void NotificationsDialog::onNotificationsListReceived()
{
    isFetching = false;

    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch notifications!";
        this->reply->deleteLater();
        this->reply = 0;

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif

        return;
    }

    QByteArray notificationsArray = reply->readAll();
    if (notificationsArray == m_notifications) {
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
        return;
    } else {
        m_notifications = notificationsArray;
    }

    if (!notificationItems.isEmpty()) {
        foreach (NotificationItem *item, notificationItems) {
            ui->layout->removeWidget(item);
            notificationItems.removeOne(item);
            delete item;
        }
    }

    QList< QMap<QString,QString> > notifications;
    QXmlStreamReader xml(notificationsArray);

    while (!xml.atEnd() && !xml.hasError()) {
        QXmlStreamReader::TokenType token = xml.readNext();
        /* If token is just StartDocument, we'll go to next.*/
        if (token == QXmlStreamReader::StartDocument) {
                continue;
        }
        /* If token is StartElement, we'll see if we can read it.*/
        if (token == QXmlStreamReader::StartElement) {
            if (xml.name().toString().contains("notifications list"))
                continue;
            if (xml.name() == "notification")
                notifications.append(this->parseNotification(xml));
        }
    }
    /* Error handling. */
    if (xml.hasError()) {
        qDebug() << xml.errorString();
    }
    /* Removes any device() or data from the reader
     * and resets its internal state to the initial state. */
    xml.clear();
    this->addNotificationsToUi(notifications);
}

QMap <QString, QString> NotificationsDialog::parseNotification(QXmlStreamReader &xml)
{
    QMap <QString, QString> notification;
    while (!(xml.tokenType() == QXmlStreamReader::EndElement &&
                xml.name() == "notification")) {
        if(xml.tokenType() == QXmlStreamReader::StartElement) {
                if (xml.name() == "notification_id")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "sender_id")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "title_text")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "body_text")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "created_time")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "object_id")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "object_type")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "icon_url")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "is_unread")
                    this->addElementDataToMap(xml, notification);
                if (xml.name() == "href")
                    this->addElementDataToMap(xml, notification);
        }
        xml.readNext();
    }
    return notification;
}

void NotificationsDialog::addElementDataToMap(QXmlStreamReader& xml, QMap<QString, QString>& map) const
{
    if (xml.tokenType() != QXmlStreamReader::StartElement) {
        return;
    }
    QString elementName = xml.name().toString();
    xml.readNext();

    if (xml.tokenType() != QXmlStreamReader::Characters) {
        return;
    }
    map.insert(elementName, xml.text().toString());
}

void NotificationsDialog::addNotificationsToUi(QList< QMap<QString,QString> >& notifications)
{
    unreadCount = 0;
    m_unreadNotifications.clear();
    m_notifier->clearNotifications();
    while (!notifications.isEmpty()) {
        QMap<QString,QString> notification = notifications.takeFirst();
        QString text = notification["title_text"];
        if (!text.isEmpty()) {
            NotificationItem *item = new NotificationItem(this, accessToken, nam);
            connect(item, SIGNAL(markedAsRead(NotificationItem*)), this, SLOT(onMarkedAsRead(NotificationItem*)));
            connect(item, SIGNAL(clicked(NotificationItem*)), this, SLOT(onNotificationClicked(NotificationItem*)));
            QString objectType = notification["object_type"];
            if (objectType == "photo") {
                // Not sure which ID facebook returns in Object ID, but it doesn't seem valid
                QString href = notification["href"];
                href = href.remove("http://www.facebook.com/photo.php?fbid=");
                int position = href.indexOf("&");
                href.remove(position, href.length()-position);
                item->setObjectId(href);
            } else {
                item->setObjectId(notification["object_id"]);
            }

            item->setObjectType(objectType);
            item->setUserId(notification["sender_id"]);
            item->setText(notification["title_text"]);
            item->setNotificationId(notification["notification_id"]);
            bool isUnread = false;
            if (notification["is_unread"].toInt() == 1) {
                isUnread = true;
                m_unreadNotifications.append(item);
                m_notifier->createNotification("Facebook", item->text(), "general_facebook");
                m_notifier->vibrate();
                unreadCount++;
            }
            if (unreadCount > 0)
                m_notifier->createSoundNotification();
            item->setUnread(isUnread);
            ui->layout->addWidget(item);
            notificationItems.append(item);
        }
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
    emit gotNotificationsCount(unreadCount);
}

void NotificationsDialog::onNotificationClicked(NotificationItem *item)
{
    item->markAsRead();
    QString type = item->objectType();
    if (type == "stream") {
        SinglePostWindow *post = new SinglePostWindow(this->parentWidget(), accessToken);
        post->loadPost(item->objectId());
#ifdef Q_WS_S60
        post->showMaximized();
#else
        post->show();
#endif
    } else if (type == "group") {
        ProfileWindow *group = new ProfileWindow(this->parentWidget(), accessToken, ProfileWindow::GroupProfile);
        group->browseProfile(item->objectId());
        group->setWindowTitle(tr("Group"));
#ifdef Q_WS_S60
        group->showMaximized();
#else
        group->show();
#endif
    } else if (type == "photo") {
        PhotoWindow *photo = new PhotoWindow(this->parentWidget(), accessToken);
        photo->showPhotoFromId(item->objectId());
        photo->setWindowTitle(tr("Photo"));
#ifdef Q_WS_S60
        photo->showMaximized();
#else
        photo->show();
#endif
    } else if (type == "friend") {
        ProfileWindow *profile = new ProfileWindow(this->parentWidget(), accessToken);
        profile->browseProfile(item->objectId());
        profile->setWindowTitle(tr("Profile"));
#ifdef Q_WS_S60
        profile->showMaximized();
#else
        profile->show();
#endif
    }
}

void NotificationsDialog::markRead()
{
    foreach (NotificationItem *item, m_unreadNotifications) {
        item->markAsRead();
    }
}

void NotificationsDialog::onMarkedAsRead(NotificationItem *item)
{
    m_unreadNotifications.removeOne(item);
    emit gotNotificationsCount(m_unreadNotifications.count());
}

void NotificationsDialog::showEvent(QShowEvent *)
{
    this->updateNotifications();
    m_notifier->clearNotifications();
    // Delay setting as read for a while, gives the user a chance to see which notificaions were unread
    this->markRead();
}
