#include "profilewindow.h"
#include "ui_profilewindow.h"
#include "albumswindow.h"

ProfileWindow::ProfileWindow(QWidget *parent, QString token, ProfileTypes profileType, bool deleteOnClose) :
    QMainWindow(parent),
    ui(new Ui::ProfileWindow),
    qfacebook(new QFacebook(token, this)),
    accessToken(token),
    nam(new QNetworkAccessManager(this)),
    m_deleteOnClose(deleteOnClose)
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    m_profileType = profileType;
    if (profileType == ProfileWindow::GroupProfile) {
        ui->menuOptions->addAction(tr("Members"), this, SLOT(showFriends()));
    } else if (profileType == ProfileWindow::FriendProfile) {
        // Argh, facebook's public API sucks! You can't browse someone's friends with it :/
        // ui->menuOptions->addAction(tr("Friends"), this, SLOT(showFriends()));
    }
    connect(ui->writeOnWallEdit, SIGNAL(clicked()), this, SLOT(onTextBoxClicked()));
    connect(ui->actionRefresh, SIGNAL(triggered()), this, SLOT(updateProfile()));
    connect(ui->userAvatar, SIGNAL(clicked()), this, SLOT(onAvatarClicked()));
    connect(ui->actionPhotos, SIGNAL(triggered()), this, SLOT(onPhotosClicked()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    m_myId = QSettings().value("main/myId").toString();
    this->orientationChanged();
    bool whiteMode = QSettings().value("main/white-mode", false).toBool();
    if (whiteMode) {
        QPalette p(palette());
        p.setColor(QPalette::Background, QColor(220, 225, 235));
        setPalette(p);
    }
}

ProfileWindow::~ProfileWindow()
{
    delete ui;
}

void ProfileWindow::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    ui->scrollAreaWidgetContents->setMaximumWidth(screenGeometry.width()-10);
}

void ProfileWindow::browseProfile(QString id)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    this->profileId = id;
    if (id != "me" && m_deleteOnClose)
        setAttribute(Qt::WA_DeleteOnClose);
    QString url = QString("https://graph.facebook.com/%1/picture?access_token=%2&type=normal").arg(id, accessToken);
    this->avatarReply = nam->get(QNetworkRequest(url));
    connect(avatarReply, SIGNAL(finished()), this, SLOT(onAvatarReceived()));

    this->infoReply = qfacebook->getObject(id);
    if (this->infoReply)
        connect(infoReply, SIGNAL(finished()), this, SLOT(onInfoReplyReceived()));

    this->reply = qfacebook->getConnections(this->profileId, "feed");
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyReceived()));
}

void ProfileWindow::showFriends()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    QString connection;
    if (m_profileType == ProfileWindow::FriendProfile)
        connection = "friends";
    else if (m_profileType == ProfileWindow::GroupProfile)
        connection = "members";
    friendsReply = qfacebook->getConnections(profileId, connection);
    if (friendsReply)
        connect(friendsReply, SIGNAL(finished()), this, SLOT(onGotFriendsList()));
}

void ProfileWindow::onGotFriendsList()
{
    QDialog *dialog = new QDialog(this);
    QVBoxLayout *layout = new QVBoxLayout(dialog);
    QListWidget *listWidget = new QListWidget(this);
    listWidget->setSortingEnabled(true);
    connect(listWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    layout->addWidget(listWidget);
    dialog->setLayout(layout);
#ifdef Q_WS_MAEMO_5
    dialog->setMinimumHeight(600);
#endif

    QVariant jsonData = friendsReply->data();
    qDebug() << jsonData;
    QVariantList listData = jsonData.toMap().value("data").toList();
    foreach (jsonData, listData) {
        QListWidgetItem *item = new QListWidgetItem(listWidget);
        item->setData(Qt::UserRole, jsonData.toMap().value("id").toString());
        item->setText(jsonData.toMap().value("name").toString());
    }

    if (m_profileType == ProfileWindow::FriendProfile)
        dialog->setWindowTitle(tr("Friends"));
    else if (m_profileType == ProfileWindow::GroupProfile)
        dialog->setWindowTitle(tr("Members"));
    dialog->setAttribute(Qt::WA_DeleteOnClose);
    dialog->show();

    friendsReply->deleteLater();
    friendsReply = 0;
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

NewsFeedPost* ProfileWindow::getPostById(QString id)
{
    if (!posts.contains(id)) {
        NewsFeedPost *post = new NewsFeedPost(this, accessToken, qfacebook, nam);
        connect(post, SIGNAL(destroyed(QObject*)), this, SLOT(onNewsFeedPostDestroyed(QObject*)));
        connect(post, SIGNAL(nameClicked(QString,QString)), this, SLOT(onNameClicked(QString,QString)));
        connect(post, SIGNAL(commentButtonClicked(QString,QString,QString)),
                this, SLOT(onCommentButtonClicked(QString,QString,QString)));
        posts.insert(id, post);
        return post;
    } else {
        return posts.value(id);
    }
}

void ProfileWindow::onFriendClicked(QListWidgetItem *item)
{
    ProfileWindow *profile = new ProfileWindow(this, accessToken);
    profile->setWindowTitle(item->text());
    profile->browseProfile(item->data(Qt::UserRole).toString());
#ifdef Q_WS_S60
    profile->showMaximized();
#else
    profile->show();
#endif
}

void ProfileWindow::onInfoReplyReceived()
{
    if (this->infoReply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch profile info!";
        this->infoReply->deleteLater();
        this->infoReply = 0;

        return;
    }

    QVariant jsonData = this->infoReply->data();
    QString name = jsonData.toMap().value("name").toString();
    if (!name.isEmpty())
        this->setWindowTitle(name);
    QString birthday = jsonData.toMap().value("birthday").toString();
    QString fromLocation = jsonData.toMap().value("hometown").toMap().value("name").toString();
    QString currentLocation = jsonData.toMap().value("location").toMap().value("name").toString();
    QString relationshipStatus = jsonData.toMap().value("relationship_status").toString();

    if (relationshipStatus.isEmpty())
        ui->relationshipLabel->hide();
    else
        ui->relationshipLabel->setText(relationshipStatus);

    if (fromLocation.isEmpty())
        ui->fromLabel->hide();
    else
        ui->fromLabel->setText(tr("From %1").arg(fromLocation));

    if (currentLocation.isEmpty())
        ui->locationLabel->hide();
    else
        ui->locationLabel->setText(tr("Lives in %1").arg(currentLocation));

    if (birthday.isEmpty())
        ui->birthdateLabel->hide();
    else
        ui->birthdateLabel->setText(tr("Born on %1").arg(birthday));
}

void ProfileWindow::onAvatarClicked()
{
    PhotoWindow *photo = new PhotoWindow(this, accessToken);
    photo->showPhoto(profileId);
    photo->setWindowTitle(this->windowTitle() + " - " + tr("Profile picture"));
    photo->showMaximized();
}

void ProfileWindow::updateProfile()
{
    this->browseProfile(this->profileId);
}

void ProfileWindow::onPhotosClicked()
{
    AlbumsWindow *albums = new AlbumsWindow(this, accessToken);
    albums->setAttribute(Qt::WA_DeleteOnClose);
    albums->browseAlbums(profileId);
#ifdef Q_WS_S60
    albums->showMaximized();
#else
    albums->show();
#endif
}

void ProfileWindow::onAvatarReceived()
{
    QUrl redir = this->avatarReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        this->avatarReply->deleteLater();
        this->avatarReply = nam->get(QNetworkRequest(redir));

        connect(this->avatarReply, SIGNAL(finished()), this, SLOT(onAvatarReceived()));
        return;
    }
    this->setAvatar(QPixmap::fromImage(QImage::fromData(this->avatarReply->readAll())));
    //this->avatarReply->deleteLater();
}

void ProfileWindow::setAvatar(QPixmap avatar)
{
    ui->userAvatar->setIcon(QIcon(avatar));
}

void ProfileWindow::onNewsFeedPostDestroyed(QObject *object)
{
    posts.remove(qobject_cast<NewsFeedPost*>(object)->postId());
}

void ProfileWindow::onReplyReceived()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch news feed!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }
    QString myId = QSettings().value("main/myId").toString();

    QVariant jsonData = this->reply->data();
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();

    QVariantList reversedList;
    QListIterator<QVariant> it(listData);
    it.toBack();
    while (it.hasPrevious())
        reversedList << it.previous();

    foreach(jsonData, reversedList) {
        QString id = jsonData.toMap().value("id").toString();
        NewsFeedPost *post = getPostById(id);
        QString name = jsonData.toMap().value("from").toMap().value("name").toString();
        QString userId = jsonData.toMap().value("from").toMap().value("id").toString();
        if (userId.contains(myId))
            post->setDeletable(true);
        if (this->profileId == "me")
            post->setDeletable(true);
        QString message = jsonData.toMap().value("message").toString();
        QString icon = jsonData.toMap().value("icon").toString();
        QString time = jsonData.toMap().value("created_time").toString();
        QString objectId = jsonData.toMap().value("object_id").toString();
        QString objectType = jsonData.toMap().value("type").toString();
        QString link = jsonData.toMap().value("link").toString();
        QString sourceLink = jsonData.toMap().value("source").toString();
        QString postType = jsonData.toMap().value("type").toString();
        if (link.isEmpty() || (postType == "video" && sourceLink.contains("fbcdn")))
            link = sourceLink;
        if (!link.isEmpty()) {
            post->addLink(link,
                          jsonData.toMap().value("name").toString(),
                          jsonData.toMap().value("description").toString(),
                          jsonData.toMap().value("picture").toString());
        }
        QVariantList toVariant = jsonData.toMap().value("to").toMap().value("data").toList();
        if (!toVariant.isEmpty()) {
            QVariant to = toVariant.last();

            foreach (QVariant taggedName, toVariant) {
                QString name = taggedName.toMap().value("name").toString();
                QString id = taggedName.toMap().value("id").toString();
                if (message.contains(name)) {
                    message.replace(name, QString("<a href=\"profile://%1\">%2</a>").arg(id, name));
                }
            }
            post->setMessage(message);

            QString toId = to.toMap().value("id").toString();
            QString toName = to.toMap().value("name").toString();

            QString thisId = profileId;
            if (thisId == "me")
                thisId = m_myId;

            if (toName != name && toId != thisId)
                post->setToName(toName, toId);
        }

        // Check if the currently logged in user likes the post
        QVariantList likes = jsonData.toMap().value("likes").toMap().value("data").toList();
        QString myId = QSettings().value("main/myId").toString();
        bool meLikesThis = false;
        foreach (QVariant likeId, likes) {
            QVariantMap map = likeId.value<QVariantMap>();
            QString likerId = map.value("id").toString();
            if (likerId.contains(myId)) {
                meLikesThis = true;
                break;
            }
        }

        // Get like and comment count.
        QString likesCount = jsonData.toMap().value("likes").toMap().value("count").toString();
        if (likesCount.isEmpty())
            likesCount = "0";
        QString comments = jsonData.toMap().value("comments").toMap().value("count").toString();
        if (comments.isEmpty())
            comments = "0";

        // Set parsed data
        post->setName(name);
        post->setObjectId(objectId);
        post->setType(objectType);
        post->setMessage(message);
        post->setId(id);
        post->setUserId(userId);
        post->getPixmapFromUrl(QString("http://graph.facebook.com/%1/picture").arg(userId));
        post->setLikedByMe(meLikesThis);
        post->setNumberOfLikesComments(likesCount, comments);
        post->setIconFromUrl(icon);
        post->setTime(time);
        ui->scrollLayout->insertWidget(0, post);
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void ProfileWindow::onTextBoxClicked()
{
    NewStatusUpdateDialog *dialog = new NewStatusUpdateDialog(this, qfacebook, profileId);
    connect(dialog, SIGNAL(posted()), this, SLOT(updateProfile()));
    dialog->setWindowTitle(tr("Post to wall"));
#ifdef Q_WS_S60
    dialog->showMaximized();
#else
    dialog->show();
#endif
}

void ProfileWindow::onNameClicked(QString id, QString name)
{
    if (id == QSettings().value("main/myId").toString() || id == "me" || id == this->profileId)
        return;

    ProfileWindow *profile = new ProfileWindow(this, accessToken);
    profile->browseProfile(id);
    profile->setWindowTitle(name);
#ifdef Q_WS_S60
    profile->showMaximized();
#else
    profile->show();
#endif
}

void ProfileWindow::onCommentButtonClicked(QString id, QString comments, QString likes)
{
    qDebug() << id;
    SinglePostWindow *window = new SinglePostWindow(this, accessToken);
    window->loadPost(id);
    window->setCountLabel(comments, likes);
#ifdef Q_WS_S60
    window->showMaximized();
#else
    window->show();
#endif
}
