#include "albumwindow.h"
#include "ui_albumwindow.h"

AlbumWindow::AlbumWindow(QWidget *parent, QString token) :
    QMainWindow(parent),
    ui(new Ui::AlbumWindow),
    qfacebook(new QFacebook(token, this)),
    accessToken(token),
    nam(new QNetworkAccessManager(this)),
    progressDialog(0)
{
    ui->setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);
    connect(ui->actionUpload_photo, SIGNAL(triggered()), this, SLOT(uploadPhoto()));
    connect(ui->actionDownload_all_photos, SIGNAL(triggered()), this, SLOT(downloadAllPhotos()));
    ui->statusBar->hide();
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
}

AlbumWindow::~AlbumWindow()
{
    delete ui;
}

void AlbumWindow::browseAlbum(QString id, QString albumCount)
{
    this->albumId = id;
    this->albumCount = albumCount;
    this->updateAlbum();
}

void AlbumWindow::uploadPhoto()
{
    ImageUploadDialog *dialog = new ImageUploadDialog(this);
    connect(dialog, SIGNAL(uploadImages(QStringList)), this, SLOT(uploadPhotos(QStringList)));
#ifdef Q_WS_S60
    dialog->showMaximized();
#else
    dialog->show();
#endif
}

void AlbumWindow::updateAlbum()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    this->reply = qfacebook->getObject(this->albumId + QString("/photos&limit=0"));
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyReceived()));
}

void AlbumWindow::onReplyReceived()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch album!";
        this->reply->deleteLater();
        this->reply = 0;
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
        return;
    }

    foreach (PhotoItem *item, photos) {
        ui->gridLayout->removeWidget(item);
        sourceLinks.clear();
        delete item;
    }

    column = 0;
    row = 0;

    QVariant jsonData = this->reply->data();
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();
    foreach(jsonData, listData) {
        // One image at a time...
        QVariantList imagesList = jsonData.toMap().value("images").toList();
        QString id = jsonData.toMap().value("id").toString();
        QString userId = jsonData.toMap().value("from").toMap().value("id").toString();
        //QString userName = jsonData.toMap().value("from").toMap().value("name").toString();
        QString normalUrl;
        QString smallUrl;
        QString thumbUrl;
        QString mediumUrl;
        QString sourceUrl = jsonData.toMap().value("source").toString();
        foreach (QVariant imageVariant, imagesList) {
            QString imageUrl = imageVariant.toMap().value("source").toString();
            if (imageUrl.endsWith("n.jpg"))
                normalUrl = imageUrl;
            else if (imageUrl.endsWith("s.jpg"))
                smallUrl = imageUrl;
            else if (imageUrl.endsWith("t.jpg"))
                thumbUrl = imageUrl;
            else if (imageUrl.endsWith("a.jpg"))
                mediumUrl = imageUrl;
        }
        sourceLinks.append(normalUrl);
        PhotoItem *item = new PhotoItem(this, accessToken, nam);
        item->setComments(jsonData.toMap().value("comments").toMap().value("data").toList());
        item->setTags(jsonData.toMap().value("tags").toMap().value("data").toList());
        connect(item, SIGNAL(clicked(PhotoItem*)), this, SLOT(onPhotoItemClicked(PhotoItem*)));
        item->setPixmapFromUrl(mediumUrl);
        item->setLargePhotoLink(normalUrl);
        item->setSmallPhotoLink(smallUrl);
        item->setPhotoId(id);
        item->setUserId(userId);
        item->setSourcePhotoLink(sourceUrl);
        if (column > MAX_COLUMNS) {
            column = 0;
            row++;
        }
        ui->gridLayout->addWidget(item, row, column);
        photos.append(item);
        column++;
    }

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif

    reply->deleteLater();
}

void AlbumWindow::onPhotoItemClicked(PhotoItem *item)
{
    PhotoWindow *photo = new PhotoWindow(this, accessToken);
    photo->setPixmapFromUrl(item->largeUrl());
    photo->setWindowTitle(tr("Photo"));
    photo->setComments(item->comments());
    photo->setTags(item->tags());
    photo->setPhotoId(item->photoId());
#ifdef Q_WS_S60
    photo->showMaximized();
#else
    photo->show();
#endif
}

void AlbumWindow::downloadAllPhotos()
{
    if (sourceLinks.isEmpty()) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr("Unable to download: album is empty"));
#else
        QMessageBox::warning(this, tr("Error"), tr("Unable to download: album is empty"));
#endif
        return;
    }

    setAttribute(Qt::WA_DeleteOnClose, false);
    m_finishedDownloads = 0;
    m_imagesCount = sourceLinks.count();
    m_downloadLocation = QFileDialog::getExistingDirectory(this, tr("Select folder to save images to"),
                                                           QDir::homePath());
    if (m_downloadLocation.isEmpty()) {
        setAttribute(Qt::WA_DeleteOnClose, true);
        return;
    }

    if (ui->statusBar->isHidden())
        ui->statusBar->show();
    ui->statusBar->showMessage(tr("%1/%2 photos downloaded").arg(m_finishedDownloads, m_imagesCount));

    foreach (QString url, sourceLinks) {
        QNetworkReply *reply = nam->get(QNetworkRequest(url));
        connect(reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
    }
}

void AlbumWindow::onDownloadFinished()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    QString url = reply->url().toString();
    int pos = url.lastIndexOf("/");
    QString baseName = url.remove(0, pos);
    baseName.remove("/");
    QString fileName = m_downloadLocation + "/" + baseName;
    QFile image(fileName, this);
    if (image.open(QFile::WriteOnly)) {
        image.write(reply->readAll());
        image.close();
    }
    m_finishedDownloads++;

    ui->statusBar->showMessage(tr("%1/%2 photos downloaded").arg(m_finishedDownloads, m_imagesCount));

    if (m_finishedDownloads == m_imagesCount) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr("Download complete"));
#else
        QMessageBox::information(this, tr("Success"), tr("Album downloaded"));
#endif
        setAttribute(Qt::WA_DeleteOnClose, true);
        ui->statusBar->hide();
        if (!isVisible())
            this->deleteLater();
        return;
    }
}

void AlbumWindow::uploadPhotos(QStringList files)
{
    foreach (QString filename, files) {
        QFileInfo info(filename);

        QString boundary = "---------------------------193971182219750";

        QByteArray datas(QString("--" + boundary + "\r\n").toAscii());
        datas += "Content-Disposition: form-data; name=\"" + info.baseName() + "\"; filename=\"" + info.baseName() + "\"\r\n";
        datas += "Content-Type: application/octet-stream\r\n\r\n";

        QFile file(filename);
        if (!file.open(QIODevice::ReadOnly))
            return;

        datas += file.readAll();
        datas += "\r\n";
        datas += QString("--" + boundary + "\r\n").toAscii();

        QString url = QString("https://graph.facebook.com/%1/photos?access_token=" + accessToken).arg(albumId);
        QNetworkRequest request(url);
        request.setAttribute(QNetworkRequest::User, info.baseName());
        request.setRawHeader("Content-Type", QString("multipart/form-data; boundary=" + boundary).toUtf8());
        request.setRawHeader("Content-Length", QString::number(datas.length()).toUtf8());

        QNetworkReply *reply = nam->post(request, datas);
        replyMap.insert(reply, info.baseName());
        connect(reply, SIGNAL(finished()), this, SLOT(onUploadFinished()));
        connect(reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(uploadProgressChanged(qint64,qint64)));
    }
}

void AlbumWindow::onUploadFinished()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error() != QNetworkReply::NoError || reply->error() != QNetworkReply::UnknownContentError) {
        qDebug() << reply->errorString();
        qDebug() << reply->readAll();
    }
}

void AlbumWindow::uploadProgressChanged(qint64 bytesSent, qint64 bytesTotal)
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (!progressDialog) {
        progressDialog = new ProgressDialog(this);
        ui->menubar->addAction(tr("Show upload progress"), progressDialog, SLOT(show()));
        progressDialog->show();
    }
    double sent = bytesSent;
    double total = bytesTotal;
    double percentagedouble = (sent/total) * 100;
    int percentage = percentagedouble;
    QString name = replyMap.value(reply);
    qDebug() << name << percentage;
    progressDialog->showProgress(name, percentage);
}
