#include "youtubeplayer.h"
#include "ui_youtubeplayer.h"

YoutubePlayer::YoutubePlayer(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::YoutubePlayer),
    m_videoWidget(new VideoWidget(this)),
    m_player(new QMediaPlayer(this))
{
    ui->setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);

    volumeTimer = new QTimer(this);
    volumeTimer->setInterval(3000);
    ui->volumeSlider->hide();

    ui->verticalLayout->addWidget(m_videoWidget);
    m_player->setVideoOutput(m_videoWidget);
    ui->volumeSlider->setValue(m_player->volume());
    connect(this, SIGNAL(gotVideoUrl(QString)), this, SLOT(onGotVideoUrl(QString)));
    connect(m_videoWidget, SIGNAL(doubleClicked()), this, SLOT(onVideoWidgetDoubleClicked()));
    connect(m_videoWidget, SIGNAL(fullScreenChanged(bool)), this, SLOT(onFullScreenChanged(bool)));
    connect(m_player, SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
    connect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    connect(m_player, SIGNAL(volumeChanged(int)), ui->volumeSlider, SLOT(setValue(int)));
    connect(m_player, SIGNAL(seekableChanged(bool)), ui->progressBar, SLOT(setEnabled(bool)));
    connect(m_player, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged(QMediaPlayer::State)));
    connect(ui->progressBar, SIGNAL(sliderMoved(int)), this, SLOT(onPositionSliderMoved(int)));
    connect(ui->volumeSlider, SIGNAL(valueChanged(int)), m_player, SLOT(setVolume(int)));
    connect(volumeTimer, SIGNAL(timeout()), this, SLOT(toggleVolumeSlider()));
    connect(ui->volumeSlider, SIGNAL(sliderPressed()), volumeTimer, SLOT(stop()));
    connect(ui->volumeSlider, SIGNAL(sliderReleased()), volumeTimer, SLOT(start()));
    connect(ui->volumeButton, SIGNAL(clicked()), this, SLOT(toggleVolumeSlider()));
    connect(ui->volumeButton, SIGNAL(clicked()), this, SLOT(volumeWatcher()));
    connect(ui->actionCopy_URL, SIGNAL(triggered()), this, SLOT(copyUrl()));
    connect(ui->actionOpen_in_browser, SIGNAL(triggered()), this, SLOT(openInBrowser()));
    connect(ui->playButton, SIGNAL(clicked()), this, SLOT(playPause()));

#ifdef Q_WS_MAEMO_5
    ui->volumeButton->setIcon(QIcon("/usr/share/icons/hicolor/64x64/hildon/mediaplayer_volume.png"));
    ui->playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
#endif

    //connect(m_player, SIGNAL(error(QMediaPlayer::Error)), this, SLOT(onErrorOccured(QMediaPlayer::Error)));
}

YoutubePlayer::~YoutubePlayer()
{
    delete ui;
}

/*
void YoutubePlayer::stateChanged(Phonon::State newState, Phonon::State oldState)
{
    switch (newState) {
        case Phonon::ErrorState:
        qDebug() << m_player->mediaObject()->errorString();
        break;
    }
}
*/

void YoutubePlayer::playPause()
{
    if (m_player->state() == QMediaPlayer::PlayingState)
        m_player->pause();
    else if (m_player->state() == QMediaPlayer::PausedState || m_player->state() == QMediaPlayer::StoppedState)
        m_player->play();
}

void YoutubePlayer::onGotVideoUrl(QString url)
{
    qDebug() << url;
    m_player->setMedia(QMediaContent(QUrl(url)));
    m_player->play();
}

void YoutubePlayer::onErrorOccured(QMediaPlayer::Error error)
{
    qDebug() << error;
}

/* ==================================================================== */
/* === Thanks to Stuart Howarth aka marxian for the code used here! === */
void YoutubePlayer::getVideoUrl(const QString &videoId) {
    qDebug() << videoId;
    QString playerUrl = "http://www.youtube.com/get_video_info?&video_id=" + videoId + "&el=detailpage&ps=default&eurl=&gl=US&hl=en";
    QNetworkAccessManager *manager = new QNetworkAccessManager(this);
    QNetworkRequest request;
    request.setUrl(QUrl(playerUrl));
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(parseVideoPage(QNetworkReply*)));
    manager->get(request);
}

void YoutubePlayer::parseVideoPage(QNetworkReply *reply) {
    QNetworkAccessManager *manager = qobject_cast<QNetworkAccessManager*>(sender());
    m_cookiesList.clear();
    m_cookiesList = manager->cookieJar()->cookiesForUrl(reply->request().url());

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));
    response = response.split("fmt_stream_map=url=").at(1);
    QList<QString> parts = response.split(",url=");
    int key;
    for (int i = 0; i < parts.length(); i++) {
        QString part = parts[i];
        QString url(QByteArray::fromPercentEncoding(part.left(part.indexOf("&type=video")).toAscii()).replace("%2C", ","));
        key = part.split("&itag=").at(1).split("&").first().toInt();
        formats[key] = url;
    }
    QList<int> flist;
    flist << 22 << 35 << 34 << 18 << 5;
    QString videoUrl;
    int index = flist.indexOf(18);
    while ((videoUrl.isEmpty()) && index < flist.size()) {
        videoUrl = formats.value(flist.at(index), "");
        index++;
    }
    if (!videoUrl.startsWith("http")) {
        emit alert(tr("Error: Unable to retrieve video"));
        emit videoUrlError();
    }
    else {
        emit gotVideoUrl(videoUrl);
    }
//        qDebug() << videoUrl;
    reply->deleteLater();
    manager->deleteLater();
}
/* ==================================================================== */

void YoutubePlayer::onVideoWidgetDoubleClicked()
{
    m_videoWidget->setFullScreen(!m_videoWidget->isFullScreen());
}

void YoutubePlayer::toggleVolumeSlider()
{
    if (ui->volumeSlider->isHidden()) {
        ui->buttonWidget->hide();
        ui->volumeSlider->show();
    } else {
        ui->volumeSlider->hide();
        ui->buttonWidget->show();
        if (volumeTimer->isActive())
            volumeTimer->stop();
    }
}

void YoutubePlayer::setOriginalUrl(QString url)
{
    m_originalUrl = url;
}

void YoutubePlayer::volumeWatcher()
{
    if (!ui->volumeSlider->isHidden())
        volumeTimer->start();
}

void YoutubePlayer::onPositionChanged(qint64 position)
{
    m_position = position / 1000;
    if (m_videoWidget->isFullScreen())
        return;
    ui->progressBar->setValue(m_position);
    QTime t(0, 0);
    t = t.addMSecs(position);
    ui->currentPositionLabel->setText(t.toString("mm:ss"));
}

void YoutubePlayer::onDurationChanged(qint64 duration)
{
    m_duration = duration / 1000;
    if (m_duration != ui->progressBar->maximum())
        ui->progressBar->setMaximum(m_duration);
    QTime t(0, 0);
    t = t.addMSecs(duration);
    ui->videoLengthLabel->setText(t.toString("mm:ss"));
}

void YoutubePlayer::onFullScreenChanged(bool fullscreen)
{
    if (!fullscreen) {
        this->activateWindow();
        ui->progressBar->setValue(m_position);
        QTime t(0, 0);
        t = t.addMSecs(m_position);
        ui->currentPositionLabel->setText(t.toString("mm:ss"));
    }
}

void YoutubePlayer::onPositionSliderMoved(int position)
{
    qint64 pos = position * 1000;
    m_player->setPosition(pos);
}

void YoutubePlayer::showExpanded()
{
#ifdef Q_WS_S60
    showMaximized();
#else
    show();
#endif
}

void YoutubePlayer::keyPressEvent(QKeyEvent *e)
{
    if (e->key() == Qt::Key_Space) {
        this->playPause();
    } else if (e->key() == Qt::Key_Left) {
        m_player->setPosition((m_position-3)*1000);
    } else if (e->key() == Qt::Key_Right) {
        m_player->setPosition((m_position+3)*1000);
    }
}

void YoutubePlayer::keyReleaseEvent(QKeyEvent *)
{

}

void YoutubePlayer::copyUrl()
{
    QClipboard *cb = QApplication::clipboard();
    cb->setText(m_originalUrl);
}

void YoutubePlayer::openInBrowser()
{
    QDesktopServices::openUrl(QUrl(m_originalUrl));
    QTimer::singleShot(100, this, SLOT(close()));
}

void YoutubePlayer::onStateChanged(QMediaPlayer::State state)
{
    switch (state) {
    case QMediaPlayer::PlayingState:
#ifdef Q_WS_MAEMO_5
        ui->playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Pause.png"));
#endif
        break;
    case QMediaPlayer::PausedState:
#ifdef Q_WS_MAEMO_5
        ui->playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
#endif
        break;
    case QMediaPlayer::StoppedState:
#ifdef Q_WS_MAEMO_5
        ui->playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
#endif
        break;
    }
}
