#include "checkinwindow.h"
#include "ui_checkinwindow.h"

CheckinWindow::CheckinWindow(QWidget *parent, QString token) :
    QMainWindow(parent),
    ui(new Ui::CheckinWindow),
    accessToken(token),
    qfacebook(new QFacebook(token, this)),
    m_mapsWidget(new MapsWidget(this)),
    markerManager(0)
{
    ui->setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    ui->mapLayout->addWidget(m_mapsWidget);
    initializeMapWidget();
    connect(ui->tagFriendsEdit, SIGNAL(clicked()), this, SLOT(onTagFriendsClicked()));
    connect(ui->checkinButton, SIGNAL(clicked()), this, SLOT(onCheckinButtonClicked()));
    //connect(ui->checkinButton, SIGNAL(clicked()), this, SLOT(checkIn()));
}

CheckinWindow::~CheckinWindow()
{
    delete ui;
}

void CheckinWindow::initializeMapWidget()
{
    // check we have a valid default provider

    QList<QString> providers = QGeoServiceProvider::availableServiceProviders();
    if (providers.size() < 1) {
        QMessageBox::information(this, tr("Maps Demo"),
                                 tr("No service providers are available"));
        QCoreApplication::quit();
        return;
    }

    foreach (QString provider, providers) {
        serviceProvider = new QGeoServiceProvider(provider);
        if (serviceProvider->mappingManager() &&
                serviceProvider->searchManager() &&
                serviceProvider->routingManager())
            break;
    }

    if (serviceProvider->error() != QGeoServiceProvider::NoError) {
        QMessageBox::information(this, tr("Maps Demo"),
                                 tr("Error loading geoservice plugin"));
        QCoreApplication::quit();
        return;
    }

    if (!serviceProvider->mappingManager() ||
            !serviceProvider->searchManager() ||
            !serviceProvider->routingManager()) {
        QMessageBox::information(this, tr("Maps Demo"),
                                 tr("No geoservice found with mapping/search/routing"));
        QCoreApplication::quit();
        return;
    }

    // start initialising things (maps, searching, routing)

    m_mapsWidget->initialize(serviceProvider->mappingManager());
    m_mapsWidget->setStyleSheet("border-radius: 9px;");

    if (markerManager)
        delete markerManager;
    markerManager = new MarkerManager(serviceProvider->searchManager());
    m_mapsWidget->setMarkerManager(markerManager);
}

void CheckinWindow::setLocation(QString name, QString id, QString longitude, QString latitude)
{
    this->setWindowTitle(tr("Check into %1").arg(name));
    //ui->nameLabel->setText(name);
    m_locationName = name;
    m_placeId = id;
    m_longitude = longitude;
    m_latitude = latitude;
    m_mapsWidget->setMyLocation(QGeoCoordinate(latitude.toDouble(), longitude.toDouble()), true);
}

void CheckinWindow::onTagFriendsClicked()
{
    FriendsWindow *friends = new FriendsWindow(this, accessToken);
    friends->setAttribute(Qt::WA_DeleteOnClose);
    connect(friends, SIGNAL(recepientsSelected(QList<QListWidgetItem*>)), this, SLOT(onRecepientsSelected(QList<QListWidgetItem*>)));
    friends->setWindowFlags(Qt::Dialog);
    friends->requestRecepientsList();
}

void CheckinWindow::onRecepientsSelected(QList<QListWidgetItem*> list)
{
    ui->tagFriendsEdit->clear();
    friendNames.clear();
    friendIds.clear();
    foreach (QListWidgetItem *item, list) {
        friendNames << item->text();
        friendIds << item->data(Qt::UserRole).toString();
    }
    ui->tagFriendsEdit->setText(friendNames.join(", "));
}

void CheckinWindow::onCheckinButtonClicked()
{
    ui->tagFriendsEdit->setEnabled(false);
    ui->statusUpdateEdit->setEnabled(false);
    ui->checkinButton->setEnabled(false);

    QByteArray data;
    if (!ui->statusUpdateEdit->text().isEmpty())
        data.append("message=" + ui->statusUpdateEdit->text().toUtf8() + "&");
    data.append(QString("place=%1").arg(m_placeId));

    Place place(this);
    place.setLongitude(m_longitude);
    place.setLatitude(m_latitude);
    QVariantMap variant = QJson::QObjectHelper::qobject2qvariant(&place);
    QJson::Serializer serializer;

    data.append("&coordinates=" + serializer.serialize(variant));
    if (!ui->tagFriendsEdit->text().isEmpty())
        data.append(QString("&tags=%1").arg(friendIds.join(", ")));
    reply = qfacebook->putObject("me", "checkins", data);
    if (reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onCheckInReplyReceived()));
}

void CheckinWindow::onCheckInReplyReceived()
{
    ui->tagFriendsEdit->setEnabled(true);
    ui->statusUpdateEdit->setEnabled(true);
    ui->checkinButton->setEnabled(true);

    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to check in!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }

#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, tr("Successfully checked into %1").arg(m_locationName));
#endif
}

Place::Place(QObject *parent) :
    QObject(parent)
{
}

Place::~Place() {}

void Place::setLongitude(const QString &longitude)
{
    m_longitude = longitude;
}

void Place::setLatitude(const QString &latitude)
{
    m_latitude = latitude;
}

QString Place::longitude() const
{
    return m_longitude;
}

QString Place::latitude() const
{
    return m_latitude;
}
