#include "chatwindow.h"
#include "ui_chatwindow.h"

ChatWindow::ChatWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::ChatWindow),
    m_xmppClient(new QXmppClient(this)),
    m_vCardCache(new vCardCache(m_xmppClient)),
    m_rosterItemModel(new rosterItemModel(m_xmppClient)),
    m_capabilitiesCache(m_xmppClient),
    m_rosterItemSortFilterModel(new rosterItemSortFilterProxyModel(m_xmppClient))
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    QXmppLogger *logger = new QXmppLogger(m_xmppClient);
    logger->setLoggingType(QXmppLogger::StdoutLogging);
    m_xmppClient->setLogger(logger);

    rosterItemDelegate *delegate = new rosterItemDelegate();
    ui->listView->setItemDelegate(delegate);

#ifdef Q_WS_MAEMO_5
    ui->listView->setAlternatingRowColors(true);
#endif

    m_rosterItemSortFilterModel->setSourceModel(m_rosterItemModel);
    m_rosterItemSortFilterModel->setShowOfflineContacts(true);
    ui->listView->setModel(m_rosterItemSortFilterModel);
    m_rosterItemSortFilterModel->sort(0);

    connect(m_xmppClient, SIGNAL(messageReceived(QXmppMessage)), this, SLOT(onMessageReceived(QXmppMessage)));
    connect(m_xmppClient, SIGNAL(presenceReceived(QXmppPresence)), this, SLOT(onPresenceReceived(QXmppPresence)));
    connect(&m_xmppClient->rosterManager(), SIGNAL(rosterReceived()), this, SLOT(rosterReceived()));
    connect(&m_xmppClient->rosterManager(), SIGNAL(rosterChanged(QString)), this, SLOT(rosterChanged(QString)));
    //connect(m_xmppClient, SIGNAL(error(QXmppClient::Error)), this, SLOT(errorClient(QXmppClient::Error)));
    connect(m_xmppClient, SIGNAL(presenceReceived(QXmppPresence)), this, SLOT(onPresenceReceived(QXmppPresence)));
    connect(&m_xmppClient->rosterManager(), SIGNAL(presenceChanged(QString,QString)), this, SLOT(presenceChanged(QString, QString)));
    connect(m_vCardCache, SIGNAL(vCardReadyToUse(QString)), this, SLOT(updateVCard(QString)));
    connect(&m_xmppClient->vCardManager(), SIGNAL(vCardReceived(const QXmppVCardIq&)),
            m_vCardCache, SLOT(vCardReceived(const QXmppVCardIq&)));
    connect(ui->listView, SIGNAL(showChatDialog(const QString&)),
                        this, SLOT(showChatWindow(const QString&)));
    connect(ui->actionGo_offline, SIGNAL(triggered()), this, SLOT(goOffline()));

    if (QSettings().contains("chat/username") && QSettings().contains("chat/password")) {
        QString username = QSettings().value("chat/username").toString();
        QString password = QSettings().value("chat/password").toString();
        m_xmppClient->connectToServer(username + "@chat.facebook.com", password);
        QTimer::singleShot(500, this, SLOT(show()));
    } else {
        QPointer<ChatLoginDialog> dialog = new ChatLoginDialog(this);
        connect(dialog, SIGNAL(login(QString,QString,bool)), this, SLOT(onLoginClicked(QString,QString,bool)));
        connect(dialog, SIGNAL(loginCanceled()), this, SIGNAL(loginCanceled()));
        if (dialog->exec() != QDialog::Accepted)
            this->deleteLater();
    }
}

void ChatWindow::onLoginClicked(QString username, QString password, bool save)
{
    qobject_cast<QDialog*>(sender())->close();
#ifdef Q_WS_S60
    this->showMaximized();
#else
    this->show();
#endif
    QSettings().setValue("chat/username", username);
    if (save) {
        QSettings().setValue("chat/password", password);
    }
    m_xmppClient->connectToServer(username + "@chat.facebook.com", password);
}

ChatWindow::~ChatWindow()
{
    delete ui;
}

void ChatWindow::onPresenceReceived(QXmppPresence presence)
{
    QString from = presence.from();

    QString message;
    switch(presence.type())
    {
    case QXmppPresence::Subscribe:
        {
            message = "<B>%1</B> wants to subscribe";

            int retButton = QMessageBox::question(
                    this, "Contact Subscription", message.arg(from),
                    QMessageBox::Yes, QMessageBox::No);

            switch(retButton)
            {
            case QMessageBox::Yes:
                {
                    QXmppPresence subscribed;
                    subscribed.setTo(from);
                    subscribed.setType(QXmppPresence::Subscribed);
                    m_xmppClient->sendPacket(subscribed);

                    // reciprocal subscription
                    QXmppPresence subscribe;
                    subscribe.setTo(from);
                    subscribe.setType(QXmppPresence::Subscribe);
                    m_xmppClient->sendPacket(subscribe);
                }
                break;
            case QMessageBox::No:
                {
                    QXmppPresence unsubscribed;
                    unsubscribed.setTo(from);
                    unsubscribed.setType(QXmppPresence::Unsubscribed);
                    m_xmppClient->sendPacket(unsubscribed);
                }
                break;
            default:
                break;
            }

            return;
        }
        break;
    case QXmppPresence::Subscribed:
        message = "<B>%1</B> accepted your request";
        break;
    case QXmppPresence::Unsubscribe:
        message = "<B>%1</B> unsubscribe";
        break;
    case QXmppPresence::Unsubscribed:
        message = "<B>%1</B> unsubscribed";
        break;
    default:
        return;
        break;
    }

    if(message.isEmpty())
        return;

    QMessageBox::information(this, "Contact Subscription", message.arg(from),
            QMessageBox::Ok);
}

void ChatWindow::rosterChanged(QString bareJid)
{
    m_rosterItemModel->updateRosterEntry(bareJid, m_xmppClient->rosterManager().
                                        getRosterEntry(bareJid));

    // if available in cache, update it else based on presence it will request if not available
    if (m_vCardCache->isVCardAvailable(bareJid))
        updateVCard(bareJid);
}

void ChatWindow::rosterReceived()
{
    QStringList list = m_xmppClient->rosterManager().getRosterBareJids();
    QString bareJid;
    foreach(bareJid, list)
        rosterChanged(bareJid);
}

void ChatWindow::presenceChanged(QString bareJid, QString resource)
{
    if(bareJid == m_xmppClient->configuration().jidBare())
        return;

    if(! m_rosterItemModel->getRosterItemFromBareJid(bareJid))
        return;

    QString jid = bareJid + "/" + resource;
    QMap<QString, QXmppPresence> presences = m_xmppClient->rosterManager().
                                             getAllPresencesForBareJid(bareJid);
    m_rosterItemModel->updatePresence(bareJid, presences);

    QXmppPresence& pre = presences[resource];

    if(pre.type() == QXmppPresence::Available)
    {
        QString node = pre.capabilityNode();
        QString ver = pre.capabilityVer().toBase64();
        QStringList exts = pre.capabilityExt();

        QString nodeVer = node + "#" + ver;
        if(!m_capabilitiesCache.isCapabilityAvailable(nodeVer))
            m_capabilitiesCache.requestInfo(jid, nodeVer);

        foreach(QString ext, exts)
        {
            nodeVer = node + "#" + ext;
            if(!m_capabilitiesCache.isCapabilityAvailable(nodeVer))
                m_capabilitiesCache.requestInfo(jid, nodeVer);
        }

        switch(pre.vCardUpdateType())
        {
        case QXmppPresence::VCardUpdateNone:
            if(!m_vCardCache->isVCardAvailable(bareJid))
                m_vCardCache->requestVCard(bareJid);
        case QXmppPresence::VCardUpdateNotReady:
            break;
        case QXmppPresence::VCardUpdateNoPhoto:
        case QXmppPresence::VCardUpdateValidPhoto:
            if(m_vCardCache->getPhotoHash(bareJid) != pre.photoHash())
                m_vCardCache->requestVCard(bareJid);
            break;
        }
    }
#if 0
    QXmppPresence::Type presenceType = presences.begin().value().getType();

    if(!m_vCardCache->isVCardAvailable(bareJid) &&
       presenceType == QXmppPresence::Available)
    {
        m_rosterItemModel->updateAvatar(bareJid,
                                   QImage::fromData(m_vCardCache->getVCard(bareJid).photo()));
    }
#endif
}

void ChatWindow::updateVCard(QString bareJid)
{
    if (bareJid != m_xmppClient->configuration().jidBare()) {
        m_rosterItemModel->updateAvatar(bareJid,
                                   m_vCardCache->getAvatar(bareJid));
        m_rosterItemModel->updateName(bareJid, m_vCardCache->getVCard(bareJid).fullName());
    } else {
        QXmppVCardIq& vCard = m_vCardCache->getVCard(m_xmppClient->configuration().jidBare());
        QString fullName = vCard.fullName();

        if(fullName.isEmpty())
            fullName = m_xmppClient->configuration().jidBare();
    }
}

ChatConversationWindow* ChatWindow::getChatWindow(const QString& bareJid)
{
    if (!m_conversationWindows.contains(bareJid)) {
        m_conversationWindows[bareJid] = new ChatConversationWindow(this);
        m_conversationWindows[bareJid]->setBareJid(bareJid);

        if (!m_rosterItemModel->getRosterItemFromBareJid(bareJid))
            return 0;

        if (!m_rosterItemModel->getRosterItemFromBareJid(bareJid)->getName().isEmpty())
            m_conversationWindows[bareJid]->setDisplayName(m_rosterItemModel->getRosterItemFromBareJid(bareJid)->getName());
        else
            m_conversationWindows[bareJid]->setDisplayName(jidToUser(bareJid));

        m_conversationWindows[bareJid]->setQXmppClient(m_xmppClient);
    }

    return m_conversationWindows[bareJid];
}

void ChatWindow::onMessageReceived(QXmppMessage message)
{
    QString from = message.from();
    QString messageBody = message.body();
    if (messageBody.isEmpty())
        return;

    getChatWindow(jidToBareJid(from))->show();
    getChatWindow(jidToBareJid(from))->messageReceived(messageBody);
}

void ChatWindow::showChatWindow(const QString &bareJid)
{
    if (!bareJid.isEmpty())
        getChatWindow(bareJid)->show();
}

void ChatWindow::goOffline()
{
    m_xmppClient->disconnectFromServer();
    this->deleteLater();
}
