#include "newsfeedwindow.h"
#include "ui_newsfeedwindow.h"

NewsFeedWindow::NewsFeedWindow(QWidget *parent, QString token) :
    QMainWindow(parent),
    ui(new Ui::NewsFeedWindow),
    accessToken(token),
    qfacebook(new QFacebook(token, this)),
    nam(new QNetworkAccessManager(this))
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif
    connect(ui->newStatusButton, SIGNAL(clicked()), this, SLOT(onNewStatusUpdateButtonClicked()));
    connect(ui->actionRefresh, SIGNAL(triggered()), this, SLOT(updateNewsFeed()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    this->orientationChanged();
    this->updateNewsFeed();
    this->myId = QSettings().value("main/myId").toString();
    bool whiteMode = QSettings().value("main/white-mode", false).toBool();
    if (whiteMode) {
        QPalette p(palette());
        p.setColor(QPalette::Background, QColor(220, 225, 235));
        setPalette(p);
    }
}

NewsFeedWindow::~NewsFeedWindow()
{
    delete ui;
}

void NewsFeedWindow::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    ui->scrollAreaWidgetContents->setMaximumWidth(screenGeometry.width()-10);
}

void NewsFeedWindow::onNewStatusUpdateButtonClicked()
{
    NewStatusUpdateDialog *dialog = new NewStatusUpdateDialog(this);
    connect(dialog, SIGNAL(postUpdate(QString,QString,QString)), this, SLOT(postUpdate(QString,QString,QString)));
#ifdef Q_WS_S60
    dialog->showMaximized();
#else
    dialog->show();
#endif
}

void NewsFeedWindow::postUpdate(QString text, QString link, QString video)
{
    QByteArray data;
    if (!text.isEmpty()) {
        data.append("message=" + text.toUtf8());
        if (!link.isEmpty() || !video.isEmpty())
            data.append("&");
    }
    if (!link.isEmpty()) {
        data.append("link=" + link);
        if (!video.isEmpty())
            data.append("&");
    }
    if (!video.isEmpty())
        data.append("video=" + video);

    statusReply = qfacebook->putObject("me", "feed", data);
    if (statusReply)
        connect(statusReply, SIGNAL(finished()), this, SLOT(onStatusUpdateReplyFinished()));
}

void NewsFeedWindow::onStatusUpdateReplyFinished()
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, tr("Status update posted"));
#else
    QMessageBox::information(this, tr("Success"), tr("Status update posted"));
#endif
}

void NewsFeedWindow::updateNewsFeed()
{
    m_loadingWidget = new LoadingWidget(this);
    m_loadingWidget->setLoading(true);

    QString limit = QSettings().value("main/newsFeedLimit", "80").toString();
    this->reply = qfacebook->getConnections("me", QString("home&limit=%1").arg(limit));
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void NewsFeedWindow::onNameClicked(QString id, QString name)
{
    ProfileWindow *profile = new ProfileWindow(this, accessToken);
    profile->browseProfile(id);
    profile->setWindowTitle(name);
#ifdef Q_WS_S60
    profile->showMaximized();
#else
    profile->show();
#endif
}

// Hackish, but oh well...
void NewsFeedWindow::onNewsFeedItemParsed(QString fromName, QString fromId, bool isDeletable,
                                          QString message, QString icon, QString time, QString link,
                                          QString linkName, QString linkDescription, QString linkPicture,
                                          QString toId, QString toName, bool meLikesThis, QString commentsCount,
                                          QString likesCount, QString postId)
{
    NewsFeedPost *post = new NewsFeedPost(ui->postsWidget, accessToken, qfacebook, nam);
    connect(post, SIGNAL(commentButtonClicked(QString,QString,QString)),
            this, SLOT(onCommentButtonClicked(QString,QString,QString)));
    connect(post, SIGNAL(nameClicked(QString,QString)), this, SLOT(onNameClicked(QString,QString)));
    post->getPixmapFromUrl(QString("http://graph.facebook.com/%1/picture").arg(fromId));
    post->setId(postId);
    post->setName(fromName);
    post->setUserId(fromId);
    post->setDeletable(isDeletable);
    post->setMessage(message);
    post->setIconFromUrl(icon);
    post->setTime(time);
    if (!link.isEmpty())
        post->addLink(link, linkName, linkDescription, linkPicture);
    if (!toName.isEmpty() && !toId.isEmpty())
        post->setToName(toName, toId);
    post->setNumberOfLikesComments(likesCount, commentsCount);
    post->setLikedByMe(meLikesThis);

    posts.append(post);
    ui->postsLayout->addWidget(post);
}

void NewsFeedWindow::onReplyFinished()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch news feed!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }

    QVariant postsVariant = reply->data();

    if (postsVariant == m_postsVariant)
        return;
    else
        m_postsVariant = postsVariant;

    if (!posts.isEmpty()) {
        foreach (NewsFeedPost *post, posts) {
            ui->postsLayout->removeWidget(post);
            delete post;
        }
        posts.clear();
    }

    QThread *thread = new QThread;
    NewsFeedParser *parser = new NewsFeedParser(postsVariant, accessToken);
    parser->moveToThread(thread);

    connect(parser, SIGNAL(finished()), parser, SLOT(deleteLater()));
    connect(thread, SIGNAL(finished()), thread, SLOT(deleteLater()));
    connect(parser, SIGNAL(feedItemParsed(QString,QString,bool,QString,QString,QString,QString,
                                          QString,QString,QString,QString,QString,bool,QString,QString,QString)), this,
            SLOT(onNewsFeedItemParsed(QString,QString,bool,QString,QString,QString,QString,QString,QString,
                                      QString,QString,QString,bool,QString,QString,QString)));
    connect(thread, SIGNAL(started()), parser, SLOT(start()));
    connect(parser, SIGNAL(finished()), m_loadingWidget, SLOT(hideWidget()));

    thread->start();
}

void NewsFeedWindow::onCommentButtonClicked(QString id, QString comments, QString likes)
{
    SinglePostWindow *window = new SinglePostWindow(this, accessToken);
    window->loadPost(id);
    window->setCountLabel(comments, likes);
#ifdef Q_WS_S60
    window->showMaximized();
#else
    window->show();
#endif
}

NewsFeedParser::NewsFeedParser(QVariant jsonData, QString accessToken)
{
    this->nam = new QNetworkAccessManager();
    this->accessToken = accessToken;
    this->qfacebook = new QFacebook(accessToken);
    m_jsonData = jsonData;
}

void NewsFeedParser::start()
{
    QVariant jsonData = m_jsonData;
    QVariantList listData;
    QString myId = QSettings().value("main/myId").toString();
    listData = jsonData.toMap().value("data").toList();
    foreach(jsonData, listData) {
        QString name = jsonData.toMap().value("from").toMap().value("name").toString();
        QString userId = jsonData.toMap().value("from").toMap().value("id").toString();
        bool isDeletable = false;
        if (userId.contains(myId))
            isDeletable = true;
        QString message = jsonData.toMap().value("message").toString();
        QString icon = jsonData.toMap().value("icon").toString();
        QString time = jsonData.toMap().value("created_time").toString();
        QString link = jsonData.toMap().value("link").toString();
        QString linkName;
        QString linkDescription;
        QString linkPicture;
        QString toId;
        QString toName;

        if (link.isEmpty())
            link = jsonData.toMap().value("source").toString();
        if (!link.isEmpty()) {
            linkName = jsonData.toMap().value("name").toString();
            linkDescription = jsonData.toMap().value("description").toString();
            linkPicture = jsonData.toMap().value("picture").toString();
        }
        QVariantList toVariant = jsonData.toMap().value("to").toMap().value("data").toList();
        if (!toVariant.isEmpty()) {
            QVariant to = toVariant.first();
            if (message.contains(to.toString()) && toVariant.count() > 1)
                to = toVariant.at(1);

            toId = to.toMap().value("id").toString();
            toName = to.toMap().value("name").toString();
        }

        // Check if the currently logged in user likes the post
        QVariantList likes = jsonData.toMap().value("likes").toMap().value("data").toList();
        bool meLikesThis = false;
        foreach (QVariant likeId, likes) {
            QVariantMap map = likeId.value<QVariantMap>();
            QString likerId = map.value("id").toString();
            if (likerId.contains(myId)) {
                meLikesThis = true;
                break;
            }
        }

        // Get like and comment count.
        QString likesCount = jsonData.toMap().value("likes").toMap().value("count").toString();
        if (likesCount.isEmpty())
            likesCount = "0";
        QString comments = jsonData.toMap().value("comments").toMap().value("count").toString();
        if (comments.isEmpty())
            comments = "0";

        QString postId = jsonData.toMap().value("id").toString();
        // Set parsed data
        emit feedItemParsed(name, userId, isDeletable, message, icon, time, link, linkName, linkDescription, linkPicture,
                            toId, toName, meLikesThis, comments, likesCount, postId);
    }
    emit finished();
}
