#include "newsfeedpost.h"
#include "ui_newsfeedpost.h"

NewsFeedPost::NewsFeedPost(QWidget *parent, QString token, QFacebook *qfacebook, QNetworkAccessManager *nam) :
    QWidget(parent),
    ui(new Ui::NewsFeedPost),
    accessToken(token),
    qfacebook(qfacebook),
    nam(nam)
{
    ui->setupUi(this);
    m_whiteMode = QSettings().value("main/white-mode", false).toBool();
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    connect(ui->commentButton, SIGNAL(clicked()), this, SLOT(onCommentButtonClicked()));
    connect(ui->likeButton, SIGNAL(clicked()), this, SLOT(onLikeButtonClicked()));
    connect(ui->deleteButton, SIGNAL(clicked()), this, SLOT(onDeleteButtonClicked()));
    connect(ui->nameLabel, SIGNAL(clicked()), this, SLOT(onNameClicked()));

    ui->widget->hide();
    ui->toLabel->hide();
    this->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    this->orientationChanged();
    this->isLikedByMe = false;
    ui->deleteButton->hide();
    QPalette p = this->palette();
    p.setColor(QPalette::Foreground, p.highlight().color());
    ui->nameLabel->setPalette(p);
    ui->toLabel->setPalette(p);

    if (m_whiteMode) {
    p.setColor(QPalette::Foreground, Qt::black);
        ui->messageLabel->setPalette(p);
        ui->timeLabel->setPalette(p);
        ui->linkDescription->setPalette(p);
    }

    QFont f(ui->linkDescription->font());
    f.setPointSize(f.pointSize()-2);
    ui->linkDescription->setFont(f);
#ifdef Q_WS_MAEMO_5
    ui->deleteButton->setText("");
    ui->deleteButton->setIcon(QIcon::fromTheme("general_close"));
#endif
}

NewsFeedPost::~NewsFeedPost()
{
    delete ui;
}

void NewsFeedPost::onCommentButtonClicked()
{
    emit commentButtonClicked(this->id, this->comments, this->likes);
}

void NewsFeedPost::onLikeButtonClicked()
{
    QByteArray data;
    if (this->isLikedByMe)
        data.append("&method=delete");
    this->reply = qfacebook->putObject(this->id, "likes", data);
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onLikeReplyReceived()));
}

void NewsFeedPost::setToName(QString name, QString id)
{
    if (name == ui->nameLabel->text() && id == userId)
        return;

    ui->toLabel->setText("-> " + name);
    m_toUserId = id;
    ui->toLabel->show();
    connect(ui->toLabel, SIGNAL(clicked()), this, SLOT(onToNameClicked()));
}

void NewsFeedPost::onToNameClicked()
{
    emit nameClicked(m_toUserId, ui->toLabel->text().remove("-> "));
}

void NewsFeedPost::onLikeReplyReceived()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to get reply!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }

    if (this->isLikedByMe) {
        this->isLikedByMe = false;
        this->setNumberOfLikesComments(QString::number(this->likes.toInt()-1).remove("\""), this->comments);
    } else {
        this->isLikedByMe = true;
        this->setNumberOfLikesComments(QString::number(this->likes.toInt()+1).remove("\""), this->comments);
    }
    reply->deleteLater();
}

void NewsFeedPost::addLink(QString url, QString linkName, QString linkDescription, QString linkPixmap)
{
    if (!linkPixmap.isEmpty()) {
        ui->linkPicture->setMinimumSize(64, 64);
        pictureReply = nam->get(QNetworkRequest(QString(linkPixmap + "&access_token=" + accessToken)));
        connect(pictureReply, SIGNAL(finished()), this, SLOT(onPictureReceived()));
    }

    if (linkName.isEmpty()) {
        QString text = QString("<a href=\"%1\">%2</a>").arg(url);
        QString shortUrl = url;
        shortUrl.truncate(20);
        shortUrl.append("...");
        text.replace("%2", shortUrl);
        ui->linkName->setText(text);
    } else {
        ui->linkName->setText(QString("<a href=\"%1\">%2</a>").arg(url, linkName));
    }
    ui->linkName->setOpenExternalLinks(false);
    connect(ui->linkName, SIGNAL(linkActivated(QString)), this, SLOT(onLinkActivated(QString)));

    if (linkDescription.isEmpty())
        ui->linkDescription->hide();
    else
        ui->linkDescription->setText(linkDescription);

    ui->widget->show();
}

void NewsFeedPost::onLinkActivated(QString url)
{
    if (url.contains("youtube.com")) {
#ifdef Q_WS_MAEMO_5
        YoutubePlayer *player = new YoutubePlayer(0);
#else
        YoutubePlayer *player = new YoutubePlayer(this->parentWidget());
#endif
        player->setOriginalUrl(url);
        if (url.contains("&")) {
            int position = url.indexOf('&');
            url = url.remove(position, url.length()-position);
        }
        player->getVideoUrl(url.remove("http://www.youtube.com/watch?v="));
        player->showExpanded();
    } else if (url.contains("youtu.be")) {
#ifdef Q_WS_MAEMO_5
        YoutubePlayer *player = new YoutubePlayer(0);
#else
        YoutubePlayer *player = new YoutubePlayer(this->parentWidget());
#endif
        player->setOriginalUrl(url);
        url = url.remove("http://youtu.be/");
        player->getVideoUrl(url.remove("http://www.youtube.com/watch?v="));
        player->showExpanded();
    } else {
        QDesktopServices::openUrl(QUrl(url));
    }
}

void NewsFeedPost::onPictureReceived()
{
    if (pictureReply->error() != QNetworkReply::NoError) {
        // W(HY)TF does facebook return forbidden for some posts???
        ui->linkPicture->hide();
        pictureReply->deleteLater();
        pictureReply = 0;

        return;
    }

    QUrl redir = pictureReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        pictureReply = nam->get(QNetworkRequest(redir));

        connect(pictureReply, SIGNAL(finished()), this, SLOT(onPictureReceived()));
        return;
    }

    ui->linkPicture->setPixmap(QPixmap::fromImage(QImage::fromData(pictureReply->readAll())));
    pictureReply->deleteLater();
}

void NewsFeedPost::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        ui->bottomLayout->setDirection(QBoxLayout::LeftToRight);
    } else {
        ui->bottomLayout->setDirection(QBoxLayout::TopToBottom);
    }
/*#ifdef Q_WS_MAEMO_5
    this->setMaximumWidth(((QMainWindow*)parent())->width()-5);
#endif*/
}

void NewsFeedPost::setName(QString name)
{
    ui->nameLabel->setText(name);
    ui->nameLabel->show();
}

void NewsFeedPost::setDate(QString date)
{
    ui->timeLabel->setText(date);
    ui->timeLabel->show();
}

void NewsFeedPost::setTime(QString time)
{
    QDateTime t = QDateTime::fromString(time, Qt::ISODate);
    t.setTimeSpec(Qt::UTC);
    QString date = QString(t.toLocalTime().toString("dddd, MMMM d %1 hh:mmap").arg(tr("at")));
    //QString today = QDateTime().currentDateTime().toString("dddd");
    //if (date.contains(today) && date.is)
    //    date.replace(today, tr("Today"));
    ui->timeLabel->setText(date);
}

void NewsFeedPost::setTimeFromString(QString timeAsString)
{
    ui->timeLabel->setText(timeAsString);
}

void NewsFeedPost::setDeletable(bool deletable)
{
    this->isDeletable = deletable;
    ui->deleteButton->setVisible(deletable);
}

void NewsFeedPost::setAvatar(QPixmap pixmap)
{
    QPropertyAnimation *animation = new QPropertyAnimation(ui->avatarLabel, "pos", this);
    animation->setStartValue(QPoint(ui->avatarLabel->x(), ui->avatarLabel->y()-(ui->avatarLabel->height()+70)));
    animation->setEndValue(ui->avatarLabel->pos());
    animation->setDuration(1000);
    animation->setEasingCurve(QEasingCurve::InOutBack);
    animation->start(QPropertyAnimation::DeleteWhenStopped);
    ui->avatarLabel->setPixmap(pixmap);
}

void NewsFeedPost::setMessage(QString message)
{
    if (message.isEmpty()) {
        ui->messageLabel->hide();
    } else {
        ui->messageLabel->setText(message.replace("\n", "<br>"));
    }
}

void NewsFeedPost::setId(QString id)
{
    this->id = id;
}

void NewsFeedPost::hideBottomBar(bool hide)
{
    ui->bottomWidget->setHidden(hide);
}

void NewsFeedPost::setNumberOfLikesComments(QString likes, QString comments)
{
    QString likeText;
    if (this->isLikedByMe)
        likeText = tr("Unlike");
    else
        likeText = tr("Like");
    this->likes = likes;
    this->comments = comments;
    ui->commentButton->setText(tr("Comment") + QString(" (%1)").arg(comments));
    ui->likeButton->setText(likeText + QString(" (%1)").arg(likes));
}

void NewsFeedPost::setLikedByMe(bool meLikesThis)
{
    this->isLikedByMe = meLikesThis;
    this->setNumberOfLikesComments(this->likes, this->comments);
}

void NewsFeedPost::setIconFromUrl(QString url)
{
    QNetworkReply *reply = nam->get(QNetworkRequest(url));
    connect(reply, SIGNAL(finished()), this, SLOT(iconReceived()));
}

void NewsFeedPost::getPixmapFromUrl(QString pixmapUrl)
{
    QNetworkReply *reply = nam->get(QNetworkRequest(pixmapUrl));
    connect(reply, SIGNAL(finished()), this, SLOT(avatarReceived()));
}

void NewsFeedPost::getPixmapFromUserId(QString userId)
{
    QString url = QString("https://graph.facebook.com/%1/picture&access_token=%2").arg(userId, accessToken);
    getPixmapFromUrl(url);
}

void NewsFeedPost::setCommentMode(bool isComment)
{
    ui->commentButton->setHidden(isComment);
}

void NewsFeedPost::setMessageMode(bool isMessage)
{
    ui->commentButton->setHidden(isMessage);
    ui->likeButton->setHidden(isMessage);
}

void NewsFeedPost::avatarReceived()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    QUrl redir = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        reply->deleteLater();

        QString url = redir.toString();
        int pos = url.lastIndexOf("/");
        QString baseName = url.remove(0, pos);
        baseName.remove("/");

        QFile cachedImage(QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName);
        if (cachedImage.exists()) {
            cachedImage.open(QIODevice::ReadOnly);
            QByteArray data = cachedImage.readAll();
            cachedImage.close();
            this->setAvatar(QPixmap::fromImage(QImage::fromData(data)));
            return;
        }

        QNetworkAccessManager *nam = qobject_cast<QNetworkAccessManager*>(sender()->parent());
        QNetworkReply *reply = nam->get(QNetworkRequest(redir));

        connect(reply, SIGNAL(finished()), this, SLOT(avatarReceived()));
        return;
    }
    QByteArray data = reply->readAll();
    this->setAvatar(QPixmap::fromImage(QImage::fromData(data)));

    QString url = reply->url().toString();
    int pos = url.lastIndexOf("/");
    QString baseName = url.remove(0, pos);
    baseName.remove("/");
    QDir tempDir(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));
    if (!tempDir.exists())
        tempDir.mkpath(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));

    QString filename = QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName;
    QFile cachedImage(filename);
    cachedImage.open(QIODevice::WriteOnly);
    cachedImage.write(data);
    cachedImage.close();

    reply->deleteLater();
}

void NewsFeedPost::onNameClicked()
{
    emit nameClicked(this->userId, ui->nameLabel->text());
}

void NewsFeedPost::iconReceived()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    ui->iconLabel->setPixmap(QPixmap::fromImage(QImage::fromData(reply->readAll())));
    reply->deleteLater();
}

void NewsFeedPost::onDeleteButtonClicked()
{
    QMessageBox confirmDelete(QMessageBox::NoIcon,
                              tr("Delete post?"),
                              tr("Are you sure you want to delete this post?"),
                              QMessageBox::Yes | QMessageBox::No);
    confirmDelete.exec();
    if (confirmDelete.result() == QMessageBox::Yes) {
        QByteArray data;
        data.append("&method=delete");
        this->deleteReply =
                qfacebook->put(QUrl(QString("https://graph.facebook.com/%1&access_token=%2").arg(this->id, accessToken)),
                               data);
        connect(this->deleteReply, SIGNAL(finished()), this, SLOT(onDeleteReplyReceived()));
    }
}

void NewsFeedPost::onDeleteReplyReceived()
{
    if (this->deleteReply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to delete post!" << deleteReply->errorString();
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, tr("Failed to delete post"));
#endif
        this->deleteReply->deleteLater();
        this->deleteReply = 0;

        return;
    }

    QPropertyAnimation *animation = new QPropertyAnimation(this, "pos", this);
    connect(animation, SIGNAL(finished()), this, SLOT(hide()));
    connect(animation, SIGNAL(finished()), this, SLOT(deleteLater()));
    animation->setStartValue(this->pos());
    animation->setEndValue(QPoint(this->x()-this->width(), this->y()));
    animation->setDuration(1000);
    animation->setEasingCurve(QEasingCurve::InOutCirc);
    animation->start(QPropertyAnimation::DeleteWhenStopped);
    emit postDeleted(id);
}

void NewsFeedPost::setUserId(QString id)
{
    this->userId = id;
}

void NewsFeedPost::showContextMenu(QPoint point)
{
    QMenu *contextMenu = new QMenu(this);
    contextMenu->setAttribute(Qt::WA_DeleteOnClose);
    contextMenu->addAction(tr("Copy post"), this, SLOT(copyPost()));
    contextMenu->exec(point);
}

void NewsFeedPost::copyPost()
{
    QString text;
    QString message = ui->messageLabel->text();
    text.append(ui->nameLabel->text() + ": ");
    text.append(message.replace("<br>", "\n"));

    QClipboard *clipboard = QApplication::clipboard();
    clipboard->setText(text);

#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, tr("Post copied"));
#endif
}

void NewsFeedPost::resizeEvent(QResizeEvent *)
{
    this->orientationChanged();
}

void NewsFeedPost::paintEvent(QPaintEvent *)
{
    QPainter painter(this);
    QRect rect = this->rect();
    QPoint left = rect.bottomLeft();
    QPoint right = rect.bottomRight();
    painter.setPen(this->palette().highlight().color());
    painter.drawLine(left, right);
}

void NewsFeedPost::mousePressEvent(QMouseEvent *ev)
{
    if (ev->button() == Qt::LeftButton) {
        m_pressed = true;
        emit pressed();
    }
}

void NewsFeedPost::mouseReleaseEvent(QMouseEvent *ev)
{
    if (ev->button() == Qt::LeftButton) {
        m_pressed = false;
        emit released();
        if (rect().contains(ev->pos())) {
            emit clicked(id);
        }
    }
}

void NewsFeedPost::showEvent(QShowEvent *)
{
#if 0
    QPropertyAnimation *animation = new QPropertyAnimation(this, "pos", this);
    animation->setStartValue(QPoint(this->x()-this->width(), this->y()));
    animation->setEndValue(this->pos());
    animation->setDuration(1000);
    animation->setEasingCurve(QEasingCurve::OutCubic);
    animation->start(QPropertyAnimation::DeleteWhenStopped);
#endif
}
