#include "gamewidget.h"

#include <QPainter>
#include <QTimer>
#include <QDebug>
#include <QKeyEvent>
#include <QMessageBox>
#include <QtDBus>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>

#include "bords.h"
#include "snake.h"
#include "motionsenser.h"
#include "util.h"


GameWidget::GameWidget()
    :mSnake(0),mMouse(-1,-1),mBonusMouse(-100,-100), mBord(0),mDisplayCounter(0),mGameState( NotStarted )
{    
    this->setMinimumSize(800,480);    
    mSenser = new MotionSenser(this);

    mTimer = new QTimer(this);
    mBonusTimer = new QTimer(this);    

    mGridPen.setColor( Qt::gray );
    mGridPen.setWidth(1);
}

GameWidget::~GameWidget()
{
    delete mSnake;
}

void GameWidget::newGame( int aBordId )
{    
    mScore = 0;

    if( mBord ) {
       delete mBord;
    }

    if( mSnake ) {
        delete mSnake;
    }

    if( aBordId ==  -1 ) {
        mBord = BordFactory::createRandomBord();
    } else {
        mBord = BordFactory::createBord( aBordId );
    }

    mSnake = new Snake;
    generateMouse();    
    mGameState = NotStarted;

    if( Util::useAccelerometer() )
        connect( mSenser,SIGNAL(keyEvent(QPoint)),this,SLOT(keyEvent(QPoint)) );

    connect( mTimer,SIGNAL(timeout()),this,SLOT(gameLoop()));
    connect( mBonusTimer,SIGNAL(timeout()),this,SLOT(generateBonusMouse()));
}

void GameWidget::endGame()
{
    mGameState = Ended;
    mTimer->stop();
    mBonusTimer->stop();
    mDisplayCounter = 0;    

    disconnect( mSenser,SIGNAL(keyEvent(QPoint)),this,SLOT(keyEvent(QPoint)) );
    disconnect(mTimer,SIGNAL(timeout()),this,SLOT(gameLoop()));
    disconnect(mBonusTimer,SIGNAL(timeout()),this,SLOT(generateBonusMouse()));

    repaint();
}

void GameWidget::gameLoop()
{    
    disableDisplayOff();

    mSnake->move();

    //check if colliding with self or bord
    if( mSnake->isCollidingWithSelf() || mBord->isColliding( mSnake) ) {                
        endGame();
        return;
    }

    //check if eating mouse
    if( mSnake->isColliding( mMouse ) ) {
        mSnake->increseLength(1);        
        mScore += Util::pointForMouse( Util::speed() );
        generateMouse();
    }

    //check if eating bonus mouse
    if( mSnake->isColliding( mBonusMouse ) ) {
        mSnake->increseLength(3);
        mScore += Util::pointForBonus( Util::speed()  );
        invalidateBonus();
    }

    repaint();
}

void GameWidget::paintEvent(QPaintEvent *)
{
    QPainter painter( this );
    painter.setRenderHint( QPainter::Antialiasing );    

     drawGame( &painter);

     if( mGameState == Ended ) {

        drawGameOver( &painter );
    } else if( mGameState == Paused || mGameState == NotStarted ) {

        drawPausedScree( &painter );
    }
}

void GameWidget::drawGame( QPainter* aPainter )
{
    aPainter->fillRect(rect(), Qt::lightGray );

    drawHighlight( aPainter );
    drawGrid( aPainter );    
    mBord->draw( aPainter );
    drawScore( aPainter );

    //draw mouse
    aPainter->setBrush( Qt::red);
    aPainter->drawEllipse( QRect( mMouse , MOUSE_SIZE ));

    //draw mouse
    aPainter->setBrush( Qt::yellow );
    aPainter->drawEllipse( QRect( mBonusMouse ,MOUSE_SIZE ));

    mSnake->draw( aPainter );
}

void GameWidget::drawGrid( QPainter* aPainter)
{
    aPainter->setPen( mGridPen );
    for( int i = 0 ; i < 24 /*480/ 20*/ ; ++i ){
        aPainter->drawLine(QPoint(0,i*20), QPoint( 800,i*20));
    }

    for( int i = 0 ; i < 40 /*800/ 20*/ ; ++i ){
        aPainter->drawLine(QPoint(i*20,0), QPoint(i*20,480));
    }    
}

void GameWidget::drawScore( QPainter* aPainter )
{
    QString score = QString("Score : %1").arg(mScore);
    aPainter->setPen( Qt::black );
    aPainter->drawText( QRect(0,0,800,100),Qt::AlignLeft,score);    
}

void GameWidget::drawHighlight( QPainter* aPainter )
{
    QPoint location = mSnake->pos();    
    aPainter->fillRect( QRect(0,location.y(),800,20), Qt::darkCyan);
    aPainter->fillRect( QRect(location.x(),0,20,480), Qt::darkCyan);
}

void GameWidget::drawGameOver( QPainter* aPainter )
{
    drawOpaqueLayer( aPainter );

    QFont font = aPainter->font();  
    font.setBold( true );

    //qDebug() << mScore <<":" << Util::highScore();
    QString greeting("Good Job!");
    if( mScore > Util::highScore() ) {
        greeting = "New High Score!!!";
        Util::setHighScore( mScore );
    }

    font.setPixelSize( 50 );
    aPainter->setFont( font );
    aPainter->setPen( Qt::yellow );
    aPainter->drawText( 0,20, 800, 200,Qt::AlignCenter, greeting );


    QString score("Your score : %1");
    score = score.arg( mScore );

    font.setPixelSize( 60 );
    aPainter->setFont( font );
    aPainter->setPen( Qt::green );
    aPainter->drawText( 0,100, 800, 200,Qt::AlignCenter, score );

    QString tapForMenu("Tap on screen to go to menu...");
    font.setPixelSize( 30 );
    aPainter->setFont( font );
    aPainter->setPen( Qt::red );
    aPainter->drawText( 0,300, 800, 200,Qt::AlignCenter, tapForMenu );
}

void GameWidget::drawOpaqueLayer( QPainter* aPainter )
{    
    QColor color( Qt::darkGray );
    color.setAlpha( 200 );
    aPainter->fillRect( rect(), color);
}

void GameWidget::drawPausedScree( QPainter* aPainter )
{
    drawOpaqueLayer( aPainter );

    QFont font = aPainter->font();
    font.setPixelSize( 50 );
    aPainter->setFont( font );
    aPainter->setPen( Qt::yellow );
    aPainter->drawText( 0,0, 800, 480,Qt::AlignCenter, " Tap screen to continue..." );

    QString tapForMenu("Tap here to go to menu...");
    font.setPixelSize( 30 );
    aPainter->setFont( font );
    aPainter->setPen( Qt::red );
    aPainter->drawText( 0,300, 800, 100,Qt::AlignCenter, tapForMenu );
}

void GameWidget::generateMouse()
{    
    mMouse = randomMousePos();
}

void GameWidget::generateBonusMouse()
{
    //bonus is not set, try to set bonus
    if( mBonusMouse.x() == -100 ) {
        int random = qrand() % 100 ;
        if( random % 2 == 0 && random > 30 /*&& random < 60*/ ) {
            mBonusMouse = randomMousePos();
            QTimer::singleShot( 5000 * Util::speed() * 10 , this, SLOT(invalidateBonus()));
        }
    }
}

void GameWidget::invalidateBonus()
{
    mBonusMouse = QPoint(-100,-100 );    
}

QPoint GameWidget::randomMousePos()
{        
    bool mousePosFound(false);
    while( !mousePosFound ) {

        QPoint mousePos(  qrand() % 40 * 20, qrand() % 24 * 20 );        

        if ( mSnake->isColliding( mousePos ) ) {
            continue;
        }

        if( mBord->isColliding( mousePos ) ) {
            continue;
        }

        mousePosFound = true;
        return mousePos;
    }

    return QPoint();
}

void GameWidget::mousePressEvent(QMouseEvent *aEvent)
{
    switch( mGameState ) {
    case NotStarted:        
    case Paused: {
            QRect rect( 0,300, 800, 100 );
            if( rect.contains( aEvent->pos() )) {
                endGame();
                emit showMenu();
            } else {
                mTimer->start( Util::speed() );
                mBonusTimer->start( 3500 );
                mGameState = Started;
            }
        }
        break;
    case Ended:
        emit showMenu();
        break;
    case Started:
        mTimer->stop();
        mBonusTimer->stop();
        mGameState = Paused;
        break;
    }

    repaint();
}

void GameWidget::keyPressEvent(QKeyEvent *aKeyEvent)
{        
    if( mGameState != Started ) {
        return;
    }

    if( Util::key("Up") == aKeyEvent->key() ) {
        mSnake->turn( QPoint(0,-1));
    } else if( Util::key("Down") == aKeyEvent->key() ) {
        mSnake->turn( QPoint(0,1));        
    } else if( Util::key("Right") == aKeyEvent->key() ) {
        mSnake->turn( QPoint(1,0));
    } else if( Util::key("Left") == aKeyEvent->key() ) {
        mSnake->turn( QPoint(-1,0));        
    }

    if( aKeyEvent->key() == 16777220 ) {
        mSnake->increseLength(3);
    }

    aKeyEvent->accept();
    repaint();
}

void GameWidget::keyEvent( QPoint aDirection )
{
    //qDebug() << "KeyEvent:" << aDirection;
    if( mGameState != Started ) {
        return;
    }

    mSnake->turn( aDirection );
    repaint();
}

void GameWidget::disableDisplayOff()
{
    mDisplayCounter++;
    if( mDisplayCounter >= 100 ) {
        mDisplayCounter = 0;
        QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                                             MCE_REQUEST_IF, MCE_PREVENT_BLANK_REQ));
    }
}
