# -*- coding: utf-8 -*-
##############################################################
# smscon - remote control library for Nokia N900 phone       #
# $Id: smsconlib.py 77 2011-12-23 23:09:47Z yablacky $
##############################################################

# standard modules
import os
import re
#import dbus
import logging
import fileinput
import signal
import time
import pexpect

##############################################################

def getSysBuildNameAndVersion(defaultName, defaultVersion = '999.1', headerInfo = None, dirLevels = 0):
    """
    Analyze information from version control system to find out name and version of a branch.
    @param string defaultName Name to return if version information not recognized.
    @param string defaultVersion Version to return if version information not recognized.
    @param string headerInfo Value from expanded $Header: https://vcs.maemo.org/svn/smscon/branches/smscon_0.9.6/src/opt/smscon/smsconlib.py 77 2011-12-23 23:09:47Z yablacky $ keyword to be analyzed.
    @param int dirLevel Number of extra dirs to walk up to find the one that contains version information (0 = parent dir).
    @return list (name, version) The name and version information.
    """
    try:
        return (headerInfo.split()[1].split('/')[-dirLevels-2].split('_') + [defaultVersion])[0:2]
    except:
        return [defaultName, defaultVersion]

NAME = 'smscon'
VERSION = '999.1'   # for trunk and private builds etc.

NAME, VERSION = getSysBuildNameAndVersion(NAME, VERSION,
    headerInfo  = '$Header: https://vcs.maemo.org/svn/smscon/branches/smscon_0.9.6/src/opt/smscon/smsconlib.py 77 2011-12-23 23:09:47Z yablacky $',
    dirLevels   = 3 )
    
XDEBUG   = False        # cross debug (use only if not running on phone)

##############################################################

TimeFormat  = '%Y-%m-%d %H:%M:%S'
InstPath    = '/opt/%s' % NAME
if XDEBUG: InstPath = '.'
BootPath    = '/etc/event.d'
DaemonFile  = '%s_daemon' % NAME
CodeFile    = '%s_code' % NAME
ConfigFile  = '%s_config' % NAME
BootFile    = '%s_boot' % NAME
ScriptFile  = '%s_script' % NAME
ScriptLog   = '%s_script.log' % NAME

PhotoName   = 'frontcam.jpg'
MapName     = 'gpsmap.png'
AudioFile   = 'alarm.wav'

LogPath    = InstPath
LogFile    = '%s.log' % NAME
BootLogFile= '%s_boot.log' % NAME
LOGGER     = None

def IF(VAL,THEN,ELSE):
    """
    Almost the same what (VAL ? THEN : ELSE) does in C, perl or php.
    Note the difference: Unlike ?: the IF() always evaluates BOTH(!)
    expressions (THEN as well as ELSE) and returns one of the results.
    """
    if VAL: return THEN
    else  : return ELSE

##############################################################
# encryption/decryption interface

import base64
from Crypto.Cipher import AES
from random import randint

_AESBlockSize = 16
_AESPadding   = '{'
_AESPadlen    = lambda l: (_AESBlockSize - l) % _AESBlockSize
_AESPad       = lambda s: s + _AESPadlen(len(s)) * _AESPadding

def EncodeAES(key, text):
    text = str(text)
    # len(salt1) == len(salt2) != _AESBlockSize must be true
    salt  = chr(randint(0,255)) + chr(randint(0,255))
    salt1 = salt + chr(randint(0,255))                      # 3 random bytes (will not be encrypted - so don't put info here)
    salt2 = salt + chr(_AESPadlen(len(salt1) + len(text)))  # 2 random bytes plus padding length info (being encrypted)
    return base64.b64encode(salt1 + key.encrypt(_AESPad(salt2 + text)))

def DecodeAES(key, code):
    code  = base64.b64decode(code)
    salt1 = code[0:len(code) % _AESBlockSize]
    text = key.decrypt(code[len(salt1):])
    salt2 = text[0:len(salt1)]
    text  = text[len(salt2):]
    if len(salt2) == 3:
        return text[0:len(text)-ord(salt2[2])]
    else:
        return text.rstrip(_AESPadding)

##############################################################
# Convenience functions. 

def strReverse(s):
    """
    @param string s String to be reversed.
    @return string string s in reverse character order.
    """
    return ''.join([s[-i] for i in range(1,1+len(s))])

def commonPrefix(*args):
    """@return string The longest same string that all passed strings are starting with"""
    for idx in range(min(map(len, args))):
        for s in args:
            if s[idx] != args[0][idx]: 
                return s[0:idx];
    if len(args):   return args[0]
    else:           return ''

def shellArg(s):
    """@return string The parameter s quoted to be safely used as a shell parameter."""
    return "'%s'" % str(s).replace("\'", "'\"'\"'")

def pidof(ProcessName):
    """
    Determine the PID of processes that match a given name.
    @return string Whitespace separator PID. Empty if none.
    """
    PIDs = os.popen('pidof %s' % shellArg(ProcessName)).read().strip()
    if PIDs != '':
        return PIDs
    PIDs = []
    for psLine in os.popen('ps|grep -v grep|grep %s' % shellArg(ProcessName)).readlines():
        PIDs.append(psLine.split()[0])
    return ' '.join(PIDs)

def NormalizePhoneNumber(phoneNumber):
    """
    @param string phoneNumber A human written phone number like '+49 1234 55 666 - 92' or '(09876) 54321'
    @return string normalized phone number; remove everything except digits and plus sign.
    """ 
    return re.sub('[^0-9+]', '', phoneNumber)

def IsSamePhoneNumber(number1, number2):
    """
    @param string number1 A human written phone number to be compared.
    @param string number2 A human written phone number to be compared.
    @return bool True iff dialing either number will ring the same phone. 
    """
    norm1 = NormalizePhoneNumber(str(number1)).replace('+','0')
    norm2 = NormalizePhoneNumber(str(number2)).replace('+','0')
    while norm1[0:2] == '00': norm1 = norm1[1:]
    while norm2[0:2] == '00': norm2 = norm2[1:]
    return norm1 == norm2

def IsHostReachable(hostname, pingCount = 3):
    """
    Check if remote host is available with a ping command and return
    @return bool false = no or bad ping; true = good ping (hostname is available)
    """
    return os.WEXITSTATUS( os.system( 'ping -q -c %d %s >/dev/null 2>&1' % (pingCount, hostname) ) ) == 0 
        
def IsInternetReachable():
    """
    Check if the phone potentially can reach computers in the internet.
    @return bool
    """
    return IsHostReachable('8.8.8.8')   # The google nameserver should always be up...

def IsFileReadable(*File):
    """
    @return bool True iff File exists and is readable. Silent check. Effective uid applies.
    """
    try:
        return open(os.path.join(*File), 'r').close() or True
    except:
        return False

def DeleteFile(*File):
    """
    Delete file. Very verbose.
    @return bool True iff the file has been (or was already) deleted.
    """
    Filename = os.path.join(*File)
    try:
        os.remove(Filename)
    except:
        if IsFileReadable(Filename):
            LOGGER.error('Failed to delete "%s".' % Filename)
            return False
        LOGGER.info('File "%s" was already deleted.' % Filename)
    else:
        LOGGER.info('File "%s" now deleted.' % Filename)
    return True

def SendSMSviaGSMmodem(RecipientNumber, Message, TryCountMax = 2):
    """
    Low Level function to send a SMS via GSM modem. The Message length must not exceed 160 chars.
    TryCountMax is number of total tries to re-activate GSM modem when the GSM modem control sequence has failed.
    @return bool Success indicator
    """
    assert(len(Message) <= 160)

    RecipientNumber = NormalizePhoneNumber(RecipientNumber)
    DelaySeconds = 2

    modem = None
    TryCount = 0
    while TryCount < TryCountMax:
        TryCount += 1

        if modem:
            modem.sendeof()
            modem.kill(9)
            time.sleep(DelaySeconds)
        modem = pexpect.spawn('pnatd', [], timeout = 2) # start pnatd
        #modem.logfile = sys.stdout

        try:
            time.sleep(1)
            modem.send('AT\r') # GSM modem returns 'OK' if working
            modem.readline()
            response = modem.readline().strip('\r\n')
        except pexpect.TIMEOUT:
            LOGGER.debug('GSM modem "AT" timeout.')
            continue

        if response != 'OK':
            LOGGER.debug('GSM modem did not reply correctly (%s).' % response)
            continue

        try:
            time.sleep(1)
            modem.send('AT+CMGF=1\r') # GSM modem returns 'OK' if working
            modem.readline()
            response = modem.readline().strip('\r\n')
        except pexpect.TIMEOUT:
            LOGGER.debug('GSM modem "AT+CMGF" timeout.')
            continue

        if response != 'OK':
            LOGGER.debug('GSM modem did not reply correctly (%s).' % response)
            continue 

        try:
            time.sleep(1)
            modem.send('AT+CMGS="%s"\r' % RecipientNumber) # GSM modem returns '> ' if working
            modem.readline()
        except pexpect.TIMEOUT:
            LOGGER.debug('GSM modem "AT+CMGS" timeout.')
            continue

        time.sleep(1)
        modem.send( '%s' % Message + chr(26) ) # set SMS message and Ctrl + Z to send message

        try:
            time.sleep(1)
            modem.readline()
            response = modem.readline().strip('\r\n')
        except pexpect.TIMEOUT:
            LOGGER.debug('GSM modem response "+CMGS:" timeout.')
            continue

        if not response.startswith('+CMGS:'):
            LOGGER.debug('GSM modem response "+CMGS:" failed (%s).' % response)
            continue 

        modem.readline()
        response = modem.readline().strip('\r\n')
        if response != 'OK':
            LOGGER.debug('GSM modem response "OK" failed (%s).' % response)
            continue 

        return True

    return False

##############################################################

class Configuration(dict):
    """
    Base class for file based configuration databases
    """
    ItemTypeName        = "configuration"
    CreateTemplate      = ''
    ConfigPrintFormat   = "%s = %s"                                     # (ConfigVar, QuotedConfigValue)
    ConfigParseFormat   = "^\\s*(\\S+)\\s*=\\s*([\"\']|)(.*?)\\2\\s*$"  # (ConfigVar, QuoteUsed, ConfigValue)
    CommentLineFormat   = "^\\s*(#.*)?$"
    CommentMakeFormat   = '# %s'
    DefaultValueFormat  = None
    TabSize             = 0     # 0 means: calc dynamically on _ConfigVars
    _OPT_CRYPT          = 'E'
    _OPT_PLAIN          = 'P'
    _ConfigVars         = {}    # known items; no restriction if empty. Structure: key: (re_pattern_of_value, default_value, options, comment).
    _Secret             = ''    # Key used to en/decrypt
    _upgradableConfigVars = {}  # class attribute served as fallback in case instance not yet loaded.

    def __init__(self, Filename = None):
        self.Filename = Filename
        if  self.TabSize == 0:
            self.TabSize = max(map(len, self._ConfigVars))
            if  self.TabSize == 0:
                self.TabSize = 18

    def copy(self):
        """@return a shallow copy (not a deep copy)"""
        other = self.__class__()
        for k in self:
            other[k] = self[k]
        for attr, value in self.__dict__.items():
            setattr(other, attr, value)
        return other 

    def _error(self, Message, ReturnCode = False):
        LOGGER.error(Message)
        return ReturnCode

    def _warning(self, Message, ReturnCode = False):
        LOGGER.warning(Message)
        return ReturnCode

    def _info(self, Message, ReturnCode = False):
        LOGGER.info(Message)
        return ReturnCode

    def _decode(self, ConfigVar, ConfigValue):
        return DecodeAES(AES.new(self._Secret), ConfigValue)
        
    def _genConfigLine(self, ConfigVar, ConfigValue):
        return self.ConfigPrintFormat % ( ConfigVar.ljust(self.TabSize), ConfigValue)

    def show(self, Password = None, PasswordVar = 'MASTERNUMBER'):
        """
        Print decoded content of configuration file. If Password is given, it must match.
        @return bool Success indicator
        """
        if Password == None:
            return self.load(PrintContent = True)
        if self.load() and self.get(PasswordVar) == Password:
            return self.load(PrintContent = True)
        return self._error('password mismatch')

    def load(self, PrintContent = False, Silent = False):
        """
        Load the configuration from file; optionally Silent about errors; optionally PrintContent while loading.
        @return bool Success indicator. The previous content of the configuration is lost in any case.
        """
        self.clear()
        self._upgradableConfigVars = {}
        Thing = '%s file "%s"' % (self.ItemTypeName, self.Filename)
        try:
            f = open(self.Filename, 'r')
        except:
            if not Silent: self._error('The %s could not be found.' % Thing)
            return False
        ConfigLines = [line.replace("\n","").replace("\r","") for line in f.readlines()] 
        f.close()
        if ConfigLines == []:
            if not Silent: self._info('The %s is empty.' % Thing)
            return False

        ConfigValuePattern = re.compile(self.ConfigParseFormat)
        CommentLinePattern = re.compile(self.CommentLineFormat)
        DefaultValuePattern= self.DefaultValueFormat and re.compile(self.DefaultValueFormat)

        ok = True
        for Line in ConfigLines:
            if CommentLinePattern.match(Line):
                if PrintContent:
                    print Line
                continue
            ErrorMsg = None
            LineTokens = ConfigValuePattern.findall(Line)
            if len(LineTokens) == 0:
                ErrorMsg = 'Bad syntax in above line. Line ignored.'
            else:
                (ConfigVar, QuoteUsed, ConfigValue) = LineTokens[0]
                if len(self._ConfigVars) > 0: 

                    if not ConfigVar in self._ConfigVars:            
                        self._upgradableConfigVars[ConfigVar] = True
                        ErrorMsg = 'Unknown %s "%s" ignored.' % (self.ItemTypeName, ConfigVar)
                    else:
                        Pattern, DefaultValue, VarOptions = self._ConfigVars[ConfigVar][0:3]
                        if DefaultValue == None:
                            self._upgradableConfigVars[ConfigVar] = True
                            ErrorMsg = 'Obsolete %s "%s" ignored.' % (self.ItemTypeName, ConfigVar)
                        else:
                            EncryptedConfigValue = ConfigValue
                            if self._OPT_CRYPT in VarOptions:
                                try:
                                    ConfigValue = self._decode(ConfigVar, ConfigValue)
                                except:
                                    ErrorMsg = 'Failed to decode %s value for %s.' % (self.ItemTypeName, ConfigVar)
                            if Pattern and not re.match(Pattern, ConfigValue):
                                # If this is a known old-style default value, upgrade to new default value:
                                if DefaultValuePattern and (re.match(DefaultValuePattern, ConfigValue)
                                                        or  re.match(DefaultValuePattern, EncryptedConfigValue)):
                                    self._upgradableConfigVars[ConfigVar] = True
                                    ErrorMsg = 'Value of %s "%s" needs upgrade.' % (self.ItemTypeName, ConfigVar)
                                else:
                                    self._warning('Value of %s "%s" does not match /%s/.' % (self.ItemTypeName, ConfigVar, Pattern))
            if ErrorMsg:
                if ok and not PrintContent: self._error('In %s:' % Thing)
                print Line
                ok = self._error(ErrorMsg)
            else:
                if PrintContent:
                    print self._genConfigLine(ConfigVar, QuoteUsed+ConfigValue+QuoteUsed)
                self[ConfigVar] = ConfigValue
        return ok

    def backup(self, Filename, Silent = False, Thing = None):
        """
        Export the configuration and generate a compact backup file.
        @param bool Silent Be silent about errors and success.
        @param string|None Thing Non-default description of what is being exported.
        @return bool Success indicator
        """
        if not Thing:
            Thing = '%s file "%s"' % (self.ItemTypeName, Filename)
        ok = True
        try:
            f = open(Filename, 'w')
        except Exception, e:
            if not Silent: self._error('Export failed to open %s for writing.' % Thing)
            return False
        try:
            temp = self.copy()
            keysToBackup = sorted(self._ConfigVars)
            keyErrorRetry = len(keysToBackup) * 2
            while keysToBackup:
                ConfigVar = keysToBackup.pop(0)
                Pattern, DefaultValue, VarOptions = self._ConfigVars[ConfigVar][0:3]
                if DefaultValue == None:
                    continue
                if ConfigVar in self:
                    ConfigValue = self[ConfigVar]
                else:
                    if not Silent:
                        ok = self._warning('Value of %s "%s" not defined. Using default.' % (self.ItemTypeName, ConfigVar))
                    try:
                        ConfigValue = temp._calcDefaultValue(DefaultValue)
                    except KeyError:
                        if  keyErrorRetry > 0 and keysToBackup:
                            keyErrorRetry -= 1
                            keysToBackup.append(ConfigVar)
                            continue
                        ConfigValue = 'default value calculation error'
                
                if Pattern and not re.match(Pattern, ConfigValue):
                    if not Silent:
                        ok = self._warning('Value of %s "%s" does not match /%s/.' % (self.ItemTypeName, ConfigVar, Pattern))
    
                Line = self._genConfigLine(ConfigVar, self._genFileValue(ConfigValue, VarOptions))
                f.writelines(Line + '\n')
                temp[ConfigVar] = ConfigValue

        except Exception, e:
            if not Silent:
                self._error('Export to %s failed: %s' % (Thing, e))
            ok = False
        finally:
            f.close
        if ok and not Silent:
            self._info('Export to %s successful.' % Thing)
        return ok

    def restore(self, Filename, Silent = False, Force = False, Thing = None):
        """
        Import configuration from an export or backup file. This is not just a file copy:
        * Only valid configuration items are imported.
        * Existing configuration items that are not present in the source remain unchanged.
        @param bool Force Continue on errors.
        @param bool Silent Be silent about errors and success.
        @param string|None Thing Non-default description of what is being imported.
        @return int Number of imported settings (even if they are not really set to a new value). Negative on errors.
        """
        if not Thing:
            Thing = '%s file "%s"' % (self.ItemTypeName, Filename)
        importCount = 0
        source = self.__class__(Filename = Filename)
        if not source.load(Silent = Silent):
            if not Silent:
                self._error('Importing %s failed.' % Thing)
            return importCount
        ok = True
        for ConfigVar in sorted(source.keys()):
            if self.update(ConfigVar, source[ConfigVar]):
                importCount += 1
                continue
            ok = False
            if not Silent:
                self._error('Importing %s failed for "%s".' % (Thing, ConfigVar))
            if not Force:
                break
        if ok and not Silent:
            self._info('Importing %s successful.' % Thing)
        return IF(ok, importCount, -importCount)

    def _calcDefaultValue(self, DefaultValue):  # may throw KeyError exception.
        val = DefaultValue
        if type(val).__name__ == 'tuple':
            if   len(val)  > 1: val = val[0] % tuple(map(lambda k: self[k], val[1:]))
            elif len(val) == 1: val = self[val[0]]
            else: val = ''
        return val

    def _genFileValue(self, ConfigValue, VarOptions):
        FileValue = str(ConfigValue)
        if self._OPT_CRYPT in VarOptions:
            FileValue = "'%s'" % EncodeAES(AES.new(self._Secret), ConfigValue)
        elif not FileValue.isdigit():
            FileValue = "'%s'" % FileValue
        return FileValue

    def getKnownConfigVars(self):
        """@return list of known (and not obsolete) config variables. Empty if no restriction."""
        return [key for key in self._ConfigVars if self._ConfigVars[key][1] != None]

    def needsUpgrade(self):
        """@return bool Indicator if configuration needs upgrade(). Detected after load()."""
        return len(self._upgradableConfigVars) > 0

    def isInitialValue(self, ConfigVar, ConfigValue):
        """@return bool Indicator if a given value is the initial value of given config var.""" 
        return IF(ConfigVar in self._ConfigVars, self._ConfigVars[ConfigVar][2] == ConfigValue, None)

    def update(self, ConfigVar, ConfigValue):
        """
        Set a config var to a new value. If required add or (if ConfigValue is None) remove the variable from configuration.
        @return bool True iff the setting was known and has been updated (or added or removed).
        """
        if ConfigVar == None:
            return self._error('Missing %s.' % (self.ItemTypeName))

        CommentLineIfNewOrOld = None
        VarOptions = ''
        # If the Configuration has known config vars, then check that the
        # ConfigVar to set is known and the ConfigValue is valid:
        if len(self._ConfigVars) > 0:
            DefaultValue = None
            if not (ConfigVar in self._ConfigVars):
                return self._error('Unknown %s "%s".' % (self.ItemTypeName, ConfigVar))

            Pattern, DefaultValue, VarOptions, CommentLineIfNewOrOld = self._ConfigVars[ConfigVar]
            if Pattern and ConfigValue != None and not re.match(Pattern, ConfigValue):
                return self._error('Value of %s "%s" does not match /%s/.' % (self.ItemTypeName, ConfigVar, Pattern))

            if DefaultValue == None and ConfigValue != None:
                return self._error('Obsolete %s "%s". %s' % (self.ItemTypeName, ConfigVar, CommentLineIfNewOrOld or ''))

        # If the ConfigVar is to be added or removed from the Configuration,
        # a comment may also be added . Prepare that comment:
        if CommentLineIfNewOrOld and ConfigValue != None:
            if not CommentLineIfNewOrOld.startswith('\n'): CommentLineIfNewOrOld = '\n' + CommentLineIfNewOrOld
            if not CommentLineIfNewOrOld.endswith('\n'):   CommentLineIfNewOrOld = CommentLineIfNewOrOld + '\n'

        # A ConfigValue of None is a request to remove that ConfigVar
        # from Configuration. If it is an known but obsolete ConfigVar,
        # it will just be commented out:
        Operation = "Updated"
        if ConfigValue == None:
            if CommentLineIfNewOrOld == None:
                UpdatedLine = None  # this will lead to removing the entry from file
                Operation = "Removed"
            else:
                UpdatedLine = self.CommentMakeFormat % self._genConfigLine(ConfigVar, CommentLineIfNewOrOld)
                Operation = "Commented out"
        else:
            UpdatedLine = self._genConfigLine(ConfigVar, self._genFileValue(ConfigValue, VarOptions))

        ConfigValuePattern = re.compile(self.ConfigParseFormat)
        CommentLinePattern = re.compile(self.CommentLineFormat)

        # Scan the file and update the line that defines the ConfigVar:        
        try:
            for Line in fileinput.FileInput(self.Filename, inplace = True):
                if not CommentLinePattern.match(Line):
                    LineTokens = ConfigValuePattern.findall(Line)    # (ConfigVar, QuoteUsed, ConfigValue)                
                    if len(LineTokens) > 0:
                        FoundVar, FoundValue = (LineTokens[0][0], LineTokens[0][2])
                        if FoundVar == ConfigVar:
                            if UpdatedLine == None:
                                continue;   # remove double entry
                            # Here it is: replace Line by UpdatedLine and consume UpdatedLine
                            Line, UpdatedLine = UpdatedLine, None
                        elif FoundVar in self._ConfigVars and not FoundVar in self:
                            # This line defines a different known config variable which
                            # is not yet load from the file. Load it implicitely if that
                            # can be done without errors (mainly for @APPVERSION key):            
                            FoundPattern, FoundDefaultValue, FoundVarOptions = self._ConfigVars[FoundVar][0:3]
                            if FoundDefaultValue != None:
                                if self._OPT_CRYPT in FoundVarOptions:
                                    try:
                                        FoundValue = self._decode(FoundVar, FoundValue)
                                    except:
                                        FoundValue = None
                                if not FoundPattern or re.match(FoundPattern, FoundValue):
                                    self[FoundVar] = FoundValue
                print Line.rstrip()
        except:
            if IsFileReadable(self.Filename):
                return self._error('Failed to update %s "%s".' % (self.ItemTypeName, ConfigVar))

        # If the line has not been updated so far, the ConfigVar has to be added to the file:
        if UpdatedLine:
            try:
                f = open(self.Filename, 'a')
            except:
                return self._error('Failed to add %s "%s".' % (self.ItemTypeName, ConfigVar))
            f.writelines(CommentLineIfNewOrOld or '');
            f.writelines(UpdatedLine + '\n')
            f.close()
            UpdatedLine = None
            Operation = "Added"

        # Keep dictionary consistent with updated value:
        if ConfigValue != None:
            self[ConfigVar] = ConfigValue
        elif ConfigVar in self:
            del(self[ConfigVar])

        if  ConfigValue == None:
            ConfigValue = "removed"
        self._info('%s %s "%s" (%s).' % (Operation, self.ItemTypeName, ConfigVar, str(ConfigValue)))
        return True

    def upgrade(self, Permanent = True):
        """
        Upgrade a new or just loaded() configuration to define all currently known config vars.
        Optionally non Permanent, e.g. keep corresponding config file unchanged.
        @return int Number of config vars upgraded (added, removed or re-encryped).
        """
        loadCount = 0
        keysToUpgrade = sorted(self._ConfigVars.keys())
        keyErrorRetry = len(keysToUpgrade) * 2
        while keysToUpgrade:
            key = keysToUpgrade.pop(0)
            Pattern, val, VarOptions = self._ConfigVars[key][0:3]
            if val != None:
                useDefaultValue = True
                if key in self:
                    if not (key in self._upgradableConfigVars):
                        continue    # up-to-date.
                    if self._OPT_CRYPT in VarOptions:
                        # This config var switched from plain to crypt
                        val = self[key]
                        useDefaultValue = False
                    else:
                        # Should be the upgrade of an old style default value
                        useDefaultValue = True
                if useDefaultValue:
                    try:
                        val = self._calcDefaultValue(val)
                    except KeyError:
                        if not (keyErrorRetry and keysToUpgrade): raise
                        keyErrorRetry -= 1
                        keysToUpgrade.append(key)
                        continue
                    if Pattern and not re.match(Pattern, val):
                        self._warning('Wrong default value or pattern for config var "%s", val="%s", pattern=%s' % (key, val, Pattern))
            elif not (key in self or key in self._upgradableConfigVars):
                continue

            if Permanent:
                if self.update(key, val):
                    loadCount += 1
            elif val != None:
                self[key] = val
            else:
                del(self[key])
        return loadCount

    def create(self, Template = None, Force = False):
        """
        Create a new configuration file that contains all known config vars with their default value.
        Optionally use a given Template; optionally Force overwrite existing config files.  
        @return bool True iff config file has been written. 
        """
        if  Template == None:
            Template = self.CreateTemplate; 
        Thing = '%s file "%s"' % (self.ItemTypeName, self.Filename)
        if not IsFileReadable(self.Filename):
            self._info('Creating new %s.' % Thing)
        elif Force:
            self._warning('Forced to create new %s.' % Thing)
        else:
            self._info('The %s already exists. Not initialized, left unchanged.' % Thing)
            return False

        try:
            f = open(self.Filename, 'w')
        except:
            return self._error('Failed to create %s.' % Thing)
        f.writelines(Template)
        f.close()
        return self.load() and self.upgrade() > 0

##############################################################

class UserConfiguration(Configuration):
    """
    The SMScon user configuration database
    """
    ItemTypeName        = "user setting"
    CreateTemplate      = '# %s user settings (initially written by version %s)\n' % (NAME, VERSION)
    SmsCommandPreFix    = 'COM_'       # _ConfigVars with that prefix define commands that can be sent via SMS
    DefaultValueFormat  = '^\+?[X.@]+$' # '+XXXXXXXXXXX', 'XXXX@XXXXXXXX.XX', 'XXXX.XXXXXXXX.XX', ...
    _PhoneNumber        = '^\+?[0-9]+$'
    _CmdPattern         = ''
    _BoolPattern        = '^(yes|no)$'
    _IntPattern         = '^\d+$'
    _FloatPattern       = '^\d+(\.\d+)?$'
    _EmailPattern       = '^[^@]+@[^@]+$'
    _IPforSSHPattern    = '^(\*|localhost|\d+(\.\d+)*)$'
    _OPT_CRYPT          = Configuration._OPT_CRYPT
    _OPT_PLAIN          = Configuration._OPT_PLAIN
    _APPVERSIONVARNAME  = '@APPVERSION'
    _ConfigVars = {
        # SETTINGNAME:      (re_pattern_of_value, default_value,    options,    comment),
        _APPVERSIONVARNAME: ('^\d+(\.\d+)*$',   VERSION,            _OPT_PLAIN, '# %s configuration file written by:' % NAME),

        # SMS command source:
        'SENDERNUMBER':     (_PhoneNumber,      '+0000000000',      _OPT_PLAIN, '# SMS number of recent controller. Receives SMS notifications:'),
        'MASTERNUMBER':     (_PhoneNumber,      ('SENDERNUMBER',),  _OPT_CRYPT, '# SMS number of master controller (encrypted):'),

        # List of known sms commands:
        'COM_CHECK':        (_CmdPattern,       'Check',            _OPT_PLAIN, '# Command to ask for help:'),
        'COM_REBOOT':       (_CmdPattern,       'Reboot',           _OPT_PLAIN, '# Command to reboot the phone:'),
        'COM_POWEROFF':     (_CmdPattern,       'Poweroff',         _OPT_PLAIN, '# Command to shutdown the phone:'),
        'COM_POWER':        (_CmdPattern,       'Power',            _OPT_PLAIN, '# Command to ask for phone battery status:'),
        'COM_LOCATION':     (_CmdPattern,       'Location',         _OPT_PLAIN, '# Command to ask current GPS location:'),
        'COM_REMOTEON':     (_CmdPattern,       'Remoteon',         _OPT_PLAIN, '# Command to start outgoing ssh connection:'),
        'COM_REMOTEOFF':    (_CmdPattern,       'Remoteoff',        _OPT_PLAIN, '# Command to stop outgoing ssh connection:'),
        'COM_CAMERA':       (_CmdPattern,       'Camera',           _OPT_PLAIN, '# Command to take picture:'),
        'COM_CALL':         (_CmdPattern,       'Call',             _OPT_PLAIN, '# Command to start an outgoing call:'),
        'COM_LOCK':         (_CmdPattern,       'Lock',             _OPT_PLAIN, '# Command to lock the phone:'), 
        'COM_UNLOCK':       (_CmdPattern,       'Unlock',           _OPT_CRYPT, '# Command to unlock the phone (encrypted):'),
        'COM_TRACKON':      (_CmdPattern,       'Trackon',          _OPT_PLAIN, '# Command to start GPS tracking:'),
        'COM_TRACKOFF':     (_CmdPattern,       'Trackoff',         _OPT_PLAIN, '# Command to stop GPS tracking:'),
        'COM_CUSTOM':       (_CmdPattern,       'Script',           _OPT_PLAIN, '# Command to run predefined user script:'),
        'COM_CUSTOMLOG':    (_CmdPattern, ('%s.log', 'COM_CUSTOM'), _OPT_PLAIN, '# Command to ask for user script output:'),
        'COM_SHELL':        (_CmdPattern,       'uptime',           _OPT_PLAIN, '# Command to execute that shell command. Returns output as notification:'),
        'COM_ALARM':        (_CmdPattern,       'Alarm',            _OPT_PLAIN, '# Command to play predefined alarm.wav:'),
        'COM_RESTART':      (_CmdPattern,       'Restart',          _OPT_PLAIN, '# Command to restart the daemon:'),
        'COM_LOG':          (_CmdPattern,       'Log',              _OPT_CRYPT, '# Command to ask for application log (encrypted):'),

        # Prefix/suffix for all command strings (in SMS message only): 
        'SMS_COMPREFIX':    (_CmdPattern,       '',                 _OPT_CRYPT, '# Prefix that must be present before all COM_XXX commands [even COM_SHELL] (encrypted):'),
        'SMS_COMSUFFIX':    (_CmdPattern,       '',                 _OPT_CRYPT, '# Suffix that must be present behind all COM_XXX commands [even COM_SHELL] (encrypted):'),

        # REMOTE command source
        'ENABLECHECKHOST':  (_BoolPattern,      'no',               _OPT_PLAIN, '# Enable checking URL for commands:'),
        'CHECKHOST':        ('.',        'http://example.com/file', _OPT_CRYPT, '# The URL to check (encoded):'),
        'CHECKTIME':        (_FloatPattern,     '15.0',             _OPT_PLAIN, '# Minutes between URL checks:'),

        # Email settings
        'EMAILADDRESS':     (_EmailPattern,     'to@example.com',   _OPT_PLAIN, '# Recipient of EMAIL notifications (regular owner of this phone, please contact him!):'),
        'USER':             ('',                '',                 _OPT_CRYPT, '# SMTP username for EMAILFROM (encrypted):'),
        'PASSWORD':         ('',                '',                 _OPT_CRYPT, '# SMTP password for EMAILFROM (encrypted):'),
        'EMAILFROM':        (_EmailPattern,     'from@example.com', _OPT_CRYPT, '# Sender address of EMAIL notification (encrypted):'),
        'MAILSERVER':       ('.',               'smtp.example.com', _OPT_CRYPT, '# SMTP server hostname or IP (encrypted):'),
        'MAILPORT':         (_IntPattern,       '25',               _OPT_PLAIN, '# SMTP server port number:'),

        # SSH settings
        'REMOTEHOST':       ('.',               'cmd.example.com',  _OPT_CRYPT, '# ssh server hostname or IP [to build tunnel] (encrypted):'),
        'REMOTEPORT':       (_IntPattern,       '22',               _OPT_PLAIN, '# ssh server port number [to build tunnel]:'),
        'REMOTEUSER':       ('',                'username',         _OPT_CRYPT, '# ssh server username [to build tunnel] (encrypted):'),
        'REMOTEPASSWORD':   ('',                'password',         _OPT_CRYPT, '# ssh server password [to build tunnel] (encrypted):'),
        'REMOTELISTENIP':   (_IPforSSHPattern,  '*',                _OPT_CRYPT, '# ssh server IP address [entry of tunnel itself] that will be forwarded to phone (*|localhost|IP):'),
        'REMOTELISTENPORT': (_IntPattern,       '8080',             _OPT_CRYPT, '# ssh server port number [entry of tunnel itself] that will be forwarded to phone:'),
        'REMOTE2LOCALPORT': (_IntPattern,       '22',               _OPT_PLAIN, '# Local port number to be forwarded to [exit of tunnel itself] on the phone:'),

        # Notification settings
        'COMMANDREPLY':     (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification if command accepted (yes|no):'),
        'MESSAGESEND':      ('^sms|email|both|none$',   'sms',      _OPT_PLAIN, '# How to send notifications (sms|email|both|none):'),
        'ENABLERESEND':     (_BoolPattern,      'no',               _OPT_PLAIN, '# On failure try to re-send reply message (yes|no):'),
        'RESENDTIME':       (_FloatPattern,     '15.0',             _OPT_PLAIN, '# Minutes between message re-send tries:'),
        'RESENDMAXTRY':     (_IntPattern,       '10',               _OPT_PLAIN, '# Maximum number of message re-send tries (0=no limit):'),
        'SMS_PARTS_LIMIT':  (_IntPattern,       '10',               _OPT_PLAIN, '# A SMS notification produces no more than this number of single SMS (0=no limit):'),
        'LOG_TIMEFORMAT':   ('%',           '%Y-%m-%d %H:%M:%S',    _OPT_PLAIN, '# Format to use for timestamps (see strftime):'),

        # GPS settings
        'GPSTIMEOUT':       (_FloatPattern,     '1800',             _OPT_PLAIN, '# Seconds until give up acquiring GPS coordinates:'),
        'GPSPOLLING':       (_IntPattern,       '3',                _OPT_PLAIN, '# Number of GSP coordinate acquisitions for COM_LOCATIONs best of:'),
        'GPSINTERVAL':      (_FloatPattern,     '60',               _OPT_PLAIN, '# Seconds between GSP coordinate acquisitions for COM_TRACKON and COM_LOCATION (10..120:10|20|30|60|120)'),
        'GPSSEND':          ('^sms|email|both$','sms',              _OPT_PLAIN, '# How to send GPS notifications (sms|email|both):'),

        # Miscellaneous settings
        'SILENCEDEVICE':    (_BoolPattern,      'no',               _OPT_PLAIN, '# Set phone silent before executing command (yes|no):'),
        'AUTODEVICELOCK':   (_BoolPattern,      'no',               _OPT_PLAIN, '# Lock phone before executing any command except un/lock (yes|no):'),
        'SIMUNLOCK':        (_BoolPattern,      'no',               _OPT_PLAIN, '# Automatically unlock phone when new SIM is inserted (yes|no):'),
    
        # Sensor detections
        'KEYBOARDDETECT':   (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification if keyboard is sliding (yes|no):'),
        'AUTOBATREPORT':    (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification on battery discharge (yes|no):'),

        # Obsolete settings: 
        'DISABLESMS':       (None,              None,               _OPT_PLAIN, 'setting removed since version 0.8.0-1'),
    }
    _Secret = '6983456712936753'

    def __init__(self, Filename = None):
        Configuration.__init__(self, Filename = Filename or os.path.join(InstPath, ConfigFile))

    def _decode(self, ConfigVar, ConfigValue):
        try:
            return Configuration._decode(self, ConfigVar, ConfigValue)
        except:
            # Handle config vars that were formerly stored un-encrypted
            if ConfigVar in ['CHECKHOST', 'REMOTEHOST', 'COM_LOG', 'COM_UNLOCK', 'EMAILFROM', 'MAILSERVER']:
                self._upgradableConfigVars[ConfigVar] = True
                return ConfigValue
            raise

    def update(self, ConfigVar, ConfigValue):
        """Same as base class with support for automatic _APPVERSION update."""
        
        # Prevent to commands being defined as he same string, except the empty string (which disables):
        if ConfigVar.startswith(self.SmsCommandPreFix) and ConfigValue != None and ConfigValue != "":
            for OtherVar in self.getSmsCommandList():
                if OtherVar != ConfigVar and self[OtherVar] == ConfigValue:
                    return self._error('Command string "%s" already used for command "%s".' % (ConfigValue, OtherVar))

        return Configuration.update(self, ConfigVar, ConfigValue) and (
                (ConfigVar == self._APPVERSIONVARNAME   # --> already updated above or ...
                 or self.get(self._APPVERSIONVARNAME) == VERSION    # --> version up to date or ...
                 or Configuration.update(self, self._APPVERSIONVARNAME, VERSION)))

    def getSmsCommandList(self):
        """@return list of config vars that are SMS commands"""
        return [cmd for cmd in self._ConfigVars if cmd in self and cmd.startswith(self.SmsCommandPreFix)]

##############################################################
# IMSI interface

class IMSIConfiguration(Configuration):
    """
    Instances of this class represent the (one and only) IMSI code database
    """ 
    ItemTypeName        = "IMSI code"
    ConfigPrintFormat   = "%s%s"
    ConfigParseFormat   = "^\\s*(.*?)\\s*()()$"
    TabSize             = 18

    def __init__(self):
        Configuration.__init__(self, Filename = os.path.join(InstPath, CodeFile))

    def _genConfigLine(self, ConfigVar, ConfigValue):
        return ConfigVar

def GetPhoneCode(CodeType):
    """
    Get current SIM IMSI code or device hardware IMEI code.
    @return string|None The requested code.
    """
    if CodeType == 'IMEI':
        Command = 'dbus-send --system --type=method_call --print-reply --dest=com.nokia.phone.SIM /com/nokia/phone/SIM/security Phone.Sim.Security.get_imei'
    elif CodeType == 'IMSI':
        Command = 'dbus-send --system --type=method_call --print-reply --dest=com.nokia.phone.SIM /com/nokia/phone/SIM Phone.Sim.get_imsi'
    else:
        LOGGER.error('unknown code type "%s"' % CodeType)
        return None

    try:
        E = re.compile('string "(\S+)"')
        return E.findall(os.popen(Command).read())[0]
    except:
        LOGGER.error('could not get %s code.' % CodeType)
        return None

##############################################################
# smscon_daemon interface

class smscon_daemon:
    """
    Instances of this class are used to control the smscon daemon.
    """
    TELL_RELOAD    = signal.SIGINT
    TELL_TERMINATE = signal.SIGTERM

    def runs(self):
        """
        Check if smscon_daemon is active.
        @return string List of PIDS. Empty if none (evaluates to false in boolean expressions).
        """
        return pidof(os.path.join(InstPath, DaemonFile))
    
    def start(self, Mode = None, Function = None):
        """
        Start new smscon_daemon instance (even if some daemon instances already run).
        @return void.
        """
        DaemonArgs = IF(Mode or Function, ' %s "%s" ' % (Mode, Function), '')
        os.system( '"%s"%s&' % (os.path.join(InstPath, DaemonFile), DaemonArgs))
        LOGGER.info('Daemon %s requested to start%s.' % (DaemonFile, DaemonArgs))
    
    def stop(self):
        """
        Kill smscon_daemon(s).
        @return: string with PIDs of daemons stopped. Empty if none were running.
        """
        PIDs = self.tell(self.TELL_TERMINATE)
        if PIDs:
            LOGGER.info('Daemon %s stopped [PID=%s].' % (DaemonFile, PIDs) )
        else:
            LOGGER.info('Daemon %s was not running.' % DaemonFile)
        return PIDs
    
    def tell(self, SignalNumber):
        """
        Signal smscon_daemon(s).
        @return: string With list of daemons signaled.
        """
        PIDs = self.runs()
        if PIDs:
            os.system('kill -%d %s' % (abs(SignalNumber), PIDs))
        return PIDs
    
##############################################################
# remote shell connection interface 

class smscon_remote_sh(dict):

    def __init__(self, args):
        for k in ('REMOTEHOST', 'REMOTEPORT', 'REMOTEUSER', 'REMOTEPASSWORD',
                  'REMOTELISTENIP', 'REMOTELISTENPORT', 'REMOTE2LOCALPORT'):
            setattr(self, k,  args.get(k) or '')

    def _getSSHcmdline(self, nParts = 3):
        return ' '.join((
            # the command and flags to tell contacted ssh server to build tunnel (-n -N -T -f):
                'ssh -nNTf',
            # definition of the entry and exit points of the tunnel to build:
                '-R %s:%s:%s:%s' % (self.REMOTELISTENIP, self.REMOTELISTENPORT, 'localhost', self.REMOTE2LOCALPORT),
            # port, user and host to contact ssh server to build tunnel:
                '-p %s %s@%s' % (self.REMOTEPORT, self.REMOTEUSER, self.REMOTEHOST)
            )[:nParts])

    def runs(self):
        """
        Check if ssh is active or not running.
        @return string list of SSH PIDs. Empty if none is running (evaluates to bool False)
        """
        # -- This would also detect ssh processes not created for reverse-ssh connection:
        # -- return os.popen('pidof ssh').read().strip('\n') != ''
        PIDs = []
        # Do not use the complete command line for grep because ps may show it only party.
        for psLine in os.popen('ps|grep -v grep|grep "%s"' % self._getSSHcmdline(2)).readlines():
            PIDs.append(psLine.split()[0])
        return ' '.join(PIDs)

    def start(self):
        """
        Silently start reverse-ssh connection to remote host.
        @return (bool, message) Success indicator, error message
        """
        (Output, ExitStatus) = pexpect.run(self._getSSHcmdline(),
            events = { '(?i)(password|passphrase)':self.REMOTEPASSWORD + '\n', '(?i)(yes/no) ?':'yes' + '\n' },
            withexitstatus = True )
        return (Output, ExitStatus)
        
    def stop(self, Verbose = False):
        """
        Kill reverse-ssh connection to remote host.
        @return: string with PIDs of connections stopped. Empty if none were running.
        """
        if Verbose:
            LOGGER.info('Trying to stop ssh connections to "%s".' % self.REMOTEHOST)
        PIDs = self.runs()
        if PIDs:
            os.system('kill %s' % PIDs)
            if Verbose:
                LOGGER.info('ssh connections stopped [PIDs=%s].' % PIDs)
        else:
            if Verbose:
                LOGGER.info("ssh connection wasn't active")
        return PIDs

##############################################################
# Custom script interface

class smscon_script:
    """
    Instances of this class represent the (one and only) user script.
    """

    def __init__(self):
        self.Filename = os.path.join(InstPath, ScriptFile)

    def init(self, Force = False):
        """
        Check if smscon_script exists, else create new default one.
        @return bool True iff the script file exists or could be created.
        """
        ScriptTemplate = """#!/bin/sh
# %s custom script (initialized by %s version %s)\n' """ % (NAME, NAME, VERSION) + """
# Script will start on COM_CUSTOM command and executes asynchronously.
# Script output (stdout and stderr) can be retrieved by COM_CUSTOMLOG command.

# Here are some example commands you might consider:
echo ps:
ps
echo netstat:
netstat -tean    
"""
        Filename = self.Filename
        Thing = 'script file "%s"' % self.Filename

        if not IsFileReadable(Filename):
            LOGGER.info('Creating new %s.' % Thing)
        elif Force:
            LOGGER.warning('Forced to create new %s.' % Thing)
        else:
            LOGGER.info('The %s already exists. Not initialized, left unchanged.' % Thing)
            return True

        try:
            f = open(Filename, 'w')
        except:
            LOGGER.error('Failed to create %s.' % Thing)
            return False            
    
        f.writelines(ScriptTemplate)
        f.close()

        if os.WEXITSTATUS( os.system( 'chmod 755 "%s"' % Filename) ) != 0:
            LOGGER.error('Failed to set user permissions on "%s".' % Filename)
            return False            

        if os.WEXITSTATUS( os.system( 'chown root "%s"' % Filename ) ) != 0:
            LOGGER.error('Failed to set owner of "%s".' % Filename)
            return False            

        LOGGER.warning("The initial %s does yet nothing. Edit it directly!" % Thing)
        return True        

    def stop(self):
        """
        Kill smscon_script file (if running).
        @return: string with PIDs of scripts stopped. Empty if none were running.
        """
        PIDs = pidof(self.Filename)
        if PIDs:
            os.system('kill %s' % PIDs)
            LOGGER.info('Script %s stopped [PIDs=%s].' % (ScriptFile, PIDs))
        else:
            LOGGER.info('Script %s was not running.' % ScriptFile)
        return PIDs

##############################################################

def CreateBootfile():
    """
    Set start smscon_daemon at device boot.
    @return bool True iff a BootFile exists or could be created.
    """
    BootCode = 'description "%s %s - nokia n900 remote control utility"\n' % (NAME, VERSION) + \
"""
start on started hildon-desktop
stop on starting shutdown

console none

script
    while [ ! -e "%s" ]; do
        sleep 3
    done
    exec "%s" -start > "%s" 2>&1
end script

""" % (InstPath, os.path.join(InstPath, NAME), os.path.join(InstPath, NAME+'_boot.log'))

    Filename = os.path.join(BootPath, BootFile)
    if IsFileReadable(Filename):
        LOGGER.info('Boot file "%s" already exists. Not changed.' % Filename)
        return True  

    try:
        f = open(Filename, "w")
    except:
        LOGGER.error('Failed to create boot file "%s".' % Filename)
        return False

    f.writelines(BootCode)
    f.close()
    LOGGER.info('Boot file "%s" created & active.' % Filename)
    return True 

##############################################################

def StartLogging(instanceName, logLevel = None, withConsole = True, withEmitterFxn = False):
    """
    Start logging for specified instance at specified level (default all)
    @return Logger the logger you have to use.
    """
    if logLevel == None:
        logLevel = logging.INFO
    if XDEBUG:
        logLevel = logging.DEBUG

    logger = logging.getLogger(instanceName)
    logger.setLevel(logLevel)

    if os.geteuid() == 0:
        setattr(logger, 'LogFilename', os.path.join(LogPath, LogFile))
        fh = logging.FileHandler(logger.LogFilename)
        fh.setLevel(logLevel)
        fh.setFormatter(logging.Formatter('%(asctime)s[%(name)s] %(levelname)s: %(message)s', datefmt = TimeFormat))
        logger.addHandler(fh)

    if withConsole:
        ch = logging.StreamHandler()
        ch.setLevel(logging.INFO)
        ch.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
        logger.addHandler(ch)

    if withEmitterFxn:

        class callbackHandler(logging.Handler):
            '''Class defining logging handler that call a foreign emit function.'''
            def __init__(self, emitter, level = logging.NOTSET):
                logging.Handler.__init__(self, level)
                self.emitter = emitter
            def emit(self, logRecord):
                self.emitter(logRecord)

        xh = callbackHandler(emitter = withEmitterFxn, level = logLevel)
        xh.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
        logger.addHandler(xh)

    global LOGGER
    if LOGGER == None:
        LOGGER = logger

    return logger
    
##############################################################

if __name__ == "__main__":
    print VERSION
