# -*- coding: utf-8 -*-
##############################################################
# smscon_service - remote control daemon for Nokia N900 phone#
# $Id: smscon_service.py 25 2011-12-14 23:58:51Z yablacky $
##############################################################

# standard modules
import os
import sys
import time
import re
import random
import signal
import dbus
from dbus.mainloop.glib import DBusGMainLoop

# for GPS control
import location
import gobject
from operator import itemgetter 

# for email sending
import smtplib
import socket
import urllib
from email.MIMEText import MIMEText
from email.MIMEImage import MIMEImage
from email.MIMEMultipart import MIMEMultipart

import smsconlib as smscon
from smsconlib import InstPath, LogPath, PhotoName, MapName, AudioFile, DaemonFile 
from smsconlib import IF, IsHostReachable, IsInternetReachable, IsFileReadable, DeleteFile, pidof, SendSMSviaGSMmodem

LOGGER = smscon.LOGGER = smscon.StartLogging('DAEMON', withConsole = False)

#############################################
#    constants

PingCountMax                = 3                                 # max number of pings for connection status check
LastKnownGoodConfigFile     = '.%s_lkg' % smscon.NAME

#############################################
#    variables

GpsCoordinateList           = []
GpsIsActive                 = False                             # true if GPS device is enabled
GpsIsTest                   = False                             # true if GPS device is enabled from test mode command
GpsDataOK                   = False                             # true if GPS device receives valid coordinates
GpsStartTime                = 0

ConnectionType              = None 
ConnectionIAP               = None
ConnectionName              = None
ConnectionBehaviour         = None

IsKeyboardSliderCheckActive = False
IsBatteryStatusCheckActive  = False
IsCheckHostActive           = False

CheckHostLastCommand        = '-1'      # "restart"
CheckHostStatusCount        = 0         # positive: successes, negative: errors
CheckHostStatusTimestamp    = time.gmtime()

smscnf = None               # the UserConfiguration actually used by the daemon.

#############################################

def GetTimeString(*fromTime):
    """
    @return string Current time in custom format.
    """
    return time.strftime(smscnf.LOG_TIMEFORMAT, *fromTime)

#############################################

def RevealThat(Subject, Text, SendMethod = None, Attachment = None, Attachhint = None):
    """
    Send message to all enabled targets. In case SMS is enabled, the
    message is send to the originating SMS SENDERNUMBER as well as to
    the SMS MASTERNUMBER if set and is a different number.
    The device hostname and a timestamp is added to the message automatically.
    @param string Subject Short description of event.
    @param string|list Text Message body.
    @param string SendMethod Defaults to smscnf.MESSAGESEND
    @param string|none Attachment Name of file to attach (to email only)
    @param string|none Attachhint Hint about the attachment file's MIME type. 
    @return void
    """
    if  SendMethod == None:
        SendMethod = smscnf.MESSAGESEND
        
    if type(Text).__name__ == 'list': Text = ''.join(Text)
    Hostname = os.uname()[1]
    Revealed = []
    if SendMethod in ['both', 'sms']:
        Message = 'N900[%s]\n' '%s\n' '-%s-\n' % (Hostname, GetTimeString(), Subject) + Text

        # Split Message into list of lines to give SMSsend hint where to split 
        TextLines = []
        while Message != '':
            Line, Sep, Message = Message.partition('\n');
            TextLines.append(Line + Sep);

        # Send to the last known SMS sender
        if SMSsend(smscnf.SENDERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
            Revealed.append(smscnf.SENDERNUMBER)

        # Send to the master if that is different 
        if smscnf.MASTERNUMBER != smscnf.SENDERNUMBER:
            if Revealed: 
                TextLines.append('\nBC:%s' % smscnf.SENDERNUMBER)   # "blind copy" information
            if SMSsend(smscnf.MASTERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
                Revealed.append("MASTERNUMBER") # do not expose the encrypted number this way!

    if SendMethod in ['both', 'email']:
        if Revealed:
            Text += '\nSMS copies have been sent to: (' + ', '.join(Revealed) + ')' 

        if EMAILsend(EmailTo     = smscnf.EMAILADDRESS,
                  Subject        = 'NOKIA N900[%s]: %s' % (Hostname, Subject),
                  Text           = 'Nokia phone "%s" at %s:\n' '%s\n' % (Hostname, GetTimeString(), Text),
                  SuccessLogMsg  = 'send email message (subject=%s) to <%s>' % (Subject, smscnf.EMAILADDRESS),
                  Attachment = Attachment,
                  Attachhint = Attachhint):
            Revealed.append(smscnf.EMAILADDRESS)

    if not Revealed:
        LOGGER.info('Message sending disabled by user setting (%s).' % SendMethod)

#############################################
#    SMS
#############################################

def OnSmsReceived(PDUmessage, MessageCenter, SomeString, SenderNumber):
    """
    Receive SMS and try to execute as command.
    @return void
    """
    n = int(PDUmessage[1])
    n = n // 2 + n % 2 + 13

    MessageArray = PDUmessage[n:len(PDUmessage)]
    try:
        MessageText = DeOctifyMessage(MessageArray)     # decode SMS message to plain text
    except Exception, e:
        LOGGER.critical("DeOctifyMessage problem: %s", e)
        LOGGER.critical('SMS Message Bytes:' + ', '.join( ["0x%02X" % int(b) for b in MessageArray] ))
        return
    LOGGER.debug('SMS Message Chars:' + ', '.join( ["0x%02X[%s]" % (ord(b), b) for b in MessageText] ))
    LOGGER.debug("SMS received:'%s'" % MessageText)

    if (    MessageText.startswith(smscnf.SMS_COMPREFIX) and
            MessageText.  endswith(smscnf.SMS_COMSUFFIX)  ):
        ProcessCommand(MessageText[len(smscnf.SMS_COMPREFIX) : len(MessageText)-len(smscnf.SMS_COMSUFFIX)],
                       SenderPhoneNumber = SenderNumber)

def SMSsend(RecipientNumber, MessageList, SuccessLogMsg):
    """
    Send SMS message to RecipientNumber.
    @return bool False iff the message has definitely not been sent. True otherwise.
    """
    if not RecipientNumber:
        return False
    MessageLengthMax = 160  # max. number of characters in one SMS message
    MessageText = ''.join(MessageList)

    if len(MessageText) <= MessageLengthMax:
        if SendToGSMmodem(RecipientNumber, MessageText, SuccessLogMsg):
            return True
        if smscnf.ENABLERESEND != 'yes':
            return False
        LOGGER.warning('Trying to re-send SMS message after %s minutes.' % float(smscnf.RESENDTIME))
        tryCount = { 'value': 0 }
        gobject.timeout_add(int(float(smscnf.RESENDTIME)) * 60000,
                            lambda a,b,c,**kwargs: not SendToGSMmodem(a,b,c,**kwargs),
                            RecipientNumber, MessageText, SuccessLogMsg, TryCount = tryCount)
        return True

    # Message must be split
    if 0:
        # Fast and simple split but may require less SMS than sophisticated split:
        part = 0
        while len(MessageText) > 0:
            part += 1
            MessageText = ('-#%s-\n' % part) + MessageText
            SMSsend(RecipientNumber, [MessageText[0:MessageLengthMax]], SuccessLogMsg + (' [part %d]' % part))
            MessageText = MessageText[MessageLengthMax:]
        return True

    # Sophisticated split:
    # Calculates n of m part header and tries not to split within messages.
    PartHeader = lambda n,m: '#%d/%d\n' % (n,m)
    def SplitMessages(MessageList, MessageLengthMax, AssumeTotalPartCount):
        """
        Join the messages in the given list and split that to parts of MessageLengthMax.
        If possible to not split within a single message.
        @return list of messages each of which is no longer than MessageLengthMax chars.
        """
        SplittedMessages = []
        MessageParts = []
        MessageList = MessageList[:]
        while MessageList:
            Msg = MessageList.pop(0)
            if len(Msg) == 0:
                continue
            if len(MessageParts) == 0:
                MessageParts = [ PartHeader(len(SplittedMessages)+1, AssumeTotalPartCount) ]
                if len(MessageParts[0]) >= MessageLengthMax:
                    MessageParts[0] = '' # no header if header alone exceeds max length.
            if sum(map(len,MessageParts)) + len(Msg) <= MessageLengthMax:
                MessageParts.append(Msg)
                continue
            lenNow = 0
            if len(MessageParts) == 1:  # No change, must split within message.
                lenNow = MessageLengthMax - len(MessageParts[0])
                MessageParts.append(Msg[0:lenNow])
            MessageList = [Msg[lenNow:]] + MessageList     # unshift
            SplittedMessages.append(''.join(MessageParts))
            MessageParts = []
        if len(MessageParts):
            SplittedMessages.append(''.join(MessageParts))
        assert(max(map(len, SplittedMessages)) <= MessageLengthMax)
        return SplittedMessages

    # Since each message part has a header that contains the part number and the total number of
    # parts, splitting is a little complicated because the total number is not known in advance: 
    assumeTotalParts, totalParts = 0, 1
    while totalParts > assumeTotalParts:
        assumeTotalParts = (assumeTotalParts + 1) * 10 - 1    # 9, 99, 999, ...
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, assumeTotalParts)
        totalParts = len(SplittedMessages)
    if totalParts < assumeTotalParts:
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, totalParts)
        while len(SplittedMessages) < totalParts:
            SplittedMessages.append(PartHeader(len(SplittedMessages)+1, totalParts))
            
    LOGGER.info('SMS message is too long (%s chars); splitting message into %d parts.' % (len(MessageText), totalParts))

    if int(smscnf.SMS_PARTS_LIMIT) > 0 and totalParts > int(smscnf.SMS_PARTS_LIMIT):
        skipCount = totalParts - int(smscnf.SMS_PARTS_LIMIT)
        skipPart = 0
        while skipPart < skipCount:
            SplittedMessages[1+skipPart] = '' # at least 1 part is allowed, leave part 0 ("header info") unskipped
            skipPart += 1
        lenAvail = MessageLengthMax - len(SplittedMessages[0])
        if lenAvail > 0:
            SplittedMessages[0] += ("[%d/%d cut]" % (skipCount, totalParts))[:lenAvail]

    for part, msg in enumerate(SplittedMessages):
        if len(msg):
            SMSsend(RecipientNumber, msg, SuccessLogMsg + ' [part %d/%d, %d chars]' % (part+1, totalParts, len(msg)))
        else:
            LOGGER.warning(SuccessLogMsg + ' [part %d/%d, %d chars SKIPPED DUE TO SMS LIMIT]' % (part+1, totalParts, len(msg)))
        
    return True

def SendToGSMmodem(RecipientNumber, Message, SuccessLogMsg, TryCount = None):
    """
    Send SMS message to GSM modem.
    @return bool Success indicator
    """
    if TryCount:
        TryCount['value'] += 1
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] > int(smscnf.RESENDMAXTRY):
            LOGGER.error('(SMS) Tried to send %d times. Limit %d reached. Giving up recipient=%s.' %
                            (TryCount['value'], int(smscnf.RESENDMAXTRY), RecipientNumber))
            return True
    if SendSMSviaGSMmodem(RecipientNumber, Message):
        LOGGER.info(SuccessLogMsg.replace('{RECIPIENTNUMBER}', RecipientNumber))
        return True
    else:
        LOGGER.error('Failed to send SMS to "%s".' % RecipientNumber)
        return False

##############################################################

def DeOctifyMessage(inputBytes):
    """
    Decode inputBytes from SMS message according to GSM 03.38, GSM 7 bit default alphabet.
    @return string.
    """
    outputCodes = []
    bitCount, lastByte = 7, 0
    for thisByte in inputBytes:
        outputCodes.append(0x7f & ((thisByte << (7 - bitCount)) | lastByte))
        lastByte = thisByte >> bitCount
        bitCount -= 1   # from the next byte, we need 1 bit less.
        if bitCount == 0:
            outputCodes.append(lastByte)    
            bitCount, lastByte = 7, 0

    # Note: for code pages below it is important that all code points evaluate to a unicode string (u-prefix):
    ESC = u'\x1b'
    GSM_03_38_codepage_std = (
        u'@' , u'£' , u'$' , u'¥' , u'è' , u'é' , u'ù' , u'ì' , u'ò' , u'Ç' , u'\n', u'Ø' , u'ø' , u'\r', u'Å' , u'å' ,
        u'Δ' , u'_' , u'Φ' , u'Γ' , u'Λ' , u'Ω' , u'Π' , u'Ψ' , u'Σ' , u'Θ' , u'Ξ' ,  ESC , u'Æ' , u'æ' , u'ß' , u'É' ,
        u' ' , u'!' , u'"' , u'#' , u'¤' , u'%' , u'&' , u'\'', u'(' , u')' , u'*' , u'+' , u',' , u'-' , u'.' , u'/' ,
        u'0' , u'1' , u'2' , u'3' , u'4' , u'5' , u'6' , u'7' , u'8' , u'9' , u':' , u';' , u'<' , u'=' , u'>' , u'?' ,
        u'¡' , u'A' , u'B' , u'C' , u'D' , u'E' , u'F' , u'G' , u'H' , u'I' , u'J' , u'K' , u'L' , u'M' , u'N' , u'O' ,
        u'P' , u'Q' , u'R' , u'S' , u'T' , u'U' , u'V' , u'W' , u'X' , u'Y' , u'Z' , u'Ä' , u'Ö' , u'Ñ' , u'Ü' , u'§' ,
        u'¿' , u'a' , u'b' , u'c' , u'd' , u'e' , u'f' , u'g' , u'h' , u'i' , u'j' , u'k' , u'l' , u'm' , u'n' , u'o' ,
        u'p' , u'q' , u'r' , u's' , u't' , u'u' , u'v' , u'w' , u'x' , u'y' , u'z' , u'ä' , u'ö' , u'ñ' , u'ü' , u'à' ,
    )

    GSM_03_38_codepage_esc = {
        0x0a: u'\f' , 
        0x14: u'^'  ,   0x1b: ESC   ,
        0x28: u'{'  ,   0x29: u'}'  ,   0x2f: u'\\' ,
        0x3c: u'['  ,   0x3d: u'~'  ,   0x3e: u']'  ,
        0x40: u'|'  ,
        0x65: u'€'  ,
    }

    outputText = []
    while outputCodes:
        code = outputCodes.pop(0)
        if code != ord(ESC) or not outputCodes:
            outputText.append(GSM_03_38_codepage_std[code])
        else:
            code = outputCodes.pop(0)
            if code in GSM_03_38_codepage_esc:
                outputText.append(GSM_03_38_codepage_esc[code])
            else:
                outputText.extend((ESC, GSM_03_38_codepage_std[code]))
    LOGGER.debug("DeOctifyMessage: have outputText: %s" % repr(outputText))
    return ''.join(outputText)

#############################################
#    email
#############################################

def EMAILsend(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
    """
    Send email; retry after <RESENDTIME> minutes if send failure.
    @return bool False iff the message has definitely not been sent. True otherwise.
    """
    if SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
        return True
    if smscnf.ENABLERESEND != 'yes' or not EmailTo:
        return False
    LOGGER.warning('Trying to re-send email message after %s minutes.' % float(smscnf.RESENDTIME))
    tryCount = { 'value': 0 }
    gobject.timeout_add(int(float(smscnf.RESENDTIME)) * 60000,
                        lambda a,b,c,d,**kwargs: not SendToSMTPserver(a,b,c,d,**kwargs),
                        EmailTo, Subject, Text, SuccessLogMsg, TryCount=tryCount, **kwargs)
    return True

def SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, Attachment = None, Attachhint = None, TryCount = None):
    """
    Send email to SMTP server.
    @return bool Success indicator
    """
    if TryCount:
        TryCount['value'] += 1
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] > int(smscnf.RESENDMAXTRY):
            LOGGER.error('(EMAIL) Tried to send %d times. Limit %d reached. Giving up recipient=%s, subject=%s.' %
                            (TryCount['value'], int(smscnf.RESENDMAXTRY), EmailTo, Subject))
            return True

    if not EmailTo:
        LOGGER.error('(EMAIL) No recipient email address')
        return False
    # convert string list to string
    if type(Text).__name__ == 'list': Text = ''.join(Text)

    # poor convertion of text to html text:
    MailBodyHtml = Text.replace('\n', '<br>')
    Hostname     = os.uname()[1]
    Message      = MIMEMultipart()

    if Attachment:
        Attachhint = Attachhint or 'text'
        ConvertToMIME         = MIMEText
        AttachmentCID         = None

        if Attachhint.startswith('image'):
            Message               = MIMEMultipart('related') 
            ConvertToMIME         = MIMEImage
            AltText               = '%s image' % os.path.basename(Attachment)
            AttachmentCID         = 'image%s' % random.randint(1, 100000)
            MailBodyHtml += '<br><img src="cid:%s" alt="%s" /><br>' % (AttachmentCID, AltText)

    Message['Subject'] = Subject
    Message['To']      = EmailTo
    Message['From']    = 'NokiaN900.SMSCON.%s <SMSCON@%s>' % (Hostname, Hostname)

    MailBodyHtml = '<html><head></head><body>\n'\
                    '<p>' + MailBodyHtml + '</p>\n'\
                    '</body></html>'
    Message.attach( MIMEText(MailBodyHtml, 'html') )

    if Attachment:
        try:
            File = open(Attachment, 'rb')
        except:
            LOGGER.error('(EMAIL) Attachment file "%s" not found.' % Attachment)
            AttachmentData = MIMEText('Attachment file "%s" not found' % Attachment)
            AttachmentCID = None
        else:
            try:
                AttachmentData = ConvertToMIME( File.read() )
            except:
                LOGGER.error('(EMAIL) Attachment file "%s" could not be read.' % Attachment)
                File.close()
                AttachmentData = MIMEText('Attachment file "%s" could not be read' % Attachment)
                AttachmentCID = None
            else:
                File.close()
        if AttachmentCID:
            AttachmentData.add_header('Content-ID', '<%s>' % AttachmentCID )
        else:
            AttachmentData.add_header('Content-Disposition', 'attachment; filename="%s"' % os.path.basename(Attachment))    
        Message.attach(AttachmentData)

    try:
        LOGGER.info('(EMAIL) Trying to connect to "%s:%s" (%s).' % ( smscnf.MAILSERVER, smscnf.MAILPORT, GetTimeString() ) )

        # connect to mail server
        Server = smtplib.SMTP(smscnf.MAILSERVER, smscnf.MAILPORT) 
        #Server.set_debuglevel(True)
    except (socket.error, smtplib.SMTPConnectError, smtplib.SMTPException):
        LOGGER.error('(EMAIL) Mailserver connection failure.')
        return False

    # identify to mail server
    Server.ehlo() 
        
    # if smtp server requires secure authentication
    if Server.has_extn('STARTTLS'):
        Server.starttls()
        Server.ehlo()

    # if smtp server requires user/password authentication
    if smscnf.USER != '' and smscnf.PASSWORD != '':
        try:
            LOGGER.info('(EMAIL) Trying to login to "%s".' % smscnf.MAILSERVER)
            Server.login(smscnf.USER, smscnf.PASSWORD) 
        except smtplib.SMTPAuthenticationError:
            LOGGER.error('(EMAIL) Wrong username/password.')
            Server.quit()
            return False

    try:
        LOGGER.info('(EMAIL) Trying to send email to "%s".' % smscnf.MAILSERVER)
        Server.sendmail(smscnf.EMAILFROM, EmailTo, Message.as_string()) 
    except smtplib.SMTPException, e:
        LOGGER.error('(EMAIL) [%s].' % e)
        Server.quit()
        LOGGER.error('(EMAIL) Mailserver connection closed.')
        return False

    LOGGER.info(SuccessLogMsg)
    LOGGER.info('(EMAIL) Mailserver connection closed successful.')
    Server.quit()
    return True

#############################################
#   save new number
#############################################

def SaveSenderPhoneNumber(PhoneNumber):
    """
    Save new SENDERNUMBER as smscnf.SENDERNUMBER in "smscon_config" file.
    """ 
    if smscnf.SENDERNUMBER == PhoneNumber:
        return

    if smscnf.update('SENDERNUMBER', PhoneNumber):
        LOGGER.info( 'New SMS SENDERNUMBER (%s) saved.' % PhoneNumber)
    else:
        LOGGER.error("Failed to save new SMS SENDERNUMBER (%s)." % PhoneNumber)

    # update smscnf.SENDERNUMBER in any case! We use it to send replies. 
    smscnf.SENDERNUMBER = PhoneNumber

#############################################
#    command processing
#############################################

def ProcessCommand(Command, SenderPhoneNumber=None, CommandOrigin=None, ExtraInfo='', LogIfCmdUnknown=False):
    """
    Process the received command message.
    """
    # Check if Command is a specific valid SMSCON command:
    CommandList = [smscnf[cmd] for cmd in smscnf.getSmsCommandList() if smscnf[cmd] != ""]
    if not Command in CommandList:
        if LogIfCmdUnknown:
            LOGGER.error( '"%s" was not a valid %s command.' % (Command, smscon.NAME) )
            LogBatteryChargeInfo()
        return False

    if SenderPhoneNumber:
        SaveSenderPhoneNumber(SenderPhoneNumber)
        CommandOrigin = CommandOrigin or "SMS"
        ExtraInfo = 'from %s (%s) %s' % (SenderPhoneNumber,
                                        IF(SenderPhoneNumber == smscnf.MASTERNUMBER, 'the master', 'NOT master'),
                                        ExtraInfo)
    else:
        CommandOrigin = str(CommandOrigin)

    LOGGER.info('Received %s command [%s] via %s %s.' % (smscon.NAME, Command, CommandOrigin, ExtraInfo))
    LogBatteryChargeInfo()

    # First of all, make phone silent
    EnableDeviceSilence(smscnf.SILENCEDEVICE == 'yes')

    # Command reply
    if smscnf.COMMANDREPLY == 'yes':
        RevealThat('reply', 'Command (%s) accepted.' % Command)
    else:
        LOGGER.info('Command (%s) accepted. Acknowledge message disabled.' % Command)

    # Explicit device locking
    if Command == smscnf.COM_UNLOCK:
        PerformDeviceUnlock()
        return

    if Command == smscnf.COM_LOCK:
        PerformDeviceLock()
        return

    # Automatic device locking
    if smscnf.AUTODEVICELOCK == 'yes':
        PerformDeviceLock()

    # Dispatch on other commands
    if Command == smscnf.COM_CHECK:
        ok = False
        if smscnf.MESSAGESEND in ['sms', 'both']:
            ok = ShowSMScommands('sms')
        if smscnf.MESSAGESEND in ['email', 'both']:
            ok = ShowSMScommands('email')
        if not ok:
            LOGGER.warning('No notification sent. Message sending is disabled by user setting: %s.' % smscnf.MESSAGESEND)

    elif Command == smscnf.COM_REBOOT:
        PerformReboot()

    elif Command == smscnf.COM_POWEROFF:
        PerformPoweroff()            

    elif Command == smscnf.COM_POWER:
        RevealThat('battery', GetBatteryChargeInfo())

    elif Command == smscnf.COM_LOCATION:
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        EnableGPStracking('single')

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_REMOTEON:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for enabling SSH connection.')
        else:
            if not IsInternetReachable():
                SetNetworkConnectionType('GPRS') # enable GPRS data network
                SetAutoConnectBehaviour('GPRS') # disable switching to other network
            EnableSSHcontrol()
            
    elif Command == smscnf.COM_REMOTEOFF:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for disabling SSH connection.')
        else:
            DisableSSHcontrol()
            
    elif Command == smscnf.COM_CAMERA:
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        PerformCamera()

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_CALL:
        PerformPhonecall(SenderPhoneNumber or smscnf.MASTERNUMBER)

    elif Command == smscnf.COM_TRACKON:
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            SetAutoConnectBehaviour('GPRS') # disable switching to other network
        EnableGPStracking('multi')

    elif Command == smscnf.COM_TRACKOFF:
        DisableGPStracking()
        SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_CUSTOM:
        PerformShellScript()

    elif Command == smscnf.COM_CUSTOMLOG:
        PerformShellScriptResult()

    elif Command == smscnf.COM_LOG:
        RevealThat('log', 'The smscon Log file', Attachment = LOGGER.LogFilename, Attachhint = 'text/plain')

    elif Command == smscnf.COM_SHELL:
        PerformDirectShellCommand(Command)

    elif Command == smscnf.COM_ALARM:
        PerformAlarm()

    elif Command == smscnf.COM_RESTART:
        RestartDeamon()

    else:
        LOGGER.debug('Unknown command: [%s].' % Command)

#############################################
#   silence phone
#############################################

def PerformAlarm():
    """
    Make phone play sound file for alarm.
    """ 
    # get current active phone profile
    Output = os.popen('dbus-send --print-reply \
                      --session --dest=com.nokia.profiled \
                      --type=method_call \
                      /com/nokia/profiled \
                      com.nokia.profiled.get_profile').read()
    E = re.compile('string "(\S+)"')

    try:
        CurrentProfile = E.findall(Output)[0]
    except:
        LOGGER.error('(ALARM) Could not get current phone profile.')
        return

    if CurrentProfile != 'general':
        # set "General" profile
        os.system('dbus-send --type=method_call \
                  --dest=com.nokia.profiled \
                  /com/nokia/profiled \
                  com.nokia.profiled.set_profile string:"general"')
        LOGGER.info('(ALARM) Set phone profile to "General".')
    else:
        LOGGER.info('(ALARM) current profile is "%s".' % CurrentProfile)

    # set volume to maximum
    Volume = 100
    os.system('dbus-send --type=method_call \
               --dest=com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer \
               /com/nokia/mafw/renderer/gstrenderer \
               com.nokia.mafw.extension.set_extension_property \
               string:volume variant:uint32:%s' % Volume)
    
    # play audio
    #gobject.timeout_add( PlayAudio(AudioFile) )
    PlayAudio(AudioFile, Wait=True)

    # set previous profile
    os.system('dbus-send --type=method_call \
              --dest=com.nokia.profiled \
              /com/nokia/profiled \
              com.nokia.profiled.set_profile string:"%s"' % CurrentProfile)

    RevealThat('alarm', 'Alarm sound activated.')        

#############################################

def PlayAudio(File, Wait=False):
    """
    Play audio file in background.
    @return bool Success indicator.
    """ 
    try:
        os.system( '/usr/bin/aplay -q "%s" &' % os.path.join(InstPath, File) )
    except:
        LOGGER.error('(ALARM) Failed to play audio file.')
        return False
    LOGGER.info('(ALARM) Start playing audio file...')
    if Wait:
        # loop until aplay quits
        while os.popen('ps ax | grep -v grep | grep aplay').read() != '':
            time.sleep(1)
    LOGGER.info('(ALARM) Stopped playing audio file.')
    return True

#############################################
#   silence phone
#############################################

def EnableDeviceSilence(Enable = True):
    """
    Make the phone silent for email, instand-messaging, phone ringing & SMS.
    """    

    if not Enable:
        LOGGER.warning('(SILENCE) Silencing is disabled: Phone probably not silent while controlled remotely.')
        return

    KeyList = [('email.alert.volume'  , '0'),
               ('im.alert.volume'     , '0'),
               ('ringing.alert.volume', '0'),
               ('sms.alert.volume'    , '0')]

    # get current active phone profile
    Output = os.popen('dbus-send --print-reply \
                      --session --dest=com.nokia.profiled \
                      --type=method_call \
                      /com/nokia/profiled \
                      com.nokia.profiled.get_profile').read()
    E = re.compile('string "(\S+)"')

    if E:
        LOGGER.debug('(SILENCE) CurrentProfile = %s.' % E.findall(Output) )
        try:
            CurrentProfile = E.findall(Output)[0]
        except:
            LOGGER.error('(SILENCE) Could not get current phone profile')
            return
        else:
            LOGGER.info('(SILENCE) Current profile is "%s".' % CurrentProfile)
    else:
        LOGGER.error('(SILENCE) Could not get current phone profile.')
        return

    # set items in current profile
    for Key, Value in KeyList:
        try:
            LOGGER.debug( '(SILENCE) Set current profile "%s" = "%s".' % (Key, Value) )
            os.system( 'dbus-send --session --dest=com.nokia.profiled \
                        --type=method_call \
                        /com/nokia/profiled \
                        com.nokia.profiled.set_value string:"%s" string:"%s" string:"%s"' % (CurrentProfile, Key, Value) )
        except:
            LOGGER.error( '(SILENCE) Could not set profile: %s = "%s".' % (Key, Value) )
            return
        
    LOGGER.info('(SILENCE) Phone is silenced in current "%s" profile.' % CurrentProfile)

#############################################
#   run custom scripts
#############################################

def PerformShellScript():
    """
    Execute user editable script.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    # Delete old log (may be retrieved before by COM_CUSTOMLOG command)
    if pidof(Filename) == '':
        DeleteFile(Logname)

    if IsFileReadable(Filename):
        # run script file
        os.system( '"%s" 1>>"%s" 2>&1&' % (Filename, Logname) )
        LOGGER.info(         'Script file (%s) executes in BG.' % Filename)
        RevealThat('script', 'script file (%s) executes in BG.' % Filename)        
    else:
        LOGGER.error(        'script file (%s) not found.' % Filename)
        RevealThat('script', 'script file (%s) not found.' % Filename)        
        DeleteFile(Logname)

def PerformShellScriptResult():
    """
    Send back logfile of user editable script execution.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    PIDs = pidof(Filename)
    if not IsFileReadable(Logname):
        RevealThat('script.log', 'Log file %s not present. Script processes running: (%s).' % (Logname, PIDs))
    else:
        RevealThat('script.log', 'Log file %s is attached. Script processes running: (%s).' % (Logname, PIDs),
                   Attachment = Logname, Attachhint = 'text/plain')

#############################################

def PerformDirectShellCommand(Command):
    """
    Run direct shell command and reply with notification message containing the command's output.
    @return void.
    """
    try:
        Result = os.popen(Command).read()
    except:
        LOGGER.Error('Failed to run shell command (%s).' % Command)
        RevealThat('shell', ['command failed: %s.' % Command])        
    else:
        LOGGER.info('Shell command (%s) executed.' % Command)
        RevealThat('shell', ['command: %s\n' % Command, 'result:\n', Result])        

#############################################
#   show SMS commands
#############################################

def ShowSMScommands(SendMethod):
    """
    Log or SMS list of SMS commands.
    @return: bool Success status.
    """
    CommandVars = sorted(smscnf.getSmsCommandList())
    if SendMethod == 'sms':
        Format = '%s=%s\n'
        skip = len(smscnf.SmsCommandPreFix)  # Skip the command prefix to keep SMS message as short as possible
    else:
        Format = '%' + str(-smscnf.TabSize) + 's = %s\n'
        skip = 0
    Lines = []
    Lines.append(IF(len(smscnf.SMS_COMPREFIX), 'Prefix:%s\n' % smscnf.SMS_COMPREFIX, 'No prefix.\n'))
    Lines.append(IF(len(smscnf.SMS_COMSUFFIX), 'Suffix:%s\n' % smscnf.SMS_COMSUFFIX, 'No suffix.\n'))
    Lines.extend([Format % (CmdVar[skip:], smscnf[CmdVar]) for CmdVar in CommandVars])
    if SendMethod == 'log':
        for Line in Lines: 
            LOGGER.info(Line)
    else:
        RevealThat('commands', Lines, SendMethod = SendMethod)
    return True

#############################################
#   device lock
#############################################

def PerformDeviceLock():
    """
    Lock the phone.
    @return void
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_open \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"3"')

    LOGGER.info(             'Phone has been locked.')
    if smscnf.COMMANDREPLY == 'yes':
        RevealThat('locked', 'Phone has been locked.')
        
def PerformDeviceUnlock():
    """
    Unlock the phone.
    @return void
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_close \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"0"')

    LOGGER.info(         'Phone has been unlocked.')
    RevealThat('unlock', 'Phone has been unlocked.')

#############################################
#   system
#############################################

def PerformReboot():
    """
    Reboot the phone.
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30

    LOGGER.info('Executing "%s" command: rebooting in %d seconds.' % (smscnf.COM_REBOOT, DelaySeconds))
    RevealThat('reboot',       'Phone is rebooting in %d seconds.' % DelaySeconds)

    time.sleep(DelaySeconds)
    os.system('reboot')

def PerformPoweroff():
    """
    Shutdown phone (switch off).
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30 

    LOGGER.info('Executing "%s" command: power off in %d seconds.' % (smscnf.COM_POWEROFF, DelaySeconds))
    RevealThat('shutdown',   'Phone will power off in %d seconds.' % DelaySeconds)
    
    time.sleep(DelaySeconds)
    os.system('poweroff')

#############################################
#   camera
#############################################

def PerformCamera():
    """
    Get picture of front camera and send it to email address.
    """
    ImageFilename = TakeFrontCamPicture()
    if ImageFilename:
        Hostname = os.uname()[1]
        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                  Subject =  'From "%s" command / N900[%s]' % (smscnf.COM_CAMERA, Hostname),
                  Text    =  ['Frontcam picture from Nokia N900 phone:\n',
                              'Taken on %s at %s\n' % (Hostname, GetTimeString())],
                  SuccessLogMsg  = '(EMAIL) camera picture send to "%s"' % smscnf.EMAILADDRESS,
                  Attachment =  ImageFilename,
                  Attachhint = 'image' )

def TakeFrontCamPicture():
    """
    Acquire front camera picture.
    @return string|bool Filename where image has been stored. False on errors.
    """
    ImageFilename       = os.path.join(LogPath, PhotoName)
    ImageFileEncoding   = 'jpegenc'     # TODO: encoding should depend on desired image type (png, jpeg, ...) 
    CamDevice           = '/dev/video1'
    DelaySeconds        = 1
    GStreamerPipeline   = (
        "v4l2camsrc device='%s' num-buffers=16" % CamDevice,
        "video/x-raw-yuv,width=640,height=480",
        "ffmpegcolorspace",
        "gamma gamma=1.8",
        "videobalance hue=0.1 brightness=0.2 saturation=1 contrast=1",
        ImageFileEncoding,
        "filesink location='%s'" % ImageFilename,
    )

    try:    os.remove(ImageFilename)
    except: pass

    if os.WEXITSTATUS(os.system('gst-launch ' + ' \! '.join(GStreamerPipeline))) != 0:
        LOGGER.error('Camera picture failed (image capture error).')
        return False

    time.sleep(DelaySeconds)        # i guess this this not really required... but to be on save side...

    if not os.path.isfile(ImageFilename):
        LOGGER.error('Taking camera picture failed (image file not found).')
        return False
    LOGGER.info('Camera picture acquired.')
    return ImageFilename

#############################################
#    phone call
#############################################

def PerformPhonecall(PhoneNumber):
    """
    Make phone call to PhoneNumber.
    @return void.
    """
    try:
        os.system('run-standalone.sh dbus-send --system --dest=com.nokia.csd.Call \
                   --type=method_call /com/nokia/csd/call com.nokia.csd.Call.CreateWith \
                   string:"%s" \
                   uint32:0' % PhoneNumber )        
        LOGGER.info(         'Making phonecall to "%s".' % PhoneNumber)          
    except:
        LOGGER.error('Failed to make phonecall to "%s".' % PhoneNumber)

#############################################
#    data connection
#############################################

def GetNetworkConnectionType(): 
    """
    Get info of current data connection type
    @return string WLAN|GPRS|NONE
    """
    Wlan = os.popen('ifconfig | grep "wlan0"').read()
    Gprs = os.popen('ifconfig | grep "gprs0"').read() 

    if   Wlan != '' and Gprs == '':     Type = 'WLAN'
    elif Wlan == '' and Gprs != '':     Type = 'GPRS'
    else:                               Type = 'NONE' 
    return Type

###############################

def GetGPRSConnectionStatus():
    """
    Get status of GPRS data network.
    @return: mixed True|False|None
    """
    Output = os.popen('dbus-send --print-reply --system --dest=com.nokia.csd.GPRS /com/nokia/csd/gprs com.nokia.csd.GPRS.GetStatus').read()

    E = re.compile(' boolean (\S+)')
    R = E.findall(Output)
    if R:
        if R[0] == 'false': return False
        if R[0]  == 'true': return True
    return None

###############################

def SetNetworkConnectionType(NewType):
    """
    Connect to GPRS data network.
    @param string Network type: GPRS|WLAN
    @return bool Success indicator
    """
    DelaySeconds = 3  # in sec. between connection status checks
    MaxT  = 30 # in sec. before timeout
    MaxR  = 3  # number of retries

    Type = GetNetworkConnectionType()

    if Type != 'NONE':
        if Type == NewType:
            if Type == 'GPRS':
                Type, IAP, Name = GetGPRSname()
            elif Type == 'WLAN':
                Type, IAP, Name = GetWLANname()
            else:
                LOGGER.error('Unknown network type "%s".' % Type)
                Name = 'unknown'
            LOGGER.info( '%s network (%s) already connected.' % (Type, Name) )
            return True 
        DisableConnection()

    if   NewType == 'GPRS': Type, IAP, Name = GetGPRSname()
    elif NewType == 'WLAN': Type, IAP, Name = GetWLANname()
    else:                   Type, IAP, Name = 'NONE', 'NONE', 'NONE'

    if Type == 'NONE':
        LOGGER.error('No network available.')
        return False

    # initiate network connecting
    LOGGER.info( 'Connecting to %s network (%s).' % (Type, Name) )
    os.system( 'dbus-send --system --type=method_call --dest=com.nokia.icd /com/nokia/icd com.nokia.icd.connect string:"%s" uint32:0' % IAP )

    Connected = IF(Type == 'GPRS', GetGPRSConnectionStatus, lambda:GetNetworkConnectionType() == Type)
    T, R = 0, 0
    while not Connected():
        if T >= MaxT:
            R += 1
            if R >= MaxR:
                LOGGER.error('Failed to connect to %s network after %s attempts.' % (Type, R))
                return False
            LOGGER.warning('No connection to %s network after %s sec. Retrying...' % (Type, T))
        time.sleep(DelaySeconds)
        T += DelaySeconds

    LOGGER.info( '%s network (%s) connection successful.' % (Type, Name) )
    return True

def DisableConnection(): 
    """
    Disconnect any current data connection.
    """
    DelaySeconds = 1

    os.system('dbus-send --system --dest=com.nokia.icd2 /com/nokia/icd2 com.nokia.icd2.disconnect_req uint32:0x8000')
    LOGGER.info('Current network disconnected.')

    time.sleep(DelaySeconds)

###############################

def GetGPRSname(): 
    """
    Get name of GPRS data connection from current inserted SIM card in Nokia phone.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()
    CurrentIMSI = smscon.GetPhoneCode('IMSI')

    for Network in NetworkList:
        Type, IAP, Name, Imsi = Network[0:4]
        if Type == 'GPRS' and Imsi == CurrentIMSI:
            LOGGER.debug('original IAP=[%s]' % IAP)
            return Type, IAP.replace('@32@', ' '), Name

    return 'NONE', 'NONE', 'NONE'

def GetWLANname():
    """
    Get name of active WLAN data connection.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()

    # get last used data network IAP
    LastUsedNetwork = os.popen(' gconftool -g /system/osso/connectivity/IAP/last_used_network').read().strip('\n')
    LOGGER.debug('LastUsedNetwork = [%s].' % LastUsedNetwork)

    for Network in NetworkList:
        Type, IAP, Name = Network[0:3]
        if Type == 'WLAN' and LastUsedNetwork == IAP:
            LOGGER.debug('WLAN: %s' % Name)
            return Type, IAP, Name

    return 'NONE', 'NONE', 'NONE'

def GetNetworks():
    """
    Get list of all available data connections (stored under "Internet connections" in menu).
    @return list of (Type, IAP, Name [, Imsi, NameAP]) tuples.
    """
    Output = os.popen('gconftool -R /system/osso/connectivity/IAP | grep IAP').read()
    E = re.compile('/system/osso/connectivity/IAP/(\S+):')
    IAPlist = E.findall(Output) 

    DiscardList = ['wap', 'mms']
    NetworkList = []

    for IAP in IAPlist:
        # get connection type
        Type = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/type' % IAP).read().replace('_INFRA', '').strip('\n')

        if Type == 'GPRS':
            # get gprs_accespointname
            NameAP = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/gprs_accesspointname' % IAP).read().strip('\n')

            # discard unwanted connections
            if not [Discard for Discard in DiscardList if NameAP.find(Discard) >= 0]:
                # get connection name & it's IMSI number
                Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
                Imsi = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/sim_imsi' % IAP).read().strip('\n')

                # convert decimal value of ASCII character in IAP string to normal ASCII character.
                E = re.compile('@(\d+)@')
                DecList = E.findall(IAP)

                for Dec in DecList:
                    IAP = IAP.replace( '@%s@' % Dec, chr( int(Dec) ) )

                # add GPRS connection to list
                LOGGER.debug( 'Type=%s, IAP=%s, Name=%s, IMSI=%s, AP=%s' % (Type, IAP, Name, Imsi, NameAP) )
                NetworkList.append( (Type, IAP, Name, Imsi, NameAP) )
            
        elif Type == 'WLAN':
            Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
            # add WLAN connection to list
            LOGGER.debug( 'Type=%s, IAP=%s, Name=%s' % (Type, IAP, Name) )
            NetworkList.append( (Type, IAP, Name) )
        
    return NetworkList

###############################

def RemindNetwork():
    """
    Save current active data network.
    """
    global ConnectionType
    global ConnectionIAP
    global ConnectionName

    Type = GetNetworkConnectionType()
    if   Type == 'WLAN':    ConnectionType, ConnectionIAP, ConnectionName = GetWLANname()
    elif Type == 'GPRS':    ConnectionType, ConnectionIAP, ConnectionName = GetGPRSname()
    else:                   ConnectionType, ConnectionIAP, ConnectionName = 'NONE', 'NONE', 'NONE'

    LOGGER.info( 'Saved current %s network (%s) for restore.' % (Type, ConnectionName) )

###############################

def RestoreNetwork():
    """
    Re-enable previous stored data network.
    """
    global ConnectionType
    global ConnectionName

    if ConnectionType == 'GPRS' or ConnectionType == 'WLAN':
        LOGGER.info( 'Restoring saved %s network (%s)' % (ConnectionType, ConnectionName) )
        SetNetworkConnectionType(ConnectionType)
    elif ConnectionType != 'NONE':
        LOGGER.info( 'Disabling connection to %s network (%s).' % (ConnectionType, ConnectionName) )
        DisableConnection()

#############################################
#    connect behaviour
#############################################

def SetAutoConnectBehaviour(NewBehaviour):
    """
    Set Nokia phone data connection behavior.
    The old behaviour is saved in global ConnectionBehaviour and can be used to restore.
    @return bool True if behaviour changed.
    """    

    def _getConnectBehaviour():
        """
        Get current Nokia phone data connection behaviour.
        @return string ASK|ANY|GPRS|WLAN|None
        """ 
        Mode = os.popen('gconftool-2 -g --list-type string /system/osso/connectivity/network_type/auto_connect').read().strip('\n') 
    
        if Mode == '[]':            return 'ASK'
        if Mode == '[*]':           return 'ANY'
        if Mode == '[GPRS]':        return 'GPRS'    
        if Mode == '[WLAN_INFRA]':  return 'WLAN'
        LOGGER.error('Unexpected connect behaviour "%s".' % str(Mode))
        return None
    
    def _setConnectBehaviour(Behaviour):
        """
        Set new Nokia device data connection behaviour (ASK|WLAN|GPRS|ANY).
        @return void
        """
        DelaySeconds = 2
    
        # options: [WLAN_INFRA], [GPRS], [*] <=='any', [] <=='ask'
        if   Behaviour == 'ASK':     String = ''
        elif Behaviour == 'WLAN':    String = 'WLAN_INFRA'
        elif Behaviour == 'GPRS':    String = 'GPRS'
        elif Behaviour == 'ANY':     String = '*'
        else:
            LOGGER.error('Unknown connect behaviour "%s".' % str(Behaviour))
            return
    
        LOGGER.debug('Setting connect behaviour to "%s".' % Behaviour)
        # change setting "Connect automatically" to given Behaviour
        os.system('gconftool-2 -s --type list --list-type string /system/osso/connectivity/network_type/auto_connect [%s]' % String)
        time.sleep(DelaySeconds)

    global ConnectionBehaviour

    OldBehaviour = _getConnectBehaviour()
    if  ConnectionBehaviour == None:
        ConnectionBehaviour = OldBehaviour

    if NewBehaviour == None:
        LOGGER.info('Changing internet connection behaviour not enabled (is "%s").' % OldBehaviour)
        return False
    elif NewBehaviour == OldBehaviour:
        LOGGER.info('Internet connection behaviour already set to "%s".' % NewBehaviour)
        return False

    # New behaviour is different from current. Save the current and try to establish the new one:
    ConnectionBehaviour = OldBehaviour
    _setConnectBehaviour(NewBehaviour)

    # Check new behaviour setting
    if _getConnectBehaviour() == NewBehaviour:
        LOGGER.info('Internet connection behaviour set from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return True
    else:
        LOGGER.error('Failed to set internet connection behaviour from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return False

#############################################
#    power status
#############################################

def GetBatteryChargeInfo(TextOnly = True):
    """
    Determine battery charge info.
    @return string Charge info text or
    @return (string, number, bool) if TextOnly is false.
    """
    global dev_obj
    chargePercentMin = 10

    # Try getting charge info, not all devices support all properties: 
    try:
        chargePercent = dev_obj.GetProperty('battery.charge_level.percentage')  # charge amount of battery in percentage
        chargePercentText = '%s%%' % chargePercent
        chargeHint = IF(chargePercent > chargePercentMin, '', ' (almost empty!)')
    except Exception, e:
        LOGGER.warning('%s' % e)
        chargePercent = None
        chargePercentText = 'unknown'
        chargeHint = ''

    try:
        isCharging = dev_obj.GetProperty('battery.rechargeable.is_charging') # charging: 1 / discharging: 0
        isChargingText = IF(isCharging, 'charging', 'discharging')
    except Exception, e:
        LOGGER.warning('%s' % e)
        isCharging = None
        isChargingText = ''

    Text = 'Phone battery charge is %s, %s%s.' % (chargePercentText, isChargingText, chargeHint)
    if TextOnly: return Text
    return (Text, chargePercent, isCharging)

def LogBatteryChargeInfo():
    """
    Log battery charge info.
    """
    LOGGER.info(GetBatteryChargeInfo())

#############################################
#    SSH 

def EnableSSHcontrol():
    """
    Start the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    
    if remote_sh.runs():
        LOGGER.info('ssh connection is already active.')
        return

    if not IsHostReachable(smscnf.REMOTEHOST):
        LOGGER.warning('No ping response from "%s", will try to connect anyway.' % smscnf.REMOTEHOST)

    LOGGER.info('Connecting to "%s" for ssh.' % smscnf.REMOTEHOST)

    (Output, ExitStatus) = remote_sh.start() 
    
    LOGGER.debug('ssh Output = %s' % Output.strip('\n') )
    LOGGER.debug('ssh ExitStatus = %s' % ExitStatus)

    if ExitStatus == 0:
        LOGGER.info(  'ssh connection to "%s" established.' % smscnf.REMOTEHOST)
        RevealThat('ssh', 'connection to "%s" established.' % smscnf.REMOTEHOST)
    else:
        LOGGER.error( 'ssh connection to "%s" failed (%s).' % (smscnf.REMOTEHOST, Output) )
        RevealThat('ssh', 'connection to "%s" failed.'       % smscnf.REMOTEHOST)
        PIDs = remote_sh.stop(Verbose=False)
        if PIDs:
            LOGGER.info('ssh stopped [PIDs=%s].' % PIDs)        

def DisableSSHcontrol(Verbose = True):
    """
    Stop the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    if remote_sh.stop(Verbose = Verbose):
        RevealThat('ssh', 'connection to "%s" terminated.' % smscnf.REMOTEHOST)

#############################################
#    GPS
#############################################

def EnableGPStracking(Mode):
    """
    Start the GPS device mainloop.
    """
    global GpsMode
    GpsMode = Mode
    
    global GpsIsActive
    if not GpsIsActive:
        GpsIsActive = True
        # enable GPS device control
        gobject.idle_add(GPSstart, GpsControl)
    else:
        LOGGER.warning('GPS device already active.')

def DisableGPStracking():
    """
    Stop the GPS device mainloop.
    """
    GpsData = GpsControl

    global GpsIsActive
    if GpsIsActive:
        GpsIsActive = False
        # stop the GPS device
        GpsData.stop()
        LOGGER.info('Stopped acquiering GPS coordinates.')
    else:
        LOGGER.info('GPS device already stopped.')

def GPSstart(GpsData):
    """
    Start the GPS device.
    """
    global GpsDataOK
    GpsDataOK = False

    if GpsMode == 'single':
        LOGGER.info('Starting GPS device in location mode.')
    elif GpsMode == 'multi':
        LOGGER.info('Starting GPS device in tracking mode.')

    global GpsStartTime
    GpsStartTime = time.time()
    GpsData.start()
    
    return False

def OnGPSstop(GpsControl, GpsData):
    """
    Stop the GPS device.
    """
    global GpsIsActive
    GpsIsActive = False

    global GpsIsTest
    if GpsIsTest:
        GpsIsTest = False
        RestoreNetwork()
        ExitDaemon()        
    
    LOGGER.info('Stopping GPS device.')
    GpsData.quit()

def OnGPSchanged(GpsDevice, GpsData):
    """
    Get GPS device data.
    """
    global GpsCoordinateList
    global GpsDataOK
    global GpsStartTime

    if not GpsIsActive:
        return
    # check GPS device timout
    TimePassedSeconds = time.time() - GpsStartTime
    if not GpsDataOK:
        LOGGER.debug('GSP GpsStartTime = %s.' % GpsStartTime)
        if TimePassedSeconds >= float(smscnf.GPSTIMEOUT):
            LOGGER.error(     'GPS device timeout after %.1f sec.' % TimePassedSeconds)
            RevealThat('GPS', 'GPS device timeout after %.1f sec.' % TimePassedSeconds, smscnf.GPSSEND)                    
            # stop the GPS device
            GpsData.stop()
            return

    # adjust start time of GPS device
    GpsStartTime = time.time()
    LOGGER.debug('GpsDevice.status = %s' % GpsDevice.status) # LOCATION_GPS_DEVICE_STATUS_NO_FIX  or  LOCATION_GPS_DEVICE_STATUS_FIX

    if (GpsDevice.fix[1] & location.GPS_DEVICE_LATLONG_SET) == 0:
        LOGGER.info('Waiting for GPS device satellite fix.')
        return
    # GPS device has fix.
    Latitude  = GpsDevice.fix[4]
    Longitude = GpsDevice.fix[5]
    Accuracy  = GpsDevice.fix[6] / 100

    # Check if coordinates acquired by GPS device are valid
    GpsDataOK = str(Latitude) != 'nan' and str(Longitude) != 'nan' and str(Accuracy) != 'nan'
    if not GpsDataOK:
        LOGGER.info('Waiting for valid GPS coordinate (search time = %.1f sec.).' % TimePassedSeconds)
        return

    if GpsMode == 'single':
        if len(GpsCoordinateList) == 0:
            LOGGER.info('Start collecting %s GPS coordinate(s)...' % smscnf.GPSPOLLING)

        GpsCoordinateList.append( (Latitude, Longitude, Accuracy) )
        LOGGER.info( 'GPS coordinate #%d acquired (%f, %f, %.1f m.).'
                        % (len(GpsCoordinateList), Latitude, Longitude, Accuracy) )                
        if len(GpsCoordinateList) < int(smscnf.GPSPOLLING):
            return
        # sort GPS coordinate list on 'Accuracy'
        GpsCoordinateList = sorted( GpsCoordinateList, key=itemgetter(2) )
        Latitude, Longitude, Accuracy = GpsCoordinateList[0]
        LOGGER.info( 'Most accurate GPS coordinate out of %d = (%f, %f, %.1f m.).'
                        % (len(GpsCoordinateList), Latitude, Longitude, Accuracy) )

        GpsCoordinateList = []
        # stop the GPS device
        GpsData.stop()

    # tracking mode
    elif GpsMode == 'multi':
        LOGGER.info( 'GPS coordinate acquired (%f, %f, %.1f m.).' % (Latitude, Longitude, Accuracy) )

    GpsText = ['Lat: %f\n' % Latitude,
               'Long: %f\n' % Longitude,
               'Accuracy: %.1f mtr\n' % Accuracy,
               'Duration: %.1f sec\n' % TimePassedSeconds]

    LinkText = ['<a href="http://maps.google.com/maps?q=%f,%f">GoogleMap</a>\n'% (Latitude, Longitude)]

    RevealThat('GPS Location', GpsText + LinkText,
                SendMethod = smscnf.GPSSEND,
                Attachment = smscnf.GPSSEND in ['email', 'both'] and CreateGoogleMap(Latitude, Longitude),
                Attachhint = 'image')

def OnGPSerror(GpsControl, GpsError, GpsData):
    """
    GPS device error detected.
    """
    global GpsIsActive
    GpsIsActive = False

    if GpsError == location.ERROR_USER_REJECTED_DIALOG:
        LOGGER.error("GPS device error (User didn't enable requested methods).")
    elif GpsError == location.ERROR_USER_REJECTED_SETTINGS:
        LOGGER.error('GPS device error (User changed settings, which disabled location).')
    elif GpsError == location.ERROR_BT_GPS_NOT_AVAILABLE:
        LOGGER.error('GPS device error (Problems with BT GPS).')
    elif GpsError == location.ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
        LOGGER.error('GPS device error (Requested method is not allowed in offline mode).')
    elif GpsError == location.ERROR_SYSTEM:
        LOGGER.error('GPS device error (System error).')
    else:
        LOGGER.error('GPS device error (Unknown error).')
        
    LOGGER.error('Stopping failed GPS device.')
    GpsData.quit()

###############################

def CreateGoogleMap(Latitude, Longitude):
    """
    Create map of current GPS coordinate.
    @return string|False Filename of map or False if no map.
    """
    MapZoom     = 16
    MapSize     = (600,600)
    MarkerColor = 'red'
    MarkerLabel = 'N'

    MapFilename = os.path.join(LogPath, MapName)
    MapUrl = 'http://maps.google.com/maps/api/staticmap?center=%s,%s&zoom=%s&size=%sx%s&format=png&markers=color:%s|label:%s|%s,%s&sensor=false' \
              % (Latitude,
                 Longitude,
                 MapZoom,
                 MapSize[0],
                 MapSize[1],
                 MarkerColor,
                 MarkerLabel,
                 Latitude,
                 Longitude)
    try:
        urllib.urlretrieve( MapUrl, MapFilename )
    except:
        LOGGER.error('Failed to create GPS coordinate map (retrieve error).')
        return False
    if not os.path.isfile(MapFilename):
        LOGGER.error('Failed to create GPS coordinate map (file not found).')
        return False
    else:
        LOGGER.info('Created GPS coordinate map.')
        return MapFilename

#############################################
#    SIM card
#############################################

def GetOperatorName():
    """
    Get current SIM telecom operator name.
    """
    try:
        Output = os.popen('dbus-send --system --print-reply=literal --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_registration_status').read()
        E = re.compile('uint32 (\d+)')
        R = E.findall(Output)
        Output = os.popen( 'dbus-send --system --print-reply --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_operator_name byte:0 uint32:"%s" uint32:"%s"' % (R[1], R[2]) ).read()
        E = re.compile('string "(.*)"')
        R = E.findall(Output)[0]
    except:
        LOGGER.error('Could not get telecom network operator name.')
        return 'UNKNOWN'        

    return R

def ReadIMSI():
    """
    Load IMSI code(s) from file.
    @return list of IMSI codes.
    """
    imsi = smscon.IMSIConfiguration()
    if imsi.load(Silent = True):
        LOGGER.info('Found %d IMSI code(s) in "%s" file.' % (len(imsi.keys()), imsi.Filename))
    else:
        LOGGER.warning('Initalizing new "%s" file with current valid IMSI code.' % imsi.Filename)
        imsi.update(smscon.GetPhoneCode('IMSI'), '')
    return imsi.keys()

def WriteIMSI(newIMSI):
    """
    Save new IMSI code to file.
    @return void
    """
    imsi = smscon.IMSIConfiguration()
    if imsi.update(newIMSI, ''):
        LOGGER.info( 'Added IMSI code (%s) to "%s" file.' % (newIMSI, imsi.Filename) )
    else:
        LOGGER.error( 'Failed to add IMSI code (%s) to "%s" file.' % (newIMSI, imsi.Filename) )

def ValidateIMSI():
    """
    Check if current used IMSI code from SIM card is authorized.
    """
    CurrentIMSI = smscon.GetPhoneCode('IMSI')

    if CurrentIMSI == None:
        LOGGER.info('No SIM card present.')
        ExitDaemon()

    # compare current IMSI code with saved one or more in code file.
    elif CurrentIMSI in ReadIMSI():
        LOGGER.info('Authorized IMSI code found.')

    else:
        LOGGER.warning('IMSI code has changed to "%s".' % CurrentIMSI)

        # save new IMSI code to file
        WriteIMSI(CurrentIMSI)    

        # auto unlock phone after new SIM is inserted
        if smscnf.SIMUNLOCK == 'yes':
            PerformDeviceUnlock()                 

        SendMethod = smscnf.MESSAGESEND
        if   SendMethod == 'none':  SendMethod = 'sms'
        elif SendMethod == 'email': SendMethod = 'both'

        RevealThat('New SIM', '\n'.join([
                        'IMSI = %s' % CurrentIMSI,
                        'IMEI = %s' % smscon.GetPhoneCode('IMEI'),
                        'Telecom Operator = "%s".' % GetOperatorName()
                    ]), SendMethod)

#############################################

def DoCommandLine():
    """
    Check arguments for smscon_daemon and perform corresponding actions.
    @return bool True iff ok
    """    
    def TellError(Msg):
        LOGGER.error(Msg)
        return False

    def PrepareNetwork():
        if  GetNetworkConnectionType() != 'WLAN':
            SetNetworkConnectionType('GPRS')
            SetAutoConnectBehaviour('GPRS') # disable switching to other network

    args = sys.argv[1:]
    if len(args) > 0:
        Mode = args.pop(0)
        if Mode == '-test':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Func = args.pop(0)
    
            if Func == 'gps1' or Func == 'gps2': 
                RemindNetwork()
                PrepareNetwork();
    
                global GpsIsTest
                GpsIsTest = True
                
                if Func == 'gps1': # test mode for gps (location mode)
                    EnableGPStracking('single')
                elif Func == 'gps2': # test mode for gps (tracking mode)
                    EnableGPStracking('multi')
                return True
    
            if Func == 'ssh': # test mode for ssh connection
                RemindNetwork()
                PrepareNetwork();
                DisableSSHcontrol(Verbose = False)
                EnableSSHcontrol()
                return True

            if Func == 'ssh-off': # test for ssh disconnect
                DisableSSHcontrol(Verbose = True)
                return True

            if Func == 'cam': # test mode for camera
                RemindNetwork()
                PrepareNetwork();
                PerformCamera()
                RestoreNetwork()
                SetAutoConnectBehaviour(ConnectionBehaviour)

            elif Func == 'sms': # test mode for sms
                SMSsend(smscnf.MASTERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func == 'sms2': # test mode for sms
                SMSsend(smscnf.SENDERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func in ('email1', 'email2'): 
                RemindNetwork()
                PrepareNetwork();

                if Func == 'email1': # test mode1 for email (text only)
                    EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject = 'Test email / Nokia N900',
                              Text = 'Email sending from Nokia N900 phone is successful!',
                              SuccessLogMsg = 'send test email message to "%s"' % smscnf.EMAILADDRESS)
                elif Func == 'email2': # test mode2 for email + frontcam image
                    ImageFilename = TakeFrontCamPicture()
                    if ImageFilename:
                        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject =  'Test email / Nokia N900',
                              Text =  'Frontcam picture from Nokia N900 phone:<br>%s<br>' % GetTimeString(),
                              SuccessLogMsg = 'send email camera picture send to "%s"' % smscnf.EMAILADDRESS,
                              Attachment =  ImageFilename,
                              Attachhint = 'image' )

                RestoreNetwork()
                SetAutoConnectBehaviour(ConnectionBehaviour)
    
            elif Func == 'call': # test mode for phone call
                PerformPhonecall(smscnf.MASTERNUMBER)

            elif Func == 'script': # test mode for script
                PerformShellScript()

            else:
                LOGGER.error('Test option error (%s).' % Func)

        # /* ONLY FOR DEVELOPMENT USAGE */
        elif Mode == '-comtest':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Command = args.pop(0)

            LOGGER.warning('Simulating received SMS message with "%s" command.' % Command)
            ProcessCommand(Command, SenderPhoneNumber = smscnf.SENDERNUMBER,
                           ExtraInfo = 'SIMULATION!', LogIfCmdUnknown = True)    
        else:
            LOGGER.warning('Option error (%s) - ignored' % Mode)
            return True
    if len(args) > 0:
        LOGGER.warning('Options ignored: %s' % ' '.join(args))
    return True                

#############################################

def CheckBootLoader():
    """
    Check if smscon auto-loads at system boot.
    @return void.
    """
    if IsFileReadable(smscon.BootPath, smscon.BootFile):
        LOGGER.info('%s auto-loads at boot. OK.' % smscon.NAME)
    else:
        LOGGER.warning(    "%s does not auto-load at boot." % smscon.NAME)
        RevealThat('boot', "%s does not auto-load at boot." % smscon.NAME)        

def ExitDaemon(ExitReason = 'regularly'):
    """
    Exit smscon_daemon.
    @return NO RETURN.
    """
    LOGGER.critical('%s is stopping %s.' % (DaemonFile, ExitReason))
    sys.exit(1)

def RestartDeamon():
    """
    Restart smscon_daemon.
    @return void (does not return on success).
    """ 
    DelaySeconds = 10

    LOGGER.critical('(RESTART) %s tries to restart itself w/o regular stop.' % DaemonFile)
    try:
        time.sleep(DelaySeconds)
        os.system( '"%s" -restart' % os.path.join(InstPath, smscon.NAME) )
    finally:
        LOGGER.critical('(RESTART) Not restarted. Failed to kill %s.' % DaemonFile)

#############################################
#   keyboard detect (dbus trigger)
#############################################

def OnKeyboardSliderChange(Action, Type):
    """
    Check and reveal state of keyboard slider.
    @return void. 
    """
    KeyState = os.popen('cat /sys/devices/platform/gpio-switch/slide/state').read().strip('\n') 
    RevealThat('Keyboard', 'Phone is being used: keyboard now %s.' % str(KeyState))            

#############################################
#   battery charge (dbus trigger)
#############################################

def OnBatteryStatusChange(BarLevel, BarLevelMax):
    """
    Check and reveal battery charge level.
    @return void. 
    """
    Message, ChargePercent, Charging = GetBatteryChargeInfo(TextOnly = False)        
    LOGGER.info(Message)
    if (Charging != None and not Charging) or (ChargePercent != None and ChargePercent < 10): 
        RevealThat('battery warning', Message)

#############################################
#   remote host file checking (remote smscon trigger)
#############################################

def PerformCheckHost(CheckHostURL):
    """
    Check remote host file for triggering smscon
    @return True
    """
    global CheckHostLastCommand, CheckHostStatusCount, CheckHostStatusTimestamp

    CommandList = {
        '01' : smscnf.COM_CHECK,
        '02' : smscnf.COM_REBOOT,
        '03' : smscnf.COM_POWEROFF,
        '04' : smscnf.COM_POWER,
        '05' : smscnf.COM_LOCATION,
        '06' : smscnf.COM_REMOTEON,
        '07' : smscnf.COM_REMOTEOFF,
        '08' : smscnf.COM_CAMERA,
        '09' : smscnf.COM_CALL,
        '10' : smscnf.COM_LOCK,
        '11' : smscnf.COM_UNLOCK,
        '12' : smscnf.COM_TRACKON,
        '13' : smscnf.COM_TRACKOFF,
        '14' : smscnf.COM_CUSTOM,
        '15' : smscnf.COM_SHELL,
        '16' : smscnf.COM_ALARM,
        '17' : smscnf.COM_RESTART,
        '18' : smscnf.COM_CUSTOMLOG,
        '19' : smscnf.COM_LOG,
    }  

    # Supply default protokol "http:"
    if  CheckHostURL.find(":/") < 0:
        CheckHostURL = 'http://' + CheckHostURL

    def eventsOf(what):
        return "%d %s since %s GMT" % (abs(CheckHostStatusCount), what, GetTimeString(CheckHostStatusTimestamp))

    LOGGER.debug('(CHECKHOST) Opening remote host file (%s).' % CheckHostURL)
    try:
        UrlFile = urllib.urlopen('%s?LASTMCD=%s&CHECKTIME=%s' % (CheckHostURL, CheckHostLastCommand, smscnf.CHECKTIME))
        Command = UrlFile.read(2)
        UrlFile.close()
    except Exception, e:
        if CheckHostStatusCount >= 0:
            LOGGER.error('(CHECKHOST) After %s: Failed to read remote file: %s' % (eventsOf('OK'), e))
            CheckHostStatusCount = 0
            CheckHostStatusTimestamp = time.gmtime()
        CheckHostStatusCount -= 1
        if -CheckHostStatusCount % 25 == 0:
            RevealThat('checkhost', 'Got %s: %s' % eventsOf('errors'), e)
        return True

    if CheckHostStatusCount <= 0:
        LOGGER.info('(CHECKHOST) After %s: Successfully read remote file.' % eventsOf('errors'))
        CheckHostStatusCount = 0
        CheckHostStatusTimestamp = time.gmtime()
    CheckHostStatusCount += 1

    LOGGER.debug('(CHECKHOST) Got command "%s". Last command was "%s".' % (Command, CheckHostLastCommand))
    if CheckHostLastCommand == Command:
        return True
    CheckHostLastCommand = Command
    
    if Command in CommandList:
        Command = CommandList[Command]
        LOGGER.info( '(CHECKHOST) %s is activated (command "%s") by remote host file.' % (DaemonFile, Command) )
        ProcessCommand(Command, CommandOrigin = 'CHECKHOST', ExtraInfo = CheckHostURL)

    elif Command == '00':
        LOGGER.info( '(CHECKHOST) %s is deactivated (%s) by remote host file.' % (DaemonFile, Command) )

    elif Command == '-1':
        LOGGER.info( '(CHECKHOST) %s will restart (%s) by remote host file.' % (DaemonFile, Command) )
        try:
            os.system( '"%s" -restart' % os.path.join(InstPath, smscon.NAME) )
        except:
            LOGGER.error('(CHECKHOST) Failed to kill %s' % DaemonFile)
    
    else:
        LOGGER.error( '(CHECKHOST) %s unknown command (%s) by remote host file.' % (DaemonFile, Command) )

    return True # run it continiously

#############################################

def InstallSignalReceivers():
    """
    Enable or disable signal receivers and polling timers to detect
    various device sensors and control command sources.
    @return void
    """
    
    global IsKeyboardSliderCheckActive
    global IsBatteryStatusCheckActive
    global IsCheckHostActive

    if (not IsKeyboardSliderCheckActive) == (smscnf.KEYBOARDDETECT == 'yes'):
            IsKeyboardSliderCheckActive   =  smscnf.KEYBOARDDETECT == 'yes'
            set_signal_receiver = IF(IsKeyboardSliderCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnKeyboardSliderChange,
                                    path           = '/org/freedesktop/Hal/devices/platform_slide',
                                    dbus_interface = 'org.freedesktop.Hal.Device',
                                    signal_name    = 'Condition')
    LOGGER.info('Keyboard slider change detection is %s.' % IF(IsKeyboardSliderCheckActive, 'on', 'off'))

    if (not IsBatteryStatusCheckActive) == (smscnf.AUTOBATREPORT == 'yes'):
            IsBatteryStatusCheckActive   =  smscnf.AUTOBATREPORT == 'yes'
            set_signal_receiver = IF(IsBatteryStatusCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnBatteryStatusChange, 
                                    path           = '/com/nokia/bme/signal',
                                    dbus_interface = 'com.nokia.bme.signal',
                                    signal_name    = 'battery_state_changed')
    LOGGER.info('Battery status change detection is %s.' % IF(IsBatteryStatusCheckActive, 'on', 'off'))

    def CheckHost(checktime):
        global IsCheckHostActive
        if smscnf.ENABLECHECKHOST == 'yes':
            IsCheckHostActive = True
            PerformCheckHost(smscnf.CHECKHOST)
            if smscnf.CHECKTIME == checktime:
                return True
            LOGGER.info('(CHECKHOST) Checkhost is on. Checking each %s minutes.' % float(smscnf.CHECKTIME))
            gobject.timeout_add(int(float(smscnf.CHECKTIME) * 60000), CheckHost, smscnf.CHECKTIME)
        else:
            IsCheckHostActive = False
            LOGGER.info('(CHECKHOST) Checkhost is off.')
        return False
    if not IsCheckHostActive:
        CheckHost(None)

def LoadConfiguration():
    """"
    Load and establish user settings from smscon_config file
    @return bool Success indicator
    """
    global smscnf
    isReload = not not smscnf;
    LastKnownGoodConfigPath = os.path.join(InstPath, LastKnownGoodConfigFile)

    theConf = smscon.UserConfiguration()
    if  theConf.load():
        theConf.upgrade()
    else:
        LOGGER.critical( 'Failed to load "%s" file.' % theConf.Filename)
        if isReload:
            return False
        LOGGER.warning('Trying last known good config.')
        theConf = smscon.UserConfiguration(Filename = LastKnownGoodConfigPath)
        if not theConf.load():
            return False
        theConf.upgrade(Permanent = False)  # Only in RAM; Do not change last known good on disk.

    # For convinience code make config settings available as attributes:
    for key in theConf.getKnownConfigVars():
        if not key in theConf:
            LOGGER.critical( 'Failed to load "%s" file.' % theConf.Filename)
            return False
        setattr(theConf, key, theConf[key])

    LOGGER.info('Successfully loaded "%s" file.' % theConf.Filename)
    theConf.backup(LastKnownGoodConfigPath, Silent = 'last known good config')

    smscnf = theConf
    if isReload:
        InstallSignalReceivers()
    return True


def main():
    """
    Main function of smscon daemon. Does not return to caller.
    """
    global smscnf, bus, dev_obj, GpsControl

    LOGGER.critical('%s is starting.' % DaemonFile)
    
    signal.signal(smscon.smscon_deamon.TELL_TERMINATE,
                    lambda signum, stack_frame: ExitDaemon("on signal %s" % str(signum))  )
    signal.signal(smscon.smscon_deamon.TELL_RELOAD,
                    lambda signum, stack_frame: LoadConfiguration()                       )
    
    LoadConfiguration() or ExitDaemon('on config load error')
    ValidateIMSI()
    CheckBootLoader()
    
    # Connect to various signal triggers:
    try:            
        DBusGMainLoop(set_as_default = True)
    
        bus = dbus.SystemBus()
    
        bus.add_signal_receiver(OnSmsReceived,
                                path           = '/com/nokia/phone/SMS',
                                dbus_interface = 'Phone.SMS',
                                signal_name    = 'IncomingSegment')
    
        # set battery charge measurment
        hal_obj = bus.get_object('org.freedesktop.Hal', '/org/freedesktop/Hal/Manager')
        hal = dbus.Interface(hal_obj, 'org.freedesktop.Hal.Manager')
    
        uids = hal.FindDeviceByCapability('battery')
        dev_obj = bus.get_object('org.freedesktop.Hal', uids[0])
        LogBatteryChargeInfo()
    
        # set GPS device control
        GpsControl = location.GPSDControl.get_default()
        GpsDevice  = location.GPSDevice()
    
        # set GPS method & most suitable interval
        gps_interval = int(float(smscnf.GPSINTERVAL))
        if   gps_interval <= 10:  gps_interval = location.INTERVAL_10S
        elif gps_interval <= 20:  gps_interval = location.INTERVAL_20S
        elif gps_interval <= 30:  gps_interval = location.INTERVAL_30S
        elif gps_interval <= 60:  gps_interval = location.INTERVAL_60S
        else:                     gps_interval = location.INTERVAL_120S
        GpsControl.set_properties(preferred_method   = location.METHOD_GNSS|location.METHOD_AGNSS,
                                  preferred_interval = gps_interval)
    
        GpsControl.connect('error-verbose', OnGPSerror, gobject.MainLoop() )
        GpsDevice.connect('changed', OnGPSchanged, GpsControl )
        GpsControl.connect('gpsd-stopped', OnGPSstop, gobject.MainLoop() )
    
        if not DoCommandLine():
            ExitDaemon('bad command line')
    
        InstallSignalReceivers()
    
        gobject.MainLoop().run()
    
    except KeyboardInterrupt:
        ExitDaemon('on KeyboardInterrupt')
    
    except Exception, e:
        LOGGER.critical('<<< SMSCON FATAL ERROR:\n%s >>>' % e)
        # Disable resending of messages (would fail anyway when exiting): 
        smscnf.ENABLERESEND = 'no' 
        RevealThat('fatal error', ['smscon_daemon crashed.\n', '%s' % e])        
        ExitDaemon('on fatal error')
    
    ExitDaemon('on mainloop exit')

#############################################

"""

 0.9.1
 REFACTURING:
         * Extracted common code to a library (smsconlib.py) which is now used by
           smscon_daemon, smscon control program and smscon-editor.
         * Added a lot of code comments.
         * Encapsulated many code into classes. E.g. the complete encyption stuff
           is now transparent to smscon control program and smsmcon editor.
         * Restructured code, written new and renamed a lot of old stuff.
         * Moved code from files "smscon" and "smscon_daemon" to pure
           python-modules (smscon_master.py and smscon_service.py respectively).
           The original files are now just jackets calling main() in associated
           python module. This allows to compile the modules and hiding source code.
 NEW:    Encryption now uses salt (formerly encrypted data can be read).
 FIX:    Encryption does no longer fail on text ending with curly open brace.
 NEW:    "MASTERNUMBER" concept: A SMS number stored in the config file that
         does NOT change due to SMS commands like "SENDERNUMBER" does.
         Each SMS is send to both (only once if they are the same numbers, of course).
         SMS replies to commands which are not introduced by SMS (e.g. via CHECKHOST)
         are sent to the MASTERNUMBER only.
         On upgrading, the default MASTERNUMBER is set to the current SENDERNUMBER.
 NEW:    "COM_CUSTOMLOG" command: A SMS command to retreive the output of the last
         "COM_CUSTOM" command. Will be sent via email only - would send too much SMS...
 NEW:    "COM_LOG" command: A SMS command to retreive the smscon.log file.
         Will be sent via email only - would send too much SMS...
 NEW:    The output of "COM_SHELL" SMS command is now sent as a command reply.
         You may have to remove output redirections from your shell command to
         get non-empty reply.
 NEW:    "SMS_PARTS_LIMIT": A number to limit the number of single SMS that would
         otherwise be generated to send a too long SMS message. Zero means no limit.
 NEW:    "LOG_TIMEFORMAT": Format to be used in time stamps of SMS and EMAIL messages.
 NEW:    "SMS_COMPREFIX" and "SMS_COMSUFFIX": Instead of explicitly define
         all your SMS control commands (COM_xxx) to start with a unique prefix
         and/or suffix, you might now separately define a prefix and/or suffix
         that is implicitely used. While "SMS_COMPREFIX" and "SMS_COMSUFFIX" are
         empty, all your commands do as in older versions.
         Advantage of using "SMS_COMPREFIX" and "SMS_COMSUFFIX":
         * Prefix and suffix can be changed easyly while embedded COM_xxx remain unchanged.
         * Prefix and suffix apply to COM_SHELL command as well. Example:
             With the configuration set to:
                 COM_SHELL=uptime
                 SMS_COMPREFIX=aaa.
                 SMS_COMSUFFIX=.bbb
             The recognized SMS command to execute the shell command "uptime" is:
                 aaa.uptime.bbb
 FIX:    It is no longer possible to define two SMS commands as the same string except
         the empty string, which means to disable that command being given via SMS.
 FIX:    SMS command recognition is no longer restricted to ASCII characters.
         The complete GSM 7 bit default alphabet may be used now (but not unicode).
         This gives you a greather variete of chars to use for your SMS commands.
 NEW:    The config file will now upgrade automagically from any previous version.
         New options are set to their default, obsolete options are commented out.
 NEW:    New function to export and import settings to/from arbitrary file.
         (see smscon -export and -import options). smscon editor makes
         now use of this functions to retrieve a and deploy editable configuration.
 NEW:    The smscon_daemon does no longer need to be restarted explicitly if
         smscon changes a setting. The daemon detects changes automagically and
         will read and apply the new configuration automatically.
 NEW:    The smscon_daemon now maintains a "last known good" configuration and uses
         that as a fallback in case the "official" configuration of smscon can't
         properly loaded by the deamon.
 NEW:    Camera picture and GSP Location requests do no longer force unconditional
         switching to GPRS network connection. The current connection will be used
         if the internet is reachable.
         Should a GPRS connection be required to reach internet, the GPRS
         connection is established but the previous network is re-activated
         after picture and/or location has been replied.
 NEW:    A LOGGER is now used wherever possible. So the smscon.log file becomes
         really interesting.
 NEW:    The "CHECKHOST" URL is now extended by "?LASTMCD=nn&CHECKTIME=minutes"
         which will allow the URL being a script that may use this information.
 CHANGE: "CHECKHOST" now works more clear as follows: While polling for
         commands, a command is executed once, if it differs from the
         last polled command. The last polled command is initialized
         to "-1" on daemon start. Code "-1" is "restart daemon" and
         this obviously is what happened on daemon start.
         Code "17" (COM_RESTART) is deprecated as a CHECKHOST command.
         Since it is different from "-1" it may restart the daemon repeatedly.
         Use code "-1" to restart the daemon via CHECKHOST instead. Of course,
         before a change to code "-1" is recognized by the daemon, you must
         have give it a different command code. If in doubt use "01" (COM_CHECK)
         because it gives you feedback that the command has been seen.
 CHANGE: "CHECKTIME" may now be a float to specify fractional parts
         of a minute: 0.25 is 15 seconds.  
 NEW:    The "CHECKHOST" URL may now have a protocol prefix and it may be
         different from "http://". If the URL has no protocol prefix, then
         "http://" is supplied as default as in former versions (e.g. your
         config file remains valid in this point as well).
 NEW:    The battery status is logged before a command is about to be executed.
 NEW:    "REMOTELISTENIP", "REMOTELISTENPORT" and "REMOTE2LOCALPORT": These
         settings give you greather flexibility of setting up a tunnel to
         remote control your phone via a reverse ssh-connection:
         Use "REMOTELISTENIP" and "REMOTELISTENPORT" to define the tunnel
         entry maintained by the remote ssh-server (which must be setup to
         allow setting LISTENIP and PORT this way).
         "REMOTELISTENPORT" defaults to 8080 and "REMOTELISTENIP" defaults
         to "localhost" as in former versions. With these defaults you must
         be logged in at the ssh-server to start a remote control connection
         of your phone. While setting up "REMOTELISTENIP" you may even
         control the phone from somewhere else (in which case the ssh-server
         act as a proxy or relay). On the tunnel exit side, which is on your
         phone, the "REMOTE2LOCALPORT" is the port number where your phone's
         ssh-daemeon is listening (usually 22).
 CHANGE: The "smscon -devconfig" command now requires a password to show the
         uncrypted configuration. The MASTERNUMBER is the password.
 NEW:    "RESENDMAXTRY": This option defines the maximum number of message
         re-send tries. Applies to SMS and EMAIL as well. (0=no limit)
         On configuration upgrade this is initialized to 10.
 NEW:    smscon now has a -sendsms option to send SMS immediately.
         Despite being a nice feature by itself, it makes smscon editor
         independent from smssend program which requires an extra package.

 0.8.1-1
 FIX: shutdown & reboot routine (PerformPoweroff & PerformReboot) give internal error.
 NEW: added remote activation of smscon_daemon; content of file CHECKHOST (hostname+filename)
      on server will be checked every CHECKTIME minutes. If other then 0 it will execute the corresponding command.

 0.8.0-1
 NEW/CHANGE: expanded user setting MESSAGESEND ('sms', 'email', 'both' & 'none').
             removed DISABLESMS user setting. This is superseded by MESSAGESEND user setting!
             SIM card change SMS always will be send regardless of MESSAGESEND setting.
 NEW: added smscon command (RESTART); user can restart smscon_daemon by SMS command.
 NEW: added smscon command (COM_ALARM); phone will play user configurable sound file (only WAV!) when command is received. 
 NEW: smscon_daemon sends SMS/email message if smscon_daemon crashes.
 CHANGE: changed user setting ENABLERESEND ('yes', 'no'). Resending of messages will again be experimental feature!!!
 CHANGE: removed MAXRESEND user setting (didn't work properly).
 CHANGE: some log text changed/added.
 CHANGE: removed updatechecker during installation of smscon (failed to work previously).
 CHANGE: some changes in GPS message sending code.
 CHANGE: AUTODEVICELOCK message is only send if COMMANDREPLY is enabled.
 FIX: small bug in SMSsend().
 FIX: bug in GPSINTERVAL handeling; GPSINTERVAL was always set at 60 sec. regardless of user setting.

 0.7-2
 FIX: bug in RestoreNetwork(); (None must be 'NONE').
 FIX: bug in heckArguments(); EnableScript() must be PerformShellScript().
 EXPERIMENTAL: updatechecker to save "smscon_config" file when updating to new version.
 
 0.7-1
 NEW: multiple IMSI codes are allowed in smscon_code file; use new option "smscon -add imsi" to store current IMSI code in smscon_code file.
 NEW: added new user option (DISABLESMS); WARNING: disables ALL SMS sending in smscon program.
 NEW: added new smscon command (COM_SHELL); run directly shell commands. (no check & feedback from command!).
 NEW: added new user option (SIMUNLOCK); auto unlock phone after new SIM is inserted (to fool a possible thief of your phone).
 NEW: added new user option (SILENCEDEVICE); disable all phone sounds (calendar alarm, clock, email alert, IM alert, ringing alert & SMS alert) if smscon is active.
 NEW/CHANGE: new user option (AUTOBATREPORT); user can enable/disable sending automatically a SMS message when smscon is activated & device battery is depleting.
 CHANGE: when initializing a new smscon_config file all user settings wil be set default to 'no'.
 REFACTORING: some parts in both scripts.
 FIX: better number handeling in SMSrecieve() routine (thanks to maemo.org user!).
 FIX: small bug in GSMmodem() in smscon_daemon script.
 FIX: small text bug in smscon script

 EXPERIMENTAL/CHANGE: when smscon fails to send SMS message (no cell reception) it will retry after 900 seconds.
 EXPERIMENTAL/FIX: change in getting the right GPRS connections; MMS & WAP will be discarded in available GPRS network list.
    
 0.5-9
 NEW: added new smscon command: COM_CUSTOM to run a user configurable shell script (/opt/smscon/smscon_script)
 NEW: if user setting REMOTEUSER, REMOTEPASSWORD is empty, "Remoteon" & "Remoteoff" function is disabled in smscon. 
 NEW: smscon will send automatically a SMS message when smscon is activated & device battery is depleting.
 CHANGE: added new options in smscon menu.
 CHANGE: added time notation in GPS email & SMS messages.
 CHANGE: improved the way smscon reads user settings from smscon_config file.
 EXPERIMENTAL/CHANGE: when smscon fails to send SMS message (no cell reception) it will retry after 900 seconds.
 EXPERIMENTAL/FIX: change in getting the right GPRS connections; MMS & WAP will be discarded in available GPRS network list.
 FIX: picture of frontcam & GoogleMap image are named wrong & placed therefore in wrong directory.
 FIX: smscon fails to send SMS message when message is too long (> 160 chars.)
 FIX: "smscon -test2" test function would fail if a valid smscon command has spaces in it. /*Developer usage only*/

 0.5-8 /* UNRELEASED VERSION */
 NEW: smscon will send automatically a SMS message when smscon is activated & device battery is depleting.
 CHANGE: added time notation in GPS email & SMS messages.
 EXPERIMENTAL/FIX: change in getting the right GPRS connections; MMS will be discarded in available network list. 
 FIX: smscon fails to send SMS message when message is too long.
 FIX: "smscon -test2" test function would fail if a valid smscon command has spaces in it.

 0.5-7
 FIX: activation of the current active/available GPRS connection could fail in specific cases (bug in acquiering the syntax correct name).
 FIX: better logging/behaviour when locking/unlocking of device.

 0.5-6
 NEW: added new command-line option for smscon to set user settings (encrypted & non-encrypted) in smscon_config file.
 NEW: added package icon
 CHANGE: improved user setting reading from smscon_config file.
 CHANGE: cosmetic changes in LOGGER.
 REFACTORING: parts of script code.
 FIX: smscon can send to wrong phone number when activated.
 FIX: added time & date of frontcam picture taken in email.

 0.5-5
 NEW: new test mode for simulating all smscon commands. /*Developer usage only*/
 NEW: in reply SMS from COM_LOCATION command is http link added to GoogleMaps.
 NEW/CHANGE: optification of smscon; new install path of package is now /opt/smscon.
 REFACTORING: improved the way SMS is send (more stable).
 CHANGE: in smscon test modes usage of WLAN connection only if already connected, otherwise usage of GPRS.
 CHANGE: in smscon test modes if usage of GPRS connection, previous connection is saved & re-enabled after GPRS usage.
 CHANGE: if usage of GPRS connection, force Internet connections to GPRS connection (needed for keeping the connection alive/stable).
 CHANGE: new smscon icon (meets requirements of maemo package)
 CHANGE: new package description (meets requirements of maemo Extra-testing package)
 FIX: execption in email sending code failed.
 FIX: internal fixes (better consistency in logging).
 FIX: email attachment (camera picture or Google map) is displayed completely wrong (alt text was wrong/mixed etc.)
 FIX: remove unused variable in ConfigVars list.
 FIX: syntax bug fixed which could crash smscon.

 0.4.4-6
 NEW: GPSSEND has new extra option 'both'. (options are now: 'sms' / 'email' / 'both')

 0.4.4-5
 NEW: COM_CHECK returns all valid SMS commands
 NEW: improved email smtp server handling
 FIX: ssh connection passfrase for auth. keys added
 FIX: "Internet connection" behavior set to initial setting when smscon_daemon stops.
 FIX: email sending works with or without secure authentication (tested with Google gmail)
"""

