# -*- coding: utf-8 -*-
##############################################################
# smscon - remote control library for Nokia N900 phone       #
# $Id: smsconlib.py 251 2012-08-12 21:01:58Z yablacky $
##############################################################

# standard modules
import os
import re
#import dbus
import logging
import logging.handlers # for RotatingFileHandler etc.
import fileinput
import signal
import time
import pexpect

##############################################################

def getSysBuildNameAndVersion(defaultName, defaultVersion = '999.1', headerInfo = None, dirLevels = 0):
    """
    Analyze information from version control system to find out name and version of a branch.
    @param string defaultName Name to return if version information not recognized.
    @param string defaultVersion Version to return if version information not recognized.
    @param string headerInfo Value from expanded \$Header\$ keyword to be analyzed.
    @param int dirLevel Number of extra dirs to walk up to find the one that contains version information (0 = parent dir).
    @return list (name, version) The name and version information.
    """
    try:
        return (headerInfo.split()[1].split('/')[-dirLevels-2].split('_') + [defaultVersion])[0:2]
    except:
        return [defaultName, defaultVersion]

NAME = 'smscon'
VERSION = '999.1'   # for trunk and private builds etc.

NAME, VERSION = getSysBuildNameAndVersion(NAME, VERSION,
    headerInfo  = '$Header: https://vcs.maemo.org/svn/smscon/branches/smscon_0.10.9/src/opt/smscon/smsconlib.py 251 2012-08-12 21:01:58Z yablacky $',
    dirLevels   = 3 )
    
XDEBUG   = False        # cross debug (use only if not running on phone)

##############################################################

TimeFormat  = '%Y-%m-%d %H:%M:%S'
InstPath    = '/opt/%s' % NAME
if XDEBUG: InstPath = '.'
DaemonFile  = '%s_daemon' % NAME
CommandFile = '%s_cmds' % NAME
CodeFile    = '%s_code' % NAME
ConfigFile  = '%s_config' % NAME
ScriptFile  = '%s_script' % NAME
ScriptLog   = '%s_script.log' % NAME
BootPath    = '/etc/event.d'
BootFile    = '%s' % NAME       # This will be the system wide job name.

PhotoName   = 'frontcam.jpg'
MapName     = 'gpsmap.png'
DefaultAudioFile   = 'alarm.wav'
AudioFile   = '%s_alarm.sound' % NAME

LogPath    = InstPath
LogFile    = '%s.log' % NAME
BootLogFile= '%s_boot.log' % NAME
LOGGER     = None

def IF(VAL,THEN,ELSE):
    """
    Almost the same what (VAL ? THEN : ELSE) does in C, perl or php.
    Note the difference: Unlike ?: the IF() always evaluates BOTH(!)
    expressions (THEN as well as ELSE) and returns one of the results.
    """
    if VAL: return THEN
    else  : return ELSE

##############################################################
# encryption/decryption interface

import base64
from Crypto.Cipher import AES
from random import randint

_AESBlockSize = 16
_AESPadding   = '{'
_AESPadlen    = lambda l: (_AESBlockSize - l) % _AESBlockSize
_AESPad       = lambda s: s + _AESPadlen(len(s)) * _AESPadding

def EncodeAES(key, text):
    text = str(text)
    # len(salt1) == len(salt2) != _AESBlockSize must be true
    salt  = chr(randint(0,255)) + chr(randint(0,255))
    salt1 = salt + chr(randint(0,255))                      # 3 random bytes (will not be encrypted - so don't put info here)
    salt2 = salt + chr(_AESPadlen(len(salt1) + len(text)))  # 2 random bytes plus padding length info (being encrypted)
    return base64.b64encode(salt1 + key.encrypt(_AESPad(salt2 + text)))

def DecodeAES(key, code):
    code  = base64.b64decode(code)
    salt1 = code[0:len(code) % _AESBlockSize]
    text = key.decrypt(code[len(salt1):])
    salt2 = text[0:len(salt1)]
    text  = text[len(salt2):]
    if len(salt2) == 3:
        return text[0:len(text)-ord(salt2[2])]
    else:
        return text.rstrip(_AESPadding)

##############################################################
# Convenience functions. 

def strReverse(s):
    """
    @param string s String to be reversed.
    @return string string s in reverse character order.
    """
    return ''.join([s[-i] for i in range(1,1+len(s))])

def commonPrefix(*args):
    """@return string The longest same string that all passed strings are starting with"""
    for idx in range(min(map(len, args))):
        for s in args:
            if s[idx] != args[0][idx]: 
                return s[0:idx];
    if len(args):   return args[0]
    else:           return ''

def csvList(s):
    """
    @param string s A comma separated value.
    @return list of trimmed strings extracted from comma separated value s.
    """
    return map(lambda s: s.strip(), s.split(','))
    
def shellArg(s):
    """@return string The parameter s quoted to be safely used as a shell parameter."""
    return "'%s'" % str(s).replace("\'", "'\"'\"'")

def pidof(ProcessName):
    """
    Determine the PID of processes that match a given name.
    @return string Whitespace separated PID(s). Empty if none.
    """
    PIDs = os.popen('pidof %s' % shellArg(ProcessName)).read().strip()
    if PIDs != '':
        return PIDs
    PIDs = []
    for psLine in os.popen('busybox ps|grep -v grep|grep %s' % shellArg(' ' + ProcessName)).readlines():
        PIDs.append(psLine.split()[0])
    return ' '.join(PIDs)

def NormalizePhoneNumber(phoneNumber):
    """
    @param string phoneNumber A human written phone number like '+49 1234 55 666 - 92' or '(09876) 54321'
    @return string normalized phone number; remove everything except digits and plus sign.
    """ 
    return re.sub('[^0-9+]', '', phoneNumber)

def IsSamePhoneNumber(number1, number2):
    """
    @param string number1 A human written phone number to be compared.
    @param string number2 A human written phone number to be compared.
    @return bool True iff dialing either number will ring the same phone. 
    """
    norm1 = NormalizePhoneNumber(str(number1)).replace('+','0')
    norm2 = NormalizePhoneNumber(str(number2)).replace('+','0')
    while norm1[0:2] == '00': norm1 = norm1[1:]
    while norm2[0:2] == '00': norm2 = norm2[1:]
    return norm1 == norm2

def IsHostReachable(hostname, pingCount = 3):
    """
    Check if remote host is available with a ping command and return
    @return bool false = no or bad ping; true = good ping (hostname is available)
    """
    return os.WEXITSTATUS( os.system( 'ping -q -c %d %s >/dev/null 2>&1' % (pingCount, hostname) ) ) == 0 
        
def IsInternetReachable():
    """
    Check if the phone potentially can reach computers in the internet.
    @return bool
    """
    return IsHostReachable('8.8.8.8')   # The google nameserver should always be up...

def IsFileReadable(*File):
    """
    @return bool True iff File exists and is readable. Silent check. Effective uid applies.
    """
    try:
        return open(os.path.join(*File), 'r').close() or True
    except:
        return False

def DeleteFile(*File):
    """
    Delete file. Very verbose.
    @return bool True iff the file has been (or was already) deleted.
    """
    Filename = os.path.join(*File)
    try:
        os.remove(Filename)
    except:
        if IsFileReadable(Filename):
            LOGGER.error('Failed to delete "%s".' % Filename)
            return False
        LOGGER.info('File "%s" was already deleted.' % Filename)
    else:
        LOGGER.info('File "%s" now deleted.' % Filename)
    return True

def SendSMSviaGSMmodem(RecipientNumber, Message, TryCountMax = 2):
    """
    Low Level function to send a SMS via GSM modem. The Message length must not exceed 160 chars.
    TryCountMax is number of total tries to re-activate GSM modem when the GSM modem control sequence has failed.
    @return bool Success indicator
    """
    assert(len(Message) <= 160)

    RecipientNumber = NormalizePhoneNumber(RecipientNumber)

    modem = None
    TryCount = 0
    while TryCount < TryCountMax:
        TryCount += 1
        LOGGER.debug('GSM modem: try count=%d/%d' % (TryCount, TryCountMax))

        if modem:
            modem.sendeof()
            time.sleep(1)
            modem.kill(9)
            time.sleep(1)
        modem = pexpect.spawn('pnatd', [], timeout = 8) # start pnatd
        #modem.logfile = sys.stdout
        time.sleep(1)   # must give pnatd a chance to start reading from stdin.

        try:
            cmd = 'AT'
            LOGGER.debug('GSM modem: %s' % cmd)
            modem.send(cmd + '\r') and modem.readline() # eat echo
            response = modem.readline().strip()
            if response != 'OK':
                LOGGER.debug('GSM modem: expected "OK" - got "%s".' % response)
                continue

            cmd = 'AT+CMGF=1'
            LOGGER.debug('GSM modem: %s' % cmd)
            modem.send(cmd + '\r') and modem.readline() # eat echo
            response = modem.readline().strip()
            if response != 'OK':
                LOGGER.debug('GSM modem: expected "OK" - got "%s".' % response)
                continue

            cmd = 'AT+CMGS="%s"' % RecipientNumber
            LOGGER.debug('GSM modem: %s' % cmd)
            modem.send(cmd + '\r') and modem.readline() # eat echo
            modem.expect_exact('> ')

            cmd = '(sms text)'
            LOGGER.debug('GSM modem: %s' % cmd)
            modem.send( Message + chr(26) ) and modem.readline()    # message plus ctrl-Z; eat echo

            response = modem.readline().strip()
            if not response.startswith('+CMGS:'):
                LOGGER.debug('GSM modem: expected "+CMGS:" - got "%s".' % response)
                continue 

            cmd = '(empty line)'
            modem.readline()        # eat empty line
            response = modem.readline().strip()
            if response != 'OK':
                LOGGER.debug('GSM modem: expected "OK" - got "%s".' % response)
                continue 

        except pexpect.EOF:
            LOGGER.debug('GSM modem: got eof after %s' % cmd)
            continue
        except pexpect.TIMEOUT:
            LOGGER.debug('GSM modem: timeout after %s' % cmd)
            continue
        LOGGER.debug('GSM modem: success.')
        return True

    LOGGER.debug('GSM modem: giving up.')
    if modem:
        modem.sendeof()
        time.sleep(1)
        modem.kill(9)
    return False

##############################################################

class Configuration(dict):
    """
    Base class for file based configuration databases
    """
    ItemTypeName        = "configuration"
    Thing               = None      # --> use default from ItemTypeName and Filename
    Silent              = False
    CreateTemplate      = ''
    ConfigPrintFormat   = "%s = %s"                                     # (ConfigVar, QuotedConfigValue)
    ConfigParseFormat   = "^\\s*(\\S+)\\s*=\\s*([\"\']|)(.*?)\\2\\s*$"  # (ConfigVar, QuoteUsed, ConfigValue)
    CommentLineFormat   = "^\\s*(#.*)?$"
    CommentMakeFormat   = '# %s'
    DefaultValueFormat  = None
    TabSize             = 0     # 0 means: calculate dynamically on _ConfigVars
    _OPT_CRYPT          = 'E'
    _OPT_PLAIN          = 'P'
    _ConfigVars         = {}    # known items; no restriction if empty. Structure: key: (re_pattern_of_value, default_value, options, comment).
    _Secret             = ''    # Key used to en/decrypt
    _upgradableConfigVars = {}  # class attribute served as fallback in case instance not yet loaded.

    def __init__(self, Filename = None):
        self.Filename = Filename
        if  self.TabSize == 0:
            self.TabSize = max(map(len, self._ConfigVars))
            if  self.TabSize == 0:
                self.TabSize = 18

    def _thing(self, Thing = None):
        return Thing or self.Thing or '%s file "%s"' % (self.ItemTypeName, self.Filename)

    def copy(self):
        """@return a shallow copy (not a deep copy)"""
        other = self.__class__()
        for k in self:
            other[k] = self[k]
        for attr, value in self.__dict__.items():
            setattr(other, attr, value)
        return other 

    def _setSilent(self, Silent):
        self.Silent = Silent
        if self.Silent:
            self.SilentLog = []

    def _error(self, Message, ReturnCode = False):
        if self.Silent:
            self.SilentLog.append(('E', Message))
        else:
            LOGGER.error(Message)
        return ReturnCode

    def _warning(self, Message, ReturnCode = False):
        if self.Silent:
            self.SilentLog.append(('W', Message))
        else:
            LOGGER.warning(Message)
        return ReturnCode

    def _info(self, Message, ReturnCode = False):
        if self.Silent:
            self.SilentLog.append(('I', Message))
        else:
            LOGGER.info(Message)
        return ReturnCode

    def _getPatternMismatchHint(self, ConfigVar, Pattern):
        return getattr(Pattern, 'HelpText', 'Must match /%s/.' % Pattern)

    def _decode(self, ConfigVar, ConfigValue):
        return DecodeAES(AES.new(self._Secret), ConfigValue)
        
    def _genConfigLine(self, ConfigVar, ConfigValue):
        return self.ConfigPrintFormat % ( ConfigVar.ljust(self.TabSize), ConfigValue)

    def show(self, Password = None, PasswordVar = 'MASTERNUMBER'):
        """
        Print decoded content of configuration file. If Password is given, it must match.
        @return bool Success indicator
        """
        if Password == None:
            return self.load(PrintContent = True)
        if self.load() and self.get(PasswordVar) == Password:
            return self.load(PrintContent = True)
        return self._error('password mismatch')

    def load(self, PrintContent = False, Silent = False):
        """
        Load the configuration from file; optionally Silent about errors; optionally PrintContent while loading.
        @return bool Success indicator. The previous content of the configuration is lost in any case.
        """
        self._setSilent(Silent)
        ok = self._load(PrintContent = PrintContent)
        self._setSilent(False)
        return ok

    def _load(self, PrintContent = False):
        self.clear()
        self._upgradableConfigVars = {}
        Thing = self._thing()
        try:
            f = open(self.Filename, 'r')
        except:
            return self._error('The %s could not be found.' % Thing)

        ConfigLines = [line.replace("\n","").replace("\r","") for line in f.readlines()] 
        f.close()
        if ConfigLines == []:
            return self._info('The %s is empty.' % Thing, False)

        ConfigValuePattern = re.compile(self.ConfigParseFormat)
        CommentLinePattern = re.compile(self.CommentLineFormat)
        DefaultValuePattern= self.DefaultValueFormat and re.compile(self.DefaultValueFormat)

        ok = True
        for Line in ConfigLines:
            if CommentLinePattern.match(Line):
                if PrintContent:
                    print Line
                continue
            ErrorMsg = None
            LineTokens = ConfigValuePattern.findall(Line)
            if len(LineTokens) == 0:
                ErrorMsg = 'Bad syntax in above line. Line ignored.'
            else:
                (ConfigVar, QuoteUsed, ConfigValue) = LineTokens[0]
                if len(self._ConfigVars) > 0: 

                    if not ConfigVar in self._ConfigVars:            
                        self._upgradableConfigVars[ConfigVar] = True
                        ErrorMsg = 'Unknown %s "%s" ignored.' % (self.ItemTypeName, ConfigVar)
                    else:
                        Pattern, DefaultValue, VarOptions = self._ConfigVars[ConfigVar][0:3]
                        if DefaultValue == None:
                            self._upgradableConfigVars[ConfigVar] = True
                            ErrorMsg = 'Obsolete %s "%s" ignored.' % (self.ItemTypeName, ConfigVar)
                        else:
                            EncryptedConfigValue = ConfigValue
                            if self._OPT_CRYPT in VarOptions:
                                try:
                                    ConfigValue = self._decode(ConfigVar, ConfigValue)
                                except:
                                    ErrorMsg = 'Failed to decode %s value for %s.' % (self.ItemTypeName, ConfigVar)
                            if Pattern and not re.match(Pattern, ConfigValue):
                                # If this is a known old-style default value, upgrade to new default value:
                                if DefaultValuePattern and (re.match(DefaultValuePattern, ConfigValue)
                                                        or  re.match(DefaultValuePattern, EncryptedConfigValue)):
                                    self._upgradableConfigVars[ConfigVar] = True
                                    ErrorMsg = 'Value of %s "%s" needs upgrade.' % (self.ItemTypeName, ConfigVar)
                                else:
                                    if type(DefaultValue).__name__ == 'function':
                                        self._upgradableConfigVars[ConfigVar] = ConfigValue
                                    self._warning('Value of %s "%s" is invalid. ' % (self.ItemTypeName, ConfigVar)
                                                            + self._getPatternMismatchHint(ConfigVar, Pattern))
            if ErrorMsg:
                if ok and not PrintContent: self._error('In %s:' % Thing)
                print Line
                ok = self._error(ErrorMsg)
            else:
                if PrintContent:
                    print self._genConfigLine(ConfigVar, QuoteUsed+ConfigValue+QuoteUsed)
                self[ConfigVar] = ConfigValue
        return ok

    def backup(self, Filename, Silent = False, Thing = None):
        """
        Export the configuration and generate a compact backup file.
        @param bool Silent Be silent about errors and success.
        @param string|None Thing Non-default description of what is being exported.
        @return bool Success indicator
        """
        self._setSilent(Silent)
        ok = self._backup(Filename, Thing)
        self._setSilent(False)
        return ok

    def _backup(self, Filename, Thing = None):        
        Thing = self._thing(Thing)
        ok = True
        try:
            f = open(Filename, 'w')
        except Exception, e:
            return self._error('Export failed to open %s for writing.' % Thing)

        try:
            temp = self.copy()
            keysToBackup = sorted(self._ConfigVars)
            keyErrorRetry = len(keysToBackup) * 2
            while keysToBackup:
                ConfigVar = keysToBackup.pop(0)
                Pattern, DefaultValue, VarOptions = self._ConfigVars[ConfigVar][0:3]
                if DefaultValue == None:
                    continue
                if ConfigVar in self:
                    ConfigValue = self[ConfigVar]
                else:
                    if not self.Silent:    # don't change ok if silent
                        ok = self._warning('Value of %s "%s" not defined. Using default.' % (self.ItemTypeName, ConfigVar))
                    try:
                        ConfigValue = temp._calcDefaultValue(DefaultValue, self)
                    except KeyError:
                        if  keyErrorRetry > 0 and keysToBackup:
                            keyErrorRetry -= 1
                            keysToBackup.append(ConfigVar)
                            continue
                        ConfigValue = 'default value calculation error'
                
                if Pattern and not re.match(Pattern, ConfigValue):
                    if not self.Silent:    # don't change ok if silent
                        ok = self._warning('Value of %s "%s" is invalid. ' % (self.ItemTypeName, ConfigVar)
                                                        + self._getPatternMismatchHint(ConfigVar, Pattern))
    
                Line = self._genConfigLine(ConfigVar, self._genFileValue(ConfigValue, VarOptions))
                f.writelines(Line + '\n')
                temp[ConfigVar] = ConfigValue

        except Exception, e:
            ok = self._error('Export to %s failed: %s' % (Thing, e))
        finally:
            f.close
        if ok:
            self._info('Export to %s successful.' % Thing)
        return ok

    def restore(self, Filename, Silent = False, Force = False, Thing = None):
        """
        Import configuration from an export or backup file. This is not just a file copy:
        * Only valid configuration items are imported.
        * Existing configuration items that are not present in the source remain unchanged.
        @param bool Force Continue on errors.
        @param bool Silent Be silent about errors and success.
        @param string|None Thing Non-default description of what is being imported.
        @return int Number of imported settings (even if they are not really set to a new value). Negative on errors.
        """
        self._setSilent(Silent)
        ok = self._restore(Filename, Force = Force, Thing = Thing)
        self._setSilent(False)
        return ok

    def _restore(self, Filename, Force = False, Thing = None):        
        Thing = self._thing(Thing)
        importCount = 0
        source = self.__class__(Filename = Filename)
        if not source.load(Silent = self.Silent):
            self._error('Importing %s failed.' % Thing)
            return importCount
        ok = True
        for ConfigVar in sorted(source.keys()):
            if self.update(ConfigVar, source[ConfigVar]):
                importCount += 1
                continue
            ok = self._error('Importing %s failed for "%s".' % (Thing, ConfigVar))
            if not Force:
                break
        if ok:
            self._info('Importing %s successful.' % Thing)
        return IF(ok, importCount, -importCount)

    def _calcDefaultValue(self, DefaultValue, OldDict = None, NewValue = None):  # may throw KeyError exception.
        val = DefaultValue
        if type(val).__name__ == 'tuple':
            if   len(val)  > 1: val = val[0] % tuple(map(lambda k: self[k], val[1:]))
            elif len(val) == 1: val = self[val[0]]
            else: val = ''
        elif type(val).__name__ == 'function':
            OldDict = (type(OldDict).__name__ == 'dict' and OldDict) or self
            val = val(OldDict, self, IF(NewValue == True, None, NewValue))
        return val

    def _genFileValue(self, ConfigValue, VarOptions):
        FileValue = str(ConfigValue)
        if self._OPT_CRYPT in VarOptions:
            FileValue = "'%s'" % EncodeAES(AES.new(self._Secret), ConfigValue)
        elif not FileValue.isdigit():
            FileValue = "'%s'" % FileValue
        return FileValue

    def getKnownConfigVars(self):
        """@return list of known (and not obsolete) config variables. Empty if no restriction."""
        return [key for key in self._ConfigVars if self._ConfigVars[key][1] != None]

    def needsUpgrade(self):
        """@return bool Indicator if configuration needs upgrade(). Detected after load()."""
        return len(self._upgradableConfigVars) > 0

    def isInitialValue(self, ConfigVar, ConfigValue):
        """@return bool Indicator if a given value is the initial value of given config var.""" 
        return IF(ConfigVar in self._ConfigVars, self._ConfigVars[ConfigVar][2] == ConfigValue, None)

    def update(self, ConfigVar, ConfigValue, Silent = False):
        """
        Set a config var to a new value. If required add or (if ConfigValue is None) remove the variable from configuration.
        @return bool True iff the setting was known and has been updated (or added or removed).
        """
        self._setSilent(Silent)
        ok = self._update(ConfigVar, ConfigValue)
        self._setSilent(False)
        return ok

    def _update(self, ConfigVar, ConfigValue):        
        if ConfigVar == None:
            return self._error('Missing %s.' % (self.ItemTypeName))

        CommentLineIfNewOrOld = None
        VarOptions = ''
        # If the Configuration has known config vars, then check that the
        # ConfigVar to set is known and the ConfigValue is valid:
        if len(self._ConfigVars) > 0:
            DefaultValue = None
            if not (ConfigVar in self._ConfigVars):
                return self._error('Unknown %s "%s".' % (self.ItemTypeName, ConfigVar))

            Pattern, DefaultValue, VarOptions, CommentLineIfNewOrOld = self._ConfigVars[ConfigVar]
            if Pattern and ConfigValue != None and not re.match(Pattern, ConfigValue):
                return self._error('Value of %s "%s" is invalid. ' % (self.ItemTypeName, ConfigVar)
                                                    + self._getPatternMismatchHint(ConfigVar, Pattern))

            if DefaultValue == None and ConfigValue != None:
                return self._error('Obsolete %s "%s". %s' % (self.ItemTypeName, ConfigVar, CommentLineIfNewOrOld or ''))

        # If the ConfigVar is to be added or removed from the Configuration,
        # a comment may also be added . Prepare that comment:
        if CommentLineIfNewOrOld and ConfigValue != None:
            if not CommentLineIfNewOrOld.startswith('\n'): CommentLineIfNewOrOld = '\n' + CommentLineIfNewOrOld
            if not CommentLineIfNewOrOld.endswith('\n'):   CommentLineIfNewOrOld = CommentLineIfNewOrOld + '\n'

        # A ConfigValue of None is a request to remove that ConfigVar
        # from Configuration. If it is an known but obsolete ConfigVar,
        # it will just be commented out:
        Operation = "Updated"
        if ConfigValue == None:
            if CommentLineIfNewOrOld == None:
                UpdatedLine = None  # this will lead to removing the entry from file
                Operation = "Removed"
            else:
                UpdatedLine = self.CommentMakeFormat % self._genConfigLine(ConfigVar, CommentLineIfNewOrOld)
                Operation = "Commented out"
        else:
            UpdatedLine = self._genConfigLine(ConfigVar, self._genFileValue(ConfigValue, VarOptions))

        ConfigValuePattern = re.compile(self.ConfigParseFormat)
        CommentLinePattern = re.compile(self.CommentLineFormat)

        # Scan the file and update the line that defines the ConfigVar:        
        try:
            for Line in fileinput.FileInput(self.Filename, inplace = True):
                if not CommentLinePattern.match(Line):
                    LineTokens = ConfigValuePattern.findall(Line)    # (ConfigVar, QuoteUsed, ConfigValue)                
                    if len(LineTokens) > 0:
                        FoundVar, FoundValue = (LineTokens[0][0], LineTokens[0][2])
                        if FoundVar == ConfigVar:
                            if UpdatedLine == None:
                                continue;   # remove double entry
                            # Here it is: replace Line by UpdatedLine and consume UpdatedLine
                            Line, UpdatedLine = UpdatedLine, None
                        elif FoundVar in self._ConfigVars and not (FoundVar in self):
                            # This line defines a different known config variable which
                            # is not yet load from the file. Load it implicitely if that
                            # can be done without errors (mainly for @APPVERSION key):            
                            FoundPattern, FoundDefaultValue, FoundVarOptions = self._ConfigVars[FoundVar][0:3]
                            if FoundDefaultValue != None:
                                if self._OPT_CRYPT in FoundVarOptions:
                                    try:
                                        FoundValue = self._decode(FoundVar, FoundValue)
                                    except:
                                        FoundValue = None
                                if not FoundPattern or re.match(FoundPattern, FoundValue):
                                    self[FoundVar] = FoundValue
                print Line.rstrip()
        except:
            if IsFileReadable(self.Filename):
                return self._error('Failed to update %s "%s".' % (self.ItemTypeName, ConfigVar))

        # If the line has not been updated so far, the ConfigVar has to be added to the file:
        if UpdatedLine:
            try:
                f = open(self.Filename, 'a')
            except:
                return self._error('Failed to add %s "%s".' % (self.ItemTypeName, ConfigVar))
            f.writelines(CommentLineIfNewOrOld or '');
            f.writelines(UpdatedLine + '\n')
            f.close()
            UpdatedLine = None
            Operation = "Added"

        # Keep dictionary consistent with updated value:
        if ConfigValue != None:
            self[ConfigVar] = ConfigValue
        elif ConfigVar in self:
            del(self[ConfigVar])

        if  ConfigValue == None:
            ConfigValue = "removed"
        self._info('%s %s "%s" (%s).' % (Operation, self.ItemTypeName, ConfigVar, str(ConfigValue)))
        return True

    def upgrade(self, Permanent = True):
        """
        Upgrade a new or just loaded() configuration to define all currently known config vars.
        Optionally non Permanent, e.g. keep corresponding config file unchanged.
        @return int Number of config vars upgraded (added, removed or re-encryped).
        """
        loadCount = 0
        oldSelf = self.copy()
        keysToUpgrade = sorted(self._ConfigVars.keys())
        keyErrorRetry = len(keysToUpgrade) * 2
        while keysToUpgrade:
            key = keysToUpgrade.pop(0)
            Pattern, val, VarOptions = self._ConfigVars[key][0:3]
            if val != None:
                useDefaultValue = True
                if key in self:
                    if not (key in self._upgradableConfigVars):
                        continue    # up-to-date.
                    if self._OPT_CRYPT in VarOptions:
                        # This config var switched from plain to crypt
                        val = self[key]
                        useDefaultValue = False
                    else:
                        # Should be the upgrade of an old style default value
                        useDefaultValue = True
                if useDefaultValue:
                    try:
                        val = self._calcDefaultValue(val, oldSelf, self._upgradableConfigVars.get(key))
                    except KeyError:
                        if not (keyErrorRetry and keysToUpgrade): raise
                        keyErrorRetry -= 1
                        keysToUpgrade.append(key)
                        continue
                    if Pattern and not re.match(Pattern, val):
                        self._warning('Wrong default value or pattern for config var "%s", val="%s", pattern=%s' % (key, val, Pattern))
            elif not (key in self or key in self._upgradableConfigVars):
                continue

            if Permanent:
                if self.update(key, val):
                    loadCount += 1
            elif val != None:
                self[key] = val
            else:
                del(self[key])
        return loadCount

    def create(self, Template = None, Force = False, Load = True):
        """
        Create a new configuration file that contains all known config vars with their default value.
        Optionally use a given Template; optionally Force overwrite existing config files.  
        @return bool True iff config file has been written. 
        """
        if  Template == None:
            Template = self.CreateTemplate; 
        Thing = self._thing()
        if not IsFileReadable(self.Filename):
            self._info('Creating new %s.' % Thing)
        elif Force:
            self._info('Deleting old and creating new %s.' % Thing)
        else:
            self._info('The %s already exists. Not changed.' % Thing)
            return False

        try:
            f = open(self.Filename, 'w')
        except:
            return self._error('Failed to create %s.' % Thing)
        f.writelines(Template)
        f.close()
        return (not Load) or (self.load() and self.upgrade() > 0)

##############################################################

class UserConfiguration(Configuration):
    """
    The SMScon user configuration database
    """
    class pat(str):
        """
        Internal helper class for pattern to transport a user friendly help.
        """
        def help(self, HelpText):
            self.HelpText = HelpText
            return self

    ItemTypeName        = "user setting"
    CreateTemplate      = '# %s user settings (initially written by version %s)\n' % (NAME, VERSION)
    SmsCommandPreFix    = 'COM_'       # _ConfigVars with that prefix define commands that can be sent via SMS
    DefaultValueFormat  = '^\\+?[X.@]+$' # '+XXXXXXXXXXX', 'XXXX@XXXXXXXX.XX', 'XXXX.XXXXXXXX.XX', ...

    __IPorHostPattern   = '(?:[A-Za-z0-9]+(?:[-.][A-Za-z0-9]+)*)'       # see RFC 952, RFC 1123.
    _PhoneNumber        = pat('^(\\+{0,2}[0-9]+|)$')                    .help('Must be + and digits or empty.')
    _CmdPattern         = pat('')
    _BoolPattern        = pat('^(?:yes|no)$')                           .help('Must be "yes" or "no".')
    _IntPattern         = pat('^\\d+$')                                 .help('Must be integer (no sign).')
    _FloatPattern       = pat('^\\d+(?:\\.\\d+)?$')                     .help('Must be float-number (no sign).')
    _HostnamePattern    = pat('^(?:'       + __IPorHostPattern + '|)$') .help('Must be IP addr or hostname or empty.')
    _EmailPattern       = pat('^(?:[^@]+@' + __IPorHostPattern + '|)$') .help('Must be email address or empty.')
    _URIPattern         = pat('^(?:([^:/?#]+:|)(?://|)(' + __IPorHostPattern + ')(:\\d+|)([/?#].*|)|)$'   # more restrictive than RFC 2396 but allow to be empty.
                                                                    )   .help('Must be an URI (internet address) or empty.')
    _IPforSSHPattern    = pat('^(?:\\*|localhost|\\d+(?:\\.\\d+)*)$')   .help('Must be "*" or "localhost" or integer.')

    _OPT_CRYPT          = Configuration._OPT_CRYPT
    _OPT_PLAIN          = Configuration._OPT_PLAIN
    _APPVERSIONVARNAME  = '@APPVERSION'
    _ConfigVars = {
        # SETTINGNAME:      (re_pattern_of_value, default_value,    options,    comment),
        _APPVERSIONVARNAME: ('^\d+(\.\d+)*$',   VERSION,            _OPT_PLAIN, '# %s configuration file written by:' % NAME),

        # SMS command source:
        'SENDERNUMBER':     (_PhoneNumber,      '+0000000000',      _OPT_PLAIN, '# SMS number of recent controller. Receives SMS notifications:'),
        'MASTERNUMBER':     (_PhoneNumber,      ('SENDERNUMBER',),  _OPT_CRYPT, '# SMS number of master controller (encrypted):'),

        # List of known sms commands:
        'COM_CHECK':        (_CmdPattern,       'Check',            _OPT_PLAIN, '# Command to ask for help:'),
        'COM_REBOOT':       (_CmdPattern,       'Reboot',           _OPT_PLAIN, '# Command to reboot the phone:'),
        'COM_POWEROFF':     (_CmdPattern,       'Poweroff',         _OPT_PLAIN, '# Command to shutdown the phone:'),
        'COM_POWER':        (_CmdPattern,       'Power',            _OPT_PLAIN, '# Command to ask for phone battery status:'),
        'COM_LOCATION':     (_CmdPattern,       'Location',         _OPT_PLAIN, '# Command to ask current GPS location:'),
        'COM_REMOTEON':     (_CmdPattern,       'Remoteon',         _OPT_PLAIN, '# Command to start outgoing ssh connection:'),
        'COM_REMOTEOFF':    (_CmdPattern,       'Remoteoff',        _OPT_PLAIN, '# Command to stop outgoing ssh connection:'),
        'COM_CAMERA':       (_CmdPattern,       'Camera',           _OPT_PLAIN, '# Command to take picture:'),
        'COM_CALL':         (_CmdPattern,       'Call',             _OPT_PLAIN, '# Command to start an outgoing call:'),
        'COM_LOCK':         (_CmdPattern,       'Lock',             _OPT_PLAIN, '# Command to lock the phone:'), 
        'COM_UNLOCK':       (_CmdPattern,       'Unlock',           _OPT_CRYPT, '# Command to unlock the phone (encrypted):'),
        'COM_TRACKON':      (_CmdPattern,       'Trackon',          _OPT_PLAIN, '# Command to start GPS tracking:'),
        'COM_TRACKOFF':     (_CmdPattern,       'Trackoff',         _OPT_PLAIN, '# Command to stop GPS tracking:'),
        'COM_CUSTOM':       (_CmdPattern,       'Script',           _OPT_PLAIN, '# Command to run predefined user script:'),
        'COM_CUSTOMLOG':    (_CmdPattern, ('%s.log', 'COM_CUSTOM'), _OPT_PLAIN, '# Command to ask for user script output:'),
        'COM_SHELL':        (_CmdPattern,       'uptime',           _OPT_PLAIN, '# Command to execute that shell command. Returns output as notification:'),
        'COM_ALARM':        (_CmdPattern,       'Alarm',            _OPT_PLAIN, '# Command to play predefined alarm.wav:'),
        'COM_RESTART':      (_CmdPattern,       'Restart',          _OPT_PLAIN, '# Command to restart the daemon:'),
        'COM_LOG':          (_CmdPattern,       'Log',              _OPT_CRYPT, '# Command to ask for application log (encrypted):'),
        'COM_SILENTON':     (_CmdPattern,       'Silent',           _OPT_PLAIN, '# Command to make phone silent on incomming commands:'),
        'COM_SILENTOFF':    (_CmdPattern,       'NotSilent',        _OPT_PLAIN, '# Command to undo silence and restore phone audibility:'),
        'COM_STOLEN':       (_CmdPattern,       'Stolen',           _OPT_PLAIN, '# Command to permanently enable stolen mode:'),

        # Prefix/suffix for all command strings (in SMS message only): 
        'SMS_COMPREFIX':    (_CmdPattern,       '',                 _OPT_CRYPT, '# Prefix that must be present before all COM_XXX commands [even COM_SHELL] (encrypted):'),
        'SMS_COMSUFFIX':    (_CmdPattern,       '',                 _OPT_CRYPT, '# Suffix that must be present behind all COM_XXX commands [even COM_SHELL] (encrypted):'),

        # REMOTE command source
        'ENABLECHECKHOST':  (_BoolPattern,      'no',               _OPT_PLAIN, '# Enable checking URL for commands:'),
        'CHECKHOST':        (_URIPattern, 'http://example.com/file',_OPT_CRYPT, '# The URL to check (encoded):'),
        'CHECKTIME':        (_FloatPattern,     '15.0',             _OPT_PLAIN, '# Minutes between URL checks:'),

        # Email settings
        'EMAILADDRESS':     (_EmailPattern,     'to@example.com',   _OPT_PLAIN, '# Recipient of EMAIL notifications (regular owner of this phone, please contact him!):'),
        'USER':             ('',                '',                 _OPT_CRYPT, '# SMTP username for EMAILFROM (encrypted):'),
        'PASSWORD':         ('',                '',                 _OPT_CRYPT, '# SMTP password for EMAILFROM (encrypted):'),
        'EMAILFROM':        (_EmailPattern,     'from@example.com', _OPT_CRYPT, '# Logical sender address of EMAIL notification (encrypted):'),
        'EMAILSENDER':      (_EmailPattern,     '',                 _OPT_CRYPT, '# Technical sender address of EMAIL notification (encrypted):'),
        'MAILSERVER':       (_HostnamePattern,  'smtp.example.com', _OPT_CRYPT, '# SMTP server hostname or IP (encrypted):'),
        'MAILPORT':         (_IntPattern,       '25',               _OPT_PLAIN, '# SMTP server port number:'),

        # SSH settings
        'REMOTEHOST':       (_HostnamePattern,  'cmd.example.com',  _OPT_CRYPT, '# ssh server hostname or IP [to build tunnel] (encrypted):'),
        'REMOTEPORT':       (_IntPattern,       '22',               _OPT_PLAIN, '# ssh server port number [to build tunnel]:'),
        'REMOTEUSER':       ('',                'username',         _OPT_CRYPT, '# ssh server username [to build tunnel] (encrypted):'),
        'REMOTEPASSWORD':   ('',                'password',         _OPT_CRYPT, '# ssh server password [to build tunnel] (encrypted):'),
        'REMOTELISTENIP':   (_IPforSSHPattern,  '*',                _OPT_CRYPT, '# ssh server IP address [entry of tunnel itself] that will be forwarded to phone (*|localhost|IP):'),
        'REMOTELISTENPORT': (_IntPattern,       '8080',             _OPT_CRYPT, '# ssh server port number [entry of tunnel itself] that will be forwarded to phone:'),
        'REMOTE2LOCALPORT': (_IntPattern,       '22',               _OPT_PLAIN, '# Local port number to be forwarded to [exit of tunnel itself] on the phone:'),

        # Notification settings
        'COMMANDREPLY':     (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification if command accepted (yes|no):'),
        'MESSAGESEND':      (pat('^(sms|email|both|none)$').help('Must be one of "none", "sms", "email" or "both".'),
                                                'sms',              _OPT_PLAIN, '# How to send notifications (sms|email|both|none):'),
        'ENABLERESEND':     (_BoolPattern,      'no',               _OPT_PLAIN, '# On failure try to re-send reply message (yes|no):'),
        'RESENDTIME':       (_FloatPattern,     '15.0',             _OPT_PLAIN, '# Minutes between message re-send tries:'),
        'RESENDMAXTRY':     (_IntPattern,       '10',               _OPT_PLAIN, '# Maximum number of message re-send tries (0=no limit):'),
        'SMS_PARTS_LIMIT':  (_IntPattern,       '10',               _OPT_PLAIN, '# A SMS notification produces no more than this number of single SMS (0=no limit):'),
        'LOG_TIMEFORMAT':   ('%',           '%Y-%m-%d %H:%M:%S',    _OPT_PLAIN, '# Format to use for timestamps (see strftime):'),

        # GPS settings
        'GPSTIMEOUT':       (_IntPattern,       '1800',             _OPT_PLAIN, '# Seconds until give up acquiring GPS coordinates:'),
        'GPSPOLLING':       (_IntPattern,       '8',                _OPT_PLAIN, '# Number of minimum GPS coordinate acquisitions for COM_LOCATIONs best of:'),
        'GPSINTERVAL':      (_IntPattern,       '10',               _OPT_PLAIN, '# Seconds between GPS coordinate acquisitions for COM_TRACKON and COM_LOCATION:'),
        'GPSMAXDEVIATION':  (_IntPattern,       '50',               _OPT_PLAIN, '# Meters of minimum accuracy of a GPS coordinate for COM_LOCATION request:'),  
        'GPSSEND':          (pat('^(sms|email|both)$').help('Must be one of "sms", "email" or "both".'),
                                                'sms',              _OPT_PLAIN, '# How to send GPS notifications (sms|email|both):'),

        # Miscellaneous settings
        'SAVEDAUDIBILITY':  ('',                '',                 _OPT_PLAIN, '# Audibility information to be used by COM_SILENTOFF (internal use only):'),
        'SILENCEDEVICE':    (_BoolPattern,      'no',               _OPT_PLAIN, '# Set phone silent before executing command (yes|no):'),
        'AUTODEVICELOCK':   (_BoolPattern,      'no',               _OPT_PLAIN, '# Lock phone before executing any command except un/lock (yes|no):'),
        'DEBUGFLAGS':       ('',                '',                 _OPT_PLAIN, '# A csv with debugging options (for development use only):'),

        # Settings for device un/locking on boot
        'LOCKMODE_OKSIM':   (pat('^(keep|lockifnopin|lockalways)$').help('Must be one of "keep", "lockifnopin" or "lockalways"'),
                            lambda oldd, curd, newv: IF(oldd.get('SIMUNLOCK')=='lockalways','lockalways','keep'),
                                                                    _OPT_PLAIN, '# Automatic device locking on authorized SIM: (keep|lockifnopin|lockalways):'),
        'LOCKMODE_NEWSIM':  (pat('^(keep|unlock|lock)$').help('Must be one of "keep", "unlock", "lock"'),
                            lambda oldd, curd, newv: {'no':'keep', 'yes':'unlock', 'locknewsim':'lock'}.get(oldd.get('SIMUNLOCK'), 'keep'),
                                                                    _OPT_PLAIN, '# Automatic device un/locking on unauthorized SIM with wrong PIN: (keep|unlock|lock):'),
        'LOCKMODE_BADSIM':  (pat('^(keep|unlock|lock)$').help('Must be one of "keep", "unlock", "lock"'),
                            lambda oldd, curd, newv: IF(oldd.get('SIMUNLOCK')=='lockifnopin','lock','keep'),
                                                                    _OPT_PLAIN, '# Automatic device un/locking on unauthorized SIM with wrong PIN: (keep|unlock|lock):'),
        # 'SIMUNLOCK' is obsolete. It is no longer used and defined only for smsconlib's API downward compatibility:
        'SIMUNLOCK':        (pat('^(no|yes|locknewsim)$').help('Must be one of "no", "yes", "locknewsim"'),
                            lambda oldd, curd, newv: IF(newv in ('no','yes','locknewsim'), newv, 'no'),
                                                                    _OPT_PLAIN, '# Automatic device un/locking on unauthorized SIM with right PIN: (no|yes|locknewsim):'),
        'DEVICELOCKLATE':   (_BoolPattern,      'yes',              _OPT_PLAIN, '# Booting phone should ask for SIM PIN before device-lock code (yes|no):'),
        'STOLENIFNOSIM':    (_BoolPattern,      'yes',              _OPT_PLAIN, '# Activate stolen mode if no SIM is present:'),
    
        # Sensor detections
        'KEYBOARDDETECT':   (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification if keyboard is sliding (yes|no):'),
        'AUTOBATREPORT':    (_BoolPattern,      'no',               _OPT_PLAIN, '# Send notification on battery discharge (yes|no):'),

        # Obsolete settings: 
        'DISABLESMS':       (None,              None,               _OPT_PLAIN, 'setting removed since version 0.8.0-1'),
        'SIMAUTHORIZATION': (None,              None,               _OPT_PLAIN, 'setting removed')
    }
    _Secret = '6983456712936753'

    def __init__(self, Filename = None):
        Configuration.__init__(self, Filename = Filename or os.path.join(InstPath, ConfigFile))

    def _decode(self, ConfigVar, ConfigValue):
        try:
            return Configuration._decode(self, ConfigVar, ConfigValue)
        except:
            # Handle config vars that were formerly stored un-encrypted
            if ConfigVar in ['CHECKHOST', 'REMOTEHOST', 'COM_LOG', 'COM_UNLOCK', 'EMAILFROM', 'MAILSERVER']:
                self._upgradableConfigVars[ConfigVar] = True
                return ConfigValue
            raise

    def update(self, ConfigVar, ConfigValue):
        """Same as base class with support for automatic _APPVERSION update."""
        
        # Prevent to commands being defined as the same string, except the empty string (which disables):
        if ConfigVar.startswith(self.SmsCommandPreFix) and ConfigValue != None and ConfigValue != "":
            for OtherVar in self.getSmsCommandList():
                if OtherVar != ConfigVar and self[OtherVar] == ConfigValue:
                    return self._error('Command string "%s" already used for command "%s".' % (ConfigValue, OtherVar))

        return Configuration.update(self, ConfigVar, ConfigValue) and (
                (ConfigVar == self._APPVERSIONVARNAME   # --> already updated above or ...
                 or self.get(self._APPVERSIONVARNAME) == VERSION    # --> version up to date or ...
                 or Configuration.update(self, self._APPVERSIONVARNAME, VERSION)))

    def getSmsCommandList(self):
        """@return list of config vars that are SMS commands"""
        return [cmd for cmd in self._ConfigVars if cmd in self and cmd.startswith(self.SmsCommandPreFix)]

    def parseSmsCommand(self, CmdText, CmdPattern, ParametersFound = {}):
        """
        Check if a text matches a given command pattern and extract command parameters.
        Command parameters are defined by * in the pattern. Many * may occur and each defines
        a separate parameter unless they are consecutive. Two or more consecutive * define
        a single parameter which declares itself and all the parameters up to the last one
        on the right hand side to be optional.
        If optional parameters are omitted, the non-* text between them must be omitted as well.
        @param string CmdText: command text being checked.
        @param string CmdPattern: pattern to check against.
        @param dict ParametersFound: ['args'] slot receives array of command parameters. 
        @return bool true iff command detected.
        """
        ParametersFound['args'] = []
        if CmdText == '' or CmdPattern == '':   # reject empty or disabled commands
            return False;
        head, sep, tail = CmdPattern.partition('*')
        if sep == '':
            return CmdText == CmdPattern    # simple case, no '*' in CmdPattern
        pat, patTail, ellipsisSeen = '', '', False;
        while sep != '':
            ellipsisSeen = ellipsisSeen or tail[0:1] == '*'
            tail = tail.lstrip('*')
            if pat == '':   # text before 1st parameter (optional or not) must always be present.
                pat += re.escape(head) + '(.*?)'
                if ellipsisSeen: pat += '(?:' 
            else:   # text before omitted optional parameter must be omitted as well.
                if ellipsisSeen: pat += '(?:' 
                pat += re.escape(head) + '(.*?)'
            if ellipsisSeen:
                patTail = ')?' + patTail;
            head, sep, tail = tail.partition('*')
        pat = '^' + pat + patTail + re.escape(head) + '$';
        args = re.findall(pat, CmdText)
        if len(args) > 0 and type(args[0]).__name__ in ('tuple', 'list'):
            args = args[0];
        ParametersFound['args'] = args;
        return len(args) > 0

##############################################################
# IMSI interface

class IMSIConfiguration(Configuration):
    """
    Instances of this class represent the (one and only) database of authorized SIM cards.
    """ 
    ItemTypeName        = "SIM code"
    Thing               = "SIM database"
    ConfigPrintFormat   = "%s%s"
    ConfigParseFormat   = "^\\s*(.*?)\\s*()()$"
    TabSize             = 18
    Loaded              = False

    def __init__(self):
        Configuration.__init__(self, Filename = os.path.join(InstPath, CodeFile))

    def _genConfigLine(self, ConfigVar, ConfigValue):
        return ConfigVar

    def load(self, **kwargs):
        self.Loaded = Configuration.load(self, **kwargs)
        return self.Loaded

    def PackCodeType(self, CodeType, ConfigValue):
        """Assemble type and value to a string actually stored in database."""
        if CodeType == 'IMSI':  return ConfigValue
        else:                   return CodeType + ':' + ConfigValue

    def UnpackCodeType(self, ConfigValue):
        """Disassemble string value from database to real type and value."""
        CodeType, sep, CodeValue = ConfigValue.partition(':')
        if sep: return [CodeType, CodeValue]
        else:   return ['IMSI', ConfigValue]

    def updateCode(self, CodeType, ConfigName, ConfigValue, Silent = False):
        self.Loaded = False
        return self.update(self.PackCodeType(CodeType, ConfigName), ConfigValue, Silent)

    def IsAuthorizedSIM(self, ICCID, IMSI):
        """
        Check if SIM with given ICCID and IMSI is an authorized SIM.
        Implicitely convert (upgrade) SIM database: remove IMSI code, store ICCID code.
        @param string|None ICCID code from SIM.
        @param string|None IMSI code from SIM.
        @return bool
        """
        if not self.Loaded and not self.load(Silent = True):
            return False
        ICCID = ICCID and self.PackCodeType('ICCID', ICCID)
        IMSI  = IMSI  and self.PackCodeType('IMSI',  IMSI)
        if ICCID == None:
            return False    # no IMSI in this case.
        if ICCID in self.keys():
            # Ensure only ICCID but no IMSI code stored for this SIM:
            if IMSI and IMSI in self.keys():
                self.update(IMSI, None, Silent = True)
            return True
        if not (IMSI and IMSI in self.keys()): 
            return False
        # Convert entry: remove IMSI code, add ICCID code:
        self.update(IMSI, None, Silent = True)
        self.update(ICCID, '')
        return True

    def GrantAuthorizationToSIM(self, ICCID, IMSI):
        """
        Add SIM of specified card codes to SIM database.
        @param string|None ICCID code from SIM.
        @param string|None IMSI code from SIM.
        @return bool True iff SIM card data written to SIM database.
        """
        if ICCID == None or IMSI == None:
            return False
        self.updateCode('IMSI', IMSI, None, Silent = True) # Ensure IMSI code of this SIM is removed from database.
        return self.updateCode('ICCID', ICCID, '')

    def RevokeAuthorizationFromSIM(self, ICCID, IMSI):
        """
        Remove SIM of specified card codes from SIM database.
        @param string|None ICCID code from SIM.
        @param string|None IMSI code from SIM.
        @return bool True iff specified SIM code no longer present in SIM database.
        """
        ok = ICCID != None and self.updateCode('ICCID', ICCID, None)
        ok = (IMSI != None and self.updateCode('IMSI', IMSI, None, Silent = True)) or ok
        return ok

    def RevokeAuthorizationFromAllSIM(self):
        """
        Remove all SIM card codes from SIM database. 
        @return bool True iff no SIM codes present in SIM database.
        """
        self.Loaded = False
        return self.create(Force = True)


def GetPhoneCode(CodeType):
    """
    Get device hardware IMEI code (always available)
    or current SIM ICCID code (available if SIM present, et all)
    or SIM IMSI code (available if correct SIM PIN was entered).
    @return string|None The requested code. None if code is not available.
    """
    if CodeType == 'IMEI':
        Command = 'dbus-send --system --type=method_call --print-reply --dest=com.nokia.phone.SIM /com/nokia/phone/SIM/security Phone.Sim.Security.get_imei'
    elif CodeType == 'ICCID':
        Command = 'dbus-send --system --type=method_call --print-reply --dest=com.nokia.phone.SSC /com/nokia/phone/SSC com.nokia.phone.SSC.get_iccid'
    elif CodeType == 'IMSI':
        Command = 'dbus-send --system --type=method_call --print-reply --dest=com.nokia.phone.SIM /com/nokia/phone/SIM Phone.Sim.get_imsi'
    else:
        LOGGER.error('Unknown code type "%s"' % CodeType)
        return None

    try:
        E = re.compile('string "(\S+)"')    # need non-empty string!
        return E.findall(os.popen(Command).read())[0]
    except:
        LOGGER.warning('No %s code available.' % CodeType)
        return None

##############################################################
# smscon_daemon interface

class smscon_daemon:
    """
    Instances of this class are used to control the smscon daemon.
    """
    TELL_RELOAD    = signal.SIGHUP
    TELL_TERMINATE = signal.SIGTERM
    TELL_COMMAND   = signal.SIGINT
    _cmdPrefix     = '<cmd>'
    _cmdSuffix     = '</cmd>'

    def __init__(self):
        self.commandChannel = None

    def __del__(self):
        self.closeCommandChannel()

    def runs(self):
        """
        Check if smscon_daemon is active.
        @return string List of PIDS. Empty if none (evaluates to false in boolean expressions).
        """
        return pidof(os.path.join(InstPath, DaemonFile))
    
    def start(self, Mode = None, Function = None):
        """
        Start new smscon_daemon instance (even if some daemon instances already run).
        @return void.
        """
        Mode = Mode or ''
        Function = Function or ''
        DaemonArgs = ''
        if Mode or Function:
            DaemonArgs = 'CLIMODE=%s CLIFUNCTION=%s' % (shellArg(Mode), shellArg(Function))

        if IsBootfileExistent():
#            if Mode or Function:
#                DaemonArgs = shellArg('CLIMODE=' + Mode) + ' ' + shellArg('CLIFUNCTION=' + Function)
#            os.system('/sbin/initctl --quiet start %s %s' % (NAME, DaemonArgs))
            # --- Above code won't work with initctl version 0.3.8 because it didn't pass environment
            # variables to the started job.  Neither by passing them as command arg nor by setting
            # them up in callers environment :-( Workaround: start and pass command via command channel:
            os.system('/sbin/initctl --quiet start %s' % BootFile)
            if Mode or Function:
                time.sleep(2)
                self.writeCliCommand(Mode, Function)
        else:
            os.system('%s "%s"&' % (DaemonArgs, os.path.join(InstPath, DaemonFile)))
        LOGGER.info('Daemon %s requested to start%s.' % (DaemonFile, IF(Mode or Function,' '+Mode+' '+Function,'')))
    
    def stop(self):
        """
        Kill all smscon_daemon(s).
        @return: string with PIDs of daemons stopped. Empty if none were running.
        """
        if IsBootfileExistent():
            PIDs = self.runs()
            os.system('/sbin/initctl --quiet stop %s' % BootFile)
        else:
            PIDs = self.tell(self.TELL_TERMINATE)
        if PIDs:
            LOGGER.info('Daemon %s stopped [PID=%s].' % (DaemonFile, PIDs) )
        else:
            LOGGER.info('Daemon %s was not running.' % DaemonFile)
        return PIDs
    
    def tell(self, SignalNumber):
        """
        Send a signal to all running smscon_daemon(s).
        @param int SignalNumber The signal to be sent to the daemons.
        @return: string With list of daemons signaled.
        """
        PIDs = self.runs()
        if PIDs:
            os.system('kill -%d %s' % (abs(SignalNumber), PIDs))
        return PIDs

    def openCommandChannel(self, Mode, Silent = False):
        if self.commandChannel != None:
            return True     # TODO: Check if is open in the requested Mode.  

        isReadMode = not 'w' in Mode
        ModeName = IF(isReadMode, 'reading', 'writing')
        try:
            self.commandChannel = os.open(os.path.join(InstPath, CommandFile),
                                IF(isReadMode, os.O_RDONLY, os.O_WRONLY) | os.O_NONBLOCK)
            if isReadMode:
                self.commandChannel = os.fdopen(self.commandChannel)
        except OSError, e:
            if e.errno == os.errno.ENXIO: # may happen os.O_WRONLY only
                if not Silent:
                    LOGGER.warning('Failed to open instruction channel for %s: Daemon %s was not running.' % (ModeName, DaemonFile))
                return False
            if not Silent:
                LOGGER.error('Failed to open instruction channel for %s: %s' % (ModeName, e))
            return False
        except Exception, e:
            if not Silent:
                LOGGER.error('Failed to open instruction channel for %s: %s' % (ModeName, e))
            return False
        return True

    def closeCommandChannel(self):
        """
        Close the command channel if it was open.
        @return bool Success indicator.
        """
        if self.commandChannel == None:
            return True
        try:
            if type(self.commandChannel).__name__ == 'int':
                os.close(self.commandChannel)
            else:
                self.commandChannel.close()
            self.commandChannel = None
        except Exception, e:
            if LOGGER:
                LOGGER.error('Failed to close instruction channel: %s' % e)
            self.commandChannel = None  # fail only once.
            return False
        return True

    def writeCommand(self, Command = None):
        """
        Send an synchroneous command to control a running daemon.
        @param string|None Command. A None-Command may be used to check if a daemon is
                    currently listening for commands without actually sending a command. 
        @return bool True if the command has been send. False on errors or if no daemon is listening.
        """
        if not self.openCommandChannel('w'):
            return False
        if Command == None:
            return True;
        Command = str(Command).strip()
        if Command != '':
            while True:
                try:
                    os.write(self.commandChannel, self._cmdPrefix + Command + self._cmdSuffix + '\n')
                except IOError, e:
                    if e.errno == os.errno.EPIPE:
                        self.closeCommandChannel()
                        if not self.openCommandChannel('w'):
                            return False
                        continue
                    LOGGER.error('Failed to write instruction channel: %s' % e)
                    return False
                except Exception, e:
                    LOGGER.error('Failed to write instruction channel: %s' % e)
                    return False
                break
        self.tell(self.TELL_COMMAND)
        return True 

    def writeCliCommand(self, *args):
        """
        Send an synchroneous "command-line-style" command to control a running daemon.
        @return bool True if the command has been send. False on errors or if no daemon is listening.
        """
        if len(args) == 0:
            return self.writeCommand(None)
        else:   # TODO: quote xml entities etc...
            return self.writeCommand("<cli><a>%s</a></cli>" % '</a><a>'.join(args))

    def readCommand(self):
        """
        Read an asynchroneous command from a daemon controller.
        This method must be called repeatedly unless it returns empty string.
        @return string One command string. Empty if no command.
        """
        if not self.openCommandChannel('r'):
            return ''
        while True:
            try:
                Command = self.commandChannel.readline()
            except IOError, e:
                if e.errno == os.errno.EAGAIN:
                    return ''
                if e.errno == os.errno.EPIPE:
                    self.closeCommandChannel()
                    if not self.openCommandChannel('r'):
                        return ''
                    continue
                LOGGER.error('Failed read instruction channel: %s' % e)
                return ''
            except Exception, e:
                LOGGER.error('Failed read instruction channel: %s' % e)
                return ''
            if Command == '':
                return ''
            Command = Command.strip()
            if not Command.startswith(self._cmdPrefix) or not Command.endswith(self._cmdSuffix):
                LOGGER.warning('Garbage found in command channel: %s' % Command)
            else:
                Command = Command[len(self._cmdPrefix):-len(self._cmdSuffix)]
                if Command != '':   # dont return empty command; caller would thing it was eof
                    return Command

    def isCliCommand(self, cmd):
        """
        Check if string received by readCommand() is a command-line-style command
        @param string cmd The command received.
        @return list|False The list of command line args or False if not command line args.  
        """ 
        if cmd.startswith('<cli><a>') and cmd.endswith('</a></cli>'):
            return cmd[8:-10].split('</a><a>')  # TODO: convert quoted xml entities
        return False

##############################################################
# remote shell connection interface 

class smscon_remote_sh(dict):

    def __init__(self, args):
        for k in ('REMOTEHOST', 'REMOTEPORT', 'REMOTEUSER', 'REMOTEPASSWORD',
                  'REMOTELISTENIP', 'REMOTELISTENPORT', 'REMOTE2LOCALPORT'):
            setattr(self, k,  args.get(k) or '')

    def _getSSHcmdline(self, nParts = 3):
        return ' '.join((
            # the command and flags to tell contacted ssh server to build tunnel (-n -N -T -f):
                'ssh -nNTf',
            # definition of the entry and exit points of the tunnel to build:
                '-R %s:%s:%s:%s' % (self.REMOTELISTENIP, self.REMOTELISTENPORT, 'localhost', self.REMOTE2LOCALPORT),
            # port, user and host to contact ssh server to build tunnel:
                '-p %s %s@%s' % (self.REMOTEPORT, self.REMOTEUSER, self.REMOTEHOST)
            )[:nParts])

    def runs(self):
        """
        Check if ssh is active or not running.
        @return string list of SSH PIDs. Empty if none is running (evaluates to bool False)
        """
        # -- This would also detect ssh processes not created for reverse-ssh connection:
        # -- return os.popen('pidof ssh').read().strip('\n') != ''
        PIDs = []
        # Do not use the complete command line for grep because ps may show it only party.
        for psLine in os.popen('busybox ps|grep -v grep|grep "%s"' % re.escape(self._getSSHcmdline(2))).readlines():
            PIDs.append(psLine.split()[0])
        return ' '.join(PIDs)

    def start(self):
        """
        Silently start reverse-ssh connection to remote host.
        @return (bool, message) Success indicator, error message
        """
        (Output, ExitStatus) = pexpect.run(self._getSSHcmdline(),
            events = { '(?i)(password|passphrase)':self.REMOTEPASSWORD + '\n', '(?i)(yes/no) ?':'yes' + '\n' },
            withexitstatus = True )
        return (Output, ExitStatus)
        
    def stop(self, Verbose = False):
        """
        Kill reverse-ssh connection to remote host.
        @return: string with PIDs of connections stopped. Empty if none were running.
        """
        if Verbose:
            LOGGER.info('Trying to stop ssh connections to "%s".' % self.REMOTEHOST)
        PIDs = self.runs()
        if PIDs:
            os.system('kill %s' % PIDs)
            if Verbose:
                LOGGER.info('ssh connections stopped [PIDs=%s].' % PIDs)
        else:
            if Verbose:
                LOGGER.info("ssh connection wasn't active")
        return PIDs

##############################################################
# Custom script interface

class smscon_script:
    """
    Instances of this class represent the (one and only) user script.
    """

    def __init__(self):
        self.Filename = os.path.join(InstPath, ScriptFile)

    def init(self, Force = False):
        """
        Check if smscon_script exists, else create new default one.
        @return bool True iff the script file exists or could be created.
        """
        ScriptTemplate = """#!/bin/sh
# %s custom script (initialized by %s version %s) """ % (NAME, NAME, VERSION) + """
# Script will start on COM_CUSTOM command and executes asynchronously.
# Script output (stdout and stderr) can be retrieved by COM_CUSTOMLOG command.
#
# Use non-consecutive '*' characters in COM_CUSTOM command definition to define placeholders.
# Placeholder accept arbitrary text which is passed to this script as positional parameter.
# Consecutive '**' indicate that this and all following placeholder are optional.
# Example: The COM_CUSTOM command definition "MYCMD *,**,*" will recognize all these incoming SMS:
# "MYCMD myparam1,myparam2,myparam3" --> "myparam1" "myparam2" "myparam3"
# "MYCMD t h i s,that,"              --> "t h i s" "that" ""    (text for 3rd placeholder given but empty)
# "MYCMD t h i s,that"               --> "t h i s" "that" ""    (text for 3rd placeholder not given)
# "MYCMD a, b, c"                    --> "a" " b" " c"          (note the blanks)
# "MYCMD a,b,c,d,e,f"                --> "a" "b" "c,d,e,f"      (right most placeholder is greedy)
# It will not recognize:
# "MYCMD,a,b,c"        (comma rather than space behind MYCMD)
# "MYCMD"              (missing space behind MYCMD)
# Consider to use a case statement to dispatch to any number of your own scripts.

echo Number of parameters passed to this script: $#. The parameter values, one per line, are:
for p in "$@"
do
    echo "$p"
done

# Here are some example commands you might consider:
echo Process list on $(date):
ps
echo netstat:
netstat -tean

# Another idea might be to add commands to clean or remove all
# personal data files from phone: multimedia files, picutes, music,
# mail, contact data, etc.. But this is up to you. 
"""
        Filename = self.Filename
        Thing = 'script file "%s"' % self.Filename

        if not IsFileReadable(Filename):
            LOGGER.info('Creating new %s.' % Thing)
        elif Force:
            LOGGER.warning('Forced to create new %s.' % Thing)
        else:
            LOGGER.info('The %s already exists. Not changed.' % Thing)
            return True

        try:
            f = open(Filename, 'w')
        except:
            LOGGER.error('Failed to create %s.' % Thing)
            return False            
    
        f.writelines(ScriptTemplate)
        f.close()

        if os.WEXITSTATUS( os.system( 'chmod 755 "%s"' % Filename) ) != 0:
            LOGGER.error('Failed to set user permissions on "%s".' % Filename)
            return False            

        if os.WEXITSTATUS( os.system( 'chown root "%s"' % Filename ) ) != 0:
            LOGGER.error('Failed to set owner of "%s".' % Filename)
            return False            

        LOGGER.warning("The initial %s does yet nothing. Edit it directly!" % Thing)
        return True        

    def stop(self):
        """
        Kill smscon_script file (if running).
        @return: string with PIDs of scripts stopped. Empty if none were running.
        """
        PIDs = pidof(self.Filename)
        if PIDs:
            os.system('kill %s' % PIDs)
            LOGGER.info('Script %s stopped [PIDs=%s].' % (ScriptFile, PIDs))
        else:
            LOGGER.info('Script %s was not running.' % ScriptFile)
        return PIDs

##############################################################

def IsBootfileExistent():
    """
    @return bool True iff a BootFile exists and is readable.
    """
    return IsFileReadable(os.path.join(BootPath, BootFile))

def CreateBootfile():
    """
    Setup to start smscon_daemon at device boot.
    @return bool True iff a BootFile exists or could be created.
    """
    BootCode = """\
description "%(NAME)s - nokia n900 remote control utility"
author "Lutz Schwarz, Frank Visser"
version "$Id: smsconlib.py 251 2012-08-12 21:01:58Z yablacky $"

start on started 00smscon
stop on starting shutdown

env LATELOCK="/root/.%(NAME)s_latedevlock"
env BOOTLOCK="/root/.%(NAME)s_bootdevlock"

console none

pre-start script
    LOG() { test ! -e "%(LogFile)s" || echo $* >>"%(LogFile)s" ; }
    until [ -e "%(InstPath)s" -a -f /tmp/dbus-info ] ; do
        sleep 3
    done
    LOG pre-start
    rm -f "$BOOTLOCK"
    if [ -f "$LATELOCK" ]
    then
        LOG "Doing late dev lock."
        rm -f "$LATELOCK"
        nice -2 run-standalone.sh dbus-send --system --type=method_call \\
            --dest=com.nokia.mce /com/nokia/mce/request \\
            com.nokia.mce.request.devlock_callback int32:0 || true
    fi
end script

script
    test -f /tmp/dbus-info && . /tmp/dbus-info   
    export DBUS_SESSION_BUS_ADDRESS          
    export DBUS_SESSION_BUS_PID              
    test -e       "%(LogFile)s" || exec "%(DaemonFile)s"
    echo start >> "%(LogFile)s"
    exec "%(DaemonFile)s" >> "%(LogFile)s" 2>&1
end script

post-stop script
    LOG() { test ! -e "%(LogFile)s" || echo $* >>"%(LogFile)s" ; }
    LOG post-stop
    if [ ! -r "%(BootFile)s" ]
    then
        rm -f "$LATELOCK"
        LOG "Late dev lock not possible. Preparing for normal dev lock."    
    elif [ -r "$BOOTLOCK" ]
    then
        LOCKMODE_OKSIM=unknown
        DEVICELOCKLATE=unknown
        if [ -r "%(ConfigFile)s" ]
        then
            LOCKMODE_OKSIM=$(awk -F \\' "/^\\\\s*LOCKMODE_OKSIM\\\\s*=\\\\s*'.*'/ { print \$2 }" "%(ConfigFile)s")
            DEVICELOCKLATE=$(awk -F \\' "/^\\\\s*DEVICELOCKLATE\\\\s*=\\\\s*'.*'/ { print \$2 }" "%(ConfigFile)s")
        fi
        DEVLOCK="$(cat "$BOOTLOCK")"
        rm -f "$BOOTLOCK"
        LOG "Dev lock status = $DEVLOCK, lock mode = $LOCKMODE_OKSIM, lock late = $DEVICELOCKLATE."
        >"$LATELOCK"
        if [ -f "$LATELOCK" ] &&
           [ \( "$DEVICELOCKLATE" == yes -a "$DEVLOCK" != unlocked \) -o "$LOCKMODE_OKSIM" == lockalways ]
        then
            run-standalone.sh dbus-send --system --type=method_call \\
                --dest=com.nokia.mce /com/nokia/mce/request \\
                com.nokia.mce.request.devlock_callback int32:2 || true
            test "$LOCKMODE_OKSIM" != lockalways || rm -f "$LATELOCK"
            LOG "Preparing for late dev lock."
        else
            rm -f "$LATELOCK"
            LOG "Preparing for normal dev lock."
        fi
    else
        rm -f "$LATELOCK"
        LOG "Dev lock status unknown. Preparing for normal dev lock."
    fi
end script

respawn
""" % {'NAME': NAME, 'VERSION': VERSION, 'InstPath': InstPath,
       'ConfigFile' : os.path.join(InstPath, ConfigFile),
       'DaemonFile' : os.path.join(InstPath, DaemonFile),
       'LogFile'    : os.path.join(InstPath, NAME+'_boot.log'),
       'BootFile'   : os.path.join(BootPath, BootFile)
       }

    # Above script does the following: Log the daemon output in smscon_boot.log only
    # if a smscon_boot.log did exist already. Since installation of smscon 0.10.4 such
    # a file does normally no longer exist. It may be created for debugging purposes.

    Filename = os.path.join(BootPath, BootFile)
    if IsFileReadable(Filename):
        LOGGER.info('Boot file "%s" already exists. Not changed.' % Filename)
        return True  

    try:
        f = open(Filename, "w")
    except:
        LOGGER.error('Failed to create boot file "%s".' % Filename)
        return False

    f.writelines(BootCode)
    f.close()
    LOGGER.info('Boot file "%s" created & active.' % Filename)
    return True 

##############################################################

class OurRotatingFileHandler(logging.handlers.RotatingFileHandler):
    """
    A RotatingFileHandler that cares for umask when we need it.
    """
    def doRollover(self):
        """
        Do a rollover, as described in __init__() of /usr/lib/python2.5/logging/handlers.py
        but use self.mode to open the file and use umask to hide log from world.
        """
        self.stream.close()
        if self.backupCount > 0:
            for i in range(self.backupCount - 1, 0, -1):
                sfn = "%s.%d" % (self.baseFilename, i)
                dfn = "%s.%d" % (self.baseFilename, i + 1)
                if os.path.exists(sfn):
                    #print "%s -> %s" % (sfn, dfn)
                    if os.path.exists(dfn):
                        os.remove(dfn)
                    os.rename(sfn, dfn)
            dfn = self.baseFilename + ".1"
            if os.path.exists(dfn):
                os.remove(dfn)
            os.rename(self.baseFilename, dfn)
            #print "%s -> %s" % (self.baseFilename, dfn)
        umaskOld = os.umask(0137)
        if self.encoding:
            self.stream = codecs.open(self.baseFilename, self.mode, self.encoding)
        else:
            self.stream = open(self.baseFilename, self.mode)
        os.umask(umaskOld)

def StartLogging(instanceName, logLevel = None, withConsole = True, withEmitterFxn = False):
    """
    Start logging for specified instance at specified level (default all)
    @return Logger the logger you have to use.
    """
    if logLevel == None:
        logLevel = logging.INFO
    if XDEBUG:
        logLevel = logging.DEBUG

    logger = logging.getLogger(instanceName)
    logger.setLevel(logLevel)

    if os.geteuid() == 0:
        setattr(logger, 'LogFilename', os.path.join(LogPath, LogFile))
#        fh = logging.FileHandler(logger.LogFilename, 'a')
        fh = OurRotatingFileHandler(logger.LogFilename, 'a', 96*1024, 5)
#        fh.setLevel(logLevel)            -- NO! dynamically inherit logLevel from parent
        fh.setFormatter(logging.Formatter('%(asctime)s[%(name)s] %(levelname)s: %(message)s', datefmt = TimeFormat))
        logger.addHandler(fh)

    if withConsole:
        ch = logging.StreamHandler()
#        ch.setLevel(logging.INFO)        -- NO! dynamically inherit logLevel from parent
        ch.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
        logger.addHandler(ch)

    if withEmitterFxn:

        class callbackHandler(logging.Handler):
            '''Class defining logging handler that call a foreign emit function.'''
            def __init__(self, emitter, level = logging.NOTSET):
                logging.Handler.__init__(self, level)
                self.emitter = emitter
            def emit(self, logRecord):
                self.emitter(logRecord)

        xh = callbackHandler(emitter = withEmitterFxn, level = logLevel)
        xh.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
        logger.addHandler(xh)

    global LOGGER
    if LOGGER == None:
        LOGGER = logger

    return logger

def EnableDebugLogging(userConfig = None, enableDebugLogging = True):
    """
    Enable or disable debug logging for the current global logger, if any.
    @param UserConfiguration userConfig with info about logging flags.
    @param bool enableDebugLogging Indicates to enable or disable debug logging.
    @return void.
    """
    global LOGGER
    if not LOGGER or not userConfig:
        return

    if not getattr(userConfig, 'DEBUGFLAGS', None):
           setattr(userConfig, 'DEBUGFLAGS', userConfig.get('DEBUGFLAGS', '').upper())

    if LOGGER.name in csvList(userConfig.DEBUGFLAGS) or '*' in csvList(userConfig.DEBUGFLAGS):
        LOGGER.setLevel(IF(enableDebugLogging, logging.DEBUG, logging.INFO))
        LOGGER.debug("Debug logging "+IF(enableDebugLogging,'enabled.','disabled.'))

##############################################################

if __name__ == "__main__":
    print VERSION
