# -*- coding: utf-8 -*-
##############################################################
# smscon_service - remote control daemon for Nokia N900 phone#
# $Id: smscon_service.py 232 2012-07-07 11:51:25Z yablacky $
##############################################################

# standard modules
import os
import sys
import time
import re
import random
import signal
import dbus
from dbus.mainloop.glib import DBusGMainLoop

# for GPS control
import location
import gobject
from operator import itemgetter 

# for email sending
import smtplib
import socket
import urllib
from email.MIMEText import MIMEText
from email.MIMEImage import MIMEImage
from email.MIMEMultipart import MIMEMultipart

import smsconlib as smscon
from smsconlib import InstPath, LogPath, PhotoName, MapName, DefaultAudioFile, AudioFile, DaemonFile, smscon_daemon
from smsconlib import IF, IsHostReachable, IsInternetReachable, IsSamePhoneNumber, IsFileReadable, DeleteFile, \
                        csvList, shellArg, pidof, SendSMSviaGSMmodem

LOGGER = smscon.LOGGER = smscon.StartLogging('DAEMON', withConsole = False)

#############################################
#    constants

PingCountMax                = 3                                 # max number of pings for connection status check
LastKnownGoodConfigFile     = '.%s_lkg' % smscon.NAME
GpsModeLOCATION             = 'location'
GpsModeTRACKING             = 'tracking'
AmbientLightSensorDevice    = '/sys/class/i2c-adapter/i2c-2/2-0029/lux'

#############################################
#    variables

GpsCoordinateList           = None
GpsMode                     = False                             # False if not active or mode-name if GPS device is enabled
GpsIsTest                   = False                             # true if GPS device is enabled from test mode command
GpsStartTime                = 0
GpsTimeoutTime              = 0
GpsLocationBasisTime        = 0
GspTrackingBasisTime        = 0
GpsControl                  = None

ConnectionType              = None 
ConnectionIAP               = None
ConnectionName              = None
ConnectionBehaviour         = None

IsKeyboardSliderCheckActive = False
IsBatteryStatusCheckActive  = False
IsCheckHostActive           = False

LastBatteryStatusMessage    = None

CheckHostLastCommand        = '-1'      # almost COM_RESTART but special "restart"
CheckHostStatusCount        = 0         # positive: successes, negative: errors
CheckHostStatusTimestamp    = time.gmtime()

daemon = None               # Our daemon object
smscnf = None               # the UserConfiguration actually used by the daemon.

IsStolenModeEnabled          = False

#############################################

def GetTimeString(*fromTime):
    """
    @return string Current time in custom format.
    """
    return time.strftime(smscnf.LOG_TIMEFORMAT, *fromTime)

def SafeReadFileLine(Filename, DefaultValue = 'unknown',  ErrorMessage = None):
    """
    Read one line of text with EOL char stripped of. Exception save.
    @return string.
    """
    try:
        return open(Filename).read().strip()
    except Exception, e:
        LOGGER.error(IF(ErrorMessage,  ErrorMessage + ': %s', '%s') % e)
    return DefaultValue

def dbQuote(text, quote="'"):
    if text == None: return 'NULL'
    return quote + text.replace(quote, quote + quote) + quote

def GetPhoneCurrentProfileName(ErrorPrefix = True, TryCount = 2):
    """
    Get name of current active phone profile.
    @param string|bool ErrorPrefix String to be used as prefix in error message. False for silent op.
    @param int TryCount count in case of errors.
    @return None|String The profile name or None on errors.
    """
    E = re.compile('string "(\S+)"')
    Output = 'Try count < 1.'
    try:
        while TryCount > 0:
            TryCount -= 1
            Output = os.popen('dbus-send --session --print-reply --type=method_call \
                          --dest=com.nokia.profiled \
                          /com/nokia/profiled \
                          com.nokia.profiled.get_profile ' + IF(TryCount>0, '', '2>&1')).read()
            CurrentProfile = E.findall(Output)
            if len(CurrentProfile) > 0:
                return CurrentProfile[0]
            time.sleep(1)
    except Exception, e:
        Output = e;
    if ErrorPrefix:
        ErrorPrefix = IF(ErrorPrefix == True, '', ErrorPrefix + ' ')
        LOGGER.error('%sCould not get current phone profile: %s ' % (ErrorPrefix, Output))
    return None

#############################################

def RevealThat(Subject, Text, SendMethod = None, Attachment = None, Attachhint = None):
    """
    Send message to all enabled targets. In case SMS is enabled, the
    message is send to the originating SMS SENDERNUMBER as well as to
    the SMS MASTERNUMBER if set and is a different number.
    The device hostname and a timestamp is added to the message automatically.
    @param string Subject Short description of event.
    @param string|list Text Message body.
    @param string SendMethod Defaults to smscnf.MESSAGESEND
    @param string|none Attachment Name of file to attach (to email only)
    @param string|none Attachhint Hint about the attachment file's MIME type. 
    @return array with brief recipient information. Empty if nothing sent.
    """
    if  SendMethod == None:
        SendMethod = smscnf.MESSAGESEND
        
    if type(Text).__name__ == 'list': Text = ''.join(Text)
    Hostname = os.uname()[1]
    Revealed = []
    if SendMethod in ['both', 'sms']:
        Message = 'N900[%s]\n' '%s\n' '%s:\n' % (Hostname, GetTimeString(), Subject) + Text

        # Split Message into list of lines to give SMSsend hint where to split 
        TextLines = []
        while Message != '':
            Line, Sep, Message = Message.partition('\n');
            TextLines.append(Line + Sep);

        # Send to the last known SMS sender
        if SMSsend(smscnf.SENDERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
            Revealed.append(smscnf.SENDERNUMBER)

        # Send to the master if that is different 
        if not IsSamePhoneNumber(smscnf.MASTERNUMBER, smscnf.SENDERNUMBER):
            if Revealed: 
                TextLines.append('\nBC:%s' % smscnf.SENDERNUMBER)   # "blind copy" information
            if SMSsend(smscnf.MASTERNUMBER, TextLines, 'send SMS message (subject=%s) to {RECIPIENTNUMBER}' % Subject):
                Revealed.append("MASTERNUMBER") # do not expose the encrypted number this way!

    if SendMethod in ['both', 'email']:
        if Revealed:
            Text += '\nSMS copies have been sent to: (' + ', '.join(Revealed) + ')' 

        if EMAILsend(EmailTo     = smscnf.EMAILADDRESS,
                  Subject        = 'NOKIA N900[%s]: %s' % (Hostname, Subject),
                  Text           = 'Nokia phone "%s" at %s:\n%s\n' % (Hostname, GetTimeString(), Text),
                  SuccessLogMsg  = 'Sent email message (subject=%s) to <%s>.' % (Subject, smscnf.EMAILADDRESS),
                  Attachment = Attachment,
                  Attachhint = Attachhint):
            Revealed.append(smscnf.EMAILADDRESS)

    if not Revealed:
        LOGGER.info('Message sending disabled by user setting (%s).' % SendMethod)
    return Revealed

#############################################
#    SMS
#############################################

def OnSmsReceived(PDUmessage, MessageCenter, MessageToken, SenderNumber):
    """
    Receive SMS and try to execute as command.
    @return void
    """
    LOGGER.debug("OnSmsReceived: PDUmessage=%s; MessageCenter=%s; MessageToken=%s;" % (
                                repr(PDUmessage), MessageCenter, MessageToken))
    n = int(PDUmessage[1])
    n = (n + 1) // 2 + 13

    MessageArray = PDUmessage[n:len(PDUmessage)]
    try:
        MessageText = DeOctifyMessage(MessageArray)     # decode SMS message to plain text
        MessageText = MessageText[:int(PDUmessage[n-1])]    # apply length information
    except Exception, e:
        LOGGER.critical("DeOctifyMessage problem: %s", e)
        LOGGER.critical('SMS Message Bytes:' + ', '.join( ["0x%02X" % int(b) for b in MessageArray] ))
        return
    LOGGER.debug('SMS Message Chars:' + ', '.join( ["0x%02X[%s]" % (ord(b), b) for b in MessageText] ))
    LOGGER.debug("SMS received:'%s'" % MessageText)

    timeNow = int(time.time())
    isKnownCommand = False

    if (    MessageText.startswith(smscnf.SMS_COMPREFIX) and
            MessageText.  endswith(smscnf.SMS_COMSUFFIX)  ):
        isKnownCommand = ProcessCommand(MessageText[len(smscnf.SMS_COMPREFIX) :
                                                    len(MessageText)-len(smscnf.SMS_COMSUFFIX)],
                       SenderPhoneNumber = SenderNumber)
    if isKnownCommand:
        try:
            if MessageToken:
                RtComElRemoveSMS(Outgoing = False, MessageToken = MessageToken)
            else:
                RtComElRemoveSMS(Outgoing = False, MessageText = MessageText,
                                 SenderNumber = SenderNumber, MaxAgeUnixtime = timeNow - 120)
        except Exception, e:
            LOGGER.critical('RtComElRemoveSMS: ' % e)

def RtComElRemoveSMS(MessageText = None, SenderNumber = None, MaxAgeUnixtime = None,
                     MessageToken = None, Outgoing = None):
    """
    Remove one or more SMS from RTCOM event log. All SMS that match all given criteria are deleted:
    @param string|None MessageText Text of sms
    @param string|None SenderNumber Sender of sms
    @param int|None MaxAgeUnixtime Max age (when sms has been stored in event log).
    @param string|None MessageToken Token of message (primary critria to delete 1 SMS).
    @param bool|None Outgoing Required direction True=Outgoing, False=Incoming, None=Both.
    @return int Number of deleted messages.
    """
    LOGGER.debug("RtComElRemoveSMS start.")
    rtcom_el_DB = '/home/user/.rtcom-eventlogger/el-v1.db'
    fields = {
            'event_id'  : 'ev.id',
            'group_uid' : 'IFNULL(ev.group_uid,"")',
            'local_uid' : 'IFNULL(re.local_uid,"")',
            'remote_uid': 'IFNULL(ev.remote_uid,"")',
            'abook_uid' : 'IFNULL(re.abook_uid,"")',
            'service'   : 'IFNULL(sv.name,"")',
        }
    fieldNames = sorted(fields.keys())
    fieldSep = "<'>"

    sql = 'SELECT ' + ','.join([fields[k] for k in fieldNames])
    sql += ' FROM (Services sv, EventTypes et, Events ev' + IF(MessageToken, ', Headers hd', '')
    sql += ') LEFT JOIN Remotes re' \
                ' ON  ev.remote_uid=re.remote_uid' \
                ' AND ev.local_uid =re.local_uid' \
        ' WHERE et.name="RTCOM_EL_EVENTTYPE_SMS_MESSAGE" AND et.id=ev.event_type_id' \
          ' AND sv.name="RTCOM_EL_SERVICE_SMS" AND ev.service_id=sv.id'

    if MessageToken:            sql += ' AND hd.event_id=ev.id'\
                                       ' AND hd.name="message-token"'\
                                       ' AND value=' + dbQuote(MessageToken)
    if SenderNumber:            sql += ' AND ev.remote_uid=' + dbQuote(SenderNumber)
    if MessageText  != None:    sql += ' AND ev.free_text='  + dbQuote(MessageText)
    if MaxAgeUnixtime:          sql += ' AND ev.storage_time>' + str(MaxAgeUnixtime)
    if Outgoing != None:        sql += ' AND ' + IF(Outgoing, 'ev.outgoing', 'NOT ev.outgoing')

    sql += ' ORDER BY ev.storage_time DESC'  
    LOGGER.debug("RtComElRemoveSMS sql=%s" % sql)
    try:
        Result = os.popen('sqlite3 -list -separator %s %s %s' % (
                    shellArg(fieldSep), shellArg(rtcom_el_DB), shellArg(sql))).read()
    except:
        LOGGER.error('RtComElRemoveSMS sqlite3 failed to run.')
        return 0
    else:
        LOGGER.debug('RtComElRemoveSMS sqlite3 result: %s' % Result)

    dbus_signal="dbus-send --type=signal --session /rtcomeventlogger/signal rtcomeventlogger.signal"
    countDeleted = 0
    needRefreshHint = False
    for rec in [dict(zip(fieldNames, rec.split(fieldSep))) for rec in Result.split("\n")]:
        # See reference for what we are doing here:
        # http://maemo.gitorious.org/maemo-rtcom/rtcom-eventlogger/blobs/master/src/eventlogger.c
        sql = 'BEGIN;'\
            'INSERT OR REPLACE INTO GroupCache'\
            ' SELECT max(id), service_id, group_uid, count(*), sum(is_read), sum(flags)'\
            ' FROM Events'\
            ' WHERE group_uid=%(group_uid)s AND NOT (id=%(event_id)s)'\
            ' GROUP BY group_uid;'\
            'DELETE FROM Events WHERE (id=%(event_id)s);'\
            'SELECT changes();'\
            'COMMIT; ' % rec
        Result = os.popen('sqlite3 -list -separator %s %s %s' % (
                    shellArg(fieldSep), shellArg(rtcom_el_DB), shellArg(sql))).read()
        LOGGER.debug("RtComElRemoveSMS SMS deleted: %s" % Result)
        countDeleted += Result
        if Result > 1:
            needRefreshHint = True  # bundle all to one.
        elif Result == 1:
            os.system(dbus_signal + '.EventDeleted'\
                    ' int32:%(event_id)s string:"%(local_uid)s"'\
                    ' string:"%(remote_uid)s" string:"%(abook_uid)s"'\
                    ' string:"%(group_uid)s" string:"%(service)s"' % rec)

    if needRefreshHint:
        os.system(dbus_signal + '.RefreshHint int32:-1 string: string: string: string: string:')
    LOGGER.debug('RtComElRemoveSMS done.')
    return countDeleted

def SMSsend(RecipientNumber, MessageList, SuccessLogMsg):
    """
    Send SMS message to RecipientNumber. Handles re-sending on errors.
    @param string RecipientNumber Phone number.
    @param list_of_strings MessageList Message to send. If total text is too long for
                one sms, the message is split into multiple sms thereby trying not
                to split single list entries.
    @param string SuccessLogMsg Message for LOGGER in case of successful sending.
                A placeholder '{RECIPIENTNUMBER}' in that message is replaced
                by the actual phone number.
    @return bool False if the message has definitely not been sent. True if sent or queued for sending.
    """
    if not RecipientNumber:
        return False
    MessageLengthMax = 160  # max. number of characters in one SMS message
    MessageText = ''.join(MessageList)

    if len(MessageText) <= MessageLengthMax:
        if SendToGSMmodem(RecipientNumber, MessageText, SuccessLogMsg):
            return True
        if smscnf.ENABLERESEND != 'yes':
            return False
        LOGGER.warning('Trying to re-send SMS message after %s minutes.' % float(smscnf.RESENDTIME))
        tryCount = { 'value': 0 }
        gobject.timeout_add(max(1000, int(float(smscnf.RESENDTIME) * 60000)),
                            lambda args: not SendToGSMmodem(**args), dict(
                                RecipientNumber   = RecipientNumber,
                                Message           = MessageText,
                                SuccessLogMsg     = SuccessLogMsg,
                                TryCount          = tryCount,
                            ))
        return True

    # Message must be split
    if 0:
        # Fast and simple split but may require less SMS than sophisticated split:
        part = 0
        while len(MessageText) > 0:
            part += 1
            MessageText = ('-#%s-\n' % part) + MessageText
            SMSsend(RecipientNumber, [MessageText[0:MessageLengthMax]], SuccessLogMsg + (' [part %d]' % part))
            MessageText = MessageText[MessageLengthMax:]
        return True

    # Sophisticated split:
    # Calculates n of m part header and tries not to split within messages.
    PartHeader = lambda n,m: '#%d/%d\n' % (n,m)
    def SplitMessages(MessageList, MessageLengthMax, AssumeTotalPartCount):
        """
        Join the messages in the given list and split that to parts of MessageLengthMax.
        If possible do not split within a single message.
        @return list of messages each of which is no longer than MessageLengthMax chars.
        """
        SplittedMessages = []
        MessageParts = []
        MessageList = MessageList[:]
        while MessageList:
            Msg = MessageList.pop(0)
            if len(Msg) == 0:
                continue
            if len(MessageParts) == 0:
                MessageParts = [ PartHeader(len(SplittedMessages)+1, AssumeTotalPartCount) ]
                if len(MessageParts[0]) >= MessageLengthMax:
                    MessageParts[0] = '' # no header if header alone exceeds max length.
            if sum(map(len,MessageParts)) + len(Msg) <= MessageLengthMax:
                MessageParts.append(Msg)
                continue
            lenNow = 0
            if len(MessageParts) == 1:  # No change, must split within message.
                lenNow = MessageLengthMax - len(MessageParts[0])
                MessageParts.append(Msg[0:lenNow])
            MessageList = [Msg[lenNow:]] + MessageList     # unshift
            SplittedMessages.append(''.join(MessageParts))
            MessageParts = []
        if len(MessageParts):
            SplittedMessages.append(''.join(MessageParts))
        assert(max(map(len, SplittedMessages)) <= MessageLengthMax)
        return SplittedMessages

    # Since each message part has a header that contains the part number and the total number of
    # parts, splitting is a little complicated because the total number is not known in advance: 
    assumeTotalParts, totalParts = 0, 1
    while totalParts > assumeTotalParts:
        assumeTotalParts = (assumeTotalParts + 1) * 10 - 1    # 9, 99, 999, ...
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, assumeTotalParts)
        totalParts = len(SplittedMessages)
    if totalParts < assumeTotalParts:
        SplittedMessages = SplitMessages(MessageList, MessageLengthMax, totalParts)
        while len(SplittedMessages) < totalParts:
            SplittedMessages.append(PartHeader(len(SplittedMessages)+1, totalParts))
            
    LOGGER.info('SMS message is too long (%s chars); splitting message into %d parts.' % (len(MessageText), totalParts))

    if int(smscnf.SMS_PARTS_LIMIT) > 0 and totalParts > int(smscnf.SMS_PARTS_LIMIT):
        skipCount = totalParts - int(smscnf.SMS_PARTS_LIMIT)
        skipPart = 0
        while skipPart < skipCount:
            SplittedMessages[1+skipPart] = '' # at least 1 part is allowed, leave part 0 ("header info") unskipped
            skipPart += 1
        lenAvail = MessageLengthMax - len(SplittedMessages[0])
        if lenAvail > 0:
            SplittedMessages[0] += ("[%d/%d cut]" % (skipCount, totalParts))[:lenAvail]

    for part, msg in enumerate(SplittedMessages):
        if len(msg):
            SMSsend(RecipientNumber, msg, SuccessLogMsg + ' [part %d/%d, %d chars]' % (part+1, totalParts, len(msg)))
        else:
            LOGGER.warning(SuccessLogMsg + ' [part %d/%d, %d chars SKIPPED DUE TO SMS LIMIT]' % (part+1, totalParts, len(msg)))
        
    return True

def SendToGSMmodem(RecipientNumber, Message, SuccessLogMsg, TryCount = None):
    """
    Send SMS message to GSM modem.
    @return bool Success indicator
    """
    if TryCount:
        TryCount['value'] += 1

    if SendSMSviaGSMmodem(RecipientNumber, Message):
        LOGGER.info(SuccessLogMsg.replace('{RECIPIENTNUMBER}', RecipientNumber))
        return True

    RetryInfo = ''
    if TryCount:
        RetryInfo = " (%d. retry out of %d)" % (TryCount['value'], int(smscnf.RESENDMAXTRY))
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] >= int(smscnf.RESENDMAXTRY):
            LOGGER.error('Failed to send SMS to "%s"%s. Giving up.' % (RecipientNumber, RetryInfo))
            return True     # fake success to stop further timer calls.

    LOGGER.error('Failed to send SMS to "%s"%s.' % (RecipientNumber, RetryInfo))
    return False

def DeOctifyMessage(inputBytes):
    """
    Decode inputBytes from SMS message according to GSM 03.38, GSM 7 bit default alphabet.
    @param list inputBytes Bytes containing packed 7 bit chars (from "PDUMessage").
    @return string.
    """
    LOGGER.debug("DeOctifyMessage: have inputBytes: %s" % repr(inputBytes))

    outputCodes = []
    bitCount, lastByte = 7, 0
    for thisByte in inputBytes:
        outputCodes.append(0x7f & ((thisByte << (7 - bitCount)) | lastByte))
        lastByte = thisByte >> bitCount
        bitCount -= 1   # from the next byte, we need 1 bit less.
        if bitCount == 0:
            outputCodes.append(lastByte)    
            bitCount, lastByte = 7, 0

    # Note: for code pages below it is important that all code points evaluate to a unicode string (u-prefix):
    ESC = u'\x1b'
    GSM_03_38_codepage_std = (
        u'@' , u'£' , u'$' , u'¥' , u'è' , u'é' , u'ù' , u'ì' , u'ò' , u'Ç' , u'\n', u'Ø' , u'ø' , u'\r', u'Å' , u'å' ,
        u'Δ' , u'_' , u'Φ' , u'Γ' , u'Λ' , u'Ω' , u'Π' , u'Ψ' , u'Σ' , u'Θ' , u'Ξ' ,  ESC , u'Æ' , u'æ' , u'ß' , u'É' ,
        u' ' , u'!' , u'"' , u'#' , u'¤' , u'%' , u'&' , u'\'', u'(' , u')' , u'*' , u'+' , u',' , u'-' , u'.' , u'/' ,
        u'0' , u'1' , u'2' , u'3' , u'4' , u'5' , u'6' , u'7' , u'8' , u'9' , u':' , u';' , u'<' , u'=' , u'>' , u'?' ,
        u'¡' , u'A' , u'B' , u'C' , u'D' , u'E' , u'F' , u'G' , u'H' , u'I' , u'J' , u'K' , u'L' , u'M' , u'N' , u'O' ,
        u'P' , u'Q' , u'R' , u'S' , u'T' , u'U' , u'V' , u'W' , u'X' , u'Y' , u'Z' , u'Ä' , u'Ö' , u'Ñ' , u'Ü' , u'§' ,
        u'¿' , u'a' , u'b' , u'c' , u'd' , u'e' , u'f' , u'g' , u'h' , u'i' , u'j' , u'k' , u'l' , u'm' , u'n' , u'o' ,
        u'p' , u'q' , u'r' , u's' , u't' , u'u' , u'v' , u'w' , u'x' , u'y' , u'z' , u'ä' , u'ö' , u'ñ' , u'ü' , u'à' ,
    )

    GSM_03_38_codepage_esc = {
        0x0a: u'\f' , 
        0x14: u'^'  ,   0x1b: ESC   ,
        0x28: u'{'  ,   0x29: u'}'  ,   0x2f: u'\\' ,
        0x3c: u'['  ,   0x3d: u'~'  ,   0x3e: u']'  ,
        0x40: u'|'  ,
        0x65: u'€'  ,
    }

    outputText = []
    while outputCodes:
        code = outputCodes.pop(0)
        if code != ord(ESC) or not outputCodes:
            outputText.append(GSM_03_38_codepage_std[code])
        else:
            code = outputCodes.pop(0)
            if code in GSM_03_38_codepage_esc:
                outputText.append(GSM_03_38_codepage_esc[code])
            else:
                outputText.extend((ESC, GSM_03_38_codepage_std[code]))
    LOGGER.debug("DeOctifyMessage: have outputText: %s" % repr(outputText))
    return ''.join(outputText)

#############################################
#   send email
#############################################

def EMAILsend(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
    """
    Send email; retry after <RESENDTIME> minutes if send failure.
    @return bool False if the message has definitely not been sent. True otherwise.
    """
    if SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, **kwargs):
        return True
    if smscnf.ENABLERESEND != 'yes' or not EmailTo:
        return False
    LOGGER.warning('Trying to re-send email message after %s minutes.' % float(smscnf.RESENDTIME))
    tryCount = { 'value': 0 }
    gobject.timeout_add(max(1000, int(float(smscnf.RESENDTIME) * 60000)),
                        lambda args: not SendToSMTPserver(**args), dict(
                            EmailTo       = EmailTo,
                            Subject       = Subject,
                            Text          = Text,
                            SuccessLogMsg = SuccessLogMsg,
                            TryCount      = tryCount,
                            **kwargs
                        ))
    return True

def SendToSMTPserver(EmailTo, Subject, Text, SuccessLogMsg, Attachment = None, Attachhint = None, TryCount = None):
    """
    Send email to SMTP server.
    @return bool Success indicator
    """
    RetryInfo = ''
    if TryCount:
        TryCount['value'] += 1
        if int(smscnf.RESENDMAXTRY) > 0 and TryCount['value'] > int(smscnf.RESENDMAXTRY):
            LOGGER.error('(EMAIL) Tried to send %d times. Limit %d reached. Giving up recipient=%s, subject=%s.' %
                            (TryCount['value'], int(smscnf.RESENDMAXTRY), EmailTo, Subject))
            return True
        RetryInfo = " (%d. retry)" % TryCount['value']

    if not EmailTo:
        LOGGER.error('(EMAIL) No recipient email address%s.' % RetryInfo)
        return False
    # convert string list to string
    if type(Text).__name__ == 'list': Text = ''.join(Text)

    # poor convertion of text to html text:
    MailBodyHtml = Text.replace('\n', '<br>')
    Hostname     = os.uname()[1]
    Message      = MIMEMultipart()

    if Attachment:
        Attachhint = Attachhint or 'text'
        ConvertToMIME         = MIMEText
        AttachmentCID         = None

        if Attachhint.startswith('image'):
            Message               = MIMEMultipart('related') 
            ConvertToMIME         = MIMEImage
            AltText               = '%s image' % os.path.basename(Attachment)
            AttachmentCID         = 'image%s' % random.randint(1, 100000)
            MailBodyHtml += '<br><img src="cid:%s" alt="%s" /><br>' % (AttachmentCID, AltText)

    # For historical reasons, EMAILFROM contains the technical sender if EMAILSENDER is empty
    if smscnf.EMAILSENDER:
        TechnicalSender, LogicalSender = smscnf.EMAILSENDER, smscnf.EMAILFROM
    else:
        TechnicalSender, LogicalSender = smscnf.EMAILFROM, None

    Message['Subject'] = Subject
    Message['To']      = EmailTo
    Message['From']    = LogicalSender or 'NokiaN900.SMSCON.%s <SMSCON@%s>' % (Hostname, Hostname)

    LOGGER.debug('(EMAIL) LogicalSender=%s, TechnicalSender=%s, Recipient=%s, Subject=%s' % (LogicalSender, TechnicalSender, EmailTo, Subject))  

    MailBodyHtml = '<html><head></head><body>\n'\
                    '<p>' + MailBodyHtml + '</p>\n'\
                    '</body></html>'
    Message.attach( MIMEText(MailBodyHtml, 'html') )

    if Attachment:
        try:
            File = open(Attachment, 'rb')
        except:
            LOGGER.error('(EMAIL) Attachment file "%s" not found.' % Attachment)
            AttachmentData = MIMEText('Attachment file "%s" not found' % Attachment)
            AttachmentCID = None
        else:
            try:
                AttachmentData = ConvertToMIME( File.read() )
            except:
                LOGGER.error('(EMAIL) Attachment file "%s" could not be read.' % Attachment)
                File.close()
                AttachmentData = MIMEText('Attachment file "%s" could not be read' % Attachment)
                AttachmentCID = None
            else:
                File.close()
        if AttachmentCID:
            AttachmentData.add_header('Content-ID', '<%s>' % AttachmentCID )
        else:
            AttachmentData.add_header('Content-Disposition', 'attachment; filename="%s"' % os.path.basename(Attachment))    
        Message.attach(AttachmentData)

    try:
        LOGGER.info('(EMAIL) Connecting mail server "%s:%s".' % (smscnf.MAILSERVER, smscnf.MAILPORT))
        # connect to mail server
        Server = smtplib.SMTP(smscnf.MAILSERVER, smscnf.MAILPORT) 
        #Server.set_debuglevel(True)
    except (socket.error, smtplib.SMTPConnectError, smtplib.SMTPException):
        LOGGER.error('(EMAIL) Failed to connect mail server%s.' % RetryInfo)
        return False

    # identify to mail server
    Server.ehlo() 
        
    # if smtp server requires secure authentication
    if Server.has_extn('STARTTLS'):
        Server.starttls()
        Server.ehlo()

    # if smtp server requires user/password authentication
    if smscnf.USER != '' and smscnf.PASSWORD != '':
        try:
            LOGGER.debug('(EMAIL) Login at mail server "%s" as "%s", PW:%s...' % (smscnf.MAILSERVER, smscnf.USER, IF(smscnf.PASSWORD,'yes','no')))
            Server.login(smscnf.USER, smscnf.PASSWORD) 
        except smtplib.SMTPAuthenticationError:
            LOGGER.error('(EMAIL) Mail not sent. Wrong username/password%s.' % RetryInfo)
            Server.quit()
            return False

    try:
        LOGGER.debug('(EMAIL) Transferring email...')
        Server.sendmail(TechnicalSender, EmailTo, Message.as_string()) 
    except smtplib.SMTPException, e:
        LOGGER.error('(EMAIL) Transfer failed: %s' % e)
        Server.quit()
        LOGGER.error('(EMAIL) Mail not sent%s.' % RetryInfo)
        return False

    Server.quit()
    LOGGER.info('(EMAIL) %s' % (SuccessLogMsg or 'Mail sent.'))
    return True

#############################################
#   save new number
#############################################

def SaveSenderPhoneNumber(PhoneNumber):
    """
    Save new SENDERNUMBER as smscnf.SENDERNUMBER in "smscon_config" file.
    """ 
    if smscnf.SENDERNUMBER == PhoneNumber:
        return

    if smscnf.update('SENDERNUMBER', PhoneNumber):
        LOGGER.info( 'New SMS SENDERNUMBER (%s) saved.' % PhoneNumber)
    else:
        LOGGER.error("Failed to save new SMS SENDERNUMBER (%s)." % PhoneNumber)

    # update smscnf.SENDERNUMBER in any case! We use it to send replies. 
    smscnf.SENDERNUMBER = PhoneNumber

#############################################
#    command processing
#############################################

def ProcessCommand(Command, SenderPhoneNumber=None, CommandOrigin=None, ExtraInfo='', LogIfCmdUnknown=False):
    """
    Process the received command message.
    @return bool True iff the command was a known command.
    """
    # Check if Command is a specific valid SMSCON command:
    if not (Command in [smscnf[cmd] for cmd in smscnf.getSmsCommandList() if smscnf[cmd] != '']
            or smscnf.parseSmsCommand(Command, smscnf.COM_CUSTOM)
            or smscnf.parseSmsCommand(Command, smscnf.COM_SHELL)
            or smscnf.parseSmsCommand(Command, smscnf.COM_CAMERA)):
        if LogIfCmdUnknown:
            LOGGER.error( '"%s" is not a valid %s command.' % (Command, smscon.NAME) )
            LogBatteryChargeInfo()
        return False

    if SenderPhoneNumber:
        SaveSenderPhoneNumber(SenderPhoneNumber)
        CommandOrigin = CommandOrigin or "SMS"
        CommandOriginDetail = '%s from %s (%s)' % (CommandOrigin, SenderPhoneNumber,
                IF(IsSamePhoneNumber(SenderPhoneNumber, smscnf.MASTERNUMBER), 'the master', 'NOT master'))
    else:
        CommandOriginDetail = CommandOrigin = CommandOrigin or "UNKNOWN"

    if ExtraInfo != '':
        ExtraInfo = ' [%s]' % ExtraInfo.strip()        
    LOGGER.info('Received command "%s" via %s%s. Acknowledge message %s.' % (Command, CommandOriginDetail, ExtraInfo,
                    IF(smscnf.COMMANDREPLY == 'yes', 'enabled', 'disabled')))
    LogBatteryChargeInfo()
    CommandArgs = {}

    # First of all, general command reply
    if smscnf.COMMANDREPLY == 'yes':
        RevealThat('reply', 'Command (%s) accepted.' % Command)

    # Make phone silent
    if smscnf.SILENCEDEVICE == 'yes' or Command == smscnf.COM_SILENTON:
        EnableDeviceSilence()

    # Explicit device locking
    if Command == smscnf.COM_UNLOCK:
        PerformDeviceUnlock()
        return True

    if Command == smscnf.COM_LOCK:
        PerformDeviceLock()
        return True

    # Automatic device locking
    if smscnf.AUTODEVICELOCK == 'yes':
        PerformDeviceLock()

    # Dispatch on other commands
    if Command == smscnf.COM_CHECK:
        ok = False
        if smscnf.MESSAGESEND in ['sms', 'both']:
            ok = ShowSMScommands('sms')
        if smscnf.MESSAGESEND in ['email', 'both']:
            ok = ShowSMScommands('email')
        if not ok:
            LOGGER.warning('No notification sent. Message sending is disabled by user setting: %s.' % smscnf.MESSAGESEND)

    elif Command == smscnf.COM_STOLEN:
        global IsStolenModeEnabled
        LogMessage = IF(IsStolenModeEnabled, 'Stolen mode was already enabled.', 'Stolen mode enabled.')
        IsStolenModeEnabled = True
        LOGGER.warning(LogMessage)
        # Prepare for permanent stolen mode:  
        simDB = smscon.IMSIConfiguration()
        simDB.load(Silent = True)
        knownSIMsCount = len(simDB.keys())
        if not simDB.create(Force = True, Load = False):
            InfoMessage = "Failed to revoke authorization of all SIM cards (%d remain)." % knownSIMsCount
        elif knownSIMsCount:
            InfoMessage = "Authorization of all SIM cards (%d) revoked." % knownSIMsCount
        else:
            InfoMessage = "Authorization of all SIM cards already revoked."
        LOGGER.warning(InfoMessage)
        RevealThat('Stolen', '\n'.join([LogMessage, InfoMessage]))
        del simDB

    elif Command == smscnf.COM_REBOOT:
        PerformReboot()

    elif Command == smscnf.COM_POWEROFF:
        PerformPoweroff()            

    elif Command == smscnf.COM_POWER:
        PerformStatusReport()

    elif Command == smscnf.COM_LOCATION:
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        EnableGpsAcquireMode(GpsModeLOCATION)

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_REMOTEON:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for enabling SSH connection.')
            RevealThat('ssh', 'No user/password entered for enabling SSH connection.')
        else:
            if not IsInternetReachable():
                SetNetworkConnectionType('GPRS') # enable GPRS data network
                SetAutoConnectBehaviour('GPRS') # disable switching to other network
            EnableSSHcontrol()
            
    elif Command == smscnf.COM_REMOTEOFF:
        if smscnf.REMOTEUSER == '' and smscnf.REMOTEPASSWORD == '':
            LOGGER.warning('No user/password entered for disabling SSH connection.')
            RevealThat('ssh', 'No user/password entered for disabling SSH connection.')
        else:
            DisableSSHcontrol()
            
    elif smscnf.parseSmsCommand(Command, smscnf.COM_CAMERA, CommandArgs):
        os.putenv('SMSCON_COMMANDORIGIN', CommandOrigin)
        os.putenv('SMSCON_SENDERNUMBER', smscnf.SENDERNUMBER)
        os.putenv('SMSCON_MASTERNUMBER', smscnf.MASTERNUMBER)
        behaviourChanged = False
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            behaviourChanged = SetAutoConnectBehaviour('GPRS') # disable switching to other network

        PerformCamera(CommandArgs['args'])

        if behaviourChanged:
            SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_CALL:
        PerformPhonecall(SenderPhoneNumber or smscnf.MASTERNUMBER)

    elif Command == smscnf.COM_TRACKON:
        if not IsInternetReachable():
            SetNetworkConnectionType('GPRS') # enable GPRS data network
            SetAutoConnectBehaviour('GPRS') # disable switching to other network
        EnableGpsAcquireMode(GpsModeTRACKING)

    elif Command == smscnf.COM_TRACKOFF:
        DisableGpsAcquiring()
        SetAutoConnectBehaviour(ConnectionBehaviour)

    elif Command == smscnf.COM_SILENTON:
        pass  # already done above.
    
    elif Command == smscnf.COM_SILENTOFF:
        RestoreDeviceSilence()

    elif Command == smscnf.COM_CUSTOMLOG:
        PerformShellScriptResult()

    elif Command == smscnf.COM_LOG:
        RevealThat('log', 'The %s log file' % smscon.NAME, Attachment = LOGGER.LogFilename, Attachhint = 'text/plain')

    elif Command == smscnf.COM_ALARM:
        PerformAlarm()

    elif Command == smscnf.COM_RESTART:
        RestartDaemon()

    elif smscnf.parseSmsCommand(Command, smscnf.COM_CUSTOM, CommandArgs):
        os.putenv('SMSCON_COMMANDORIGIN', CommandOrigin)
        os.putenv('SMSCON_SENDERNUMBER', smscnf.SENDERNUMBER)
        os.putenv('SMSCON_MASTERNUMBER', smscnf.MASTERNUMBER)
        PerformShellScript(CommandArgs['args'])

    elif smscnf.parseSmsCommand(Command, smscnf.COM_SHELL):
        os.putenv('SMSCON_COMMANDORIGIN', CommandOrigin)
        os.putenv('SMSCON_SENDERNUMBER', smscnf.SENDERNUMBER)
        os.putenv('SMSCON_MASTERNUMBER', smscnf.MASTERNUMBER)
        PerformDirectShellCommand(Command)

    else:
        LOGGER.debug('Ignoring unknown command: [%s].' % Command)
        return False

    return True

#############################################
#   play sound file as an alarm
#############################################

def PerformAlarm():
    """
    Make phone play sound file for alarm.
    """

    AudioFilename = os.path.join(InstPath, AudioFile)
    if not IsFileReadable(AudioFilename):
        DefaultAudioFilename = os.path.join(InstPath, DefaultAudioFile)
        if not IsFileReadable(DefaultAudioFilename):
            LOGGER.error('(ALARM) Sound does not exist or is not readable: %s' % AudioFilename)
            RevealThat('alarm', 'Sound does not exist or is not readable.')
            return
        AudioFilename = DefaultAudioFilename

    PrePlayCommand = []
    PostPlayCommand = []
    
    # Before dealing with phone profiles and playing our alarm sound,
    # wait a little until the sounds are done that the phone will
    # play on receiving the sms message (the COM_ALARM message).... 
    PrePlayCommand.append("sleep 10")

    CurrentProfile = GetPhoneCurrentProfileName(ErrorPrefix = '(ALARM)')
    if CurrentProfile == None:
        CurrentProfile = 'general'

    SetPhoneProfileCommand = 'dbus-send --type=method_call' \
                              ' --dest=com.nokia.profiled' \
                              ' /com/nokia/profiled' \
                              ' com.nokia.profiled.set_profile string:%s'

    if CurrentProfile != 'general':
        LOGGER.info('(ALARM) Temporarily switching phone profile from "%s" to "General".' % CurrentProfile)
        PrePlayCommand .append(SetPhoneProfileCommand % shellArg('general'))
        PostPlayCommand.append(SetPhoneProfileCommand % shellArg(CurrentProfile))

    # set volume to maximum
    Volume = 100
    PrePlayCommand.append('dbus-send --type=method_call' \
               ' --dest=com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer' \
               ' /com/nokia/mafw/renderer/gstrenderer' \
               ' com.nokia.mafw.extension.set_extension_property' \
               ' string:volume variant:uint32:%s' % shellArg(Volume))
    
    # play audio
    ok = PlayAudio(AudioFilename,
                PrePlayCommand  = ';'.join(PrePlayCommand),
                PostPlayCommand = ';'.join(PostPlayCommand))
    RevealThat('alarm', IF(ok, 'Alarm sound activated.', 'Alarm sound activation failed.'))        

def PlayAudio(AudioFilename, PrePlayCommand = None, PostPlayCommand = None):
    """
    Play audio file in background.
    @return bool Success indicator.
    """ 
    PrePlayCommand  = PrePlayCommand  or 'true'
    PostPlayCommand = PostPlayCommand or 'true'
    try:
        os.system( '(%s; /usr/bin/gst-launch filesrc location=%s ! decodebin2 ! autoaudiosink; %s)&' % (
            PrePlayCommand, shellArg(AudioFilename), PostPlayCommand) )
    except Exception, e:
        LOGGER.error('(ALARM) Failed to play sound: %s' % e)
        return False
    LOGGER.info('(ALARM) Sound is playing now.')
    return True

#############################################
#   silence phone
#############################################

def SetPhoneProfileAudibility(Profile, NoisyItems, OldValues = None):
    """
    Modify volume values of a phone profile.
    @param string Profile Name of profile to modify.
    @param dict NoisyItems Which items to set and their volume value.
    @param dict|None The old volume values are returned here.
    @return bool Success indicator.
    """
    E = re.compile('string "(\S+)"')
    Errors = 0

    for Item, Volume in NoisyItems.iteritems():
        # Query old volume values for restore:
        if type(OldValues).__name__ == 'dict':        
            Result = os.popen('dbus-send --session --print-reply --type=method_call \
                        --dest=com.nokia.profiled \
                        /com/nokia/profiled \
                        com.nokia.profiled.get_value string:"%s" string:"%s" 2>&1' % (Profile, Item)
                    ).read()
            try:
                OldValues[Item] = int(E.findall(Result)[0])
                # Skip setting new value if old and new is same:
                if str(Volume) == str(OldValues[Item]):
                    continue
            except:
                if Item in OldValues: del OldValues[Item]
                LOGGER.error('(SILENCE) Could not get profile "%s"\'s %s: %s' % (Profile, Item, Result))
                Errors = Errors + 1
        # Set new volume values:
        try:
            Result = os.popen('dbus-send --session --print-reply --type=method_call \
                        --dest=com.nokia.profiled \
                        /com/nokia/profiled \
                        com.nokia.profiled.set_value string:"%s" string:"%s" string:"%s" 2>&1' %
                            (Profile, Item, Volume)
                    ).read()
            LOGGER.debug('(SILENCE) Modify profile "%s": %s = %s: %s' % (Profile, Item, Volume, Result))
            if 'boolean true' in Result:
                continue
        except Exception, e:
            Result = e
        LOGGER.error('(SILENCE) Could not modify profile "%s": %s = %s failed: %s' % (Profile, Item, Volume, Result))
        Errors = Errors + 1
    return Errors == 0

def EnableDeviceSilence():
    """
    Make the phone silent for email, instant-messaging, phone ringing & SMS.
    Save previous volume values if they are non-silent for restore. 
    """    

    NoisyItems = {
       'email.alert.volume'     : 0,
       'im.alert.volume'        : 0,
       'ringing.alert.volume'   : 0,
       'sms.alert.volume'       : 0,
    }

    CurrentProfile = GetPhoneCurrentProfileName(ErrorPrefix = '(SILENCE)')
    if CurrentProfile == None:
        return
    LOGGER.debug('(SILENCE) Current profile is "%s".' % CurrentProfile)

    # set items in current profile
    OldNoise = {}
    SetPhoneProfileAudibility(CurrentProfile, NoisyItems, OldNoise)
    
    if [1 for Item in OldNoise if OldNoise[Item] > 0]:
        LOGGER.debug('(SILENCE) storing SAVEDAUDIBILITY.')
        OldNoise['profile'] = CurrentProfile
        smscnf.update('SAVEDAUDIBILITY', repr(OldNoise).strip('{}').replace("'",'"').replace("\n"," "))
    else:
        LOGGER.debug('(SILENCE) not storing SAVEDAUDIBILITY.')

    LOGGER.info('(SILENCE) Phone is silent in current profile "%s".' % CurrentProfile)
    RevealThat('silence', 'Phone is silent in current profile "%s".' % CurrentProfile)

def RestoreDeviceSilence():
    """
    Use saved information to restore volume values for alerting 
    email, instant-messaging, phone ringing & SMS. Restores values in
    the profile that was active when information was saved. This may
    be different from current profile which is not changed in this case.
    """
    CurrentProfile = SavedProfile = GetPhoneCurrentProfileName(ErrorPrefix = '(SILENCE)')
    LOGGER.debug('(SILENCE) Current profile is "%s".' % CurrentProfile)
    try:
        OldNoise = eval('{' + smscnf.SAVEDAUDIBILITY + '}')
        if 'profile' in OldNoise:
            SavedProfile = OldNoise['profile']
            del OldNoise['profile']
        
        if not OldNoise:
            raise Exception

        if not [1 for Item in OldNoise if int(OldNoise[Item]) > 0]:
            LOGGER.error('(SILENCE) Phone audibility not restored. Would be silent.')
            RevealThat('silence',  'Phone audibility not restored. Would be silent.')
            return    

        if SavedProfile == None: 
            LOGGER.error('(SILENCE) Phone audibility not restored. No profile name.')
            RevealThat('silence',  'Phone audibility not restored. No profile name.')
            return
    
        if not SetPhoneProfileAudibility(SavedProfile, OldNoise):
            LOGGER.error('(SILENCE) Phone audibility not restored.') # errors already logged.
            RevealThat('silence',  'Phone audibility not restored due to errors.')
            return
    except:
        LOGGER.error('(SILENCE) Phone audibility not restored. Was not saved or corrupt.')
        RevealThat('silence',  'Phone audibility not restored. Was not saved or corrupt.')
        return

    ProfileInfo = IF(CurrentProfile == SavedProfile, 'current', 'inactive')
    LOGGER.info('(SILENCE) Phone audibility restored for %s profile "%s".' % (ProfileInfo, SavedProfile))
    RevealThat('silence', 'Phone audibility restored for %s profile "%s".' % (ProfileInfo, SavedProfile))

#############################################
#   run custom scripts
#############################################

def PerformShellScript(args = []):
    """
    Execute user editable script.
    @param array args Parameters to pass to the script.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    # Delete old log (may be retrieved before by COM_CUSTOMLOG command)
    if pidof(Filename) == '':
        DeleteFile(Logname)

    if not IsFileReadable(Filename):
        LOGGER.error(        'script file (%s) not found.' % Filename)
        RevealThat('script', 'script file (%s) not found.' % Filename)        
        DeleteFile(Logname)
        return;
    # run script file
    try:
        os.system( '"%s" %s 1>>"%s" 2>&1&' % (Filename, ' '.join([shellArg(a) for a in args]), Logname) )
        PIDs = pidof(Filename)
        LOGGER.info(         'Script %s executes in BG [PIDs=%s].' % (str(args), PIDs))
        RevealThat('script', 'Script %s executes in BG [PIDs=%s].' % (str(args), PIDs))
    except Exception, e:
        LOGGER.error('Failed to execute script %s: %s' % (str(args), e))
        RevealThat('shell', ['Failed to execute script %s.' % str(args)])        

def PerformShellScriptResult():
    """
    Send back logfile of user editable script execution.
    @return void
    """
    Filename = os.path.join(InstPath, smscon.ScriptFile)
    Logname  = os.path.join(InstPath, smscon.ScriptLog)
    PIDs = pidof(Filename)
    ScriptRunInfo = IF(PIDs, 'Scripts are running [PIDs=%s].' % PIDs, 'No scripts running.')
    if not IsFileReadable(Logname):
        RevealThat('script.log', 'Log file %s not present. %s' % (Logname, ScriptRunInfo))
    else:
        RevealThat('script.log', 'Log file %s is attached. %s' % (Logname, ScriptRunInfo),
                   Attachment = Logname, Attachhint = 'text/plain')

def PerformDirectShellCommand(Command):
    """
    Run direct shell command and reply with notification message containing the command's output.
    @return void.
    """
    try:
        Result = os.popen(Command).read()
    except:
        LOGGER.error('Failed to run shell command (%s).' % Command)
        RevealThat('shell', ['Command failed: %s.' % Command])        
    else:
        LOGGER.info('Shell command (%s) executed.' % Command)
        RevealThat('shell', ['Command: %s\n' % Command, 'Result:\n', Result])        

#############################################
#   show SMS commands
#############################################

def ShowSMScommands(SendMethod):
    """
    Log or SMS list of SMS commands.
    @return: bool Success status.
    """
    CommandVars = sorted(smscnf.getSmsCommandList())
    if SendMethod == 'sms':
        Format = '%s=%s\n'
        skip = len(smscnf.SmsCommandPreFix)  # Skip the command prefix to keep SMS message as short as possible
    else:
        Format = '%' + str(-smscnf.TabSize) + 's = %s\n'
        skip = 0
    Lines = []
    Lines.append(IF(len(smscnf.SMS_COMPREFIX), 'Prefix:%s\n' % smscnf.SMS_COMPREFIX, 'No prefix.\n'))
    Lines.append(IF(len(smscnf.SMS_COMSUFFIX), 'Suffix:%s\n' % smscnf.SMS_COMSUFFIX, 'No suffix.\n'))
    Lines.extend([Format % (CmdVar[skip:], smscnf[CmdVar]) for CmdVar in CommandVars])
    if SendMethod == 'log':
        for Line in Lines: 
            LOGGER.info(Line)
    else:
        RevealThat('commands', Lines, SendMethod = SendMethod)
    return True

#############################################
#   device lock
#############################################

def __DoDeviceLock():
    """
    Silently lock the phone.
    @return string Message.
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_open \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"3"')
    return 'Phone has been locked.'

def __DoDeviceUnlock():
    """
    Silently unlock the phone.
    @return string Message.
    """
    os.system ('run-standalone.sh \
               dbus-send --system --type=method_call \
               --dest=com.nokia.system_ui \
               /com/nokia/system_ui/request \
               com.nokia.system_ui.request.devlock_close \
               string:"com.nokia.mce" \
               string:"/com/nokia/mce/request" \
               string:"com.nokia.mce.request" \
               string:"devlock_callback" \
               uint32:"0"')
    return 'Phone has been unlocked.'

def PerformDeviceLock(Reason = ''):
    """
    Perform full phone locking operation.
    @param string Reason.
    @return void.
    """
    Msg = __DoDeviceLock() + Reason
    LOGGER.info(Msg)
    RevealThat('locked', Msg)
        
def PerformDeviceUnlock(Reason = ''):
    """
    Perform full phone unlocking operation.
    @param string Reason.
    @return void.
    """
    Msg = __DoDeviceUnlock() + Reason
    LOGGER.info(Msg)
    RevealThat('unlock', Msg)

#############################################
#   system
#############################################

def PerformStatusReport():
    """
    Gather and reveal all cheap phone status information.
    """
    # Assemble short battery status text 

    (BatteryState, chargePercent, isCharging) = GetBatteryChargeInfo(TextOnly = False)
    BatteryState = '%s%%' % IF(chargePercent == None, '?', chargePercent)
    if isCharging != None:
        BatteryState = BatteryState + IF(isCharging, '++', '--')

    # Query device lock mode

    Output = os.popen('dbus-send --system --type=method_call --print-reply \
                    --dest="com.nokia.mce" "/com/nokia/mce/request" \
                    com.nokia.mce.request.get_devicelock_mode').read()
    E = re.compile('string "(\S+)"')
    try:
        DeviceLockState = E.findall(Output)[0]
    except Exception, e:
        DeviceLockState = 'unknown'
        LOGGER.warning('Could not determine device lock state: %s' % e)

    # Query screen & keyboard lock mode

    Output = os.popen('dbus-send --system --type=method_call --print-reply \
                    --dest="com.nokia.mce" "/com/nokia/mce/request" \
                    com.nokia.mce.request.get_tklock_mode').read()
    E = re.compile('string "(\S+)"')
    try:
        ScreenLockState = E.findall(Output)[0]
    except Exception, e:
        ScreenLockState = 'unknown'
        LOGGER.warning('Could not determine screen lock state: %s' % e)

    # Query device orientation (with interpretation)

    Output = os.popen('dbus-send --system --print-reply \
                    --dest=com.nokia.mce /com/nokia/mce/request \
                    com.nokia.mce.request.get_device_orientation').read()
    E = re.compile('(?:string|int32)\s+"?(\S+?)"?\s+')
    try:
        DeviceOrientation = E.findall(Output)
    except Exception, e:
        DeviceOrientation = ['unknown', 'unknown', 'unknown', '?', '?', '?']
        LOGGER.warning('Could not determine current device orientation: %s' % e)

    # Assemble status report as short as possible so that it may fit in one SMS

    StatusReport = [
        'Pwr:%s' % BatteryState,
        'Kbrd:%s' % GetKeyboardSliderState(),
        '/'.join([v for v in DeviceOrientation[0:3] if v != 'unknown']),    # interpreted orientation
#        'Roll:%s Nick:%s Rota:%s' % (DeviceOrientation[3], DeviceOrientation[4], DeviceOrientation[5]),
        'Light:%s' %  GetAmbientLightValue(),
        'Proxmty:%s' % GetProximityState(),
        'Screen:%s' % ScreenLockState,
        'Device:%s' % DeviceLockState
        ]

    RevealThat('status', '\n'.join(StatusReport))

def PerformReboot():
    """
    Reboot the phone.
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30

    LOGGER.info('Executing "%s" command: rebooting in %d seconds.' % (smscnf.COM_REBOOT, DelaySeconds))
    RevealThat('reboot',       'Phone is rebooting in %d seconds.' % DelaySeconds)

    time.sleep(DelaySeconds)
    os.system('reboot')

def PerformPoweroff():
    """
    Shutdown phone (switch off).
    """
    # disable resending of messages (would fail anyway when rebooting) 
    smscnf.ENABLERESEND = 'no' 
    DelaySeconds = 30 

    LOGGER.info('Executing "%s" command: power off in %d seconds.' % (smscnf.COM_POWEROFF, DelaySeconds))
    RevealThat('shutdown',   'Phone will power off in %d seconds.' % DelaySeconds)
    
    time.sleep(DelaySeconds)
    os.system('poweroff')

#############################################
#   camera
#############################################

def PerformCamera(args = []):
    """
    Get picture of front camera and send it to email address.
    """
    LOGGER.debug('PerformCamera(' + ','.join(args) + ')')
    
    # Should there be a running live stream, stop it in any case.
    # Otherwise neither a new stream could be started nor a picture may be taken. 
    LiveCameraCmd = os.path.join(InstPath, 'live-camera.sh')
    if os.path.isfile(LiveCameraCmd):
        PIDs = pidof(LiveCameraCmd)
        LOGGER.debug('%s PIDS: (%s)' % (LiveCameraCmd, PIDs))
        if PIDs:
            os.system('kill -TERM %s' % PIDs)
            LOGGER.info(         'Camera live stream stopped.')
            RevealThat('Camera', 'Camera live stream stopped.')                    
            time.sleep(2)
    
        if not (len(args) < 2 or args[0].upper() in ('', 'P', 'PIC', 'IMG')):
            # Interpret this as command to control camera live stream.
            if args[0] and not (args[0].upper() in ('off', '0', '-')):
                if os.WEXITSTATUS(os.system('"%s" 1 %s &' %(LiveCameraCmd, ' '.join(shellArg(x) for x in args[1:])))) == 0:
                    LiveViewerCmd = os.path.join(InstPath, 'live-viewer.sh')
                    LOGGER.info(         'Camera live stream started.')
                    RevealThat('Camera',['Camera live stream started (%s).\n' % ','.join(args),
                                         'Viewer is attached. Call with --help for details.'],
                               Attachment = os.path.isfile(LiveViewerCmd) and LiveViewerCmd,
                               Attachhint = 'text/plain')
                else:
                    LOGGER.error(        'Camera live stream did not start.')
                    RevealThat('Camera', 'Camera live stream did not start.')
            return

    # Take a single image:
    ImageFilename = TakeFrontCamPicture()
    if ImageFilename:
        Hostname = os.uname()[1]
        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                  Subject =  'NOKIA N900[%s]: %s' % (Hostname, smscnf.COM_CAMERA),
                  Text    =  ['Frontcam picture from Nokia N900 phone:\n',
                              'Taken on %s at %s\n' % (Hostname, GetTimeString())],
                  SuccessLogMsg  = 'Sent camera picture to "%s".' % smscnf.EMAILADDRESS,
                  Attachment =  ImageFilename,
                  Attachhint = 'image' )

def TakeFrontCamPicture():
    """
    Acquire front camera picture.
    @return string|bool Filename where image has been stored. False on errors.
    """
    ImageFilename       = os.path.join(LogPath, PhotoName)
    ImageFileEncoding   = 'jpegenc'     # TODO: encoding should depend on desired image type (png, jpeg, ...) 
    CamDevice           = '/dev/video1'
    
    # If the camctrl tool is available, then before taking picture, let it measure
    # ambient light and set appropriate gain and exposure values for the camera: 
    CamCtrl = os.path.join(InstPath, 'camctrl')
    if os.path.isfile(CamCtrl):
        try:    os.WEXITSTATUS(os.system(CamCtrl
                        + " --device '%s'" % CamDevice
                        + " --ambient-light-sensor '%s'" % AmbientLightSensorDevice
                        + " --all-auto-via-als 1"))
        except: pass

    GStreamerPipeline   = (
        # For an acceptable image let the camera warm up for approx 0.5 seconds.
        # This is achieved by taking 16 buffers (rather than only one) and then
        # use the last one for the image.
        "v4l2camsrc device='%s' num-buffers=16" % CamDevice,
#        # framerate-cap requires videorate element:
#       "videorate",
#       "video/x-raw-yuv,width=640,height=480,framerate=3/1",
        "video/x-raw-yuv,width=640,height=480",
        "ffmpegcolorspace",
        "gamma gamma=2",
        ImageFileEncoding,
        # Let multifilesink store all images to the same file; so the last one wins:
        "multifilesink render-delay=1000000000 location='%s'" % ImageFilename,
    )

    try:    os.remove(ImageFilename)
    except: pass

    if os.WEXITSTATUS(os.system('gst-launch ' + ' \! '.join(GStreamerPipeline))) != 0:
        LOGGER.error('Camera picture failed (image capture error).')
        return False

    if not os.path.isfile(ImageFilename):
        LOGGER.error('Taking camera picture failed (image file not found).')
        return False
    LOGGER.info('Camera picture acquired.')
    return ImageFilename

#############################################
#    phone call
#############################################

def PerformPhonecall(PhoneNumber):
    """
    Make phone call to PhoneNumber.
    @return void.
    """
    try:
        os.system('run-standalone.sh dbus-send --system --dest=com.nokia.csd.Call \
                   --type=method_call /com/nokia/csd/call com.nokia.csd.Call.CreateWith \
                   string:"%s" \
                   uint32:0' % PhoneNumber )        
        LOGGER.info(         'Making phonecall to "%s".' % PhoneNumber)          
    except:
        LOGGER.error('Failed to make phonecall to "%s".' % PhoneNumber)

#############################################
#    data connection
#############################################

def GetNetworkConnectionType(): 
    """
    Get info of current data connection type
    @return string WLAN|GPRS|NONE
    """
    Wlan = os.popen('ifconfig | grep "wlan0"').read()
    Gprs = os.popen('ifconfig | grep "gprs0"').read() 

    if   Wlan != '' and Gprs == '':     Type = 'WLAN'
    elif Wlan == '' and Gprs != '':     Type = 'GPRS'
    else:                               Type = 'NONE' 
    return Type

def GetGPRSConnectionStatus():
    """
    Get status of GPRS data network.
    @return: mixed True|False|None
    """
    Output = os.popen('dbus-send --print-reply --system --dest=com.nokia.csd.GPRS /com/nokia/csd/gprs com.nokia.csd.GPRS.GetStatus').read()

    E = re.compile(' boolean (\S+)')
    R = E.findall(Output)
    if R:
        if R[0] == 'false': return False
        if R[0]  == 'true': return True
    return None

def SetNetworkConnectionType(NewType):
    """
    Connect to GPRS or WLAN data network.
    @param string Network type: GPRS|WLAN
    @return bool Success indicator
    """
    DelaySeconds = 3  # in sec. between connection status checks
    MaxT  = 30 # in sec. before timeout
    MaxR  = 3  # number of retries

    Type = GetNetworkConnectionType()

    if Type != 'NONE':
        if Type == NewType:
            if Type == 'GPRS':
                Type, IAP, Name = GetGPRSname()
            elif Type == 'WLAN':
                Type, IAP, Name = GetWLANname()
            else:
                LOGGER.error('Unknown network type "%s".' % Type)
                Name = 'unknown'
            LOGGER.info( '%s network (%s) already connected.' % (Type, Name) )
            return True 
        DisableConnection()

    if   NewType == 'GPRS': Type, IAP, Name = GetGPRSname()
    elif NewType == 'WLAN': Type, IAP, Name = GetWLANname()
    else:                   Type, IAP, Name = 'NONE', 'NONE', 'NONE'

    if Type == 'NONE':
        LOGGER.error('No network available.')
        return False

    # initiate network connecting
    LOGGER.info( 'Connecting to %s network (%s).' % (Type, Name) )
    os.system( 'dbus-send --system --type=method_call --dest=com.nokia.icd /com/nokia/icd com.nokia.icd.connect string:"%s" uint32:0' % IAP )

    Connected = IF(Type == 'GPRS', GetGPRSConnectionStatus, lambda:GetNetworkConnectionType() == Type)
    T, R = 0, 0
    while not Connected():
        if T >= MaxT:
            R += 1
            if R >= MaxR:
                LOGGER.error('Failed to connect to %s network after %s attempts.' % (Type, R))
                return False
            LOGGER.warning('No connection to %s network after %s sec. Retrying...' % (Type, T))
        time.sleep(DelaySeconds)
        T += DelaySeconds

    LOGGER.info( '%s network (%s) connection successful.' % (Type, Name) )
    return True

def DisableConnection(): 
    """
    Disconnect any current data connection.
    """
    DelaySeconds = 1

    os.system('dbus-send --system --dest=com.nokia.icd2 /com/nokia/icd2 com.nokia.icd2.disconnect_req uint32:0x8000')
    LOGGER.info('Current network disconnected.')

    time.sleep(DelaySeconds)

def GetGPRSname(): 
    """
    Get name of GPRS data connection from current inserted SIM card in Nokia phone.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()
    CurrentIMSI = smscon.GetPhoneCode('IMSI')

    for Network in NetworkList:
        Type, IAP, Name, Imsi = (list(Network) + ['NONE', 'NONE', 'NONE', 'NONE'])[0:4]
        if Type == 'GPRS' and Imsi == CurrentIMSI:
            LOGGER.debug('original IAP=[%s]' % IAP)
            return Type, IAP.replace('@32@', ' '), Name

    return 'NONE', 'NONE', 'NONE'

def GetWLANname():
    """
    Get name of active WLAN data connection.
    @return string tuple (Type, IAP, Name). Each of which is string 'NONE' if not found.
    """
    NetworkList = GetNetworks()

    # get last used data network IAP
    LastUsedNetwork = os.popen('gconftool -g /system/osso/connectivity/IAP/last_used_network').read().strip('\n')
    LOGGER.debug('LastUsedNetwork = [%s].' % LastUsedNetwork)

    for Network in NetworkList:
        Type, IAP, Name = Network[0:3]
        if Type == 'WLAN' and LastUsedNetwork == IAP:
            LOGGER.debug('WLAN: %s' % Name)
            return Type, IAP, Name

    return 'NONE', 'NONE', 'NONE'

def GetNetworks():
    """
    Get list of all available data connections (stored under "Internet connections" in menu).
    @return list of (Type, IAP, Name [, Imsi, NameAP]) tuples.
    """
    Output = os.popen('gconftool -R /system/osso/connectivity/IAP | grep /IAP/').read()
    E = re.compile('/system/osso/connectivity/IAP/(\S+):')
    IAPlist = E.findall(Output) 

    DiscardList = ['wap', 'mms']
    NetworkList = []

    for IAP in IAPlist:
        if IAP == '0000-0000-0000-0000':
            continue
        # get connection type
        Type = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/type' % IAP).read().replace('_INFRA', '').strip('\n')

        if Type == 'GPRS':
            # get gprs_accespointname
            NameAP = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/gprs_accesspointname' % IAP).read().strip('\n')

            # discard unwanted connections
            if not [Discard for Discard in DiscardList if NameAP.find(Discard) >= 0]:
                # get connection name & it's IMSI number
                Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
                Imsi = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/sim_imsi' % IAP).read().strip('\n')

                # convert decimal value of ASCII character in IAP string to normal ASCII character.
                E = re.compile('@(\d+)@')
                DecList = E.findall(IAP)

                for Dec in DecList:
                    IAP = IAP.replace( '@%s@' % Dec, chr( int(Dec) ) )

                # add GPRS connection to list
                LOGGER.debug( 'Type=%s, IAP=%s, Name=%s, IMSI=%s, AP=%s' % (Type, IAP, Name, Imsi, NameAP) )
                NetworkList.append( (Type, IAP, Name, Imsi, NameAP) )
            
        elif Type == 'WLAN':
            Name = os.popen('gconftool -g /system/osso/connectivity/IAP/%s/name' % IAP).read().strip('\n')
            # add WLAN connection to list
            LOGGER.debug( 'Type=%s, IAP=%s, Name=%s' % (Type, IAP, Name) )
            NetworkList.append( (Type, IAP, Name) )

    return NetworkList

def RememberNetworkSettings():
    """
    Save current active data network settings.
    """
    global ConnectionType
    global ConnectionIAP
    global ConnectionName

    Type = GetNetworkConnectionType()
    if   Type == 'WLAN':    ConnectionType, ConnectionIAP, ConnectionName = GetWLANname()
    elif Type == 'GPRS':    ConnectionType, ConnectionIAP, ConnectionName = GetGPRSname()
    else:                   ConnectionType, ConnectionIAP, ConnectionName = 'NONE', 'NONE', 'NONE'

    LOGGER.info( 'Saved current %s network (%s) for restore.' % (Type, ConnectionName) )

def RestoreNetworkSettings():
    """
    Re-enable previous stored data network settings.
    """
    global ConnectionType
    global ConnectionName

    if ConnectionType == 'GPRS' or ConnectionType == 'WLAN':
        LOGGER.info( 'Restoring saved %s network (%s)' % (ConnectionType, ConnectionName) )
        SetNetworkConnectionType(ConnectionType)
    elif ConnectionType != 'NONE':
        LOGGER.info( 'Disabling connection to %s network (%s).' % (ConnectionType, ConnectionName) )
        DisableConnection()

#############################################
#   connect behaviour
#############################################

def SetAutoConnectBehaviour(NewBehaviour):
    """
    Set Nokia phone data connection behavior.
    The old behaviour is saved in global ConnectionBehaviour and can be used to restore.
    @param string NewBehaviour One of [ASK|WLAN|GPRS|ANY]
    @return bool True if behaviour changed.
    """

    def _getConnectBehaviour():
        """
        Get current Nokia phone data connection behaviour.
        @return string ASK|ANY|GPRS|WLAN|None
        """ 
        Mode = os.popen('gconftool-2 -g --list-type string /system/osso/connectivity/network_type/auto_connect').read().strip('\n') 
    
        if Mode == '[]':            return 'ASK'
        if Mode == '[*]':           return 'ANY'
        if Mode == '[GPRS]':        return 'GPRS'    
        if Mode == '[WLAN_INFRA]':  return 'WLAN'
        LOGGER.error('Unexpected connect behaviour "%s".' % str(Mode))
        return None
    
    def _setConnectBehaviour(Behaviour):
        """
        Set new Nokia device data connection behaviour (ASK|WLAN|GPRS|ANY).
        @param string Behaviour One of [ASK|WLAN|GPRS|ANY]
        @return void
        """
        DelaySeconds = 2
    
        # options: [WLAN_INFRA], [GPRS], [*] <=='any', [] <=='ask'
        if   Behaviour == 'ASK':     String = ''
        elif Behaviour == 'WLAN':    String = 'WLAN_INFRA'
        elif Behaviour == 'GPRS':    String = 'GPRS'
        elif Behaviour == 'ANY':     String = '*'
        else:
            LOGGER.error('Unknown connect behaviour "%s".' % str(Behaviour))
            return
    
        LOGGER.debug('Setting connect behaviour to "%s".' % Behaviour)
        # change setting "Connect automatically" to given Behaviour
        os.system('gconftool-2 -s --type list --list-type string /system/osso/connectivity/network_type/auto_connect [%s]' % String)
        time.sleep(DelaySeconds)

    global ConnectionBehaviour

    OldBehaviour = _getConnectBehaviour()
    if  ConnectionBehaviour == None:
        ConnectionBehaviour = OldBehaviour

    if NewBehaviour == None:
        LOGGER.info('Changing internet connection behaviour not enabled (still is "%s").' % OldBehaviour)
        return False
    elif NewBehaviour == OldBehaviour:
        LOGGER.info('Internet connection behaviour already set to "%s".' % NewBehaviour)
        return False

    # New behaviour is different from current. Save the current and try to establish the new one:
    ConnectionBehaviour = OldBehaviour
    _setConnectBehaviour(NewBehaviour)

    # Check new behaviour setting
    if _getConnectBehaviour() == NewBehaviour:
        LOGGER.info('Internet connection behaviour set from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return True
    else:
        LOGGER.error('Failed to set internet connection behaviour from "%s" to "%s".' % (OldBehaviour, NewBehaviour))
        return False

#############################################
#   power status
#############################################

def GetBatteryChargeInfo(TextOnly = True, ChargePercentMin = 10):
    """
    Determine battery charge info.
    @param bool TextOnly
    @param int ChargePercentMin Warn if battery change level is below this value.
    @return string Charge info text or
    @return (string, number, bool) if TextOnly is false.
    """
    global dev_obj

    # Try getting charge info, not all devices support all properties: 
    try:
        chargePercent = dev_obj.GetProperty('battery.charge_level.percentage')  # charge amount of battery in percentage
        chargePercent = int(float(chargePercent))
        chargePercentText = '%s%%' % chargePercent
        chargeHint = IF(chargePercent >= ChargePercentMin, '', ' (almost empty!)')
    except Exception, e:
        LOGGER.warning('%s' % e)
        chargePercent = None
        chargePercentText = 'unknown'
        chargeHint = ''

    try:
        isCharging = dev_obj.GetProperty('battery.rechargeable.is_charging') # charging: 1 / discharging: 0
        isChargingText = IF(isCharging, ', charging', ', discharging')
    except Exception, e:
        LOGGER.warning('%s' % e)
        isCharging = None
        isChargingText = ''

    Text = 'Phone battery charge is %s%s%s.' % (chargePercentText, isChargingText, chargeHint)
    if TextOnly: return Text
    return (Text, chargePercent, isCharging)

def LogBatteryChargeInfo():
    """
    Log battery charge info.
    """
    LOGGER.info(GetBatteryChargeInfo())

#############################################
#   SSH 
#############################################

def EnableSSHcontrol():
    """
    Start the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    
    if remote_sh.runs():
        LOGGER.info('ssh connection is already active.')
        return

    if not IsHostReachable(smscnf.REMOTEHOST):
        LOGGER.warning('No ping response from "%s", will try to connect anyway.' % smscnf.REMOTEHOST)

    LOGGER.info('Connecting to "%s" for ssh.' % smscnf.REMOTEHOST)

    (Output, ExitStatus) = remote_sh.start() 
    
    LOGGER.debug('ssh Output = %s' % Output.strip('\n') )
    LOGGER.debug('ssh ExitStatus = %s' % ExitStatus)

    if ExitStatus == 0:
        LOGGER.info(  'ssh connection to "%s" established.' % smscnf.REMOTEHOST)
        RevealThat('ssh', 'connection to "%s" established.' % smscnf.REMOTEHOST)
    else:
        LOGGER.error( 'ssh connection to "%s" failed (%s).' % (smscnf.REMOTEHOST, Output) )
        RevealThat('ssh', 'connection to "%s" failed.'       % smscnf.REMOTEHOST)
        PIDs = remote_sh.stop(Verbose=False)
        if PIDs:
            LOGGER.info('ssh stopped [PIDs=%s].' % PIDs)        

def DisableSSHcontrol(Verbose = True):
    """
    Stop the ssh routine.
    @return void.
    """
    remote_sh = smscon.smscon_remote_sh(smscnf)
    if remote_sh.stop(Verbose = Verbose):
        RevealThat('ssh', 'connection to "%s" terminated.' % smscnf.REMOTEHOST)

#############################################
#   GPS
#############################################

def EnableGpsAcquireMode(Mode):
    """
    Schedule starting the GPS device.
    """
    LOGGER.debug('Enabling GPS device for %s mode (current mode is %s).' % (Mode, GpsMode))
    gobject.idle_add(OnGPSstart, Mode)

def DisableGpsAcquiring():
    """
    Stop the GPS device mainloop.
    """
    global GpsMode, GpsControl

    if GpsMode:
        LOGGER.info('Stopping GPS device from %s mode.' % GpsMode)
        GpsMode = False
        # stop the GPS device
        GpsControl.stop()
    else:
        LOGGER.info('GPS device already stopped.')

def OnGPSstart(Mode):
    """
    Callback to start the GPS device in given Mode or switch already started device to given Mode.
    @return False
    """
    global GpsMode, GpsControl, GpsStartTime, GpsTimeoutTime, GpsLocationBasisTime, GspTrackingBasisTime

    if GpsMode:
        LOGGER.info('Switching GPS device mainloop to %s mode (was %s).' % (Mode, GpsMode))
        GpsMode = Mode
    else:
        LOGGER.info('Starting GPS device mainloop in %s mode.' % Mode)
        GpsMode = Mode
        GpsStartTime = GpsTimeoutTime = time.time()
        GpsLocationBasisTime = GspTrackingBasisTime = 0
        GpsControl.start()
    return False    # single shot.

def OnGPSstop(GpsControl, notUsed):
    """
    Callback from GPS subsystem in case GPS device got stopped.
    @return False
    """
    global GpsMode
    GpsMode = False

    global GpsIsTest
    if GpsIsTest:
        GpsIsTest = False
        RestoreNetworkSettings()
    return False    # single shot.

def OnGPSchanged(GpsDevice, GpsControl):
    """
    Callback from GPS subsystem in case GPS device data changed.
    @return void
    """
    global GpsCoordinateList, GpsStartTime, GpsTimeoutTime, GpsLocationBasisTime, GspTrackingBasisTime

    LOGGER.debug('OnGPSchanged() called for mode "%s".' % str(GpsMode))
    # The next check is essential because we may even get callbacks if another app turns GPS device on!
    if not (GpsMode in (GpsModeLOCATION, GpsModeTRACKING)):
        return

    timeNow = time.time()
    StartedSinceSeconds = timeNow - GpsStartTime
    TimeoutSeconds      = timeNow - GpsTimeoutTime
    LOGGER.debug('GpsStartTime = %s, StartedSinceSeconds = %.1f, TimeoutSeconds = %.1f'
                                % (GpsStartTime, StartedSinceSeconds, TimeoutSeconds))
    LOGGER.debug('GpsDevice.status = %s' % GpsDevice.status) # location.GPS_DEVICE_STATUS_NO_FIX  or  location.GPS_DEVICE_STATUS_FIX

    # Check if data is OK and log why it it not OK.

    GpsDataOK = True
    if GpsDataOK:
        GpsDataOK = (GpsDevice.fix[1] & location.GPS_DEVICE_LATLONG_SET) != 0
        if not GpsDataOK:
            LOGGER.info('Waiting for GPS satellite fix (since %.1f of %d sec).' % (TimeoutSeconds, int(smscnf.GPSTIMEOUT)))

    if GpsDataOK:
        # GPS device has fix.
        Latitude  = GpsDevice.fix[4]
        Longitude = GpsDevice.fix[5]
        Accuracy  = GpsDevice.fix[6]    # centimeters

        if str(Accuracy) != 'nan':
            Accuracy  = Accuracy / 100  # centimeters to meters
        elif GpsMode == GpsModeTRACKING: 
            Accuracy = 999999  # 999999 meters. A very high value should be better than no coords et all.

        # Check if coordinates acquired by GPS device are valid
        GpsDataOK = str(Latitude) != 'nan' and str(Longitude) != 'nan' and str(Accuracy) != 'nan'
        if not GpsDataOK:
            LOGGER.info('Waiting for valid GPS coordinate (since %.1f of %d sec).' % (TimeoutSeconds, int(smscnf.GPSTIMEOUT)))

    if GpsDataOK and GpsDevice.status == location.GPS_DEVICE_STATUS_NO_FIX:
        # Ignore those coordinates; they are valid but might be just the last coordinate
        # from last long-time-ago fix and has nothing to do with current position:
        GpsDataOK = False
        LOGGER.info('Artificial GPS coordinate (%f, %f, %.1f m.) after %.1f sec since start. Ignored.'
                                % (Latitude, Longitude, Accuracy, StartedSinceSeconds) )
    # Check timeout condition

    if GpsDataOK:
        GpsTimeoutTime = timeNow  # restart timeout
    IsTimeoutDetected = TimeoutSeconds > int(smscnf.GPSTIMEOUT)

    # Now handle things differently depending on GpsMode:

    if GpsMode == GpsModeLOCATION:
        LocationTimeoutSeconds = max(int(smscnf.GPSINTERVAL), 30) * int(smscnf.GPSPOLLING)
        LocationSinceSeconds = timeNow - GpsLocationBasisTime 
        if GpsDataOK:

            if type(GpsCoordinateList).__name__ != 'list' or len(GpsCoordinateList) == 0:
                GpsCoordinateList = []
                GpsLocationBasisTime = timeNow
                LOGGER.info('Start collecting min. %s GPS coordinate(s) for max. %d sec...' % (smscnf.GPSPOLLING, LocationTimeoutSeconds))

            GpsCoordinateList.append( (Latitude, Longitude, Accuracy) )
            LOGGER.info('GPS coordinate #%d (%f, %f, %.1f m.) after %.1f sec since fix.'
                                % (len(GpsCoordinateList), Latitude, Longitude, Accuracy, LocationSinceSeconds) )                

            if len(GpsCoordinateList) < int(smscnf.GPSPOLLING):
                LOGGER.debug('gather more coordinates (GPSPOLLING count).')
                return  # gather more coordinates
            ReportTheLocation = True
        else:
            # If this is timeout but we have some coords already, report them.
            ReportTheLocation = IsTimeoutDetected and len(GpsCoordinateList or []) > 0

        if ReportTheLocation:
            # Sort GPS coordinate list on 'Accuracy'
            GpsCoordinateList = sorted( GpsCoordinateList, key=itemgetter(2) )
            Latitude, Longitude, Accuracy = GpsCoordinateList[0]
            GpsDataOK = True

            # If phone is in move, then GSP change reports may arrive faster then as
            # specified in smscnf.GPSINTERVAL. If the GPS device was started in this
            # situation, we get a lot of less than possibe accurate coords very fast.
            # If best accuracy is not in tolecance and we still have time, try further:
            if Accuracy > int(smscnf.get('GPSMAXDEVIATION', 50)) and LocationSinceSeconds < LocationTimeoutSeconds:
                LOGGER.debug('gather more coordinates (Accuracy > GPSMAXTOLERANCE and %d < %d sec).'
                                                                % (LocationSinceSeconds, LocationTimeoutSeconds))
                return  # gather more coordinates

            LOGGER.info( 'Most accurate GPS coordinate out of %d = (%f, %f, %.1f m.) after %.1f sec since start.'
                            % (len(GpsCoordinateList), Latitude, Longitude, Accuracy, StartedSinceSeconds) )
            GpsCoordinateList = None
            # stop the GPS device
            GpsControl.stop()

    elif GpsMode == GpsModeTRACKING:
        if GpsDataOK:
            if GspTrackingBasisTime and timeNow < GspTrackingBasisTime + int(smscnf.GPSINTERVAL):
                LOGGER.debug('gather more coordinates (data not OK but ealier then GPSINTERVAL).')
                return
            GspTrackingBasisTime = timeNow
            LOGGER.info( 'GPS coordinate acquired (%f, %f, %.1f m.) after %.1f sec since start.'
                            % (Latitude, Longitude, Accuracy, StartedSinceSeconds) )

    # Common report part. 

    if not GpsDataOK:
        if IsTimeoutDetected:
            LOGGER.error(     'GPS device timeout after %.1f sec.' % StartedSinceSeconds)
            RevealThat('GPS', 'GPS device timeout after %.1f sec.' % StartedSinceSeconds, smscnf.GPSSEND)                    
            # stop the GPS device
            GpsControl.stop()
        LOGGER.debug('gather more coordinates (data not OK but no timeout).')
        return

    GpsText = ['Lat: %f\n' % Latitude,
               'Long: %f\n' % Longitude,
               'Accuracy: %.1f mtr\n' % Accuracy,
               'Duration: %.1f sec\n' % StartedSinceSeconds]

    LinkText = ['<a href="http://maps.google.com/maps?q=%f,%f">GoogleMap</a>\n'% (Latitude, Longitude)]

    RevealThat('GPS Location', GpsText + LinkText,
                SendMethod = smscnf.GPSSEND,
                Attachment = smscnf.GPSSEND in ['email', 'both'] and CreateGoogleMap(Latitude, Longitude),
                Attachhint = 'image')

def OnGPSerror(GpsControl, GpsError, notUsed):
    """
    Callback from GPS subystem to handle GPS device error detected.
    """

    if GpsError == location.ERROR_USER_REJECTED_DIALOG:
        ErrorMsg = ("GPS device error (User didn't enable requested methods).")
    elif GpsError == location.ERROR_USER_REJECTED_SETTINGS:
        ErrorMsg =('GPS device error (User changed settings, which disabled location).')
    elif GpsError == location.ERROR_BT_GPS_NOT_AVAILABLE:
        ErrorMsg =('GPS device error (Problems with BT GPS).')
    elif GpsError == location.ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
        ErrorMsg =('GPS device error (Requested method is not allowed in offline mode).')
    elif GpsError == location.ERROR_SYSTEM:
        ErrorMsg =('GPS device error (System error).')
    else:
        ErrorMsg =('GPS device error (Unknown error %s).' % str(GpsError))
    LOGGER.error(ErrorMsg)

    LOGGER.error('Stopping failed GPS device.')
    DisableGpsAcquiring()

    RevealThat('GPS Error', ErrorMsg)

###############################

def CreateGoogleMap(Latitude, Longitude):
    """
    Create map of current GPS coordinate.
    @return string|False Filename of map or False if no map.
    """
    MapZoom     = 16
    MapSize     = (600,600)
    MarkerColor = 'red'
    MarkerLabel = 'N'

    MapFilename = os.path.join(LogPath, MapName)
    MapUrl = 'http://maps.google.com/maps/api/staticmap?center=%s,%s&zoom=%s&size=%sx%s&format=png&markers=color:%s|label:%s|%s,%s&sensor=false' \
              % (Latitude,
                 Longitude,
                 MapZoom,
                 MapSize[0],
                 MapSize[1],
                 MarkerColor,
                 MarkerLabel,
                 Latitude,
                 Longitude)
    try:
        urllib.urlretrieve( MapUrl, MapFilename )
    except:
        LOGGER.error('Failed to create GPS coordinate map (retrieve error).')
        return False
    if not os.path.isfile(MapFilename):
        LOGGER.error('Failed to create GPS coordinate map (file not found).')
        return False
    else:
        LOGGER.info('Created GPS coordinate map.')
        return MapFilename

#############################################
#   SIM card
#############################################

def GetOperatorName():
    """
    Get current SIM telecom operator name.
    """
    try:
        Output = os.popen('dbus-send --system --print-reply=literal --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_registration_status').read()
        E = re.compile('uint32 (\d+)')
        R = E.findall(Output)
        Output = os.popen( 'dbus-send --system --print-reply --dest=com.nokia.phone.net /com/nokia/phone/net Phone.Net.get_operator_name byte:0 uint32:"%s" uint32:"%s"' % (R[1], R[2]) ).read()
        E = re.compile('string "(.*)"')
        R = E.findall(Output)[0]
    except:
        LOGGER.error('Could not get telecom network operator name.')
        return 'UNKNOWN'        

    return R

def IsAuthorizedSIM(ICCID = None, IMSI = None):
    """
    Check if SIM with given ICCID and IMSI is an authorized SIM.
    Implicitely convert SIM database: remove IMSI code, store ICCID code.
    @param string|None ICCID code from SIM.
    @param string|None IMSI code from SIM.
    @return bool
    """
    simDB = smscon.IMSIConfiguration()
    simDB.load(Silent = True)
    LOGGER.info('Loaded authorization info for %d SIM card(s).' %  len(simDB.keys()))
    ICCID = ICCID and simDB.PackCodeType('ICCID', ICCID)
    IMSI  = IMSI  and simDB.PackCodeType('IMSI',  IMSI)
    if ICCID == None:
        return False    # no IMSI in this case.
    if ICCID in simDB.keys():
        # Ensure only ICCID but no IMSI code stored for this SIM:
        if IMSI and IMSI in simDB.keys():
            simDB.update(IMSI, None, Silent = True)
        return True
    if not (IMSI and IMSI in simDB.keys()): 
        return False
    # Convert simDB entry: remove IMSI code, add ICCID code:
    simDB.update(IMSI, None, Silent = True)
    simDB.update(ICCID, '')
    return True

def IsAuthorizedSIMcard():
    """
    @return bool Iff SIM card is authorized.
    """
    return IsAuthorizedSIM(smscon.GetPhoneCode('ICCID'), smscon.GetPhoneCode('IMSI'))

def ValidateSIMCard():
    """
    Check if there is a current SIM card, if that SIM card is authorized and act accordingly.
    @return void
    """
    
    def HandleSIMValidation(IsSIMPresent = True, IsSIMPINok = False, IsKnownSIM = False):
        """
        Perform actions depending on SIM validation status.
        @param bool IsSIMPresent Whether a SIM is present in the phone et all.
        @param bool IsSIMPINok Whether a present SIM was authorized by a correct SIM PIN.
        @param bool IsKnownSIM Whether the SIM was known and authorized.
        @return string Device lock status.
        """
        # Use silent calls to lock/unlock and RevealThat situation once only:
        DeviceLockStatus = ''
        if smscnf.SIMUNLOCK == 'lockalways':
            return 'Lock always: ' + __DoDeviceLock()
        elif smscnf.SIMUNLOCK == 'lockifnosim':
            if not IsSIMPresent:
                return 'Lock if no SIM: ' + __DoDeviceLock()
        elif smscnf.SIMUNLOCK == 'lockifnopin':
            if IsSIMPresent and not IsSIMPINok:
                return 'Lock if no SIM PIN.' + __DoDeviceLock()
        elif smscnf.SIMUNLOCK == 'locknewsim':
            if IsSIMPresent and not IsKnownSIM:
                return 'Lock if new SIM: ' + __DoDeviceLock() 
        elif smscnf.SIMUNLOCK == 'yes':
            if not IsKnownSIM:
                return 'Unlock if new SIM: ' + __DoDeviceUnlock() 
        return DeviceLockStatus

    global IsStolenModeEnabled
    IsSIMPresent = IsSIMPINok = IsKnownSIM = False
    
    CurrentICCID = smscon.GetPhoneCode('ICCID');
    IsSIMPresent = CurrentICCID != None
    if not IsSIMPresent:
        DeviceLockStatus = HandleSIMValidation(IsSIMPresent, IsSIMPINok, IsKnownSIM)
        LOGGER.warning('SIM card is not present. %s' % DeviceLockStatus)
        if smscnf.STOLENIFNOSIM == 'no':
            return
        IsStolenModeEnabled = True
        LOGGER.warning('Stolen mode enabled.')
        # With some luck internet connection via wlan is available:
        RevealThat('No SIM', '\n'.join(['SIM card not present.',
                        DeviceLockStatus,
                        'Stolen mode enabled.',
                    ]), SendMethod = 'email')   # sending sms would not work w/o a SIM...
        return

    CurrentIMSI = smscon.GetPhoneCode('IMSI')
    IsSIMPINok = CurrentIMSI != None
    IsKnownSIM = IsAuthorizedSIM(ICCID = CurrentICCID, IMSI = CurrentIMSI)

    if not IsSIMPINok:
        simStatusText = IF(IsKnownSIM, 'authorized but inactive', 'not authorized and inactive')
        DeviceLockStatus = HandleSIMValidation(IsSIMPresent, IsSIMPINok, IsKnownSIM)
        LOGGER.warning('SIM card is %s. %s' % (simStatusText, DeviceLockStatus))
        IsStolenModeEnabled = True
        LOGGER.warning('Stolen mode enabled.')
        # With some luck internet connection via wlan is available.
        RevealThat('Skipped SIM', '\n'.join([
                        'SIM card %s (no or wrong SIM pin).' % simStatusText,
                        IF(IsKnownSIM, '', 'SIM ICCID=%s' % CurrentICCID),  # Don't send ICCID of known SIMs.
                        DeviceLockStatus,
                        'Stolen mode enabled.',
                    ]), SendMethod = 'email')   # sending sms would not work w/o SIM PIN...
        return

    if not IsKnownSIM:
        DeviceLockStatus = HandleSIMValidation(IsSIMPresent, IsSIMPINok, IsKnownSIM)
        LOGGER.warning('SIM card is not authorized. %s' % DeviceLockStatus)
        IsStolenModeEnabled = True
        LOGGER.warning('Stolen mode enabled.')

        SendMethod = smscnf.MESSAGESEND
        if   SendMethod == 'none':  SendMethod = 'sms'
        elif SendMethod == 'email': SendMethod = 'both'

        RevealThat('New SIM', '\n'.join([
                        'IMSI=%s' % CurrentIMSI,
                        'IMEI=%s' % smscon.GetPhoneCode('IMEI'),
                        'ICCID=%s' % CurrentICCID,
                        'Telecom="%s".' % GetOperatorName(),
                        DeviceLockStatus,
                        'Stolen mode enabled.'
                    ]), SendMethod)
        return

    IsStolenModeEnabled = False
    DeviceLockStatus = HandleSIMValidation(IsSIMPresent, IsSIMPINok, IsKnownSIM)
    LOGGER.info('SIM card is authorized. %s' % DeviceLockStatus)

#############################################

def DoCommandLine(args):
    """
    Check arguments for smscon_daemon and perform corresponding actions.
    @return int Status where 0|False is fatal error, 1 is success, 2 is success with test mode
    """

    DoCommandLine.BAD = False
    DoCommandLine.OK = 1
    DoCommandLine.TESTMODE = 2

    def TellError(Msg):
        LOGGER.error(Msg)
        return False

    def PrepareTestModeNetwork():
        if  GetNetworkConnectionType() != 'WLAN':
            SetNetworkConnectionType('GPRS')
            SetAutoConnectBehaviour('GPRS') # disable switching to other network

    CommandLineStatus = DoCommandLine.OK;
    if len(args) > 0:
        Mode = args.pop(0)
        if Mode == '-test':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Func = args.pop(0)
            CommandLineStatus = CommandLineStatus and DoCommandLine.TESTMODE

            if Func == 'gps1' or Func == 'gps2': 
                RememberNetworkSettings()
                PrepareTestModeNetwork();

                global GpsIsTest
                GpsIsTest = True

                if Func == 'gps1':
                    EnableGpsAcquireMode(GpsModeLOCATION)
                elif Func == 'gps2':
                    EnableGpsAcquireMode(GpsModeTRACKING)
                return CommandLineStatus

            if Func == 'ssh': # test mode for ssh connection
                RememberNetworkSettings()
                PrepareTestModeNetwork();
                DisableSSHcontrol(Verbose = False)
                EnableSSHcontrol()
                return CommandLineStatus

            if Func == 'ssh-off': # test for ssh disconnect
                DisableSSHcontrol(Verbose = True)
                return CommandLineStatus

            if Func == 'cam': # test mode for camera
                RememberNetworkSettings()
                PrepareTestModeNetwork();
                PerformCamera()
                RestoreNetworkSettings()
                SetAutoConnectBehaviour(ConnectionBehaviour)

            elif Func == 'sms': # test mode for sms
                SMSsend(smscnf.MASTERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func == 'sms2': # test mode for sms
                SMSsend(smscnf.SENDERNUMBER, ['Test SMS from Nokia N900 phone'], 'send SMS message test to {RECIPIENTNUMBER}')

            elif Func in ('email1', 'email2'): 
                RememberNetworkSettings()
                PrepareTestModeNetwork();

                if Func == 'email1': # test mode1 for email (text only)
                    EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject = 'Test email / Nokia N900',
                              Text = 'Email sending from Nokia N900 phone is successful!',
                              SuccessLogMsg = 'Sent test email message to <%s>.' % smscnf.EMAILADDRESS)
                elif Func == 'email2': # test mode2 for email + frontcam image
                    ImageFilename = TakeFrontCamPicture()
                    if ImageFilename:
                        EMAILsend(EmailTo = smscnf.EMAILADDRESS,
                              Subject =  'Test email / Nokia N900',
                              Text =  'Frontcam picture from Nokia N900 phone:<br>%s<br>' % GetTimeString(),
                              SuccessLogMsg = 'Sent camera picture to <%s>.' % smscnf.EMAILADDRESS,
                              Attachment =  ImageFilename,
                              Attachhint = 'image' )

                RestoreNetworkSettings()
                SetAutoConnectBehaviour(ConnectionBehaviour)
    
            elif Func == 'call': # test mode for phone call
                PerformPhonecall(smscnf.MASTERNUMBER)

            elif Func == 'script': # test mode for script
                PerformShellScript(args)
                args = [];

            else:
                LOGGER.error('Test option error (%s).' % Func)

        elif Mode == '-comtest':
            if len(args) == 0: return TellError('%s requires argument.' % Mode)
            Command = args.pop(0)
            # Note: does not enter test mode!

            LOGGER.warning('Simulating received SMS message from MASTERNUMBER with "%s" command.' % Command)
            ProcessCommand(Command, SenderPhoneNumber = smscnf.MASTERNUMBER,
                           ExtraInfo = 'SIMULATION!', CommandOrigin = 'CMDLINE', LogIfCmdUnknown = True)    

        elif Mode == '-sendemail':
            if len(args) < 3 : return TellError('%s requires at least 3 arguments.' % Mode)
            Recipient, Subject, Text = args.pop(0), args.pop(0), args.pop(0)
            AttachmentFileName = len(args) > 0 and args.pop(0)
            AttachmentMimeType = len(args) > 0 and args.pop(0)
            Hostname = os.uname()[1]

            EMAILsend(EmailTo = Recipient,
                  Subject        = 'NOKIA N900[%s]: %s' % (Hostname, Subject),
                  Text           = 'Nokia phone "%s" at %s:\n%s\n' % (Hostname, GetTimeString(), Text),
                  SuccessLogMsg  = 'Sent email message (subject=%s) to <%s>.' % (Subject, Recipient),
                  Attachment = AttachmentFileName,
                  Attachhint = AttachmentMimeType)
        else:
            LOGGER.warning('Option error (%s) - ignored' % Mode)
            return CommandLineStatus

    if len(args) > 0:
        LOGGER.warning('Options ignored: %s' % ' '.join(args))
    return CommandLineStatus

#############################################

def CheckBootLoader(WithNotification = True):
    """
    Check if smscon auto-loads at system boot.
    @return void.
    """
    if smscon.IsBootfileExistent():
        LOGGER.info('%s auto-loads at boot. OK. %s' % (smscon.NAME, IF(WithNotification, '', 'Silent check.')))
    else:
        LOGGER.warning("%s does not auto-load at boot. %s" % (smscon.NAME, IF(WithNotification, '', 'Silent check.')))
        if WithNotification:
            RevealThat('boot', "%s does not auto-load at boot." % smscon.NAME)        

def ExitDaemon(ExitReason = 'regularly', ExitCode = 0):
    """
    Exit smscon_daemon.
    @return NO RETURN.
    """
    LOGGER.critical('%s is stopping %s.' % (DaemonFile, ExitReason))
    sys.exit(ExitCode)

def RestartDaemon():
    """
    Restart smscon_daemon.
    @return void (does not return on success).
    """ 
    DelaySeconds = 10

    LOGGER.critical('(RESTART) %s tries to restart itself w/o regular stop.' % DaemonFile)
    try:
        time.sleep(DelaySeconds)
        os.system( '%s -restart' % shellArg(os.path.join(InstPath, smscon.NAME)) )
    finally:
        LOGGER.critical('(RESTART) Not restarted. Failed to kill %s.' % DaemonFile)

def GetAcceleratorValues():
    """
    Get current accelerator values.
    @return string. 3 signed integers X Y Z in range 0..1024 (approx) or ? ? ?
    """
    return SafeReadFileLine('/sys/class/i2c-adapter/i2c-3/3-001d/coord',
                ErrorMessage = 'Failed to read accelerator sensor',
                DefaultValue = '? ? ?')

def GetProximityState():
    """
    Get current proximity value.
    @return string. 1 open, closed or unknown
    """
    return SafeReadFileLine('/sys/devices/platform/gpio-switch/proximity/state',
                ErrorMessage = 'Failed to read proximity sensor')

def GetAmbientLightValue():
    """
    Get current ambient light value.
    @return string. 1 integer [lux] or unknown
    """
    return SafeReadFileLine(AmbientLightSensorDevice,
                ErrorMessage = 'Failed to read ambient light sensor')

#############################################
#   keyboard detect (dbus trigger)
#############################################

def GetKeyboardSliderState():
    """
    Get current keyboard slider state.
    @return string. 
    """
    return SafeReadFileLine('/sys/devices/platform/gpio-switch/slide/state',
                ErrorMessage = 'Failed to detect keyboard slider state')

def OnKeyboardSliderChange(Action, Type):
    """
    In stolen mode: Check and reveal state of keyboard slider.
    @return void. 
    """
    global IsStolenModeEnabled

    if not IsStolenModeEnabled:
        return

    SliderState = GetKeyboardSliderState()
    LOGGER.info('Phone is being used: keyboard position now %s.' % SliderState)
    RevealThat('Keyboard', 'Phone is being used: keyboard position now %s.' % SliderState)            

#############################################
#   battery charge (dbus trigger)
#############################################

def OnBatteryStatusChange(BarLevel, BarLevelMax):
    """
    In stolen mode: Check and reveal battery charge level.
    @return void. 
    """
    global LastBatteryStatusMessage, IsStolenModeEnabled

    if not IsStolenModeEnabled:
        return

    ChargePercentMin = 10
    Message, ChargePercent, Charging = GetBatteryChargeInfo(TextOnly = False, ChargePercentMin = ChargePercentMin)        

    if Message != LastBatteryStatusMessage:
        LastBatteryStatusMessage = Message
        LOGGER.info(Message)
        if (Charging == None or not Charging) and (ChargePercent != None and ChargePercent < ChargePercentMin): 
            RevealThat('battery warning', Message)

#############################################
#   remote host file checking (remote smscon trigger)
#############################################

def PerformCheckHost(CheckHostURL):
    """
    Check remote host file for triggering smscon
    @return True
    """
    global CheckHostLastCommand, CheckHostStatusCount, CheckHostStatusTimestamp

    CommandList = {
        '01' : smscnf.COM_CHECK,
        '02' : smscnf.COM_REBOOT,
        '03' : smscnf.COM_POWEROFF,
        '04' : smscnf.COM_POWER,
        '05' : smscnf.COM_LOCATION,
        '06' : smscnf.COM_REMOTEON,
        '07' : smscnf.COM_REMOTEOFF,
        '08' : smscnf.COM_CAMERA,
        '09' : smscnf.COM_CALL,
        '10' : smscnf.COM_LOCK,
        '11' : smscnf.COM_UNLOCK,
        '12' : smscnf.COM_TRACKON,
        '13' : smscnf.COM_TRACKOFF,
        '14' : smscnf.COM_CUSTOM,
        '15' : smscnf.COM_SHELL,
        '16' : smscnf.COM_ALARM,
        '17' : smscnf.COM_RESTART,
        '18' : smscnf.COM_CUSTOMLOG,
        '19' : smscnf.COM_LOG,
        '20' : smscnf.COM_SILENTON,
        '21' : smscnf.COM_SILENTOFF,
        '22' : smscnf.COM_STOLEN,
    }

    # Supply default protokol "http:"
    if  CheckHostURL.find(":/") < 0:
        CheckHostURL = 'http://' + CheckHostURL

    def eventsOf(what):
        return "%d %s since %s GMT" % (abs(CheckHostStatusCount), what, GetTimeString(CheckHostStatusTimestamp))

    LOGGER.debug('(CHECKHOST) Opening remote host file (%s).' % CheckHostURL)
    try:
        UrlFile = urllib.urlopen('%s?LASTMCD=%s&CHECKTIME=%s' % (CheckHostURL, CheckHostLastCommand, smscnf.CHECKTIME))
        Command = UrlFile.read(2)
        UrlFile.close()
    except Exception, e:
        if CheckHostStatusCount >= 0:
            LOGGER.error('(CHECKHOST) After %s: Failed to read remote command: %s' % (eventsOf('OK'), e))
            CheckHostStatusCount = 0
            CheckHostStatusTimestamp = time.gmtime()
        CheckHostStatusCount -= 1
        if -CheckHostStatusCount % 25 == 0:
            RevealThat('checkhost', 'Got %s: %s' % eventsOf('errors'), e)
        return True

    if CheckHostStatusCount <= 0:
        LOGGER.info('(CHECKHOST) After %s: Successfully read remote command.' % eventsOf('errors'))
        CheckHostStatusCount = 0
        CheckHostStatusTimestamp = time.gmtime()
    CheckHostStatusCount += 1

    # No longer treat '17' and '-1' differently. Map regular COM_RESTART to
    # our special -1 so that we can handle it and not ProcessCommand():
    if Command == '17': Command = '-1'

    LOGGER.debug('(CHECKHOST) Got command "%s". Last command was "%s".' % (Command, CheckHostLastCommand))
    if CheckHostLastCommand == Command:
        return True
    CheckHostLastCommand = Command

    if Command in CommandList:
        Command = CommandList[Command]
        LOGGER.info( '(CHECKHOST) Activating %s by remote command [%s].' % (DaemonFile, Command) )
        ProcessCommand(Command, CommandOrigin = 'CHECKHOST', ExtraInfo = CheckHostURL)

    elif Command == '00':
        LOGGER.info( '(CHECKHOST) Leave %s idle by remote command [%s].' % (DaemonFile, Command) )

    elif Command == '-1':
        LOGGER.info( '(CHECKHOST) Will restart %s by remote command [%s].' % (DaemonFile, Command) )
        try:
            os.system( '"%s" -restart' % os.path.join(InstPath, smscon.NAME) )
        except Exception, e:
            LOGGER.error('(CHECKHOST) Failed to kill %s: [%s]' % (DaemonFile, e))
    
    else:
        LOGGER.error( '(CHECKHOST) Ignoring unknown remote command [%s].' % (Command) )

    return True # run it continiously

#############################################

def InstallSignalReceivers():
    """
    Enable or disable signal receivers and polling timers to detect
    various device sensors and control command sources.
    @return void
    """
    
    global IsKeyboardSliderCheckActive
    global IsBatteryStatusCheckActive
    global IsCheckHostActive
    global daemon

    daemon.openCommandChannel('r')

    if (not IsKeyboardSliderCheckActive) == (smscnf.KEYBOARDDETECT == 'yes'):
            IsKeyboardSliderCheckActive   =  smscnf.KEYBOARDDETECT == 'yes'
            set_signal_receiver = IF(IsKeyboardSliderCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnKeyboardSliderChange,
                                    path           = '/org/freedesktop/Hal/devices/platform_slide',
                                    dbus_interface = 'org.freedesktop.Hal.Device',
                                    signal_name    = 'Condition')
    LOGGER.info('Keyboard slider change detection is %s.' % IF(IsKeyboardSliderCheckActive, 'on', 'off'))

    if (not IsBatteryStatusCheckActive) == (smscnf.AUTOBATREPORT == 'yes'):
            IsBatteryStatusCheckActive   =  smscnf.AUTOBATREPORT == 'yes'
            set_signal_receiver = IF(IsBatteryStatusCheckActive, bus.add_signal_receiver, bus.remove_signal_receiver)
            set_signal_receiver(OnBatteryStatusChange, 
                                    path           = '/com/nokia/bme/signal',
                                    dbus_interface = 'com.nokia.bme.signal',
                                    signal_name    = 'battery_state_changed')
    LOGGER.info('Battery status change detection is %s.' % IF(IsBatteryStatusCheckActive, 'on', 'off'))

    # Set CHECKHOST polling

    def CheckHost(checktime):
        global IsCheckHostActive
        if smscnf.ENABLECHECKHOST == 'yes':
            IsCheckHostActive = True
            PerformCheckHost(smscnf.CHECKHOST)
            if smscnf.CHECKTIME == checktime:
                return True
            LOGGER.info('(CHECKHOST) Checkhost is on. Checking each %s minutes.' % float(smscnf.CHECKTIME))
            gobject.timeout_add(max(1000, int(float(smscnf.CHECKTIME) * 60000)), CheckHost, smscnf.CHECKTIME)
        else:
            IsCheckHostActive = False
            LOGGER.info('(CHECKHOST) Checkhost is off.')
        return False

    if not IsCheckHostActive:
        CheckHost(None)
    else:
        LOGGER.info('Checkhost change detection is %s.' % IF(smscnf.ENABLECHECKHOST == 'yes', 'on', 'off'))

def ApplyConfiguration():
    """
    Ensure settings of globally loaded smscon configuration are used in device dependend modules.
    @return void
    """
    global smscnf, GpsControl
    
    # Set GPS method & most suitable interval. Do not select intervals above 30 seconds:
    # In tracking mode we check the GPSINTERVAL ourself and in location mode the longer
    # intervals do not make very much sense anyway. Keep in mind that the actual interval
    # is also influenced by other applications that are using the GPS device! 

    gps_interval = int(smscnf.GPSINTERVAL)
    if   gps_interval <= 1:   pref_interval = location.INTERVAL_1S
    elif gps_interval <= 2:   pref_interval = location.INTERVAL_2S
    elif gps_interval <= 5:   pref_interval = location.INTERVAL_5S
    elif gps_interval <= 10:  pref_interval = location.INTERVAL_10S
    elif gps_interval <= 20:  pref_interval = location.INTERVAL_20S
    else:                     pref_interval = location.INTERVAL_30S
    GpsControl.set_properties(#preferred_method   = location.METHOD_GNSS|location.METHOD_AGNSS,
                              preferred_method   = location.METHOD_USER_SELECTED,
                              preferred_interval = pref_interval)
    LOGGER.info('GPS query interval is %d sec.' % gps_interval)

def LoadConfiguration():
    """"
    Load and establish user settings from smscon_config file
    @return bool Success indicator
    """
    global smscnf, IsStolenModeEnabled
    isReload = not not smscnf;
    LastKnownGoodConfigPath = os.path.join(InstPath, LastKnownGoodConfigFile)

    if isReload and (not IsStolenModeEnabled) == (not IsAuthorizedSIMcard()): # use 'not' to force booleans.
        IsStolenModeEnabled = not IsStolenModeEnabled       # stolen mode changed.
        InfoMessage = "Stolen mode %s." % IF(IsStolenModeEnabled, 'enabled', 'disabled')
        LOGGER.warning(InfoMessage)
        RevealThat('Stolen', InfoMessage)   # this uses current (old) config. Should be no problem.

    theConf = smscon.UserConfiguration()
    if  theConf.load():
        theConf.upgrade()
    else:
        LOGGER.critical('Failed to load "%s" file.' % theConf.Filename)
        if isReload:
            return False
        LOGGER.warning('Trying last known good config.')
        theConf = smscon.UserConfiguration(Filename = LastKnownGoodConfigPath)
        if not theConf.load():
            return False
        theConf.upgrade(Permanent = False)  # Only in RAM; Do not change last known good on disk.

    smscon.EnableDebugLogging(theConf)

    # For convinience code make config settings available as attributes:
    for key in theConf.getKnownConfigVars():
        if not key in theConf:
            LOGGER.critical('Failed to load "%s" file.' % theConf.Filename)
            return False
        setattr(theConf, key, theConf[key])

    LOGGER.info('Loaded "%s" file.' % theConf.Filename)
    theConf.backup(LastKnownGoodConfigPath, Silent = 'last known good config')

    smscnf = theConf
    if isReload:
        ApplyConfiguration()
        InstallSignalReceivers()
    return True

def OnLocalCommandAvailable():
    """
    Try to read commands from local command pipe and execute them.
    @return False
    """
    global daemon

    while True:
        Command = daemon.readCommand()
        if Command == '':
            break
        cliArgs = daemon.isCliCommand(Command)
        if cliArgs:
            LOGGER.info("----- CLI PROCESSING '%s' BEGINS -----" % ' '.join(cliArgs))
            DoCommandLine(cliArgs[:])
            LOGGER.info("----- CLI PROCESSING '%s' DONE -----" % ' '.join(cliArgs))
        else:
            LOGGER.warning('Unknown command "%s" found in local command channel. Ignored.' % Command)
    return False

def main():
    """
    Main function of smscon daemon. Does not return to caller.
    """
    global smscnf, bus, dev_obj, GpsControl
    global daemon

    daemon = smscon_daemon()

    LOGGER.critical('%s is starting.' % DaemonFile)
    
    signal.signal(smscon.smscon_daemon.TELL_TERMINATE,
                    lambda signum, stack_frame: ExitDaemon("on signal %s" % str(signum))  )
    signal.signal(smscon.smscon_daemon.TELL_RELOAD,
                    lambda signum, stack_frame: LoadConfiguration()                       )
    signal.signal(smscon.smscon_daemon.TELL_COMMAND,
                    lambda signum, stack_frame: gobject.idle_add(OnLocalCommandAvailable) )
    
    LoadConfiguration() or ExitDaemon('on config load error', ExitCode = 1)
    ValidateSIMCard()
    
    # Connect to various signal triggers:
    try:            
        DBusGMainLoop(set_as_default = True)
    
        bus = dbus.SystemBus()
    
        bus.add_signal_receiver(OnSmsReceived,
                                path           = '/com/nokia/phone/SMS',
                                dbus_interface = 'Phone.SMS',
                                signal_name    = 'IncomingSegment')
    
        # set battery charge measurment
        hal_obj = bus.get_object('org.freedesktop.Hal', '/org/freedesktop/Hal/Manager')
        hal = dbus.Interface(hal_obj, 'org.freedesktop.Hal.Manager')
    
        uids = hal.FindDeviceByCapability('battery')
        dev_obj = bus.get_object('org.freedesktop.Hal', uids[0])
        LogBatteryChargeInfo()
    
        # set global GPS device control
        GpsControl = location.GPSDControl.get_default()
        GpsDevice  = location.GPSDevice()

        # set GPS signal receivers (need them for DoCommandLine())
        GpsControl.connect('error-verbose', OnGPSerror, None )
        GpsDevice.connect('changed', OnGPSchanged, GpsControl )
        GpsControl.connect('gpsd-stopped', OnGPSstop, None )

        ApplyConfiguration()

        # Now that we behave like a system job, take command from environment: 
        cliArgs = [os.getenv('CLIMODE', ''), os.getenv('CLIFUNCTION', '')]
        if cliArgs == ['', ''] : cliArgs = []
        cliArgs += sys.argv[1:]
        
        CommandLineStatus = DoCommandLine(cliArgs)
        if not CommandLineStatus:
            ExitDaemon('bad command line', ExitCode = 1)

        InstallSignalReceivers()

        CheckBootLoader(WithNotification = CommandLineStatus != DoCommandLine.TESTMODE)

        gobject.MainLoop().run()
    
    except KeyboardInterrupt:
        ExitDaemon('on KeyboardInterrupt')
    
    except Exception, e:
        LOGGER.critical('<<< SMSCON FATAL ERROR:\n%s >>>' % e)
        # Disable resending of messages (would fail anyway when exiting): 
        smscnf.ENABLERESEND = 'no' 
        RevealThat('fatal error', ['smscon_daemon crashed.\n', '%s' % e])        
        ExitDaemon('on fatal error')
    
    except:
        ExitDaemon('on exception [%s]: %s' % sys.exc_info()[0:2])
        
    ExitDaemon('on mainloop exit')

#############################################
